#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

import json

from shinkensolutions.api.synchronizer.source import CollectorModule, SourceException, DisplayOriginItemFormat, SourceConfiguration, GenericDictMapperOriginToSource, SERVICE_MODE
from shinkensolutions.lib_modules.configuration_reader import read_string_in_configuration

try:
    import pyexcel as pe
except ImportError:
    pe = None

properties = {
    'daemons': ['synchronizer'],
    'type'   : "source_demo",
    'tabs'   : True,
}
app = None

dcif_tabular = DisplayOriginItemFormat(DisplayOriginItemFormat.TABULAR)
dcif_string = DisplayOriginItemFormat(DisplayOriginItemFormat.STRING)
# TODO BMDEV dcif_custom = DisplayOriginItemFormat(DisplayOriginItemFormat.CUSTOM, template=complementary_information_filter_tabular, controller=controller)


source_configuration = SourceConfiguration()
source_configuration.set_mapping_origin_to_source(SERVICE_MODE.ON, GenericDictMapperOriginToSource)
# source_configuration.set_mapping_origin_to_source(SERVICE_MODE.ON, GenericDictMapperOriginToSource, ['mapper_01', 'mapper_02'])
source_configuration.set_host_template_binding_rule(SERVICE_MODE.ON)
source_configuration.set_display_origin_item(SERVICE_MODE.ON, dcif_string)


# called by the plugin manager to get a module
def get_instance(plugin):
    return SourceDemo(plugin, source_configuration)


class SourceDemo(CollectorModule):
    excel_path = ''
    sheet_name_path = ''
    sheet_name = {}
    
    
    def load_configuration(self, configuration):
        self.excel_path = read_string_in_configuration(configuration, 'excel_path', '')
        self.sheet_name_path = read_string_in_configuration(configuration, 'sheet_name', '')
        
        with open(self.sheet_name_path) as f:
            raw_sheet_name = json.load(f)
        
        self.sheet_name = dict(((v, k) for k, v in raw_sheet_name.iteritems()))
    
    
    def import_source_items(self):
        if pe is None:
            raise SourceException(self.translator.translate('errors.missing_pyexcel'))
        
        origin_items = self._read_excel_data()
        source_items = self._mapping_origin_items_to_source_item(origin_items)
        items_containers = self._add_source_items_to_items_containers(source_items)
        
        return self.prepare_finale_source_result(items_containers)
    
    
    def _read_excel_data(self):
        origin_data = pe.get_book_dict(file_name=self.excel_path)
        problem_sheet = []
        origin_items = {}
        for sheet, data_in_sheet in origin_data.iteritems():
            origin_data_parsed = pe.get_records(array=data_in_sheet)
            try:
                item_type = self._item_type_from_sheet_name(sheet)
            except Exception as e:
                problem_sheet.append(str(e))
                continue
            if item_type in self.type_to_import:
                origin_items[item_type] = origin_data_parsed
        if problem_sheet:
            raise SourceException(self.translator.translate('errors.sheet_unknown_name', ','.join(problem_sheet)))
        return origin_items
    
    
    def _item_type_from_sheet_name(self, sheet):
        item_type = self.sheet_name.get(sheet, None)
        if not item_type:
            raise Exception(sheet)
        return item_type
    
    
    def _mapping_origin_items_to_source_item(self, origin_items):
        source_items = {}
        for item_type, items in origin_items.iteritems():
            source_items[item_type] = [self.mapper_origin_to_source.map(item_type, origin_item) for origin_item in items]
        return source_items
    
    
    def _add_source_items_to_items_containers(self, source_items):
        items_containers = self.build_items_containers()
        for item_type, items in source_items.iteritems():
            for i in items:
                items_containers.add_item(item_type, i)
        return items_containers
