#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.


import os
import sys
import shutil
import socket
from subprocess import Popen, PIPE
import json

from shinken.basemodule import BaseModule, SOURCE_STATE
from shinken.log import logger
from shinkensolutions import localsystem

properties = {
    'daemons': ['synchronizer'],
    'type'   : 'sync-vmware',
}


# CF: https://www.vmware.com/support/developer/converter-sdk/conv51_apireference/vim.vm.GuestOsDescriptor.GuestOsIdentifier.html
def get_os(_os):
    oses = {'rhel4Guest'             : 'linux',
            'debian5Guest'           : 'linux',
            'debian5_64Guest'        : 'linux',
            'debian6_64Guest'        : 'linux',
            'debian6Guest'           : 'linux',
            'fedoraGuest'            : 'linux',
            'freebsd64Guest'         : 'linux',
            'oracleLinux64Guest'     : 'linux',
            'oracleLinuxGuest'       : 'linux',
            'other24xLinux64Guest'   : 'linux',
            'other24xLinuxGuest'     : 'linux',
            'other26xLinux64Guest'   : 'linux',
            'other26xLinuxGuest'     : 'linux',
            'otherLinux64Guest'      : 'linux',
            'otherLinuxGuest'        : 'linux',
            'rhel4_64Guest'          : 'linux',
            'rhel5_64Guest'          : 'linux',
            'rhel5Guest'             : 'linux',
            'rhel6_64Guest'          : 'linux',
            'rhel6Guest'             : 'linux',
            'win2000ServGuest'       : 'windows',
            'windows8Server64Guest'  : 'windows',
            'winLonghorn64Guest'     : 'windows',
            'winNetDatacenter64Guest': 'windows',
            'winNetDatacenterGuest'  : 'windows',
            'winNetEnterprise64Guest': 'windows',
            'winNetEnterpriseGuest'  : 'windows',
            'winNetStandard64Guest'  : 'windows',
            'winNetStandardGuest'    : 'windows',
            'winNTGuest'             : 'windows',
            'winVista64Guest'        : 'windows',
            'winVistaGuest'          : 'windows',
            'winXPPro64Guest'        : 'windows',
            'winXPProGuest'          : 'windows',
            'windows7Server64Guest'  : 'windows',
            'windows7ServerGuest'    : 'windows',
            'windows8_64Guest'       : 'windows',
            'windows7_64Guest'       : 'windows',
            'windows7Guest'          : 'windows',
            'centos64Guest'          : 'linux',
            'centosGuest'            : 'linux',
            'winLonghornGuest'       : 'windows',
        
            }
    r = oses.get(_os, '')
    return r


# Split and clean the rules from a string to a list
def _split_rules(rules):
    return [r.strip() for r in rules.split('|')]


# Apply all rules on the objects names
def _apply_rules(name, rules):
    if 'nofqdn' in rules:
        name = name.split('.', 1)[0]
    if 'lower' in rules:
        name = name.lower()
    return name


# Get all vmware hosts from a VCenter and return the list
def get_vmware_hosts(check_esx_path, vcenter, user, password):
    list_host_cmd = [check_esx_path, '-D', '"%s"' % vcenter, '-u', '"%s"' % user, '-p', '"%s"' % password, '-l', 'runtime', '-s', 'listhost']
    
    try:
        parts = send_cmd_to_vmware(list_host_cmd)
    except Exception as e:
        return None, e.message
    
    hsts_raw = parts[1].split('|')[0]
    hsts_raw_lst = hsts_raw.split(',')
    
    hosts = []
    for hst_raw in hsts_raw_lst:
        hst_raw = hst_raw.strip()
        # look as server4.mydomain(UP)
        elts = hst_raw.split('(')
        hst = elts[0]
        hosts.append(hst)
    
    return (hosts, '')


# For a specific host, ask all VM on it to the VCenter
def get_vm_of_host(check_esx_path, vcenter, host, user, password):
    warnings = []
    logger.debug('[sync-vmware Import] Listing guests for host: %s' % host)
    list_vm_cmd = [check_esx_path, '-D', '"%s"' % vcenter, '-H', host, '-u', '"%s"' % user, '-p', '"%s"' % password, '-l', 'runtime', '-s', 'list']
    
    try:
        parts = send_cmd_to_vmware(list_vm_cmd)
    except Exception as e:
        logger.warning('[sync-vmware Import] exception in get_vm_of_host : %s' % e.message)
        return None, warnings
    
    # Maybe we got a 'CRITICAL - There are no VMs.' message, if so, we bypass this host
    if len(parts) < 2:
        return None, warnings
    
    vms_raw = parts[1].split('|')[0]
    vms_raw_lst = vms_raw.split(',')
    
    lst = []
    for vm_raw in vms_raw_lst:
        vm_raw = vm_raw.strip()
        # look as MYVM(UP)
        elts = vm_raw.split('(')
        if len(elts) != 4:
            msg = 'Cannot read guest info : [%s]. Did your guest have a \'(\' in this name ?.' % vm_raw
            logger.warning('[sync-vmware Import] %s' % msg)
            warnings.append(msg)
            continue
        vm = elts[0]
        _os = elts[2].split(')')[0].strip()
        _ip = elts[3].split(')')[0].strip()
        v = {'name': vm, 'os': _os, 'ip': _ip}
        lst.append(v)
    return lst, warnings


def send_cmd_to_vmware(cmd):
    _cmd = ' '.join(cmd)
    
    p = Popen(_cmd, stdout=PIPE, shell=True)
    output = p.communicate()
    if p.returncode not in [0]:  # do not catch 3 because now in no SDK we want a critical
        _s = output[0].splitlines()[0]
        if p.returncode == 3:
            _s = output[0]
            if os.path.exists('/var/lib/shinken/install.path'):
                f = open('/var/lib/shinken/install.path', 'r')
                pth = f.read()
                f.close()
                pth = pth.replace('\n', '')
                _s = 'Error: Missing VMWare Perl SDK. Please install it by launching %s/tools/install-vmware-sdk.sh as root.' % pth
        raise Exception(_s)
    
    return output[0].split(':', 1)


def create_simple_host(hname, tpls, ip=''):
    host = hname
    d = {'host_name': host,
         'address'  : host,
         'use'      : tpls,
         # '_VMS':','.join(res[host])}
         }
    keys = [host]
    fqdn = ''
    try:
        if not ip:
            ip = socket.gethostbyname(host)
        try:
            fqdn = socket.gethostbyaddr(ip)[0]
        except socket.error:
            pass
    except socket.error:
        pass
    if fqdn:
        d['address'] = fqdn
        keys.append(fqdn)
    if ip:
        keys.append(ip)
        if not fqdn:
            d['address'] = ip
    # make keys uniq
    keys = list(set(keys))
    d['_SYNC_KEYS'] = (','.join(keys)).lower()
    return d


# Create all tuples of the links for the hosts
def create_all_links(res, rules, esx_templates, vm_templates):
    r = []
    for host in res:
        d = create_simple_host(host, esx_templates)
        # d['_VMS'] = ','.join([v['name'] for v in res[host]])
        r.append(d)
        for v in res[host]:
            vm = v['name']
            _os = v['os']
            _ip = v['ip']
            tpls = vm_templates
            tpls_os = get_os(_os)
            if tpls_os:
                tpls = tpls_os + ',' + tpls
            # First we apply rules on the names
            # host_name = _apply_rules(host, rules)
            # vm_name = _apply_rules(vm, rules)
            # v = (('host', host_name), ('host', vm_name))
            d = create_simple_host(vm, tpls, ip=_ip)
            # d['_ESX'] = host
            # if _os:
            #    d['_VM_GUEST_OS'] = _os
            r.append(d)
    return r


def write_output(r, path):
    try:
        with open(path + '.tmp', 'wb') as f:
            buf = json.dumps(r)
            f.write(buf)
        shutil.move(path + '.tmp', path)
        logger.debug("[sync-vmware Import] File %s wrote" % path)
    except IOError, exp:
        sys.exit("Error writing the file %s: %s" % (path, exp))


def main(check_esx_path, vcenter, user, password, rules, esx_templates, vm_templates):
    warnings = []
    rules = _split_rules(rules)
    res = {}
    hosts, err = get_vmware_hosts(check_esx_path, vcenter, user, password)
    if hosts is None:
        return None, err, warnings
    
    for host in hosts:
        lst, warnings = get_vm_of_host(check_esx_path, vcenter, host, user, password)
        if lst:
            res[host] = lst
    
    hosts = create_all_links(res, rules, esx_templates, vm_templates)
    logger.debug("[sync-vmware Import] Created %d links" % len(hosts))
    logger.debug("[sync-vmware Import] Finished!")
    
    return hosts, '', warnings


# called by the plugin manager to get a module
def get_instance(plugin):
    logger.info("[sync-vmware Import] Get a sync-vmware import module for plugin %s" % plugin.get_name())
    
    check_esx_path = plugin.check_esx_path
    vcenter = plugin.vcenter
    user = plugin.user
    password = plugin.password
    rules = ''
    esx_templates = plugin.esx_templates
    vm_templates = plugin.vm_templates
    instance = VMWare_Import(plugin, check_esx_path, vcenter, user, password, rules, esx_templates, vm_templates)
    return instance


class VMWare_Import(BaseModule):
    def __init__(self, mod_conf, check_esx_path, vcenter, user, password, rules, esx_templates, vm_templates):
        BaseModule.__init__(self, mod_conf)
        self.check_esx_path = check_esx_path
        self.vcenter = vcenter
        self.user = user
        self.password = password
        self.rules = rules
        self.esx_templates = esx_templates
        self.vm_templates = vm_templates
        self.synchronizer_daemon = None
    
    
    # Called by Arbiter to say 'let's prepare yourself guy'
    def init(self):
        logger.info("[sync-vmware Import] Initialization of the sync-vmware import module")
    
    
    def load(self, synchronizer_daemon):
        self.synchronizer_daemon = synchronizer_daemon
    
    
    def get_objects(self):
        if self.vcenter == 'vcenter.fqdn.com':
            err = self.synchronizer_daemon._('import-vmware.output_nc_module')
            res = {
                'state'   : SOURCE_STATE.NOT_CONFIGURED,
                'output'  : err,
                'objects' : {},
                'errors'  : [],
                'warnings': []
            }
            return res
        
        if not localsystem.is_vmware_sdk_installed():
            _s = self.synchronizer_daemon._('import-vmware.warn_missing_vmware_perl_sdk')
            if os.path.exists('/var/lib/shinken/install.path'):
                f = open('/var/lib/shinken/install.path', 'r')
                pth = f.read()
                f.close()
                pth = pth.replace('\n', '')
                _s = self.synchronizer_daemon._('import-vmware.cr_missing_vmware_perl_sdk') % pth
            res = {
                'state'   : SOURCE_STATE.CRITICAL,
                'output'  : self.synchronizer_daemon._('import-vmware.output_critical'),
                'objects' : {},
                'errors'  : [],
                'warnings': [_s]
            }
            return res
        
        hosts, err, warnings = main(self.check_esx_path, self.vcenter, self.user, self.password, self.rules, self.esx_templates, self.vm_templates)
        
        # Maybe there was a problem
        if hosts is None:
            res = {
                'state'   : SOURCE_STATE.CRITICAL,
                'output'  : self.synchronizer_daemon._('import-vmware.output_critical'),
                'objects' : {},
                'errors'  : [err],
                'warnings': warnings
            }
        elif warnings:
            objects = {'host': hosts, 'contact': []}
            res = {
                'state'   : SOURCE_STATE.WARNING,
                'output'  : self.synchronizer_daemon._('import-vmware.output_warning'),
                'objects' : objects,
                'errors'  : [],
                'warnings': warnings
            }
        else:
            objects = {'host': hosts, 'contact': []}
            res = {
                'state'   : SOURCE_STATE.OK,
                'output'  : self.synchronizer_daemon._('import-vmware.output_ok_load_successful'),
                'objects' : objects,
                'errors'  : [],
                'warnings': warnings
            }
        
        return res
