#!/usr/bin/python
# -*- coding: utf-8 -*-

# Fix for python 2.7 with https://github.com/vmware/pyvmomi/issues/627
# (Yes this API don't handle timeout)

import re
import ssl
import sys

from pyVim.connect import __FindSupportedVersion, _rx, SetSi, GetLocalTicket, localSslFixup
from pyVmomi import vim, vmodl, SoapStubAdapter
from pyVmomi.SoapAdapter import CONNECTION_POOL_IDLE_TIMEOUT_SEC, SOAP_ADAPTER_ARGS
from pyVmomi.VmomiSupport import GetServiceVersions, versionMap
from six import reraise

#SOAP_ADAPTER_ARGS.append('timeout')

class SoapStubAdapterWithSocketTimeout(SoapStubAdapter):
    def __init__(
            self,
            host='localhost',
            port=443,
            ns=None,
            path='/sdk',
            url=None,
            sock=None,
            poolSize=5,
            certFile=None,
            certKeyFile=None,
            httpProxyHost=None,
            httpProxyPort=80,
            sslProxyPath=None,
            thumbprint=None,
            cacertsFile=None,
            version=None,
            acceptCompressedResponses=True,
            connectionPoolTimeout=CONNECTION_POOL_IDLE_TIMEOUT_SEC,
            samlToken=None,
            sslContext=None,
            requestContext=None,
            socketTimeout=None
    ):
        SoapStubAdapter.__init__(
            self,
            host,
            port,
            ns,
            path,
            url,
            sock,
            poolSize,
            certFile,
            certKeyFile,
            httpProxyHost,
            httpProxyPort,
            sslProxyPath,
            thumbprint,
            cacertsFile,
            version,
            acceptCompressedResponses,
            connectionPoolTimeout,
            samlToken,
            sslContext,
            requestContext
        )
        if socketTimeout:
            self.schemeArgs['timeout'] = socketTimeout


def __RetrieveContent(host, port, adapter, version, path, keyFile, certFile, thumbprint, sslContext, connectionPoolTimeout=CONNECTION_POOL_IDLE_TIMEOUT_SEC, socketTimeout=None):
    # XXX remove the adapter and service arguments once dependent code is fixed
    if adapter != "SOAP":
        raise ValueError(adapter)
    
    # Create the SOAP stub adapter
    stub = SoapStubAdapterWithSocketTimeout(host, port, version=version, path=path, certKeyFile=keyFile, certFile=certFile, thumbprint=thumbprint, sslContext=sslContext, connectionPoolTimeout=connectionPoolTimeout, socketTimeout=socketTimeout)
    
    # Get Service instance
    si = vim.ServiceInstance("ServiceInstance", stub)
    content = None
    try:
        content = si.RetrieveContent()
    except vmodl.MethodFault:
        raise
    except Exception as e:
        # NOTE (hartsock): preserve the traceback for diagnostics
        # pulling and preserving the traceback makes diagnosing connection
        # failures easier since the fault will also include where inside the
        # library the fault occurred. Without the traceback we have no idea
        # why the connection failed beyond the message string.
        (type, value, traceback) = sys.exc_info()
        if traceback:
            fault = vim.fault.HostConnectFault(msg=str(e))
            reraise(vim.fault.HostConnectFault, fault, traceback)
        else:
            raise vim.fault.HostConnectFault(msg=str(e))
    
    return content, si, stub


def __Login(host, port, user, pwd, service, adapter, version, path, keyFile, certFile, thumbprint, sslContext, connectionPoolTimeout=CONNECTION_POOL_IDLE_TIMEOUT_SEC, socketTimeout=None):
    content, si, stub = __RetrieveContent(host, port, adapter, version, path, keyFile, certFile, thumbprint, sslContext, connectionPoolTimeout, socketTimeout)
    
    # Get a ticket if we're connecting to localhost and password is not specified
    if host == 'localhost' and not pwd:
        try:
            (user, pwd) = GetLocalTicket(si, user)
        except:
            pass  # This is not supported against vCenter, and connecting
            # with an empty password is fine in debug builds
    
    # Login
    try:
        x = content.sessionManager.Login(user, pwd, None)
    except vim.fault.InvalidLogin:
        raise
    except Exception as e:
        raise
    return si, stub


def __LoginBySSPI(host, port, service, adapter, version, path, keyFile, certFile, thumbprint, sslContext, b64token, connectionPoolTimeout=CONNECTION_POOL_IDLE_TIMEOUT_SEC, socketTimeout=None):
    content, si, stub = __RetrieveContent(host, port, adapter, version, path, keyFile, certFile, thumbprint, sslContext, connectionPoolTimeout, socketTimeout)
    
    if b64token is None:
        raise Exception('Token is not defined for sspi login')
    
    # Login
    try:
        x = content.sessionManager.LoginBySSPI(b64token)
    except vim.fault.InvalidLogin:
        raise
    except Exception as e:
        raise
    return si, stub


def Connect(
        host='localhost',
        port=443,
        user='root',
        pwd='',
        service="hostd",
        adapter="SOAP",
        namespace=None,
        path="/sdk",
        connectionPoolTimeout=CONNECTION_POOL_IDLE_TIMEOUT_SEC,
        version=None,
        keyFile=None,
        certFile=None,
        thumbprint=None,
        sslContext=None,
        b64token=None,
        mechanism='userpass',
        socketTimeout=None
):
    try:
        info = re.match(_rx, host)
        if info is not None:
            host = info.group(1)
            if host[0] == '[':
                host = info.group(1)[1:-1]
            if info.group(2) is not None:
                port = int(info.group(2)[1:])
    except ValueError as ve:
        pass
    
    sslContext = localSslFixup(host, sslContext)
    
    if namespace:
        assert (version is None)
        version = versionMap[namespace]
    elif not version:
        version = "vim.version.version6"
    
    si, stub = None, None
    if mechanism == 'userpass':
        si, stub = __Login(host, port, user, pwd, service, adapter, version, path, keyFile, certFile, thumbprint, sslContext, connectionPoolTimeout, socketTimeout)
    elif mechanism == 'sspi':
        si, stub = __LoginBySSPI(host, port, service, adapter, version, path, keyFile, certFile, thumbprint, sslContext, b64token, connectionPoolTimeout, socketTimeout)
    else:
        raise Exception('''The provided connection mechanism is not available, the supported mechanisms are userpass or sspi''')
    
    SetSi(si)
    
    return si


def SmartConnect(
        protocol='https',
        host='localhost',
        port=443,
        user='root',
        pwd='',
        service="hostd",
        path="/sdk",
        connectionPoolTimeout=CONNECTION_POOL_IDLE_TIMEOUT_SEC,
        preferredApiVersions=None,
        keyFile=None,
        certFile=None,
        thumbprint=None,
        sslContext=None,
        b64token=None,
        mechanism='userpass',
        socketTimeout=None
):
    if preferredApiVersions is None:
        preferredApiVersions = GetServiceVersions('vim25')
    
    sslContext = localSslFixup(host, sslContext)
    
    supportedVersion = __FindSupportedVersion(protocol, host, port, path, preferredApiVersions, sslContext)
    if supportedVersion is None:
        raise Exception("%s:%s is not a VIM server" % (host, port))
    
    portNumber = protocol == "http" and -int(port) or int(port)
    
    return Connect(
        host=host,
        port=portNumber,
        user=user,
        pwd=pwd,
        service=service,
        adapter='SOAP',
        version=supportedVersion,
        path=path,
        connectionPoolTimeout=connectionPoolTimeout,
        keyFile=keyFile,
        certFile=certFile,
        thumbprint=thumbprint,
        sslContext=sslContext,
        b64token=b64token,
        mechanism=mechanism,
        socketTimeout=socketTimeout
    )


class SmartConnectNoSSL(object):
    def __init__(self,
                 protocol='https',
                 host='localhost',
                 port=443,
                 user='root',
                 pwd='',
                 service="hostd",
                 path="/sdk",
                 connectionPoolTimeout=CONNECTION_POOL_IDLE_TIMEOUT_SEC,
                 preferredApiVersions=None,
                 keyFile=None,
                 certFile=None,
                 thumbprint=None,
                 b64token=None,
                 mechanism='userpass',
                 socketTimeout=None
                 ):
        if hasattr(ssl, '_create_unverified_context'):
            sslContext = ssl._create_unverified_context()
        else:
            sslContext = None
            
        self.session = SmartConnect(
            protocol=protocol,
            host=host,
            port=port,
            user=user,
            pwd=pwd,
            service=service,
            path=path,
            connectionPoolTimeout=connectionPoolTimeout,
            preferredApiVersions=preferredApiVersions,
            keyFile=keyFile,
            certFile=certFile,
            thumbprint=thumbprint,
            sslContext=sslContext,
            b64token=b64token,
            mechanism=mechanism,
            socketTimeout=socketTimeout
        )
        
    def smartConnectTestConnection(self, logger, translate):
        if len(self.session._stub.pool) < 1:
            msg = translate('vmware_error.connection_timeout')
            logger.error(msg)
            raise Exception(msg)
