/**
 * @author Eberhard Graether / http://egraether.com/
 */

THREE.TrackballControls = function ( object, domElement ) {

	var _this = this;
	var STATE = { NONE: -1, ROTATE: 0, ZOOM: 1, PAN: 2, TOUCH_ROTATE: 3, TOUCH_ZOOM: 4, TOUCH_PAN: 5 };

	this.object = object;
	this.domElement = ( domElement !== undefined ) ? domElement : document;

	// API

	this.enabled = true;

	this.screen = { left: 0, top: 0, width: 0, height: 0 };

	this.rotateSpeed = 1.0;
	this.zoomSpeed = 1.2;
	this.panSpeed = 0.3;

	this.noRotate = false;
	this.noZoom = false;
	this.noPan = false;
	this.noRoll = false;

	this.staticMoving = false;
	this.dynamicDampingFactor = 0.2;

	this.minDistance = 0;
	this.maxDistance = Infinity;

	this.keys = [ 65 /*A*/, 83 /*S*/, 68 /*D*/ ];

        // 360, to know when we stop moved
        this.last_move = 0;

	// internals

	this.target = new THREE.Vector3();

	var lastPosition = new THREE.Vector3();

	var _state = STATE.NONE,
	_prevState = STATE.NONE,

	_eye = new THREE.Vector3(),

	_rotateStart = new THREE.Vector3(),
	_rotateEnd = new THREE.Vector3(),

	_zoomStart = new THREE.Vector2(),
	_zoomEnd = new THREE.Vector2(),

	_touchZoomDistanceStart = 0,
	_touchZoomDistanceEnd = 0,

	_panStart = new THREE.Vector2(),
	_panEnd = new THREE.Vector2();

	// for reset

	this.target0 = this.target.clone();
	this.position0 = this.object.position.clone();
	this.up0 = this.object.up.clone();

	// events

	var changeEvent = { type: 'change' };
	var startEvent = { type: 'start'};
	var endEvent = { type: 'end'};


	// methods

	this.handleResize = function () {

		if ( this.domElement === document ) {

			this.screen.left = 0;
			this.screen.top = 0;
			this.screen.width = window.innerWidth;
			this.screen.height = window.innerHeight;

		} else {

			this.screen = this.domElement.getBoundingClientRect();

		}

	};

	this.handleEvent = function ( event ) {

		if ( typeof this[ event.type ] == 'function' ) {

			this[ event.type ]( event );

		}

	};

	this.getMouseOnScreen = function ( clientX, clientY ) {

		return new THREE.Vector2(
			( clientX - _this.screen.left ) / _this.screen.width,
			( clientY - _this.screen.top ) / _this.screen.height
		);

	};

	this.getMouseProjectionOnBall = function ( clientX, clientY ) {

		var mouseOnBall = new THREE.Vector3(
			( clientX - _this.screen.width * 0.5 - _this.screen.left ) / (_this.screen.width*.5),
			( _this.screen.height * 0.5 + _this.screen.top - clientY ) / (_this.screen.height*.5),
			0.0
		);

		var length = mouseOnBall.length();

		if ( _this.noRoll ) {
		    
			if ( length < Math.SQRT1_2 ) {

				mouseOnBall.z = Math.sqrt( 1.0 - length*length );

			} else {

				mouseOnBall.z = .5 / length;
				
			}

		} else if ( length > 1.0 ) {

			mouseOnBall.normalize();

		} else {

			mouseOnBall.z = Math.sqrt( 1.0 - length * length );

		}

		_eye.copy( _this.object.position ).sub( _this.target );

		var projection = _this.object.up.clone().setLength( mouseOnBall.y );
		projection.add( _this.object.up.clone().cross( _eye ).setLength( mouseOnBall.x ) );
		projection.add( _eye.setLength( mouseOnBall.z ) );
		
		return projection;

	};

	this.rotateCamera = function () {

		var angle = Math.acos( _rotateStart.dot( _rotateEnd ) / _rotateStart.length() / _rotateEnd.length() );

		if ( angle ) {

			var axis = ( new THREE.Vector3() ).crossVectors( _rotateStart, _rotateEnd ).normalize(),
				quaternion = new THREE.Quaternion();

			angle *= _this.rotateSpeed;

			quaternion.setFromAxisAngle( axis, -angle );

			_eye.applyQuaternion( quaternion );
			_this.object.up.applyQuaternion( quaternion );

			_rotateEnd.applyQuaternion( quaternion );

			if ( _this.staticMoving ) {

				_rotateStart.copy( _rotateEnd );

			} else {

				quaternion.setFromAxisAngle( axis, angle * ( _this.dynamicDampingFactor - 1.0 ) );
				_rotateStart.applyQuaternion( quaternion );

			}

		}

	};

	this.zoomCamera = function () {

		if ( _state === STATE.TOUCH_ZOOM ) {

			var factor = _touchZoomDistanceStart / _touchZoomDistanceEnd;
			_touchZoomDistanceStart = _touchZoomDistanceEnd;
			_eye.multiplyScalar( factor );

		} else {

			var factor = 1.0 + ( _zoomEnd.y - _zoomStart.y ) * _this.zoomSpeed;

			if ( factor !== 1.0 && factor > 0.0 ) {

				_eye.multiplyScalar( factor );

				if ( _this.staticMoving ) {

					_zoomStart.copy( _zoomEnd );

				} else {

					_zoomStart.y += ( _zoomEnd.y - _zoomStart.y ) * this.dynamicDampingFactor;

				}

			}

		}

	};

	this.panCamera = function () {

		var mouseChange = _panEnd.clone().sub( _panStart );

		if ( mouseChange.lengthSq() ) {

			mouseChange.multiplyScalar( _eye.length() * _this.panSpeed );

			var pan = _eye.clone().cross( _this.object.up ).setLength( mouseChange.x );
			pan.add( _this.object.up.clone().setLength( mouseChange.y ) );

			_this.object.position.add( pan );
			_this.target.add( pan );

			if ( _this.staticMoving ) {

				_panStart = _panEnd;

			} else {

				_panStart.add( mouseChange.subVectors( _panEnd, _panStart ).multiplyScalar( _this.dynamicDampingFactor ) );

			}

		}

	};

	this.checkDistances = function () {

		if ( !_this.noZoom || !_this.noPan ) {

			if ( _eye.lengthSq() > _this.maxDistance * _this.maxDistance ) {

				_this.object.position.addVectors( _this.target, _eye.setLength( _this.maxDistance ) );

			}

			if ( _eye.lengthSq() < _this.minDistance * _this.minDistance ) {

				_this.object.position.addVectors( _this.target, _eye.setLength( _this.minDistance ) );

			}

		}

	};

	this.update = function () {

		_eye.subVectors( _this.object.position, _this.target );

		if ( !_this.noRotate ) {

			_this.rotateCamera();

		}

		if ( !_this.noZoom ) {

			_this.zoomCamera();

		}

		if ( !_this.noPan ) {

			_this.panCamera();

		}

		_this.object.position.addVectors( _this.target, _eye );

		_this.checkDistances();

		_this.object.lookAt( _this.target );

		if ( lastPosition.distanceToSquared( _this.object.position ) > 0 ) {

			_this.dispatchEvent( changeEvent );

			lastPosition.copy( _this.object.position );

		}

	};

	this.reset = function () {

		_state = STATE.NONE;
		_prevState = STATE.NONE;
	        _this.last_move = Math.round((new Date()).getTime() / 1000);
		_this.target.copy( _this.target0 );
		_this.object.position.copy( _this.position0 );
		_this.object.up.copy( _this.up0 );

		_eye.subVectors( _this.object.position, _this.target );

		_this.object.lookAt( _this.target );

		_this.dispatchEvent( changeEvent );

		lastPosition.copy( _this.object.position );

	};

	// listeners

	function keydown( event ) {

		if ( _this.enabled === false ) return;

		window.removeEventListener( 'keydown', keydown );

		_prevState = _state;

		if ( _state !== STATE.NONE ) {

			return;

		} else if ( event.keyCode === _this.keys[ STATE.ROTATE ] && !_this.noRotate ) {

			_state = STATE.ROTATE;

		} else if ( event.keyCode === _this.keys[ STATE.ZOOM ] && !_this.noZoom ) {

			_state = STATE.ZOOM;

		} else if ( event.keyCode === _this.keys[ STATE.PAN ] && !_this.noPan ) {

			_state = STATE.PAN;

		}

	}

	function keyup( event ) {

		if ( _this.enabled === false ) return;

		_state = _prevState;

		window.addEventListener( 'keydown', keydown, false );

	}

	function mousedown( event ) {

		if ( _this.enabled === false ) return;

		event.preventDefault();
		event.stopPropagation();

		if ( _state === STATE.NONE ) {

			_state = event.button;

		}

		if ( _state === STATE.ROTATE && !_this.noRotate ) {

			_rotateStart = _this.getMouseProjectionOnBall( event.clientX, event.clientY );
			_rotateEnd.copy(_rotateStart)

		} else if ( _state === STATE.ZOOM && !_this.noZoom ) {

			_zoomStart = _this.getMouseOnScreen( event.clientX, event.clientY );
			_zoomEnd.copy(_zoomStart);

		} else if ( _state === STATE.PAN && !_this.noPan ) {

			_panStart = _this.getMouseOnScreen( event.clientX, event.clientY );
			_panEnd.copy(_panStart)

		}

		document.addEventListener( 'mousemove', mousemove, false );
		document.addEventListener( 'mouseup', mouseup, false );
		_this.dispatchEvent( startEvent );


	}

	function mousemove( event ) {

		if ( _this.enabled === false ) return;

		event.preventDefault();
		event.stopPropagation();

		if ( _state === STATE.ROTATE && !_this.noRotate ) {

			_rotateEnd = _this.getMouseProjectionOnBall( event.clientX, event.clientY );

		} else if ( _state === STATE.ZOOM && !_this.noZoom ) {

			_zoomEnd = _this.getMouseOnScreen( event.clientX, event.clientY );

		} else if ( _state === STATE.PAN && !_this.noPan ) {

			_panEnd = _this.getMouseOnScreen( event.clientX, event.clientY );

		}

	}

	function mouseup( event ) {

		if ( _this.enabled === false ) return;

		event.preventDefault();
		event.stopPropagation();

		_state = STATE.NONE;
	        _this.last_move = Math.round((new Date()).getTime() / 1000);

		document.removeEventListener( 'mousemove', mousemove );
		document.removeEventListener( 'mouseup', mouseup );
		_this.dispatchEvent( endEvent );

	}

	function mousewheel( event ) {

		if ( _this.enabled === false ) return;

		event.preventDefault();
		event.stopPropagation();

		var delta = 0;

		if ( event.wheelDelta ) { // WebKit / Opera / Explorer 9

			delta = event.wheelDelta / 40;

		} else if ( event.detail ) { // Firefox

			delta = - event.detail / 3;

		}

		_zoomStart.y += delta * 0.01;
		_this.dispatchEvent( startEvent );
		_this.dispatchEvent( endEvent );

	}

	function touchstart( event ) {

		if ( _this.enabled === false ) return;

		switch ( event.touches.length ) {

			case 1:
				_state = STATE.TOUCH_ROTATE;
				_rotateStart = _rotateEnd = _this.getMouseProjectionOnBall( event.touches[ 0 ].pageX, event.touches[ 0 ].pageY );
				break;

			case 2:
				_state = STATE.TOUCH_ZOOM;
				var dx = event.touches[ 0 ].pageX - event.touches[ 1 ].pageX;
				var dy = event.touches[ 0 ].pageY - event.touches[ 1 ].pageY;
				_touchZoomDistanceEnd = _touchZoomDistanceStart = Math.sqrt( dx * dx + dy * dy );
				break;

			case 3:
				_state = STATE.TOUCH_PAN;
				_panStart = _panEnd = _this.getMouseOnScreen( event.touches[ 0 ].pageX, event.touches[ 0 ].pageY );
				break;

			default:
				_state = STATE.NONE;
       	                        _this.last_move = Math.round((new Date()).getTime() / 1000);

		}
		_this.dispatchEvent( startEvent );


	}

	function touchmove( event ) {

		if ( _this.enabled === false ) return;

		event.preventDefault();
		event.stopPropagation();

		switch ( event.touches.length ) {

			case 1:
				_rotateEnd = _this.getMouseProjectionOnBall( event.touches[ 0 ].pageX, event.touches[ 0 ].pageY );
				break;

			case 2:
				var dx = event.touches[ 0 ].pageX - event.touches[ 1 ].pageX;
				var dy = event.touches[ 0 ].pageY - event.touches[ 1 ].pageY;
				_touchZoomDistanceEnd = Math.sqrt( dx * dx + dy * dy )
				break;

			case 3:
				_panEnd = _this.getMouseOnScreen( event.touches[ 0 ].pageX, event.touches[ 0 ].pageY );
				break;

			default:
				_state = STATE.NONE;
                 	        _this.last_move = Math.round((new Date()).getTime() / 1000);

		}

	}

	function touchend( event ) {

		if ( _this.enabled === false ) return;

		switch ( event.touches.length ) {

			case 1:
				_rotateStart = _rotateEnd = _this.getMouseProjectionOnBall( event.touches[ 0 ].pageX, event.touches[ 0 ].pageY );
				break;

			case 2:
				_touchZoomDistanceStart = _touchZoomDistanceEnd = 0;
				break;

			case 3:
				_panStart = _panEnd = _this.getMouseOnScreen( event.touches[ 0 ].pageX, event.touches[ 0 ].pageY );
				break;

		}

		_state = STATE.NONE;
	        _this.last_move = Math.round((new Date()).getTime() / 1000);
		_this.dispatchEvent( endEvent );

	}

	this.domElement.addEventListener( 'contextmenu', function ( event ) { event.preventDefault(); }, false );

	this.domElement.addEventListener( 'mousedown', mousedown, false );

	this.domElement.addEventListener( 'mousewheel', mousewheel, false );
	this.domElement.addEventListener( 'DOMMouseScroll', mousewheel, false ); // firefox

	this.domElement.addEventListener( 'touchstart', touchstart, false );
	this.domElement.addEventListener( 'touchend', touchend, false );
	this.domElement.addEventListener( 'touchmove', touchmove, false );

	window.addEventListener( 'keydown', keydown, false );
	window.addEventListener( 'keyup', keyup, false );

	this.handleResize();

    this.getState = function (){
	return _state;
    }

    
    this.getLeapProjectionOnBall = function ( normX, normY, normZ ) {
	var mouseOnBall = new THREE.Vector3(normX, normY , normZ);
	
	var length = mouseOnBall.length();
	
	if ( length > 1.0 ) {
	    mouseOnBall.normalize();
	} else {
	    //mouseOnBall.z = Math.sqrt( 1.0 - length * length );
	}
	

	_eye.copy( _this.object.position ).sub( _this.target );
	
	var projection = _this.object.up.clone().setLength( mouseOnBall.y );
	projection.add( _this.object.up.clone().cross( _eye ).setLength( mouseOnBall.x ) );
	projection.add( _eye.setLength( mouseOnBall.z ) );
	
	return projection;

    };

	



    var state_leap  = null;
    var startFrame_leap = null;
    var interactionBox = null;
    var cent_x = cent_y = cent_z = 0;
    var size_x = size_y = size_z = 0;
    _this.nb_pan = 0;
    _this.nb_null = 0;
    _this.nb_rotate = 0;
    _this.nb_zoom = 0;
    _this.history_size = 120;

    var tool_chooser = [];

    function  choose_state(){
        //console.log(tool_chooser.length);
        if(tool_chooser.length < _this.history_size){
            return null;
        }
        var nb_pan = 0;
        var nb_zoom = 0;
        var nb_rotate = 0;
        var nb_null = 0;
        for(var i=0; i<tool_chooser.length; i++){
            var v = tool_chooser[i];
            if(v == 'pan'){nb_pan++;}
            else if(v == 'zoom'){nb_zoom++;}
            else if(v == 'rotate'){nb_rotate++;}
            else nb_null++;
        }
	_this.nb_null = nb_null;
	_this.nb_pan  = nb_pan;
	_this.nb_rotate = nb_rotate;
	_this.nb_zoom = nb_zoom;
        //console.log("RESULTS"+nb_pan+" "+ nb_zoom+" "+ nb_rotate +" "+ nb_null);
        
        // Trim the chooser array
        new_tool_chooser = [];
        for(var i=1; i<_this.history_size; i++){
            new_tool_chooser.push(tool_chooser[i]);
        }
        tool_chooser = new_tool_chooser;
        
        if(_this.nb_pan > _this.history_size*0.75){
            return 'pan';
        }
        if(_this.nb_zoom > _this.history_size*0.75){
            return 'zoom';
        }
        if(_this.nb_rotate > _this.history_size*0.75){
            return 'rotate';
        }
        return null;
    }

    if ( typeof Leap != 'undefined' ) {
        Leap.loop ( function ( frame ) {

            if ( frame.valid ) {

                // But we got a valid here :)
                if ( got_controller == false ) {
                    $ ( '#leap-gauge' ).show ();
                    $ ( '#leap_detection' ).html ( 'connected' );
                }

                if ( interactionBox == null && frame.interactionBox.valid ) {
                    //only need to get one interaction box, since it appears to stay the same.
                    interactionBox = new Leap.InteractionBox ( frame.interactionBox );
                    console.log ( interactionBox );
                }

                var nb_hands     = frame.hands.length;
                var nb_fingers   = frame.pointables.length;
                var good_fingers = [];
                var j            = 0;
                for ( var i = 0; i < nb_fingers; i++ ) {
                    var finger = frame.pointables[ i ];
                    //console.log(finger);
                    // Maybe it's the pouce on a close hand, but still see, if so delete it
                    if ( finger.length > 30 ) {
                        good_fingers[ j ] = finger;
                        j += 1;
                    } else {
                        console.log ( 'DROPING FINGER' );
                    }
                }
                if ( nb_hands > 0 ) {
                    //console.log(frame.hands);
                }
                // Save our good fingers
                frame.pointables = good_fingers;
                nb_fingers       = frame.pointables.length;

                //console.log("HANDS:"+nb_hands+ "  finger:"+nb_fingers);

                var action_proposed = false;
                var prev_state      = state_leap;
                var s               = choose_state ();
                state_leap          = s;
                //console.log('STATE '+s);

                //console.log('Choosen state'+s);

                var entering_state = (prev_state != s);

                if ( entering_state ) {
                    console.log ( 'Switching from state ' + prev_state + ' to the new state ' + s );
                    $ ( '#action' ).html ( 'SET?' + s );
                    if ( !s ) {
                        _state          = STATE.NONE;
                        _this.last_move = Math.round ( (new Date ()).getTime () / 1000 );
                    }

                }

                /*
                 // We are stopping pan as soon as we open a finger, or remove the hand or add another one
                 if (state_leap == 'pan' && (nb_hands !=1 || nb_fingers >1)){
                 console.log('STOP PAN');
                 state_leap = null;
                 _state = STATE.NONE;
                 }

                 // For the rotate, we are stopping when we remove a finger
                 if (state_leap == 'rotate' && (nb_hands !=2 || (nb_hands == 2 && nb_fingers == 2))){
                 console.log('STOP ROTATE');
                 state_leap = null;
                 _state = STATE.NONE;
                 }

                 // For the rotate, we are stopping when we remove a finger
                 if (state_leap == 'zoom' && (nb_hands !=1 || nb_fingers !=2)){
                 console.log('STOP ZOOM');
                 state_leap = null;
                 _state = STATE.NONE;
                 }
                 */

                /// PAN : one closed hand
                if ( nb_hands == 1 && nb_fingers <= 1 ) {
                    tool_chooser.push ( 'pan' );
                    action_proposed = true;
                }

                if ( state_leap == 'pan' && nb_hands == 1 && nb_fingers <= 1 ) {
                    var hand = frame.hands[ 0 ];
                    if ( entering_state ) {
                        startFrame_leap = frame;
                        var norm        = interactionBox.normalizePoint ( hand.palmPosition, true );
                        var n_x         = norm[ 0 ];
                        var n_y         = -norm[ 1 ]; // Beware, the leap is moving on the top, the mouse to the down
                        var n_z = norm[ 2 ];

                        state_leap = 'pan';
                        // Now let the Trackball know it
                        _panStart = _panEnd = new THREE.Vector2 ( n_x, n_y );
                        _zoomStart = _zoomEnd = new THREE.Vector2 ( 0, n_z );
                        _state = STATE.PAN;

                    } else {
                        var norm = interactionBox.normalizePoint ( hand.palmPosition, true );
                        var n_x  = norm[ 0 ];
                        var n_y  = -norm[ 1 ];
                        var n_z  = norm[ 2 ];
                        console.log ( 'NZ' + n_z );
                        // Let update the Trackball
                        _panEnd  = new THREE.Vector2 ( n_x, n_y );
                        _zoomEnd = new THREE.Vector2 ( 0, n_z );
                        console.log ( "HANDS:" + nb_hands + "  finger:" + nb_fingers );
                    }
                }

                /// ROTATE: Two hands with fingers
                if ( nb_hands >= 2 && nb_fingers >= 2 ) {
                    tool_chooser.push ( 'rotate' );
                    action_proposed = true;
                }

                if ( state_leap == 'rotate' && nb_hands >= 2 ) {

                    if ( entering_state ) {
                        var hand1       = frame.hands[ 0 ];
                        var hand2       = frame.hands[ 1 ];
                        startFrame_leap = frame;
                        var norm1       = interactionBox.normalizePoint ( hand1.palmPosition, true );
                        var n1_x        = norm1[ 0 ];
                        var n1_y        = norm1[ 1 ]; // Beware, the leap is moving on the top, the mouse to the down
                        var n1_z = norm1[ 2 ]; // Beware, the leap is moving on the top, the mouse to the down

                        var norm2 = interactionBox.normalizePoint ( hand2.palmPosition, true );
                        var n2_x  = norm2[ 0 ];
                        var n2_y  = norm2[ 1 ]; // Beware, the leap is moving on the top, the mouse to the down
                        var n2_z = norm2[ 2 ]; // Beware, the leap is moving on the top, the mouse to the down

                        var n_x = Math.abs ( n1_x - n2_x ) / 2;
                        var n_y = Math.abs ( n1_y - n2_y ) / 2;
                        var n_z = Math.abs ( n1_z - n2_z ) / 2;

                        state_leap = 'rotate';
                        /*
                         console.log('N1X '+n1_x+" N1Y "+n1_y);
                         console.log('NX '+n_x+" NY "+n_y);
                         */
                        // Now let the Trackball know it
                        _rotateStart = _rotateEnd = _this.getLeapProjectionOnBall ( n_x, n_y, n_z );
                        _state = STATE.ROTATE;
                        console.log ( 'STARTING ROTATE' );

                    } else {
                        var hand1 = frame.hands[ 0 ];
                        var hand2 = frame.hands[ 1 ];

                        var norm1 = interactionBox.normalizePoint ( hand1.palmPosition, true );
                        var n1_x  = norm1[ 0 ];
                        var n1_y  = norm1[ 1 ]; // Beware, the leap is moving on the top, the mouse to the down
                        var n1_z = norm1[ 2 ]; // Beware, the leap is moving on the top, the mouse to the down

                        var norm2 = interactionBox.normalizePoint ( hand2.palmPosition, true );
                        var n2_x  = norm2[ 0 ];
                        var n2_y  = norm2[ 1 ]; // Beware, the leap is moving on the top, the mouse to the down
                        var n2_z = norm2[ 2 ]; // Beware, the leap is moving on the top, the mouse to the down

                        var n_x = (n1_x - n2_x) / 2;
                        var n_y = (n1_y - n2_y) / 2;
                        var n_z = (n1_z - n2_z) / 2;

                        state_leap = 'rotate';

                        /*console.log('N1X '+n1_x+" N1Y "+n1_y);
                         console.log('NX '+n_x+" NY "+n_y);
                         */
                        // Now let the Trackball know it
                        _rotateEnd = _this.getLeapProjectionOnBall ( n_x, n_y, n_z );

                    }
                }

                /// ZOOM: 2 hands, One finger each
                if ( nb_hands >= 2 && nb_fingers == 2 ) {
                    //tool_chooser.push('zoom');
                    //action_proposed = true;
                }

                if ( state_leap == 'zoom' && nb_hands == 2 && nb_fingers == 2 ) {
                    if ( entering_state ) {
                        var finger_1 = frame.pointables[ 0 ];
                        var finger_2 = frame.pointables[ 1 ];
                        if ( finger_1.handId != finger_2.handId ) {
                            startFrame_leap = frame;
                            var norm1       = interactionBox.normalizePoint ( finger_1.tipPosition, true );
                            var norm2       = interactionBox.normalizePoint ( finger_2.tipPosition, true );
                            var d           = 2 - 2 * Math.abs ( norm2[ 0 ] - norm1[ 0 ] );
                            console.log ( 'DIFFFF' + d );

                            if ( d != 0 ) {
                                state_leap = 'zoom';
                                // Now let the Trackball know it
                                _zoomStart = _zoomEnd = new THREE.Vector2 ( 0, d );
                                _state = STATE.ZOOM;
                            }
                        }

                    } else {
                        var finger_1 = frame.pointables[ 0 ];
                        var finger_2 = frame.pointables[ 1 ];
                        if ( finger_1.handId != finger_2.handId ) {
                            startFrame_leap = frame;
                            var norm1       = interactionBox.normalizePoint ( finger_1.tipPosition, true );
                            var norm2       = interactionBox.normalizePoint ( finger_2.tipPosition, true );
                            var d           = 2 - 2 * Math.abs ( norm2[ 0 ] - norm1[ 0 ] );
                            console.log ( "DIFF" + d );

                            // Let update the Trackball
                            _zoomEnd = new THREE.Vector2 ( 0, d );
                        }
                    }
                }

                // If we didn't match any position, push a none instead
                if ( !action_proposed ) {
                    tool_chooser.push ( null );
                }

            }
        } );
    }


}
;

THREE.TrackballControls.prototype = Object.create ( THREE.EventDispatcher.prototype );
