#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import os
import traceback
import base64
import zlib
import cPickle
from multiprocessing import active_children
from Queue import Empty
import time
import threading

from shinken.property import PathProp, IntegerProp
from shinken.log import logger, get_chapter_string
from shinken.external_command import ExternalCommandManager
from shinken.daemon import Interface
from shinken.withinventorysatellite import WithInventorySatellite, IArbiterToInventorySatellite
from shinken.http_client import HTTPClient, HTTPExceptions
from shinken.vmware_stats import vmware_stats_reader

CHAPTER_CONNECTION = get_chapter_string('CONNECTION')


class IStatsReceiver(Interface):
    """ 
    Interface for various stats about broker activity
    """
    
    doc = '''Get raw stats from the daemon:
  * command_buffer_size: external command buffer size
'''
    
    
    def get_raw_stats(self, param=''):
        app = self.app
        raw_stats = super(IStatsReceiver, self).get_raw_stats()
        raw_stats.update({
            'command_buffer_size': len(app.external_commands),
            'module_stats'       : self._get_module_stats(getattr(self.app, 'modules_manager', None), param),
            'http_errors_count'  : app.http_errors_count,
            'have_conf'          : app.cur_conf is not None,
            'activated'          : self.app.activated,
            'spare'              : self.app.spare,
        })
        return raw_stats
    
    
    get_raw_stats.doc = doc


class IBroks(Interface):
    """Interface for Brokers
    They connect here and get all broks (data for brokers)
    data must be ORDERED! (initial status BEFORE update...)

    """
    
    doc = 'Get broks from the daemon'
    
    
    # poller or reactionner ask us actions
    def get_broks(self, bname):
        res = self.app.get_broks()
        return base64.b64encode(zlib.compress(cPickle.dumps(res), 2))
    
    
    get_broks.doc = doc


# Our main APP class
class Receiver(WithInventorySatellite):
    properties = WithInventorySatellite.properties.copy()
    properties.update({
        'pidfile'  : PathProp(default='receiverd.pid'),
        'port'     : IntegerProp(default='7773'),
        'local_log': PathProp(default='receiverd.log'),
    })
    
    
    def __init__(self, config_file, is_daemon, do_replace, debug, debug_file, daemon_id=0):
        super(Receiver, self).__init__('receiver', config_file, is_daemon, do_replace, debug, debug_file, daemon_id)
        
        # Our arbiters
        self.arbiters = {}
        
        # Our pollers and reactionners
        self.pollers = {}
        self.reactionners = {}
        
        # Modules are load one time
        self.have_modules = False
        
        # Can have a queue of external_commands give by modules
        # will be taken by arbiter to process
        self.external_commands = []
        # and the unprocessed one, a buffer
        self.unprocessed_external_commands = []
        
        # All broks to manage
        self.broks = {}
        
        self.host_assoc = {}
        self.direct_routing = False
        
        # Declare  HTTP methods interfaces
        self._add_http_interface(IStatsReceiver(self))
        self._add_http_interface(IBroks(self))
        self._add_http_interface(IArbiterToInventorySatellite(self))
    
    
    # Give us objects we need to manage. Only 2 are done:
    # Brok -> self.broks
    # External commands -> self.external_commands
    def add(self, elt):
        cls_type = elt.__class__.my_type
        if cls_type == 'brok':
            # For brok, we TAG brok with our instance_id
            elt.instance_id = 0
            self.broks[elt.id] = elt
            return
        elif cls_type == 'externalcommand':
            self.unprocessed_external_commands.append(elt)
    
    
    def push_host_names(self, sched_id, hnames):
        for h in hnames:
            self.host_assoc[h] = sched_id
    
    
    def get_sched_from_hname(self, hname):
        i = self.host_assoc.get(hname, None)
        e = self.schedulers.get(i, None)
        return e
    
    
    # Get 'objects' from external modules
    # from now nobody use it, but it can be useful
    # for a module like livestatus to raise external
    # commands for example
    def get_objects_from_from_queues(self):
        for queue in self.modules_manager.get_external_from_queues():
            while not queue.empty():
                o = queue.get(block=False)
                self.add(o)
    
    
    def do_stop(self):
        act = active_children()
        for a in act:
            a.terminate()
            a.join(1)
        super(Receiver, self).do_stop()
    
    
    def setup_new_conf(self):
        with self.satellite_lock:
            self.really_setup_new_conf()
    
    
    # Initialize or re-initialize connection with scheduler
    def _connect_to_scheduler(self, schedulers_id):
        scheduler = self.schedulers[schedulers_id]
        
        # If sched is not active, I do not try to init it is just useless
        if not scheduler['active']:
            return
        
        sname = scheduler['name']
        uri = scheduler['uri']
        timeout = scheduler.get('timeout', 3)
        data_timeout = scheduler.get('data_timeout', 120)
        running_id = scheduler['running_id']
        
        start_time = time.time()
        try:
            scheduler_connection = scheduler['con'] = HTTPClient(uri=uri, strong_ssl=scheduler['hard_ssl_name_check'], timeout=timeout, data_timeout=data_timeout)
        except HTTPExceptions, exp:
            logger.info('%s Connection problem to the %s %s (uri="%s"): %s' % (CHAPTER_CONNECTION, 'scheduler', sname, uri, str(exp)))
            scheduler['con'] = None
            scheduler['con_info'] = str(exp)
            return
        
        # timeout of 120 s and get the running id
        try:
            new_run_id = scheduler_connection.get('get_daemon_incarnation')
            new_run_id = float(new_run_id)
        except (HTTPExceptions, cPickle.PicklingError, KeyError), exp:
            logger.warning("[executor][%s] Scheduler %s is not initialized or has network problem: %s" % (self.name, sname, str(exp)))
            scheduler['con'] = None
            scheduler['con_info'] = str(exp)
            return
        
        elapsed = time.time() - start_time
        scheduler['con_latency'] = elapsed
        
        # The schedulers have been restarted: it has a new run_id.
        # So we clear all verifs, they are obsolete now.
        if scheduler['running_id'] != 0 and new_run_id != running_id:
            logger.info("[executor][%s] The running id of the scheduler %s changed, we must clear its actions" % (self.name, sname))
            with scheduler['wait_homerun_lock']:
                scheduler['wait_homerun'].clear()
        scheduler['running_id'] = new_run_id
        logger.info('%s Connection OK to the %s %s in %.3fs (uri="%s", ping_timeout=%ss, transfert_timeout=%ss)' % (CHAPTER_CONNECTION, 'scheduler', sname, elapsed, uri, timeout, data_timeout))
    
    
    def really_setup_new_conf(self):
        conf = self.new_conf
        self.new_conf = None
        self.cur_conf = conf
        # Got our name from the globals
        if 'receiver_name' in conf['global']:
            name = conf['global']['receiver_name']
        else:
            name = 'Unnamed receiver'
        self.name = name
        logger.load_obj(self, name)
        self.save_daemon_name_into_configuration_file(name)
        global_conf = conf['global']
        self.direct_routing = global_conf['direct_routing']
        self.activated = conf.get('activated', True)
        self.spare = global_conf.get('spare', False)
        # The arbiter let us know about the realms that are allowed to talk to us
        # it let us know also if a realm that was present before did disapear and so need to be deleted
        self.known_realms = conf['known_realms']
        
        # Let the vmware stats part know if it's enabled or not. Can change while running
        vmware_stats_reader.set_enabled(global_conf.get('vmware__statistics_compute_enable', True))
        
        # Should we enable/disable human log format
        logger.set_human_format(on=global_conf.get('human_timestamp_log', True))
        
        if not self.activated:
            logger.info('Stopping all modules')
            self.modules_manager.stop_all()
            self.have_modules = False
            
            self.modules = ()
            logger.info("[receiver][configuration] Configuration received, I'm configured as Spare")
            return
        logger.info("[receiver][configuration] Configuration received")
        
        # If we've got something in the schedulers, we do not want it anymore
        for sched_id in conf['schedulers']:
            
            already_got = False
            
            # We can already got this conf id, but with another address
            if sched_id in self.schedulers:
                new_addr = conf['schedulers'][sched_id]['address']
                old_addr = self.schedulers[sched_id]['address']
                new_port = conf['schedulers'][sched_id]['port']
                old_port = self.schedulers[sched_id]['port']
                # Should got all the same to be ok :)
                if new_addr == old_addr and new_port == old_port:
                    already_got = True
            
            if already_got:
                logger.info("[%s] We already got the conf %d (%s)" % (self.name, sched_id, conf['schedulers'][sched_id]['name']))
                wait_homerun = self.schedulers[sched_id]['wait_homerun']
                wait_homerun_lock = self.schedulers[sched_id]['wait_homerun_lock']
                actions = self.schedulers[sched_id]['actions']
                external_commands = self.schedulers[sched_id]['external_commands']
                con = self.schedulers[sched_id]['con']
            
            s = conf['schedulers'][sched_id]
            self.schedulers[sched_id] = s
            
            if s['name'] in global_conf['satellitemap']:
                s.update(global_conf['satellitemap'][s['name']])
            
            proto = 'http'
            if s['use_ssl']:
                proto = 'https'
            uri = '%s://%s:%s/' % (proto, s['address'], s['port'])
            
            self.schedulers[sched_id]['uri'] = uri
            if already_got:
                self.schedulers[sched_id]['wait_homerun'] = wait_homerun
                self.schedulers[sched_id]['wait_homerun_lock'] = wait_homerun_lock
                self.schedulers[sched_id]['actions'] = actions
                self.schedulers[sched_id]['external_commands'] = external_commands
                self.schedulers[sched_id]['con'] = con
            else:
                self.schedulers[sched_id]['wait_homerun'] = {}
                self.schedulers[sched_id]['wait_homerun_lock'] = threading.RLock()
                self.schedulers[sched_id]['actions'] = {}
                self.schedulers[sched_id]['external_commands'] = []
                self.schedulers[sched_id]['con'] = None
            self.schedulers[sched_id]['running_id'] = 0
            self.schedulers[sched_id]['active'] = s['active']
            # Do not connect if we are a passive satellite
            if self.direct_routing and not already_got:
                # And then we connect to it :)
                self._connect_to_scheduler(sched_id)
        
        logger.debug("[%s] Sending us configuration %s" % (self.name, conf))
        
        self.modules = conf['global']['modules']
        logger.info("[receiver][configuration] Receiving modules:[%s] i already load modules:[%s]" % (','.join([m.get_name() for m in self.modules]), self.have_modules))
        
        if not self.have_modules:
            # Ok now start, or restart them!
            # Set modules, init them and start external ones
            self.modules_manager.set_modules(self.modules)
            self.do_load_modules()
            self.modules_manager.start_external_instances()
            self.modules_manager.start_worker_based_instances()
            self.have_modules = True
        else:  # just update the one we need
            self.modules_manager.update_modules(self.modules)
        
        # Set our giving timezone from arbiter
        self.set_tz(conf['global']['use_timezone'])
        
        # Now create the external commander. It's just here to dispatch the commands to schedulers
        e = ExternalCommandManager(None, 'receiver')
        e.load_receiver(self)
        self.external_command = e
    
    
    # Take all external commands, make packs and send them to
    # the schedulers
    def push_external_commands_to_schedulers(self):
        with self.satellite_lock:
            self.really_push_external_commands_to_schedulers()
    
    
    def really_push_external_commands_to_schedulers(self):
        # If we are not in a direct routing mode, just bailout after
        # faking resolving the commands
        if not self.direct_routing:
            self.external_commands.extend(self.unprocessed_external_commands)
            self.unprocessed_external_commands = []
            return
        
        # Now get all external commands and put them into the
        # good schedulers
        for ext_cmd in self.unprocessed_external_commands:
            self.external_command.resolve_command(ext_cmd)
            self.external_commands.append(ext_cmd)
        
        # And clean the previous one
        self.unprocessed_external_commands = []
        
        # Now for all alive schedulers, send the commands
        for sched_id in self.schedulers:
            sched = self.schedulers[sched_id]
            sched_name = sched['name']
            extcmds = sched['external_commands']
            cmds = [extcmd.cmd_line for extcmd in extcmds]
            con = sched.get('con', None)
            sent = False
            if not con:
                logger.warning("The scheduler %s is not connected" % sched_name)
                self._connect_to_scheduler(sched_id)
                con = sched.get('con', None)
            
            # If there are commands and the scheduler is alive
            if len(cmds) > 0 and con:
                logger.debug("Sending %d commands to scheduler %s" % (len(cmds), sched_name))
                try:
                    # con.run_external_commands(cmds)
                    con.post('run_external_commands', {'cmds': cmds})
                    sent = True
                # Not connected or sched is gone
                except (HTTPExceptions, KeyError), exp:
                    if getattr(exp, 'errno', None) == 503:
                        logger.info('The scheduler %s is not ready.' % sched_name)
                        return
                    else:
                        logger.error('manage_returns exception:: %s,%s ' % (type(exp), str(exp)))
                        self._connect_to_scheduler(sched_id)
                        return
                except AttributeError, exp:  # the scheduler must  not be initialized
                    logger.debug('manage_returns exception:: %s,%s ' % (type(exp), str(exp)))
                    return
                except Exception, exp:
                    logger.error("A satellite raised an unknown exception: %s (%s)" % (exp, type(exp)))
                    raise
            
            # If we sent or not the commands, just clean the scheduler list.
            self.schedulers[sched_id]['external_commands'] = []
            
            # If we sent them, remove the commands of this scheduler of the arbiter list
            if sent:
                # and remove them from the list for the arbiter (if not, we will send it twice)
                for extcmd in extcmds:
                    try:
                        self.external_commands.remove(extcmd)
                    except ValueError:
                        pass
    
    
    def do_loop_turn(self):
        
        loop_start = time.time()
        
        # Begin to clean modules
        self.check_and_del_zombie_modules()
        
        # Now we check if arbiter speak to us in the pyro_daemon.
        # If so, we listen for it
        # When it push us conf, we reinit connections
        self.watch_for_new_conf(0.0)
        if self.new_conf:
            self.setup_new_conf()
        
        # Maybe external modules raised 'objects'
        # we should get them
        self.get_objects_from_from_queues()
        
        self.push_external_commands_to_schedulers()
        
        # We need to be sure that our inventory do not keep
        # deleted realms
        self._clean_old_realms_in_inventory()
        
        # If an inventory did change, warn the modules about it
        # so they can update their own inventory about it
        self.assert_module_inventory_are_updated()
        
        if int(time.time()) % 60 == 0:
            for (realm_name, inventory) in self._realms_inventory.items():
                logger.info('The realm %s inventory have currently %s elements' % (realm_name, inventory.get_len()))
        
        diff_time = time.time() - loop_start
        # Protect it against time shifting from system
        sleep_time = 1 - diff_time
        self.sleep(sleep_time)
    
    
    #  Main function, will loop forever
    def main(self):
        try:
            self.load_config_file()
            # Look if we are enabled or not. If ok, start the daemon mode
            self.look_for_early_exit()
            
            for line in self.get_header():
                logger.info(line)
            
            logger.info("[Receiver] Using working directory: %s" % os.path.abspath(self.workdir))
            
            self.do_daemon_init_and_start()
            
            self.load_modules_manager()
            
            self._register_http_interfaces()
            
            #  We wait for initial conf
            self.wait_for_initial_conf()
            if not self.new_conf:
                return
            
            self.setup_new_conf()
            
            # Now the main loop
            self.do_mainloop()
        
        except Exception:
            logger.critical("The daemon did have an unrecoverable error. It must exit.")
            logger.critical("You can log a bug to your Shinken integrator with the error message:")
            logger.critical("%s" % (traceback.format_exc()))
            raise
