#!/usr/bin/python

# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

"""
Helper functions for some filtering, like for user based
"""
from shinken.log import logger

item_access = {}


# Return true if user have access to element.
def check_user_item_access(user, item_uuid):
    if user.is_admin:
        return True
    
    if item_uuid is None:
        return False
    
    if '-' in item_uuid:
        item_uuid = item_uuid.split('-', 1)[0]
    
    users_with_access = item_access.get(item_uuid, set())
    if users_with_access == 'everyone':
        return True
    
    return user.uuid in users_with_access


def init_user_item_access_cache(rg):
    global item_access
    _def_access = rg.get_default_property_value('host', 'view_contacts', missing='nobody')
    _item_access = {}
    
    for host in rg.hosts:
        if host.view_contacts:
            _item_access[host.uuid] = set([c.uuid for c in host.view_contacts])
        else:
            _item_access[host.uuid] = _def_access
    
    item_access = _item_access


# Return true if user have access to element.
def is_authorized(element, user, app):
    if element is None or element.__class__ is None:
        return False
    if element.__class__.my_type == 'service':
        element = element.host
    return len(only_related_to([element], user, app)) != 0


# Get only user relevant items for the user
def only_related_to(items, user, app):
    if user is None:
        return []
    # if the user is an admin, show all
    if user.is_admin:
        return items
    
    # Ok the user is a simple user, we should filter
    filter_item = []
    for item in items:
        to_check = item
        if item.__class__.my_type == 'service':
            if item.host is None:
                logger.error('A service without host have been found id: %s  service_description: %s' % (item.instance_uuid, item.service_description))
                continue
            to_check = item.host
        
        # Maybe the user is a direct contact
        if user in to_check.view_contacts:
            filter_item.append(item)
            continue
        
        # Maybe view_contacts is void and we are allowed to set access to everyone
        if not to_check.view_contacts:
            _def_access = app.rg.get_default_property_value('host', 'view_contacts', missing='nobody')
            if _def_access == 'everyone':  # ok match a void contacts and anyone access, access is granted
                filter_item.append(item)
                continue
    
    return filter_item
