import os
import threading
import time
import traceback
from Queue import Empty

from shinken.misc.type_hint import List, Union

from .basesubprocess import BaseSubProcess
from .log import LoggerFactory
from .message import Message
from .objects.inventory import ElementsInventoryInBrokerModule
from .objects.itemsummary import HostSummary, CheckSummary
from .withworkersandinventorymodule import FROM_DAEMON_TO_MODULE_MESSAGE_TYPES


class ModuleWorker(BaseSubProcess):
    def __init__(self, worker_id, mod_conf, name, from_module_to_main_daemon_queue, queue_factory, daemon_display_name):
        super(ModuleWorker, self).__init__(name, daemon_display_name)
        
        self._worker_id = worker_id
        self._queue_factory = queue_factory
        self._reset_inventories()
        self._main_process_to_worker_inventory_queue = self._queue_factory('_main_process_to_worker_inventory_queue')
        self._from_module_to_main_daemon_queue = from_module_to_main_daemon_queue
        self.myconf = mod_conf
        self.logger = LoggerFactory.get_logger(mod_conf.get_name())
        self.daemon_display_name = daemon_display_name
        self.command_from_module_to_worker = self._queue_factory('command_from_module_to_worker')
        self.command_from_worker_to_module = self._queue_factory('command_from_worker_to_module')
        self._worker_is_init = False  # some threads will wait until the worker is fully init (like manage broks)
    
    
    def get_name(self):
        return self.name
    
    
    def in_main_process_tick(self):
        pass
    
    
    def _start_sub_process_threads(self):
        super(ModuleWorker, self)._start_sub_process_threads()
        
        thr = threading.Thread(target=self._command_to_worker_handler, name='command-to-worker')
        thr.daemon = True
        thr.start()
    
    
    def _command_to_worker_handler(self):
        while True:
            # Will block so we don't hammer cpu
            try:
                command_name, command_uuid = self.command_from_module_to_worker.get(block=True, timeout=1)
            except:
                # No command was send in last sec
                command_name = None
                command_uuid = None
            if command_name:
                # Right now we have no args on commands, so they are not handled
                f = getattr(self, command_name, None)
                if callable(f):
                    self.logger.debug('Executing command %s' % command_name)
                    self.command_from_worker_to_module.put((f(), command_uuid))
                else:
                    self.logger.warning('Received unknown command %s from father process !' % command_name)
    
    
    def get_raw_stats(self):
        return {}
    
    
    def _reset_inventories(self):
        self._global_inventory = ElementsInventoryInBrokerModule()
        self._realms_inventory = {}
    
    
    # Set by worker based modules to display their worker in the process list
    def _get_worker_display_name(self):
        return '[ Worker: %s ]' % self._worker_id
    
    
    def get_process_name(self):
        return "%s [ - Module: %s ]%s" % (self.father_name, self.name, self._get_worker_display_name())
    
    
    def get_logger_name(self):
        return [self.father_name.replace('shinken-', '').strip(), self.name, 'Worker: %s (pid:%s)' % (self._worker_id, os.getpid())]
    
    
    # By default workers start no more threads than the inventory one
    def start_worker_specific_treads(self):
        pass
    
    
    # Can be implemented by end-user
    def do_stop(self):
        pass
    
    
    def send_object_to_main_daemon(self, obj):
        self._from_module_to_main_daemon_queue.put(obj)
    
    
    # Important: others threads cannot work until the worker is fully init
    def _wait_for_worker_init_to_be_done(self):
        while not self._worker_is_init:
            time.sleep(0.1)
    
    
    def _worker_main(self):
        self.logger.info('In the worker: %s' % os.getpid())
        try:
            
            # set up name, priority, father lookup and such things
            # TODO: get back this
            self._sub_process_common_warmup()
            
            # Let the user code prepare itself
            self.init_worker(self.myconf)
            
            # Also start a thread that will manage all inventory updates
            # Start a thread that is need to load/compute inventory
            thr = threading.Thread(target=self._do_in_worker_inventory_thread, name='worker-inventory-thread')
            thr.daemon = True
            thr.start()
            
            # If the class want, it can ask to start specific threads (like broks for broker worker)
            self.start_worker_specific_treads()
            
            # Let the user code prepare itself, but after we did prepare the threads and worker structure
            self.init_worker_before_main(self.myconf)
            self._worker_is_init = True
            
            self.logger.info('Now running..')
            
            # Will block here!
            self.logger.debug('go main')
            self.worker_main()
            self.do_stop()
            self.logger.info('exiting now..')
        except Exception as exp:
            self.logger.error('[%s] The worker %s did crash. Exiting. Back trace of it:' % (self.name, self._worker_id))
            self.logger.print_stack()
            msg = Message(id=0, type='ICrash', data={'name': self.get_name(), 'exception': exp, 'trace': traceback.format_exc()})
            self._from_module_to_main_daemon_queue.put(msg)
            # wait a bit so we know that the receiver got our message, and die
            time.sleep(5)
            # And force the full module process to go down
            os._exit(2)
    
    
    def _send_exception_error_to_main_process_and_exit(self, trace, exception, thread_name):
        self.logger.error('The module %s does have a fail %s thread (%s). Cannot continue.' % (self.get_name(), thread_name, trace))
        try:
            msg = Message(id=0, type='ICrash', data={'name': self.get_name(), 'exception': exception, 'trace': trace})
            self._from_module_to_main_daemon_queue.put(msg)
            # wait a bit so we know that the receiver got our message, and die
            time.sleep(5)
        except Exception as exp:  # maybe the queue was close, but in all ca
            self.logger.error('Cannot send back thread error to the main process because of an error: %s' % exp)
        # And force the full module process to go down
        os._exit(2)
    
    
    def _do_in_worker_inventory_thread(self):
        # Important: we cannot manage broks until the worker is fully init
        self._wait_for_worker_init_to_be_done()
        
        while True:
            try:
                self._do_in_worker_inventory_thread_loop()
            except Exception as exp:
                self._send_exception_error_to_main_process_and_exit(traceback.format_exc(), exp, 'inventory update')
    
    
    def _send_inventory_message_to_worker(self, message):
        queue = self._main_process_to_worker_inventory_queue
        try:
            queue.put(message)
        except Exception as exp:
            raise Exception('[%s] [INVENTORY] Cannot send the inventory message to the worker %s: %s' % (self.get_name(), self._worker_id, exp))
    
    
    def _do_in_worker_inventory_thread_loop(self):
        # self.logger.debug('_do_in_worker_inventory_thread_loop:: %s' % self._worker_id)
        # A bit of debug but not too much
        if int(time.time() % 3600 == 0):
            for (realm_name, inventory) in self._realms_inventory.items():
                self.logger.debug('Dumping realm %s about inventory: %s elements' % (realm_name, inventory.get_len()))
        
        # We get from the Queue update from the receiver.
        # NOTE: as we are blocking, we are not using CPU and so no need for sleep
        try:
            inventory_update_message = self._main_process_to_worker_inventory_queue.get(block=True, timeout=1)
        except Empty:
            return
        
        # Load all form it and update our inventories
        self._in_worker_load_inventory_update_message(inventory_update_message)
    
    
    # We are in the worker and the master process did send us a inventory update message
    # with new/deleted and updated elements
    def _in_worker_load_inventory_update_message(self, inventory_update_message):
        message_type = inventory_update_message.get_type()
        payload = inventory_update_message.get_payload()
        realm_name = payload['realm_name']
        
        self.logger.debug('[INVENTORY] [WORKER:%s] We did receive an inventory update (%s) from the daemon about realm %s' % (self._worker_id, message_type, realm_name))
        
        # New realm => add
        if message_type == FROM_DAEMON_TO_MODULE_MESSAGE_TYPES.ADD_REALM:
            realm_inventory = payload['inventory']
            self.logger.info('[INVENTORY] [WORKER:%s] Adding the new realm %s in inventory' % (self._worker_id, realm_name))
            self._realms_inventory[realm_name] = realm_inventory
            self._global_inventory.add_elements_from_realm_inventory(realm_name, realm_inventory)
            self._warn_user_code_about_new_inventory(realm_name, realm_inventory)
        
        # Delete ream => del
        elif message_type == FROM_DAEMON_TO_MODULE_MESSAGE_TYPES.DELETE_REALM:
            self.logger.info('[INVENTORY] [WORKER:%s] removing the realm %s from inventory' % (self._worker_id, realm_name))
            old_realm_inventory = self._realms_inventory.get(realm_name, None)
            if old_realm_inventory is None:
                raise Exception('Asking for to delete a not exiting realm entry? %s %s' % (payload, self._realms_inventory))
            del self._realms_inventory[realm_name]
            self._global_inventory.delete_elements_from_realm_inventory(realm_name, old_realm_inventory)
            self._warn_user_code_about_deleted_inventory(realm_name, old_realm_inventory)
        
        # Update => reset
        elif message_type == FROM_DAEMON_TO_MODULE_MESSAGE_TYPES.UPDATE_REALM:
            self.logger.info('[INVENTORY] [WORKER:%s] Updating the realm %s in inventory' % (self._worker_id, realm_name))
            current_realm_inventory = self._realms_inventory.get(realm_name, None)
            if current_realm_inventory is None:
                raise Exception('Asking for a update on a not exiting realm entry? %s %s' % (payload, self._realms_inventory))
            inventory_differences = payload['inventory_differences']
            self.logger.debug('Applying realm difference: %s' % inventory_differences)
            
            current_realm_inventory.update_from_differences(inventory_differences)
            self._global_inventory.update_from_differences(inventory_differences)
            
            # Differences have new/changed/deleted elements
            self._warn_user_code_about_changed_inventory(realm_name, inventory_differences)
        
        else:
            raise Exception('Error: the message type %s is not managed' % message_type)
    
    
    # The module did get a new inventory, let the user code know about it
    def _warn_user_code_about_new_inventory(self, realm_name, realm_inventory):
        host_uuids = realm_inventory.get_hosts_uuids()
        for host_uuid in host_uuids:
            self.callback__a_new_host_added(host_uuid)
        self.callback__a_new_realm_added(realm_name)
    
    
    # The module did get a realm/inventory deletion, let the user code know about it
    def _warn_user_code_about_deleted_inventory(self, realm_name, old_realm_inventory):
        host_uuids = old_realm_inventory.get_hosts_uuids()
        for host_uuid in host_uuids:
            self.callback__a_host_deleted(host_uuid)
        self.callback__a_realm_deleted(realm_name)
    
    
    # The module did get an updated inventory, so need to let the user code know about new/deleted/changed elements
    def _warn_user_code_about_changed_inventory(self, realm_name, inventory_differences):
        for host_uuid in inventory_differences['new'].keys():
            self.callback__a_new_host_added(host_uuid)
        
        for host_uuid in inventory_differences['changed'].keys():
            self.callback__a_host_updated(host_uuid)
        
        for host_uuid in inventory_differences['deleted']:
            self.callback__a_host_deleted(host_uuid)
        
        self.callback__a_realm_updated(realm_name)
    
    
    ######### call backs that can be implemented by the final module
    def callback__a_new_host_added(self, host_uuid):
        pass
    
    
    def callback__a_host_updated(self, host_uuid):
        pass
    
    
    def callback__a_host_deleted(self, host_uuid):
        pass
    
    
    def callback__a_new_realm_added(self, realm_name):
        pass
    
    
    def callback__a_realm_deleted(self, realm_name):
        pass
    
    
    def callback__a_realm_updated(self, realm_name):
        pass
    
    
    def get_host_from_uuid(self, host_uuid):
        # type: (str) -> HostSummary
        return self._global_inventory.get_host(host_uuid)
    
    
    def get_all_inventory(self):
        # type: () -> List[Union[HostSummary, CheckSummary]]
        return self._global_inventory.get_all()
    
    
    def known_item_uuid(self, item_uuid):
        # type: (str) -> bool
        return self._global_inventory.have_item_uuid(item_uuid)
    
    
    def get_hosts_uuids(self):
        return self._global_inventory.get_hosts_uuids()
    
    
    def worker_main(self):
        raise NotImplementedError()
    
    
    def init_worker(self, module_configuration):
        pass
    
    
    def init_worker_before_main(self, module_configuration):
        pass
