#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.


import copy
from collections import namedtuple

from shinken.log import logger
from shinken.util import make_unicode
from shinken.misc.fast_copy import fast_deepcopy
from ..def_items import METADATA, DEF_ITEMS, INTERNAL_ATTR, WORKING_AREA_STATUS, ITEM_STATE, prop_is_linked
from ..helpers import get_dict_for_key, safe_add_to_dict_but_uniq, get_item_property, set_item_property, ShinkenDatabaseConsistencyError, items_for_link, item_for_link
from ..item_saving_formatter import LINKIFY_MANAGE_STATES

FromInfo = namedtuple('FromInfo', ['value', 'origin', 'type'])
ItemReversedLink = namedtuple('ReversedLink', ['item_type', 'item_id', 'key', 'item_state'])


class BaseItem(dict):
    
    def __repr__(self):
        # There are loop in base item so you must never print or log raw content. You can print raw_item.
        try:
            return '%s-%s-%s-%s' % (self['_id'], self.get_type(), self.get_state(), self.get_name())
        except:
            return '%s' % self.get_name()
    
    
    def __str__(self):
        # There are loop in base item so you must never print or log raw content. You can print raw_item.
        return '%s-%s-%s-%s' % (self['_id'], self.get_type(), self.get_state(), self.get_name())
    
    
    def raw_repr(self):
        return super(BaseItem, self).__repr__()
    
    
    def __setitem__(self, key, value):
        if key != "@metadata":
            logger.info("[BaseItem] Please use set_value instead key:(%s)" % key)
        # raise DataException("Please use set_value instead")
        self.set_value(key, value)
    
    
    def __delitem__(self, key):
        self.del_key(key)
    
    
    def del_key(self, key):
        unicode_key = make_unicode(key)
        self.get_from().pop(key, None)
        METADATA.get_metadata(self, METADATA.RAW_ITEM, {}).pop(unicode_key, None)
        super(BaseItem, self).__delitem__(unicode_key)
    
    
    def set_value(self, key, value, _from=None, default_value=False):
        # fast setup for defaults values
        if default_value:
            self.get_from()[key] = _from
            super(BaseItem, self).__setitem__(key, value)
            return
        
        unicode_key = make_unicode(key)
        item_type = self.get_type()
        
        if key.startswith('@'):
            _from = None
        elif key in INTERNAL_ATTR:
            # if _from is not None that mean we come from a deepcopy and the unicode value can contain links
            if _from is None:
                # we need to clean the last_modification.contact and workarea_info.user
                contact_link = None
                if key == 'last_modification' and value.get('contact', None):
                    contact_link = value['contact']['links'][0].pop('@link', None)
                raw_value = fast_deepcopy(value)
                if contact_link:
                    value['contact']['links'][0]['@link'] = contact_link
                
                METADATA.get_metadata(self, METADATA.RAW_ITEM, {})[unicode_key] = raw_value
        elif _from is None and prop_is_linked(item_type, key):
            if key in DEF_ITEMS[item_type].get('fields_with_plus', set()):
                _from = []
                item_ids_or_names = (i.get('_id', i.get('name')) for i in value.get('links', ()))
                for item_link in item_ids_or_names:
                    _from.append(FromInfo(item_link, self['_id'], item_type))
            else:
                _from = [FromInfo(value, self['_id'], item_type)]
            METADATA.get_metadata(self, METADATA.RAW_ITEM, {})[unicode_key] = fast_deepcopy(value)
        elif _from is None:
            # The default FromInfo, with the given value and says tht the value come from us (our name + our type)
            _from = [FromInfo(value, self['_id'], item_type)]
            # update the raw metadata and then the value
            METADATA.get_metadata(self, METADATA.RAW_ITEM, {})[unicode_key] = value
        
        if _from is not None:
            # update the from metadata
            self.get_from()[key] = _from
        super(BaseItem, self).__setitem__(unicode_key, value)
    
    
    # This one is a real private
    def _raw_set_value(self, prop, value):
        super(BaseItem, self).__setitem__(prop, value)
    
    
    def __copy__(self):
        logger.error("We shouldn't be here __copy__")
        # raise Exception('NO COPY')
        # from . import get_item_instance
        # new_item = get_item_instance(self.get_type())
        # new_item['@metadata'] = copy.copy(self["@metadata"])
        # raw_item = METADATA.get_metadata(self, METADATA.RAW_ITEM)
        # if raw_item:
        #     METADATA.update_metadata(new_item, METADATA.RAW_ITEM, copy.deepcopy(raw_item))
        # for prop, value in self.iteritems():
        #     if prop == "@metadata":
        #         continue
        #     new_item._raw_set_value(prop, value)
        # return new_item
    
    
    def __deepcopy__(self, memodict={}):
        logger.error("We shouldn't be here __deepcopy__")
        # raise Exception('NO DEEP COPY')
        # from . import get_item_instance
        # new_item = get_item_instance(self.get_type())
        # new_item['@metadata'] = copy.copy(self["@metadata"])
        # raw_item = METADATA.get_metadata(self, METADATA.RAW_ITEM)
        # if raw_item:
        #     METADATA.update_metadata(new_item, METADATA.RAW_ITEM, copy.deepcopy(raw_item))
        # for prop, value in self.iteritems():
        #     if prop == "@metadata":
        #         continue
        #     new_item.set_value(prop, value, _from=self.get_from(prop))
        # return new_item
    
    
    def get_key(self):
        try:
            return hash((self['_id'], self.get_type(), self.get_state()))
        except:
            logger.error('Cannot make key for item [%s]' % self.raw_repr())
            raise
    
    
    def get_uuid(self):
        return self['_id']
    
    
    # =================== METADATA management ===============================
    def get_name(self):
        return METADATA.get_metadata(self, METADATA.NAME)
    
    
    def get_type(self):
        return METADATA.get_metadata(self, METADATA.ITEM_TYPE)
    
    
    def get_class(self):
        return METADATA.get_metadata(self, METADATA.TABLE)
    
    
    def get_state(self):
        return METADATA.get_metadata(self, METADATA.STATE)
    
    
    def get_changes(self):
        return METADATA.get_metadata(self, METADATA.CHANGES, {})
    
    
    def is_enabled(self):
        return self.get('enabled', '1') == '1'
    
    
    def is_existing(self):
        return self.get('exists', '1') == '1'
    
    
    def get_reverse_links(self):
        return METADATA.get_metadata(self, METADATA.REVERSE_LINKS, set())
    
    
    def check_reverse_link(self, reversed_link, linked_item):
        check_reverse_link = True
        if not linked_item or self not in linked_item.get_link_items(reversed_link.key, states=[self.get_state()], only_exist=True):
            self.get_reverse_links().remove(reversed_link)
            check_reverse_link = False
        return check_reverse_link
    
    
    def get_from(self, attr=None):
        if attr:
            return METADATA.get_metadata(self, METADATA.FROM, {}).get(attr, [])
        return METADATA.get_metadata(self, METADATA.FROM, {})
    
    
    def get_work_area_status(self, item_state=None):
        return WORKING_AREA_STATUS.get_work_area_status(self, item_state)
    
    
    def have_manual_changes(self):
        return bool(self.get_changes())
    
    
    def apply_all_diffs(self, flatten_links=LINKIFY_MANAGE_STATES):
        _raw_item = self.get_raw_item(flatten_links=flatten_links)
        changes = self.get_changes()
        for (prop, couple) in changes.iteritems():
            old_value, new_value = couple[0], couple[1]
            if new_value:
                _raw_item[prop] = new_value
            elif prop in _raw_item:
                del _raw_item[prop]
        return _raw_item
    
    
    def add_reverse_link(self, new_reverse_link):
        self_reverse_links = self.get_reverse_links()
        if new_reverse_link not in self_reverse_links:
            self_reverse_links.add(new_reverse_link)
            METADATA.update_metadata(self, METADATA.REVERSE_LINKS, self_reverse_links)
    
    
    def flatten_prop(self, prop_to_flatten, prop_item_states=LINKIFY_MANAGE_STATES, raw_item=None, for_arbiter=False):
        item = raw_item or self
        raw_item_value = get_item_property(item, prop_to_flatten)
        if not raw_item_value:
            return None
        if isinstance(raw_item_value, basestring):
            return raw_item_value
        
        specific_method = getattr(self, "_flatten_prop_%s" % prop_to_flatten, None)
        if specific_method and callable(specific_method):
            return specific_method(raw_item_value, prop_to_flatten, prop_item_states, for_arbiter)
        
        return self._flatten_basic_value(raw_item_value, prop_to_flatten, prop_item_states)
    
    
    def _flatten_basic_value(self, raw_item_value, prop_to_flatten='', prop_item_states=LINKIFY_MANAGE_STATES):
        item_name_links = []
        item_value = get_item_property(self, prop_to_flatten)
        # create an index id -> links with @links
        items_to_links_id_index = {}
        for item_link_info in item_value.get('links', []):
            if '_id' in item_link_info:
                items_to_links_id_index[item_link_info['_id']] = item_link_info
        # Loop on links and not @links to keep order ( because of none existing links)
        for link_info in raw_item_value.get('links', []):
            if link_info['exists']:
                # Find real item describe from link_info
                link_info_with_item_link = items_to_links_id_index.get(link_info['_id'], None)
                if not link_info_with_item_link or "@link" not in link_info_with_item_link:
                    raise ShinkenDatabaseConsistencyError('property=[%s] value=[%s] raw_item_value=[%s]' % (prop_to_flatten, item_value, link_info), self['_id'], self.get_type(), self.get_state(), asking_state=prop_item_states)
                link_info_links = link_info_with_item_link['@link']
                link_item = next((link_info_links[item_state] for item_state in prop_item_states if item_state in link_info_links), None)
                if link_item:
                    item_name_links.append(link_item.get_name())
            else:
                item_name_links.append(link_info['name'])
        if len(item_name_links) > 1:
            item_name_links = (n for n in item_name_links if n != 'null')
        
        plus_as_string = '+' if raw_item_value.get('has_plus', False) else ''
        return '%s%s' % (plus_as_string, ','.join(item_name_links))
    
    
    def get_flat_item(self, item_as_dict, flatten_links=LINKIFY_MANAGE_STATES, for_arbiter=False):
        for prop_to_flatten in DEF_ITEMS[self.get_type()]['props_links']:
            working_dict, working_prop_to_flatten = get_dict_for_key(item_as_dict, prop_to_flatten)
            if working_prop_to_flatten in working_dict:
                prop_value = self.flatten_prop(prop_to_flatten, prop_item_states=flatten_links, raw_item=item_as_dict, for_arbiter=for_arbiter)
                if prop_value is None:
                    raise ShinkenDatabaseConsistencyError('property=[%s] value=[%s]' % (prop_to_flatten, self[prop_to_flatten]), self['_id'], self.get_type(), self.get_state())
                working_dict[working_prop_to_flatten] = prop_value
        return item_as_dict
    
    
    def get_raw_item(self, keep_metadata=False, flatten_links=LINKIFY_MANAGE_STATES, keep_links=False, for_arbiter=False):
        raw_item = METADATA.get_metadata(self, METADATA.RAW_ITEM, {})
        if raw_item:
            raw_item = fast_deepcopy(raw_item)
            if keep_metadata:
                raw_item[u'@metadata'] = self['@metadata'].copy()
                raw_item[u'@metadata'][METADATA.FROM] = self['@metadata'][METADATA.FROM].copy()
            
            if isinstance(keep_metadata, (list, tuple, set)):
                for metadata_key in self['@metadata']:
                    if metadata_key not in keep_metadata:
                        METADATA.remove_metadata(raw_item, metadata_key)
            
            if flatten_links:
                raw_item = self.get_flat_item(raw_item, flatten_links=flatten_links, for_arbiter=for_arbiter)
            elif keep_links:
                from . import get_item_instance
                raw_item_instance = get_item_instance(self.get_type(), raw_item)
                
                for prop_to_keep_link in DEF_ITEMS[self.get_type()]['props_links']:
                    if prop_to_keep_link in raw_item_instance:
                        try:
                            # create an index id -> link_item
                            items_to_links_id_index = {}
                            for item_to_link in self.get_links(prop_to_keep_link):
                                if item_to_link['exists']:
                                    items_to_links_id_index[item_to_link['_id']] = item_to_link['@link']
                        except KeyError:
                            raise ShinkenDatabaseConsistencyError(
                                'Object exists but not linkify : property=[%s] incorrect links=[%s]' % (prop_to_keep_link, [link for link in self.get_links(prop_to_keep_link) if link['exists'] and not link.get('@link')]),
                                self['_id'],
                                self.get_type(),
                                self.get_state(),
                                asking_state=LINKIFY_MANAGE_STATES)
                        for link in raw_item_instance.get_links(prop_to_keep_link):
                            if link['exists']:
                                link['@link'] = items_to_links_id_index[link['_id']]
            return raw_item
        else:
            logger.warning('get_raw_item missing metadata [%s]' % METADATA.RAW_ITEM)
            logger.print_stack()
            return self.copy()
    
    
    def get_raw_value(self, property_name, datamanagerV2=None, flatten_links=LINKIFY_MANAGE_STATES):
        raw_item = METADATA.get_metadata(self, METADATA.RAW_ITEM, {})
        if not raw_item:
            raw_item = self
        value = raw_item[property_name]
        
        if flatten_links:
            value = datamanagerV2.flatten_prop(value, self.get_type(), property_name, flatten_links)
        return value
    
    
    def get_links(self, property_name, linked_items_type=None):
        item_value = get_item_property(self, property_name)
        if isinstance(item_value, basestring):
            return []
        
        specific_method = getattr(self, "_get_links_%s" % property_name, None)
        if specific_method and callable(specific_method):
            return specific_method(property_name, linked_items_type)
        
        if isinstance(item_value, dict):
            if linked_items_type is not None:
                links = item_value.get('links', [])
                return [link for link in links if link.get('item_type', None) == linked_items_type or link['exists'] is False]
            
            return item_value.get('links', [])
        else:
            return []
    
    
    def update_link(self, property_name, item):
        specific_method = getattr(self, "_update_link_%s" % property_name, None)
        if specific_method and callable(specific_method):
            effective = specific_method(property_name, item)
        else:
            effective, _ = self._update_link(property_name, item)
        
        self._build_reversed_link([item], property_name)
        return effective
    
    
    def _update_link(self, property_name, item_to_link):
        effective = False
        data_found = False
        value = get_item_property(self, property_name)
        if not value:
            return effective, data_found
        
        item_to_link_name = item_to_link.get_name()
        for index, link in enumerate(value['links']):
            if link.get('_id', '') == item_to_link['_id']:
                data_found = True
                link['@link'][item_to_link.get_state()] = item_to_link
                break
            
            # we are maybe linking a previously non existing item
            elif not link['exists'] and link['name'] == item_to_link_name:
                data_found = True
                link['exists'] = True
                link['item_type'] = item_to_link.get_type()
                link['_id'] = item_to_link['_id']
                del link['name']
                
                # update the metadata raw_item
                raw_value = get_item_property(METADATA.get_metadata(self, METADATA.RAW_ITEM), property_name)
                if raw_value and index < len(raw_value['links']):
                    raw_value['links'][index] = link.copy()
                    effective = True
                
                link['@link'] = {item_to_link.get_state(): item_to_link}
                break
        return effective, data_found
    
    
    def add_unique_items_links(self, property_name, items):
        link_added = []
        value = get_item_property(self, property_name)
        
        if not value:
            value = {
                'has_plus': False,
                'links'   : []
            }
            set_item_property(self, property_name, value)
        
        for item_to_add in items:
            if item_to_add.get_state() not in LINKIFY_MANAGE_STATES:
                continue
            
            _, data_found = self._update_link(property_name, item_to_add)
            if not data_found:
                link = {
                    'exists'   : True,
                    'item_type': item_to_add.get_type(),
                    '_id'      : item_to_add['_id']
                }
                value['links'].append(link)
                raw_value = get_item_property(METADATA.get_metadata(self, METADATA.RAW_ITEM), property_name)
                if not raw_value:
                    raw_value = {
                        'has_plus': False,
                        'links'   : []
                    }
                    set_item_property(METADATA.get_metadata(self, METADATA.RAW_ITEM), property_name, raw_value)
                
                raw_value['links'].append(link.copy())
                link['@link'] = {item_to_add.get_state(): item_to_add}
                
                if property_name.split('.')[0] not in INTERNAL_ATTR:
                    safe_add_to_dict_but_uniq(self.get_from(), property_name, FromInfo(item_to_add['_id'], self['_id'], self.get_type()))
                
                link_added.append(item_to_add)
        
        if link_added:
            self._build_reversed_link(link_added, property_name)
        return link_added
    
    
    def _build_reversed_link(self, items, property_name):
        # add the reverse link
        item_reversed_link = ItemReversedLink(item_type=self.get_type(), item_id=self['_id'], key=property_name, item_state=self.get_state())
        # Add item_reverse_link on each item_to_link reverse_link metadata
        for item_to_link in items:
            item_to_link.add_reverse_link(item_reversed_link)
    
    
    def remove_item_link(self, property_name, item_to_unlink):
        specific_method = getattr(self, '_remove_item_link_%s' % property_name.replace('.', '_'), None)
        if specific_method and callable(specific_method):
            return specific_method(property_name, item_to_unlink)
        
        property_value = get_item_property(self, property_name)
        if not property_value:
            return False
        
        effective = False
        
        item_to_unlink_id = item_to_unlink['_id']
        my_links = property_value['links']
        link_item_index, link_item = next(((index, link_item) for (index, link_item) in enumerate(my_links) if link_item.get('_id', None) == item_to_unlink_id), (0, {}))
        # The item to remove wasn't found
        if not link_item:
            return False
        
        link_item.get('@link', {}).pop(item_to_unlink.get_state(), None)
        if not link_item.get('@link', {}):
            # in ITEM_STATE.WORKING_AREA and ITEM_STATE.STAGGING dead link are simply remove and not pass to exist False
            if self.get_state() in (ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING):
                effective = 'must_save_with_callback'
                my_links.remove(link_item)
                _raw_item = METADATA.get_metadata(self, METADATA.RAW_ITEM, {})
                if len(my_links) or property_value['has_plus']:
                    raw_item_links = get_item_property(_raw_item, property_name).get('links', [])
                    link_item.pop('@link', None)
                    if link_item in raw_item_links:
                        raw_item_links.remove(link_item)
                else:
                    _raw_item.pop(property_name, None)
                # There aren't source_info in WORKING_AREA or STAGGING so we don't call set_source_info_property_value
            else:
                effective = True
                link_item.pop('@link', None)
                link_item.pop('_id', None)
                link_item.pop('item_type', None)
                link_item['exists'] = False
                link_item['name'] = item_to_unlink.get_name()
                raw_item = METADATA.get_metadata(self, METADATA.RAW_ITEM, {})
                raw_item_links = get_item_property(raw_item, property_name).get('links', [])
                if link_item_index < len(raw_item_links):
                    raw_item_links[link_item_index] = copy.copy(link_item)
        return effective
    
    
    def _remove_item_link_last_modification_contact(self, property_name, item_to_unlink):
        
        if not get_item_property(self, property_name):
            return False
        
        effective = False
        item_to_unlink_id = item_to_unlink['_id']
        my_links = get_item_property(self, property_name)['links']
        link_item_index, link_item = next(((_index, link_item) for (_index, link_item) in enumerate(my_links) if link_item.get('_id', None) == item_to_unlink_id), (0, {}))
        
        if not link_item:
            return False
        
        link_item.get('@link', {}).pop(item_to_unlink.get_state(), None)
        if not link_item.get('@link', {}):
            link_item.pop('@link', None)
            link_item.pop('_id', None)
            link_item.pop('item_type', None)
            link_item['exists'] = False
            link_item['name'] = item_to_unlink.get_name()
            
            _raw_item = METADATA.get_metadata(self, METADATA.RAW_ITEM, {})
            raw_item_links = get_item_property(_raw_item, property_name).get('links', [])
            raw_item_links[link_item_index] = link_item.copy()
            effective = True
        return effective
    
    
    def set_inexisting_link(self, property_name, item_to_unlink):
        specific_method = getattr(self, '_set_inexisting_link_%s' % property_name.replace('.', '_'), None)
        if specific_method and callable(specific_method):
            return specific_method(property_name, item_to_unlink)
        
        property_value = get_item_property(self, property_name)
        if not property_value:
            return False
        
        item_to_unlink_id = item_to_unlink['_id']
        my_links = property_value['links']
        link_item_index, link_item = next(((index, link_item) for (index, link_item) in enumerate(my_links) if link_item.get('_id', None) == item_to_unlink_id), (0, {}))
        # The item to remove wasn't found
        if not link_item:
            return False
        
        link_item.pop('@link', None)
        link_item.pop('_id', None)
        link_item.pop('item_type', None)
        link_item['exists'] = False
        link_item['name'] = item_to_unlink.get_name()
        raw_item = METADATA.get_metadata(self, METADATA.RAW_ITEM, {})
        raw_item_links = get_item_property(raw_item, property_name).get('links', [])
        if link_item_index < len(raw_item_links):
            raw_item_links[link_item_index] = copy.copy(link_item)
        return True
    
    
    def get_all_state_link_items(self, property_name, states=LINKIFY_MANAGE_STATES, only_exist=False, item_type=None):
        link_values = self.get_links(property_name, item_type)
        if not link_values:
            return []
        
        link_items = []
        for item_link in link_values:
            link_items.extend(items_for_link(item_link, states=states, only_exist=only_exist))
        return link_items
    
    
    def get_link_items(self, property_name, states=LINKIFY_MANAGE_STATES, only_exist=False, item_type=None):
        link_values = self.get_links(property_name, item_type)
        if not link_values:
            return []
        link_items = []
        for item_link in link_values:
            link_item = item_for_link(item_link, states=states, only_exist=only_exist)
            if link_item:
                link_items.append(link_item)
        return link_items
    
    
    def get_link_item(self, property_name, states=LINKIFY_MANAGE_STATES, only_exist=False):
        return next(iter(self.get_link_items(property_name, states, only_exist)), None)
    
    
    def get_ids_in_link(self, property_name):
        ids = []
        if not isinstance(self.get(property_name, {}), dict):
            return ids
        for link in self.get(property_name, {}).get('links', []):
            link_exists = link.get('exists', False)
            link_id = link.get('_id', '')
            if link_exists and link_id:
                ids.append(link_id)
        return ids
    
    
    def _get_use_ids(self):
        return self.get_ids_in_link('use')
    
    
    def find_all_templates_by_links(self, hosts_visited=None, level=32):
        if level <= 0 or (hosts_visited and self.get_key() in hosts_visited):
            return []
        
        if hosts_visited:
            hosts_visited.append(self.get_key())
        else:
            hosts_visited = [self.get_key()]
        
        if not isinstance(self.get('use', {}), dict):
            return []
        
        templates = []
        for template in self.get_all_state_link_items('use'):
            if template.is_existing() and template.is_enabled():
                templates.append(template)
                for i in template.find_all_templates_by_links(hosts_visited=hosts_visited, level=level - 1):
                    if i not in templates:
                        templates.append(i)
        
        return templates
