#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.
import uuid
import threading
import time
from ...business.sync_ui_common import syncuicommon
from shinken.log import logger

DB_TRANSACTION_CONTAINER_NAME = 'db_transaction_container'


def get_transaction_object():
    current_thread = threading.current_thread()
    return getattr(current_thread, DB_TRANSACTION_CONTAINER_NAME, None)


def set_transaction_container(transaction_container):
    current_thread = threading.current_thread()
    return setattr(current_thread, DB_TRANSACTION_CONTAINER_NAME, transaction_container)


def remove_transaction_container():
    current_thread = threading.current_thread()
    return delattr(current_thread, DB_TRANSACTION_CONTAINER_NAME)


class TransactionContainer(threading.local):
    def __init__(self, transaction_uuid, user):
        # logger.debug('Initializing transaction container %r %s' % (self, transaction_uuid))
        self.transaction_uuid = transaction_uuid
        self.user = user
        self.container = {}
    
    
    def get_container(self, name):
        if not name in self.container:
            self.container[name] = {}
        return self.container[name]


class DBTransaction(object):
    
    def __init__(self, datamanagerV2=None, user=None):
        self.transaction_uuid = uuid.uuid4().hex
        self.start_time = time.time()
        self.datamanager = datamanagerV2 or syncuicommon.app.datamanagerV2
        self.user = user
        # a transaction call in a transaction will be disable
        # All OP will done in Mother transaction
        self.enable = True
    
    
    def __enter__(self):
        if get_transaction_object():
            self.enable = False
            return
        
        # create the transaction container link to the current thread
        set_transaction_container(TransactionContainer(self.transaction_uuid, self.user))
        self.datamanager.start_transaction(self.transaction_uuid)
    
    
    def __exit__(self, type, value, traceback):
        if not self.enable:
            return
        nb_op = self.datamanager.commit_transaction(self.transaction_uuid)
        
        remove_transaction_container()
        syncuicommon.reset_configuration_stats_cache()
        
        try:
            path_info = syncuicommon.app.request.path
        except Exception:
            path_info = 'not in request'
        
        logger.log_perf(self.start_time, self, 'Transaction %s with %s op ended for path [%s]' % (self.transaction_uuid, nb_op, path_info))
