#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018
# This file is part of Shinken Enterprise, all rights reserved.

import copy
import socket
import sys

from shinken.log import logger
from shinken.objects.config import Config
from shinken.property import IntegerProp, StringProp, BoolProp

from .business.source.source import Source, Sources
from .business.source.listener import Listener, Listeners
from .business.source.analyzer import Analyzer, Analyzers
from .synchronizerlink import SynchronizerLink, SynchronizerLinks
from .tagger import Tagger, Taggers


class SynchronizerConfig(Config):
    properties = copy.copy(Config.properties)
    properties.update({
        'sync_history_lifespan'                        : IntegerProp(default='1440'),
        'mongodb_retry_timeout'                        : IntegerProp(default='120'),
        'service_merged_flag'                          : StringProp(default='/var/lib/shinken/service_merged_flag'),
        'master_key'                                   : StringProp(default=''),
        'protect_fields__activate_encryption'          : BoolProp(default=False),
        'protect_fields__activate_database_encryption' : BoolProp(default=False),
        'protect_fields__activate_interface_encryption': BoolProp(default=False),
        'protect_fields__encryption_keyfile'           : StringProp(default=''),
        'protect_fields__substrings_matching_fields'   : StringProp(default=u'PASSWORD,PASSPHRASE,PASSE,DOMAINUSER,MSSQLUSER,MYSQLUSER,ORACLE_USER,SSH_USER,LOGIN'),
        'protect_fields__are_viewable_by_admin_si'     : BoolProp(default='0'),
    })
    early_created_types = ['synchronizer', 'module', 'source', 'listener', 'analyzer', 'tagger', 'arbiter']
    types_creations = copy.copy(Config.types_creations)
    types_creations.update({
        'synchronizer': (SynchronizerLink, SynchronizerLinks, 'synchronizers'),
        'source'      : (Source, Sources, 'sources'),
        'listener'    : (Listener, Listeners, 'listeners'),
        'analyzer'    : (Analyzer, Analyzers, 'analyzers'),
        'tagger'      : (Tagger, Taggers, 'taggers'),
    })
    global_cfg_p = '/etc/shinken/shinken.cfg'
    
    
    # configuration_types = Config.configuration_types
    # configuration_types.extend(['source'])
    # we want to let our listeners be managed like sources, so merge them
    # into the self.sources
    def merge_listeners_into_sources(self):
        for listener in self.listeners:
            listener.prepare_before_being_merge_into_sources()
            # IMPORTANT: it's possible because the Listener and Source class
            # share the same id
            self.sources[listener.id] = listener
    
    
    # configuration_types = Config.configuration_types
    # configuration_types.extend(['source'])
    # we want to let our listeners be managed like sources, so merge them
    # into the self.sources
    def merge_analyzers_into_sources(self):
        for analyzer in self.analyzers:
            analyzer.prepare_before_being_merge_into_sources()
            # IMPORTANT: it's possible because the Listener and Source class
            # share the same id
            self.sources[analyzer.id] = analyzer
    
    
    # Here arbiter and modules objects should be prepare and link
    # before all others types
    def early_synchronizer_linking(self):
        """ Prepare the synchronizer for early operations """
        #
        self.merge_listeners_into_sources()
        self.merge_analyzers_into_sources()
        
        self.modules.create_reversed_list()
        self.sources.create_reversed_list()
        self.taggers.create_reversed_list()
        self.arbiters.create_reversed_list()
        
        if len(self.synchronizers) == 0:
            logger.warning("There is no synchronizer, I add one in localhost:7765")
            a = SynchronizerLink({
                'synchronizer_name': 'Default-Synchronizer',
                'host_name'        : socket.gethostname(),
                'address'          : 'localhost',
                'port'             : '7765',
                'spare'            : '0'})
            self.synchronizers = SynchronizerLinks([a])
        
        # First fill default
        self.synchronizers.fill_default()
        self.modules.fill_default()
        self.sources.fill_default()
        self.taggers.fill_default()
        self.arbiters.fill_default()
        
        # print "****************** Pythonize ******************"
        self.synchronizers.pythonize()
        self.sources.pythonize()
        self.taggers.pythonize()
        self.arbiters.pythonize()
        
        # print "****************** Linkify ******************"
        self.synchronizers.linkify(self.modules, self.sources, self.taggers)
        self.modules.linkify()
        self.sources.linkify(self.modules)
        self.taggers.linkify(self.modules)
        self.arbiters.linkify(self.modules)
    
    
    def check_config(self, list_elt, type):
        tmp_list = {}
        for src in list_elt:
            source_name = getattr(src, '%s_name' % type, '')
            import_from = getattr(src, 'imported_from', '')
            if source_name:
                if not source_name in tmp_list:
                    tmp_list[source_name] = import_from
                else:
                    sys.exit("The %s name [%s] is duplicate in file : %s,%s" % (type, source_name, tmp_list[source_name], import_from))
            else:
                sys.exit("A %s with a missing name was found in [%s]." % (type, import_from))
    
    
    def is_correct(self):
        r = self.conf_is_correct
        
        for s in self.sources:
            r = r and s.is_correct()
        b = self.synchronizers.is_correct()
        if not b:
            r = False
        b = self.taggers.is_correct()
        if not b:
            r = False
        b = self.realms.is_correct()
        if not b:
            self.configuration_errors.extend(self.realms.configuration_errors)
            r = False
        self.conf_is_correct = r
    
    
    def split_sources_and_modules(self, objects):
        for source in [o for o in objects.get('source', []) if 'modules' not in o]:
            if not 'source_name' in source:
                continue
            module = {'module_name': ['module-%s' % source['source_name'][0]]}
            source['modules'] = module['module_name']
            for key in source.keys():
                module[key] = source[key]
            objects['module'].append(module)
        
        for listener in [o for o in objects.get('listener', []) if 'modules' not in o]:
            if 'listener_name' not in listener:
                continue
            module = {'module_name': ['module-%s' % listener['listener_name'][0]]}
            listener['modules'] = module['module_name']
            for key in listener.keys():
                module[key] = listener[key]
            objects['module'].append(module)
        
        for analyzer in [o for o in objects.get('analyzer', []) if 'modules' not in o]:
            if 'analyzer_name' not in analyzer:
                continue
            module = {'module_name': ['module-%s' % analyzer['analyzer_name'][0]]}
            analyzer['modules'] = module['module_name']
            for key in analyzer.keys():
                module[key] = analyzer[key]
            objects['module'].append(module)
