#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019:
# This file is part of Shinken Enterprise, all rights reserved.
from shinken.log import logger
from shinken.misc.type_hint import NoReturn, List
from shinkensolutions.api.synchronizer.http_lib_external.v01_00.html.shinken_object import ShinkenObject


class MESSAGE(object):
    ICON_ERROR = '''<span class='shinken-icon-error shinkon-warning'></span>'''
    ICON_WARNING = '''<span class='shinken-icon-warning shinkon-warning'></span>'''
    ICON_BEWARE = '''<span class='shinken-icon-warning shinkon-warning-light'></span>'''
    MAX_ERROR = 5
    MAX_WARNING = 5
    STATUS_NONE = 'shinken-no-status'
    STATUS_INFO = 'shinken-info'
    STATUS_ERROR = 'shinken-error-status'
    STATUS_WARNING = 'shinken-warning-status'
    STATUS_BEWARE = 'shinken-beware-status'


class Messages(ShinkenObject):
    def __init__(self):
        # type: ()-> None
        super(Messages, self).__init__()
        self.messages_error = []
        self.messages_warning = []
        self.messages_beware = []
        self.messages_info = []
    
    
    def add_message_txt(self, type_message, text):
        # type: (str, str)-> None
        self.add_message(Message(type_message, text))
    
    
    def add_message(self, message):
        # type: (Message)-> None
        if message.get_type() == MESSAGE.STATUS_ERROR:
            self.messages_error.append(message)
        elif message.get_type() == MESSAGE.STATUS_WARNING:
            self.messages_warning.append(message)
        elif message.get_type() == MESSAGE.STATUS_BEWARE:
            self.messages_beware.append(message)
        else:
            self.messages_info.append(message)
    
    
    def add_messages(self, messages):
        # type: (List[Message])-> None
        for _message in messages:
            self.add_message(_message)
    
    
    def has_message(self):
        # type: ()-> bool
        return self.has_error() or self.has_warning() or self.has_beware() or self.has_info()
    
    
    def has_error(self):
        # type: ()-> bool
        return bool(len(self.messages_error))
    
    
    def has_warning(self):
        # type: ()-> bool
        return bool(len(self.messages_warning))
    
    
    def has_beware(self):
        # type: ()-> bool
        return bool(len(self.messages_beware))
    
    
    def has_info(self):
        # type: ()-> bool
        return bool(len(self.messages_info))
    
    
    def get_status(self):
        # type: ()-> str
        if self.has_error():
            return MESSAGE.STATUS_ERROR
        if self.has_warning():
            return MESSAGE.STATUS_WARNING
        if self.has_beware():
            return MESSAGE.STATUS_BEWARE
        return MESSAGE.STATUS_NONE
    
    
    def build_message(self):
        # type: ()-> str
        if self.get_status() == MESSAGE.STATUS_NONE:
            return ''
        
        _to_return = ['''<div class='shinken-tooltip-message text-left'>''']
        if self.has_error():
            _to_return.append('''%s%s<br><ol>%s</ol>''' % (MESSAGE.ICON_ERROR, self._('element.tooltip_message_tag_error'), self.build_message_for_list(self.messages_error, MESSAGE.MAX_ERROR)))
        if self.has_warning():
            _to_return.append('''%s%s<br><ol>%s</ol>''' % (MESSAGE.ICON_WARNING, self._('element.tooltip_message_tag_warning'), self.build_message_for_list(self.messages_warning, MESSAGE.MAX_WARNING)))
        if self.has_beware():
            _to_return.append('''%s%s<br><ol>%s</ol>''' % (MESSAGE.ICON_BEWARE, self._('element.tooltip_message_tag_warning'), self.build_message_for_list(self.messages_beware)))
        _to_return.append('''</div>''')
        return ''.join(_to_return)
    
    
    def extract_error_messages(self):
        # type: ()-> Messages
        _to_return = Messages()
        _to_return.add_messages(self.messages_error)
        return _to_return
    
    
    def extract_warning_messages(self):
        # type: ()-> Messages
        _to_return = Messages()
        _to_return.add_messages(self.messages_warning)
        return _to_return
    
    
    def build_message_for_list(self, list, max=999):
        # type: (List[Message], int)-> str
        _size = min(len(list), max)
        _to_return = []
        for index, _current in enumerate(list):
            if index > _size:
                break
            _to_return.append(_current.get_html())
        
        return ''.join(_to_return)


class Message(ShinkenObject):
    def __init__(self, type_message, text):
        # type: (str, str)-> None
        super(Message, self).__init__()
        self._type = type_message
        self._text = text
    
    
    def get_type(self):
        # type: ()-> str
        return self._type
    
    
    def get_html(self):
        # type: ()-> str
        return '''<li class='%s'>%s</li>''' % (self._type, self._text)
