#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.

import json
from datetime import datetime, timedelta

from shinkensolutions.lib_checks.common import Result, RaiseOnExitOptionParser, ParseOptionError, EXIT_STATUS, BREAK_LINE, ShinkenUtils, HTMLTag, HTMLList, HTMLTable, COLOR, Utils

VERSION = '0.1'
DAEMON_TYPE = 'broker'

TAG_FOR_STATE = {
    EXIT_STATUS.OK      : HTMLTag.OK,
    EXIT_STATUS.WARNING : HTMLTag.WARNING,
    EXIT_STATUS.CRITICAL: HTMLTag.CRITICAL
}

NO_STATS = 'NO_STATS'
NO_DATA = 'NO_DATA'

DATE_FORMAT = '%Y/%m/%d %X'

result = Result()
warnings = []
criticals = []

parser = RaiseOnExitOptionParser('%prog [options] [--help]', version='%prog ' + VERSION)
parser.add_option('-H', '--hostname', dest='hostname', help='The hostname of the shinken daemon')
parser.add_option('-p', '--port', dest='port', type='int', help='The port of the shinken daemon')
parser.add_option('-t', '--timeout', dest='timeout', type='int', default=3, help='timeout to connect to the shinken daemon. Default : 3')
parser.add_option('-m', '--minutes', dest='minutes_of_stats', type='int', default=1, help='Number of minutes worth of stats displayed. Defualt : 1 minute')
parser.add_option('--shinkenversion', dest='shinken_supervisor_version', default='', help='The shinken version number used to compare with the monitored shinken. Mandatory if in shinken mode.')


def _parse_args():
    opts = None
    try:
        opts, args = parser.parse_args()
        if args:
            parser.error('Does not accept arguments.')
        if not opts.hostname:
            parser.error('Missing parameter hostname (-H/--hostname)')
        if not opts.port:
            parser.error('Missing parameter port (-p/--port)')
    except ParseOptionError as e:
        if e.msg:
            result.hard_exit(EXIT_STATUS.CRITICAL, 'Fail to parse command argument: %s %s' % (BREAK_LINE, BREAK_LINE.join(e.msg.split('\n'))))
        exit(0)
    return opts


def check_perf(event_container_module_info, second_of_stats):
    status = EXIT_STATUS.OK
    perf_info = []
    left_headers = []
    
    str_total_duration = '%s minute' % (int(second_of_stats / 60))
    item_count = event_container_module_info.get('item_count', {})
    total_nb_item = item_count.get('items', 0)
    items = Utils.print_human_readable_number(total_nb_item)
    hosts = Utils.print_human_readable_number(item_count.get('hosts', 0))
    clusters = Utils.print_human_readable_number(item_count.get('clusters', 0))
    checks = Utils.print_human_readable_number(item_count.get('checks', 0))
    
    have_perf_stats = event_container_module_info.get('have_perf_stats', NO_STATS)
    brok_handle = event_container_module_info.get('brok_handle', 0)
    event_write = event_container_module_info.get('event_write', 0)
    work_time = event_container_module_info.get('work_time', 0)
    
    perf_text = 'Not enough data to compute performance. Please wait some time and retry to launch the check.'
    if have_perf_stats == NO_DATA:
        perf_text = 'No new events were generated in the last %s.' % str_total_duration
    elif have_perf_stats != NO_STATS:
        str_brok_handle = Utils.print_human_readable_number(brok_handle)
        str_event_write = Utils.print_human_readable_number(event_write)
        str_average_time = Utils.print_human_readable_period(work_time / brok_handle)
        
        perf_text = 'Events written in the last %s : %s changes for %s status updates (average time by status update : %s)' % (str_total_duration, str_event_write, str_brok_handle, str_average_time)
    
    global_stats = [HTMLList.header_list(
        'Element : %s' % items,
        ['Hosts: %s' % hosts, 'Clusters: %s' % clusters, 'Checks: %s' % checks]),
        perf_text]
    global_stats = HTMLList.simple_list(global_stats)
    perf_info.append([global_stats])
    left_headers.append('Global')
    
    for worker_id, worker_data in event_container_module_info.get('workers', {}).iteritems():
        item_count = worker_data.get('item_count', {})
        items = Utils.print_human_readable_number(item_count.get('items', 0))
        hosts = Utils.print_human_readable_number(item_count.get('hosts', 0))
        clusters = Utils.print_human_readable_number(item_count.get('clusters', 0))
        checks = Utils.print_human_readable_number(item_count.get('checks', 0))
        
        worker_have_perf_stats = worker_data.get('have_perf_stats', NO_STATS)
        worker_total_duration = worker_data.get('total_duration', 0)
        worker_brok_handle = worker_data.get('brok_handle', 0)
        worker_event_write = worker_data.get('event_write', 0)
        worker_work_time = worker_data.get('work_time', 0)
        
        str_perfs = ['Not enough data to compute performance. Please wait some time and retry to launch the check.']
        if worker_have_perf_stats == NO_DATA:
            str_perfs = ['No new events were generated in the last %s.' % str_total_duration]
        elif worker_have_perf_stats != NO_STATS:
            str_brok_handle = Utils.print_human_readable_number(worker_brok_handle)
            str_event_write = Utils.print_human_readable_number(worker_event_write)
            str_average_time = Utils.print_human_readable_period(worker_work_time / worker_brok_handle)
            str_load = Utils.print_percent(worker_work_time, worker_total_duration)
            str_work_time = Utils.print_human_readable_period(worker_work_time)
            
            result.add_perf_data('worker_%s_event_write_in_last_min' % worker_id, worker_event_write)
            result.add_perf_data('worker_%s_brok_handle_in_last_min' % worker_id, worker_brok_handle)
            result.add_perf_data('worker_%s_load_in_last_min' % worker_id, str_load)
            
            str_perfs = [
                'Events written in the last %s : %s changes for %s status updates (average time by status update : %s)' % (str_total_duration, str_event_write, str_brok_handle, str_average_time),
                'Worker load in the last %s seconds: %s%% (work time : %s)' % (str_total_duration, str_load, str_work_time)
            ]
        
        worker_stats = [HTMLList.header_list('Element : %s' % items, ['Hosts: %s' % hosts, 'Clusters: %s' % clusters, 'Checks: %s' % checks])]
        worker_stats.extend(str_perfs)
        worker_stats = HTMLList.simple_list(worker_stats)
        perf_info.append([worker_stats])
        left_headers.append('<p style="padding-left:20px">Worker %s<p/>' % worker_id)
    
    total_event_in_db = event_container_module_info.get('total_event_in_db', 0)
    size = event_container_module_info.get('total_size_of_event_in_db', 0)
    oldest_event_in_db = event_container_module_info.get('oldest_event_in_db', 0)
    nb_day_keep = event_container_module_info.get('nb_day_keep', 0)
    if oldest_event_in_db:
        str_oldest_event_in_db = 'Date of the oldest event : %s' % datetime.fromtimestamp(oldest_event_in_db).strftime(DATE_FORMAT)
    else:
        str_oldest_event_in_db = 'No event in database'
    event_infos = [
        'Events kept during: %s days' % nb_day_keep,
        '%s events saved ( %s )' % (Utils.print_human_readable_number(total_event_in_db), Utils.print_human_readable_size(size)),
        str_oldest_event_in_db,
    ]
    perf_info.append([HTMLList.simple_list(event_infos)])
    left_headers.append('Database')
    class_uuid = HTMLTable.generate_class_uuid()
    
    extra_style = [
        '.%s .skn-ict .skn-lfh {width: 20%% !important;}' % class_uuid
    ]
    
    result.add_check(status=status, output=perf_text, long_output=HTMLTable.table([], perf_info, left_headers=left_headers, class_uuid=class_uuid, all_col_same_width=False, extra_style=extra_style))
    
    if have_perf_stats:
        result.add_perf_data('global_event_write_in_last_min', brok_handle)
        result.add_perf_data('global_brok_handle_in_last_min', event_write)
    result.add_perf_data('total_element', total_nb_item)
    result.add_perf_data('total_event_number', total_event_in_db)
    result.add_perf_data('total_base_size', size)


def check_modules(raw_stats, second_of_stats):
    event_container_modules = raw_stats.get('module_stats', {}).get('event_container', {})
    if not event_container_modules:
        result.add_check(EXIT_STATUS.CRITICAL, '%s - There is no event manager module on the Broker daemon.' % TAG_FOR_STATE[EXIT_STATUS.CRITICAL])
        return
    if len(event_container_modules) > 1:
        result.add_check(EXIT_STATUS.CRITICAL, '%s - There is more than one event manager module on the Broker daemon.' % TAG_FOR_STATE[EXIT_STATUS.CRITICAL])
        return
    
    modules_info = raw_stats.get('modules_info', [])
    if not modules_info:
        result.add_check(EXIT_STATUS.CRITICAL, '%s - There is no module info on the Broker daemon.' % TAG_FOR_STATE[EXIT_STATUS.CRITICAL])
        return
    event_container_module_info = next(iter([module_info for module_info in modules_info if module_info['type'] == 'event_container']), None)
    if event_container_module_info is None:
        result.add_check(EXIT_STATUS.CRITICAL, '%s - There is no event manager module info on the Broker daemon.' % TAG_FOR_STATE[EXIT_STATUS.CRITICAL])
        return
    
    event_container_module_status = event_container_module_info['status']
    # Event manager module info status stuff
    if event_container_module_status == 'FATAL':
        if isinstance(event_container_module_info['output'], basestring):
            result.add_check(EXIT_STATUS.CRITICAL, '%s %s' % (HTMLTag.state_tag('FATAL'), event_container_module_info['output']))
        else:
            for output in event_container_module_info['output']:
                result.add_check(EXIT_STATUS.CRITICAL, '%s %s' % (HTMLTag.state_tag('FATAL'), output))
        return
    elif event_container_module_status == 'CRITICAL':
        if isinstance(event_container_module_info['output'], basestring):
            result.add_check(EXIT_STATUS.CRITICAL, '%s %s' % (HTMLTag.state_tag('CRITICAL'), event_container_module_info['output']))
        else:
            for output in event_container_module_info['output']:
                result.add_check(EXIT_STATUS.CRITICAL, '%s %s' % (HTMLTag.state_tag('CRITICAL'), output))
        return
    
    check_status = result.status
    check_perf(event_container_modules.values()[0], second_of_stats)
    
    output = []
    if check_status == EXIT_STATUS.CRITICAL:
        output.append('Module is in a critical state.')
        for critical in criticals:
            output.append('%s %s' % (HTMLTag.CRITICAL, critical))
        for warning in warnings:
            output.append('%s %s' % (HTMLTag.WARNING, warning))
    elif check_status == EXIT_STATUS.WARNING:
        output.append('Module has some anomalies')
        for warning in warnings:
            output.append('%s %s' % (HTMLTag.WARNING, warning))
    elif check_status == EXIT_STATUS.OK:
        output.append('Module is working as intended.')
    result.add_check(status=check_status, output='<br>'.join(output), title=True)


def main():
    opts = _parse_args()
    daemon_adr = opts.hostname
    daemon_port = opts.port
    timeout = opts.timeout
    minutes_of_stats = opts.minutes_of_stats
    if minutes_of_stats <= 0:
        minutes_of_stats = 1
    shinken_supervisor_version = opts.shinken_supervisor_version
    
    second_of_stats = (minutes_of_stats * 60)
    html, connection_time = ShinkenUtils.request_get_daemon(result, DAEMON_TYPE, '%s:%s' % (daemon_adr, daemon_port), '/get_raw_stats?param=%s' % second_of_stats, timeout=timeout)
    raw_stats = json.loads(html)
    ShinkenUtils.minimal_check(result, raw_stats, DAEMON_TYPE, shinken_supervisor_version)
    
    check_modules(raw_stats, second_of_stats)
    result.exit()


if __name__ == '__main__':
    main()
