#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

"""
This class is for linking the WebUI with active directory,
like check passwords, or get photos.
"""

import os

# Lazy load, as import ldap can be super long
ldap = None

from shinken.log import logger
from shinken.basemodule import BaseModule, ModuleState
import json

properties = {
    'daemons': ['webui', 'skonf', 'synchronizer'],
    'type'   : 'ad_webui'
}

DEFAULT_MAPPING_FILE = '/etc/shinken/_default/configuration/modules/auth-active-directory/mapping.json'


# called by the plugin manager
def get_instance(plugin):
    global ldap
    try:
        import ldap as ldap
        ldap = ldap
    except ImportError:
        ldap = None
    logger.debug("Get an Active Directory/OpenLdap UI module for plugin %s" % plugin.get_name())
    if not ldap:
        raise Exception('The module python-ldap is not found. Please install it.')
    instance = AD_Webui(plugin)
    return instance


class AD_Webui(BaseModule):
    def __init__(self, modconf):
        
        BaseModule.__init__(self, modconf)
        self.timeout = getattr(modconf, 'timeout', 5)
        self.mapping_file = getattr(modconf, 'mapping_file', '')
        self.mode = getattr(modconf, 'mode', 'ad')
        self.module_name = modconf.get_name()
        
        # Default values, should be overridden by json mapping file
        self.ldap_key = 'sAMAccountName'
        self.shinken_key = 'contact_name'
        self.login_placeholder = ''
        self.active = False
        
        if self.mode == 'ad':
            self.retrieveAttributes = ["userPrincipalName"]
            self.name_id = 'userPrincipalName'
            self.auth_key = 'userPrincipalName'
            self.ldap_key = 'sAMAccountName'
        elif self.mode == 'openldap':
            self.retrieveAttributes = ["dn", "uid"]
            self.name_id = 'uid'
            self.auth_key = 'dn'
            self.ldap_key = 'uid'
        else:
            raise Exception('[%s] WebUI Auth ldap module error, mode is not in ad or openldap' % self.module_name)
        
        self.load_mapping_file()
        self.retrieveAttributes.append(self.ldap_key)
        self.search_format = "(%s=%%s)" % self.ldap_key
        
        mandatory_fields = ('ldap_uri', 'username', 'password', 'basedn')
        self.missing_fields = []
        for field in mandatory_fields:
            value = getattr(modconf, field, None)
            setattr(self, field, value)
            if not value:
                self.missing_fields.append(field)
        
        if self.missing_fields:
            logger.error('[%s] Theses parameters are missing in the cfg file : %s' % (self.module_name, ', '.join(self.missing_fields)))
            return
        
        try:
            # Try to set the self.acttive to True
            connected = self.connect()
            if connected:
                logger.info('[%s] Ldap server [%s] with basedn [%s] is available' % (self.module_name, self.ldap_uri, self.basedn))
        except ldap.LDAPError as exp:
            raise Exception("[%s] Can't connect to server %s with error : %s" % (self.module_name, self.ldap_uri, str(exp)))
        finally:
            self.disconnect()
    
    
    def get_state(self):
        if not self.active:
            return {"status": ModuleState.CRITICAL, "output": "The module was not able to auth in the LDAP server. Please check your connection configuration and credentials."}
        return {"status": ModuleState.OK, "output": "OK"}
    
    
    # Try to connect if we got true parameter
    def load_mapping_file(self):
        try:
            if not self.mapping_file:
                self.mapping_file = DEFAULT_MAPPING_FILE
                logger.info('[%s] Mapping file not set, will use the default mapping file : %s' % self.module_name, DEFAULT_MAPPING_FILE)
            lines = []
            with open(self.mapping_file) as f:
                for l in f.readlines():
                    l = l.strip()
                    if l.startswith('#'):
                        continue
                    lines.append(l)
            conf = json.loads('\n'.join(lines))
            self.ldap_key = conf.get('ldap_key', 'samaccountname').encode('utf8', 'ignore')
            self.shinken_key = conf.get('shinken_key', 'contact_name').encode('utf8', 'ignore')
            self.login_placeholder = conf.get('login_placeholder', '').encode('utf8', 'ignore')
            logger.debug('[%s] Loaded mapping : ldap_key [%s], shinken_key [%s], login_placeholder [%s]' % (self.module_name, self.ldap_key, self.shinken_key, self.login_placeholder))
        except Exception, exp:
            logger.warning('[%s] Failed to load mapping with error : %s' % (self.module_name, exp))
            logger.info('[%s] Can\'t load custom mapping, will use default mapping : ldap_key [%s], shinken_key [%s], login_placeholder [%s]' % (self.module_name, self.ldap_key, self.shinken_key, self.login_placeholder))
    
    
    def init(self):
        if not self.active:
            return
    
    
    def connect(self):
        logger.debug("[%s] Trying to initialize the connection to [%s] with user [%s] on basedn [%s]" % (self.module_name, self.ldap_uri, self.username, self.basedn))
        try:
            self.con = ldap.initialize(self.ldap_uri)
            self.con.set_option(ldap.OPT_REFERRALS, 0)
            
            # Any errors will throw an ldap.LDAPError exception
            # or related exception so you can ignore the result
            self.con.set_option(ldap.OPT_NETWORK_TIMEOUT, self.timeout)
            self.con.simple_bind_s(self.username, self.password)
            self.active = True
            logger.debug('[%s] Connection to server %s is OK' % (self.module_name, self.ldap_uri))
            return True
        except Exception as e:
            logger.error("[%s] Connection to server %s Failed : %s" % (self.module_name, self.ldap_uri, e.message['desc'] if 'desc' in e.message else str(e)))
            return False
    
    
    def disconnect(self):
        self.con = None
    
    
    # To load the webui application
    def load(self, app):
        self.app = app
    
    
    # Give the entry for a contact
    def find_contact_entry(self, cname):
        
        if not cname:
            return None
        
        cname = cname.encode('utf8')
        # First we try to connect, because there is no "KEEP ALIVE" option available, so we will get a drop after one day...
        searchScope = ldap.SCOPE_SUBTREE
        searchFilter = (self.search_format % cname)
        
        logger.debug('[%s] The ldap request filter is : %s' % (self.module_name, str(searchFilter).decode("utf8")))
        try:
            connected = self.connect()
            ldap_result_id = self.con.search(self.basedn, searchScope, searchFilter, self.retrieveAttributes)
            
            while True:
                result_type, result_data = self.con.result(ldap_result_id, 0)
                if (result_data == []):
                    logger.warning('[%s] The user %s was not found in the ldap server [%s] in base [%s]' % (self.module_name, cname.decode('utf8'), self.ldap_uri, self.basedn))
                    return None
                
                if result_type == ldap.RES_SEARCH_ENTRY:
                    (_, elts) = result_data[0]
                    if self.mode == 'openldap':
                        elts['dn'] = str(result_data[0][0])
                    try:
                        account_name = elts[self.name_id][0]
                    except Exception:
                        account_name = str(result_data[0])
                    
                    logger.info('[%s] user account %s was found in ldap base' % (self.module_name, account_name.decode('utf8')))
                    return elts
        except ldap.LDAPError, e:
            logger.error("[%s] Error during requesting the ldap server %s : %s, %s" % (self.module_name, self.ldap_uri, e, str(e.__dict__)))
            return None
        # Always clean on exit
        finally:
            self.disconnect()
    
    
    # Try to auth a user in the ldap dir
    def check_auth(self, user, password):
        
        logger.info('[%s] Trying to authenticate user %s on server %s' % (self.module_name, user, self.ldap_uri))
        
        if not self.active:
            if self.missing_fields:
                logger.error('[%s] The module is disabled because the configuration is incomplete' % self.module_name)
                return False
            else:
                connected = self.connect()
                if not connected:
                    logger.error('[%s] The module is disabled because the configuration is invalid or server is unavailable' % self.module_name)
                    return False
        
        c = None
        if self.shinken_key == 'contact_name':
            c = self.app.datamgr.get_contact(user)
        else:
            for contact in self.app.datamgr.get_contacts():
                # If c is a dict, we were called from the Synchronizer
                # If c is an object, we were called from the Broker
                if isinstance(contact, dict):
                    login = contact.get(self.shinken_key, '')
                else:
                    if self.shinken_key.startswith('_'):
                        login = contact.customs.get(self.shinken_key, '')
                    else:
                        login = getattr(contact, self.shinken_key, '')
                if login == user:
                    c = contact
                    break
        
        if not c:
            logger.warning('[%s] The user %s is not defined in the configuration (%s)' % (self.module_name, self.shinken_key, user))
            return False
        
        # I don't know why, but ldap automagically auth void password. That's just stupid I think so we don't allow them.
        if not password:
            logger.warning('[%s] User %s doesn\'t give password. It\' not allowed' % (self.module_name, user))
            return False
        
        # first we need to find the principalname of this entry
        # because it can be a user name like j.gabes, but we should auth by ldap
        # with j.gabes@google.com for example
        elts = self.find_contact_entry(user)
        
        # no user found, exit
        if elts is None:
            return False
        
        try:
            # On AD take the uid / principalname
            if self.mode == 'ad':
                # Maybe the entry is void....
                if self.auth_key in elts:
                    account_name = elts[self.auth_key][0]
                else:
                    account_name = user
            else:  # For openldap, use the full DN
                account_name = elts[self.auth_key]
        except KeyError:
            logger.warning('[%s] The ldap object have not the attribute %s. Can\'t map the attribute with the shinken key. The user value will be used to found the user : %s' % (self.module_name, self.auth_key, user))
            account_name = user
        
        local_con = ldap.initialize(self.ldap_uri)
        local_con.set_option(ldap.OPT_REFERRALS, 0)
        
        # Any errors will throw an ldap.LDAPError exception
        # or related exception so you can ignore the result
        try:
            
            # logger.debug('[%s] password type:[%s]' % (self.module_name, type(password)))
            local_con.simple_bind_s(account_name, password.encode('utf8'))
            if isinstance(c, dict):
                contact_name = c.get('contact_name', '')
            else:
                contact_name = c.contact_name
            logger.info('[%s] Credentials for %s are OK. Authenticated as shinken user %s' % (self.module_name, user, contact_name))
            return c
        except ldap.LDAPError as e:
            logger.error("[%s] Credentials for %s FAILED : %s" % (self.module_name, user, e.message['desc'] if 'desc' in e.message else str(e)))
        except UnicodeEncodeError, exp:
            logger.error("[%s] The name or the password contain a unicode character: %s" % (self.module_name, str(exp)))
        finally:
            # Necessary for Oracle Directory Server to close connection immediately
            local_con.unbind_s()
        
        # The local_con will automatically close this connection when
        # the object will be deleted, so no close need
        
        # No good? so no auth :)
        return False
