#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.
import uuid
from datetime import datetime, timedelta

from ec_common import STATE, ITEM_TYPE, DOWNTIME, ACK, STATE_TYPE
from shinken.brok import Brok
from shinken.check import NO_END_VALIDITY
from shinkensolutions.date_helper import get_datetime_with_local_time_zone, get_datetime_from_utc_to_local_time_zone, timestamp_from_datetime


def get_item_type_from_brok_data(brok_data):
    if '-' in brok_data['instance_uuid']:
        return ITEM_TYPE.CHECK
    elif brok_data.get('got_business_rule', False):
        return ITEM_TYPE.CLUSTER
    else:
        return ITEM_TYPE.HOST


def get_state_id_from_brok_data(item_type, brok_data):
    if item_type == ITEM_TYPE.CLUSTER:
        return brok_data['bp_state']
    
    state_id = brok_data['state_id']
    # before first check we force STATE.MISSING_DATA in state_id
    if brok_data['state'] == 'PENDING':
        state_id = STATE.MISSING_DATA
    elif brok_data['state'] == 'UNREACHABLE':
        state_id = STATE.UNKNOWN
    
    if item_type == ITEM_TYPE.HOST and state_id == STATE.WARN:
        state_id = STATE.CRIT
    
    return state_id


def get_downtime_from_brok_data(brok_data):
    dt_value = DOWNTIME.NONE
    if brok_data.get('in_scheduled_downtime', None):
        dt_value = DOWNTIME.ACTIVE
    if brok_data.get('in_inherited_downtime', None):
        dt_value = DOWNTIME.INHERITED
    
    return dt_value


def get_ack_from_brok_data(brok_data):
    ack = ACK.NONE
    if brok_data['problem_has_been_acknowledged'] and not brok_data['in_inherited_acknowledged']:
        ack = ACK.ACTIVE
    elif brok_data['in_inherited_acknowledged']:
        ack = ACK.INHERITED
    
    return ack


class Event(object):
    event_uuid = u''
    item_uuid = u''
    event_since = None  # type: datetime
    event_hard_since = None  # type: datetime
    item_type = ITEM_TYPE.UNSET
    state_id = STATE.MISSING_DATA
    state_type = STATE_TYPE.HARD
    state_validity_period = NO_END_VALIDITY
    flapping = False
    acknowledged = ACK.NONE
    acknowledge_id = u''
    downtime = DOWNTIME.NONE
    partial_flapping = False
    partial_ack = False
    partial_dt = False
    active_downtime_uuids = []
    output = u''
    long_output = u''
    realm = u''
    name = u''
    host_name = u''
    check_name = u''
    ordering_uuid = u''
    reader_stats = None  # type: ECReaderStats
    
    
    def __init__(self, **kwarg):
        self.item_uuid = kwarg.get('item_uuid', Event.item_uuid)
        self.event_since = kwarg.get('event_since', Event.event_since)
        self.event_hard_since = kwarg.get('event_hard_since', Event.event_hard_since)
        self.item_type = kwarg.get('item_type', Event.item_type)
        self.state_id = kwarg.get('state_id', Event.state_id)
        self.state_type = kwarg.get('state_type', Event.state_type)
        self.state_validity_period = kwarg.get('state_validity_period', Event.state_validity_period)
        self.flapping = kwarg.get('flapping', Event.flapping)
        self.acknowledged = kwarg.get('acknowledged', Event.acknowledged)
        self.acknowledge_id = kwarg.get('acknowledge_id', Event.acknowledge_id)
        self.downtime = kwarg.get('downtime', Event.downtime)
        self.partial_flapping = kwarg.get('partial_flapping', Event.partial_flapping)
        self.partial_ack = kwarg.get('partial_ack', Event.partial_ack)
        self.partial_dt = kwarg.get('partial_dt', Event.partial_dt)
        self.active_downtime_uuids = kwarg.get('active_downtime_uuids', Event.active_downtime_uuids)
        self.output = kwarg.get('output', Event.output)
        self.long_output = kwarg.get('long_output', Event.long_output)
        self.realm = kwarg.get('realm', Event.realm)
        self.name = kwarg.get('name', Event.name)
        self.host_name = kwarg.get('host_name', Event.host_name)
        self.check_name = kwarg.get('check_name', Event.check_name)
        
        if 'ordering_uuid' in kwarg:
            self.ordering_uuid = kwarg['ordering_uuid']
        else:
            self.ordering_uuid = u'%015i-%s' % (timestamp_from_datetime(self.event_since), self.item_uuid)
        
        if 'event_uuid' in kwarg:
            self.event_uuid = kwarg['event_uuid']
        else:
            self.event_uuid = uuid.uuid4().hex
    
    
    def get_comprehensive_state(self):
        return len(self.active_downtime_uuids) * 10000000 + self.partial_flapping * 1000000 + self.partial_ack * 100000 + self.partial_dt * 10000 + self.flapping * 1000 + self.acknowledged * 100 + self.downtime * 10 + self.state_id
    
    
    @classmethod
    def from_brok_data(cls, brok):
        # type: (Brok) -> Event
        brok_data = brok.data
        creation_date = brok_data['creation_date']
        item_uuid = brok_data['instance_uuid']
        item_type = get_item_type_from_brok_data(brok_data)
        state_id = get_state_id_from_brok_data(item_type, brok_data)
        state_type = brok_data['state_type_id']
        state_validity_period = brok_data['state_validity_period']
        flapping = brok_data['is_flapping']
        acknowledged = get_ack_from_brok_data(brok_data)
        acknowledge_id = brok_data['acknowledge_id']
        downtime = get_downtime_from_brok_data(brok_data)
        partial_flapping = brok_data.get('is_partial_flapping', False)
        partial_ack = brok_data.get('is_partial_acknowledged', False)
        partial_dt = brok_data.get('in_partial_downtime', False)
        active_downtime_uuids = brok_data.get('active_downtime_uuids', None)
        output = brok_data['output']
        long_output = brok_data['long_output']
        realm = brok_data['realm']
        event_since = get_datetime_with_local_time_zone(creation_date)
        event_hard_since = event_since if state_type == STATE_TYPE.HARD else None
        host_name = brok_data['host_name']
        check_name = brok_data['service_description'] if item_type == ITEM_TYPE.CHECK else ''
        name = u'%s/%s' % (host_name, check_name) if item_type == ITEM_TYPE.CHECK else brok_data['host_name']
        
        event = Event(
            item_uuid=item_uuid,
            item_type=item_type,
            state_id=state_id,
            state_type=state_type,
            state_validity_period=state_validity_period,
            flapping=flapping,
            acknowledged=acknowledged,
            acknowledge_id=acknowledge_id,
            downtime=downtime,
            partial_flapping=partial_flapping,
            partial_ack=partial_ack,
            partial_dt=partial_dt,
            active_downtime_uuids=active_downtime_uuids,
            output=output,
            long_output=long_output,
            realm=realm,
            event_since=event_since,
            event_hard_since=event_hard_since,
            host_name=host_name,
            check_name=check_name,
            name=name,
        )
        
        return event
    
    
    @classmethod
    def from_database_entry(cls, _to_load):
        # type: (dict) -> Event
        event = Event(**_to_load)
        if event.event_since:
            event.event_since = get_datetime_from_utc_to_local_time_zone(event.event_since)
        if event.event_hard_since:
            event.event_hard_since = get_datetime_from_utc_to_local_time_zone(event.event_hard_since)
        
        if Event.reader_stats:
            Event.reader_stats.read_event_stat(event)
        return event
    
    
    def to_database_entry(self):
        database_entry = self.__dict__
        database_entry['_id'] = self.event_uuid
        return database_entry
    
    
    def get_state_expiration_time(self):
        if self.state_validity_period == NO_END_VALIDITY:
            return NO_END_VALIDITY
        return self.event_since + timedelta(seconds=self.state_validity_period)
