#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

import time
from datetime import timedelta, datetime
from collections import namedtuple
from shinken.log import logger

DaysWorked = namedtuple('DaysWorked', ['custom', 'first', 'last'])


class WorkHours(object):
    
    def __init__(self, enable, start, end, days_worked, default_rate, work_hours_rate):
        # Set the simple options
        self.enable = False if enable == '0' else True
        self.default_rate = default_rate
        self.work_hours_rate = int(work_hours_rate)
        self.days_worked = None
        self.start = self.init_work_hours(start)
        self.end = self.init_work_hours(end)
        
        # Then compute the days_worked
        self.init_days_worked(days_worked)
    
    
    def init_work_hours(self, _time):
        # The work hours are set in format HH:MM in the module. We split it and create a timedelta from this data
        if not _time:
            return None
        _time = _time.split(':')
        if len(_time) != 2:
            logger.error('The work_hours format seem malformed [%s]. Must be "HH:MM"' % _time)
            return None
        try:
            _time = [int(t) for t in _time]
        except ValueError:
            logger.error('The work_hours format seem malformed [%s]. Must be "HH:MM"' % _time)
            return None
        
        return timedelta(hours=_time[0], minutes=_time[1], seconds=0, microseconds=0)
    
    
    def init_days_worked(self, days_worked):
        # The days_worked are set in format "X-Y" where X is the first day of the week and Y the last. The day start on monday (0) and finis sunday (6)
        # If there is no days_worked set, we use all the days
        if not days_worked:
            self.days_worked = DaysWorked(False, 0, 6)
            return
        
        # Split the option and parse to int
        days = days_worked.split('-')
        if len(days) != 2:
            self.days_worked = DaysWorked(False, 0, 6)
            raise SyntaxError('The days_worked format seem malformed [%s]. Must be "X-Y"' % days_worked)
        
        # Parse to int and set it
        try:
            days = [int(day) for day in days]
        except ValueError:
            self.days_worked = DaysWorked(False, 0, 6)
            raise SyntaxError('The days_worked format seem malformed [%s]. Must be "X-Y"' % days_worked)
        
        self.days_worked = DaysWorked(True, days[0], days[1])
    
    
    def is_enabled(self):
        return self.enable
    
    
    def is_correct(self):
        if not self.enable:
            return True
        
        if not (self.start and self.end and self.work_hours_rate):
            raise AttributeError('An option is missing : work_hours_start [%s] work_hours_end [%s] rate_workings_hours [%s].' % (self.start, self.end, self.work_hours_rate))
        
        if self.end <= self.start:
            raise AttributeError('The work_hours end before start : work_hours_start [%s] work_hours_end [%s].' % (self.start, self.end))
        
        if self.days_worked.custom:
            if not ((0 <= self.days_worked.first <= 6) and (0 <= self.days_worked.last <= 6)):
                raise AttributeError('The days_worked are out of range (0-6) : days_worked [%s-%s].' % (self.days_worked.first, self.days_worked.last))
            
            if self.days_worked.first > self.days_worked.last:
                raise AttributeError('The days_worked end before start : days_worked [%s-%s].' % (self.days_worked.first, self.days_worked.last))
    
    
    def work_hours_applied(self, actual_time):
        start, end = self.get_computed_work_hours()
        return start <= actual_time <= end
    
    
    def days_worked_applied(self, actual_day):
        if self.days_worked.custom:
            return self.days_worked.first <= actual_day <= self.days_worked.last
        return True
    
    
    def get_rate(self):
        if self.enable:
            actual_time = datetime.fromtimestamp(time.time())
            actual_day = actual_time.timetuple().tm_wday
            if self.days_worked_applied(actual_day) and self.work_hours_applied(actual_time):
                return self.work_hours_rate
        return self.default_rate
    
    
    def get_computed_work_hours(self):
        today = datetime.today()
        today = today.replace(hour=0, minute=0, second=0, microsecond=0)
        start = today + self.start
        end = today + self.end
        return start, end
