#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#     Gabes Jean, naparuba@gmail.com
#     Gerhard Lausser, Gerhard.Lausser@consol.de
#     Gregory Starck, g.starck@gmail.com
#     Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

# Must be imported before any thread creation in python 2.7
import _strptime  # noqa
import datetime
import errno
import glob
import hashlib
import inspect
import json
import logging
import os
import random
import re
import shutil
import signal
import subprocess
import sys
import threading
import time
import traceback
from ctypes import c_bool
from multiprocessing import Value
from multiprocessing import active_children
from operator import add
from subprocess import Popen, PIPE
from threading import RLock
from traceback import format_exc

import select
from multiprocessing.managers import SyncManager, Token, Server, util  # noqa => this are not in __all__ but we need them to overide multiprocessing lib

# noinspection PyUnresolvedReferences
import shinken.aaaa_monkey_patch_thread_name
from shinken.basesubprocess import LookAtMyFatherThread
from shinken.exceptions.system import NotEnoughMemory
# Hack to allow old modules to be loaded by daemons (because we did move module imports)
from shinken.modules.base_module import basemodule
from shinken.modules.base_module import brokermodule
from shinken.modules.base_module import brokermoduleworker
from shinken.subprocess_helper.after_fork_cleanup import after_fork_cleanup
from shinken.thread_helper import async_call
from shinken.util import start_malloc_trim_thread, get_day, mem_wait_for_fork_possible, force_memory_trimming
from shinkensolutions.http_helper import http_get, HTTP_ERRORS
from shinkensolutions.localinstall import VERSION, get_shinken_current_version_and_patch
from shinkensolutions.os_helper import get_cur_user_name, get_cur_group_name, get_user_id_from_name, get_group_id_from_name, get_all_group, set_ownership
from shinkensolutions.system_tools import is_regular_file
from .arbiter_trace_manager import ArbiterTraceManager
from .compat import ConfigParser, StringIO
from .configuration_incarnation import ConfigurationIncarnation
from .daemoninfo import daemon_info
from .http_daemon import HTTPDaemon, InvalidWorkDir, http_daemon_set_daemon_inst
from .log import logger, get_chapter_string, get_section_string, WARNING as LOG_WARNING, LoggerFactory
from .misc.type_hint import TYPE_CHECKING
from .modulesctx import modulesctx
from .modulesmanager import ModulesManager
from .property import StringProp, BoolProp, PathProp, ConfigPathProp, IntegerProp, LogLevelProp
from .runtime_stats.memory_stats import memory_stats
from .runtime_stats.threads_dumper import thread_dumper, WatchDogThreadDumper, watchdog_fatal_status
from .safepickle import serialization_security_container
from .vmware_stats import vmware_stats_reader

if TYPE_CHECKING:
    from .misc.type_hint import Any, Dict, List, Number, Optional, Union

# Compatibility hook, these submodules have moved to shinken/modules/base_module
sys.modules.update({'shinken.basemodule': basemodule, 'shinken.brokermodule': brokermodule, 'shinken.brokermoduleworker': brokermoduleworker})

raw_logger = LoggerFactory.get_logger()
logger_configuration = raw_logger.get_sub_part(u'CONFIGURATION')
logger_about_arbiters = logger_configuration.get_sub_part(u'ARBITERS')


def get_resource_lib():
    # On unix, try to raise system resources to the max (unlimited if possible)
    try:
        import resource
    except ImportError:
        resource = None
    return resource


# DEBUG file should be rotating over 5 days
DEBUG_FILE_ROTATING_DAY_LIMIT = 5

if os.name != u'nt':
    _PATH_HTTP_ERROR_STAT = r'/var/lib/shinken/http_error_%s_%s.json'
    CONTEXT_FILE_PATH = r'/var/lib/shinken/context.json'
else:  # specific path for windows
    _PATH_HTTP_ERROR_STAT = r'c:\shinken/http_error_%s_%s.json'
    CONTEXT_FILE_PATH = r'c:\shinken\var\context.json'

DEFAULT_SHINKEN_VERSION = u'(unknown)'

# see SEF-6068 First call of time.strptime is not thread safe, so I call here first before call in thread
time.strptime('01 2020', '%j %Y')

SYSTEM_CHAPTER = get_chapter_string(u'SYSTEM')
CHAPTER_MODULES = get_chapter_string(u'MODULES')
CONFIGURATION_CHAPTER = get_chapter_string(u'CONFIGURATION')
CONFIGURATION_UPDATE_CHAPTER = u'%s %s' % (CONFIGURATION_CHAPTER, get_chapter_string(u'CONFIGURATION'))

raw_logger = LoggerFactory.get_logger()
logger_configuration = raw_logger.get_sub_part(u'CONFIGURATION')

# #########################   DAEMON PART    ###############################
# The standard I/O file descriptors are redirected to /dev/null by default.
REDIRECT_TO = getattr(os, u'devnull', u'/dev/null')

UMASK = 0o27

_MODULES_STR = get_chapter_string(u'MODULES')

DATA_TO_CLEAN_AFTER_FORK = [
    u'conf',
    u'_realms_inventory',
    u'_realms_inventory_differences',
    u'known_realms',
    u'_realms_inventory_lock',
    u'http_errors_count',
    u'broks_lock',
    u'local_module_stats',
    u'broks',
    u'_tmp_bucket',
    u'external_module_broks',
    u'modules_manager',
    u'arbiter_broks',
    u'receivers',
    u'_realms_inventory_differences',
    u'pollers',
    u'check_ntp_thread',
    u'schedulers',
    u'http_daemon',
    u'satellite_lock',
    u'known_realms',
    u'broks_internal_raised',
    u'new_conf',
    u'cur_conf',
    u'abort',
    u'arbiter_broks_lock',
    u'avg_brok_send_speed',
    u'external_commands_lock',
]


class InvalidPidFile(Exception):
    pass


class Tee(object):
    def __init__(self, fd1, fd2):
        self.fd1 = fd1
        self.fd2 = fd2
    
    
    def write(self, data):
        os.write(self.fd1, data)
        os.write(self.fd2, data)
    
    
    def flush(self):
        # We don't buffer, nothing to do here
        pass


u""" Interface for Inter satellites communications """


class Interface(object):
    RAW_STATS_API_VERSION = u'2.12'
    # The API_VERSION will be use in route. Must be use with a route prefix
    API_VERSION = 0
    
    
    #  'app' is to be set to the owner of this interface.
    def __init__(self, app, route_prefix=u'', default_lock=True, public_api=False):
        self._arbiter_traces = []
        self._arbiter_traces_lock = RLock()  # modify/read _arbiter_traces list is not thread safe
        self.__satellites_url = []
        self.app = app
        self._static_context = None
        self.running_id = u'%d.%d' % (time.time() % 1000, random.random() * 100000)
        self.route_prefix = u'/' + route_prefix if route_prefix else u''
        self.default_lock = default_lock
        self.public_api = public_api
    
    
    doc = u'Get all satellites I am configured to talk to'
    
    
    def get_satellites(self):
        res = self.app.get_satellite_connections()
        return res
    
    
    get_satellites.doc = doc
    get_satellites.need_lock = False
    
    doc = u'Return a connection status from my network location to all my satellites.'
    
    
    @staticmethod
    @async_call
    def _check_satellites_connexion(satellite, timeout):
        status = u''  # noqa => for doc => will be: ok | timeout | error
        try:
            uri = u'%sping' % satellite[u'uri']
            html = http_get(uri, timeout=int(timeout))
            j = json.loads(html)
            if j == u'pong':
                status = u'ok'
            else:
                status = u'error'
        except (HTTP_ERRORS,) as err:
            code = u''
            update = u''
            if getattr(err, u'code', u''):
                code = u'HTTP ERROR %s : ' % str(err.code)
                if err.code == 404:
                    update = u' Please update.'
            if getattr(err, u'reason', u''):
                error = str(err.reason)
            else:
                error = str(err)
            status = u'%s%s%s' % (code, error, update)
        except Exception as ex:
            status = str(ex)
        satellite[u'status'] = status
    
    
    def check_satellites_connexion(self, timeout):
        start_time = time.time()
        max_allowed_time = start_time + int(timeout)
        if getattr(self.app, u'passive', False) is True:
            return []
        reply_list = []
        thread_list = []
        for satellite in self.app.get_satellite_connections():
            satellite[u'status'] = u'timed out'
            reply_list.append(satellite)
            thread_list.append(self._check_satellites_connexion(satellite, timeout))
        for thread in thread_list:
            wait_time = max_allowed_time - time.time()
            if wait_time <= 0:
                break
            thread.join(wait_time)
        return reply_list
    
    
    check_satellites_connexion.doc = doc
    check_satellites_connexion.need_lock = False
    
    doc = u'Test the connection to the daemon. Returns: pong'
    
    
    def ping(self):
        return u'pong'
    
    
    ping.need_lock = False
    ping.doc = doc
    
    # DEPRECATED: to remove after migrating all old pollers, to switch to the new name
    doc = u'Get the current running id of the daemon (scheduler)'
    
    
    def get_running_id(self):
        return self.get_daemon_incarnation()
    
    
    get_running_id.need_lock = False
    get_running_id.doc = doc
    
    # DEPRECATED: to remove after migrating all old pollers, to switch to the new name
    doc = u'Get the current daemon incarnation number, to know if it did change since we last talk to it'
    
    
    def get_daemon_incarnation(self):
        return self.running_id
    
    
    get_daemon_incarnation.need_lock = False
    get_daemon_incarnation.doc = doc
    
    doc = u'Send a new configuration to the daemon (internal)'
    
    
    def put_conf(self, conf):
        
        arbiter_trace = conf.get(u'arbiter_trace', None)
        if arbiter_trace is None:
            logger.error(u'Incompatible daemon version : Your Arbiter daemon is too old and do not manage arbiter trace. Refusing this configuration.')
            return
        
        self.app.set_arbiter_trace(arbiter_trace)
        self.app.arbiter_version = arbiter_trace.get(u'version', u'')
        if self.app.arbiter_version != self.get_context()[u'current_version']:
            logger.error(u'Incompatible daemon version : Your Arbiter daemon is in version [%s] while this daemon is in version [%s].' % (self.app.arbiter_version, self.get_context()['current_version']))
        
        daemon_name = conf.get(u'name', None)  # maybe it's an old arbiter, don't crash for it
        if daemon_name is not None:
            self.app.set_daemon_name(daemon_name)
        
        # We need the configuration incarnation now to directly answer to the arbiter about what we are managing (or will manage)
        configuration_incarnation = conf.get(u'configuration_incarnation', None)
        if configuration_incarnation is None:
            logger.error(u'The configuration is missing the configuration incarnation information. Your arbiter daemon must be outdated or unpatched. Cannot load this configuration.')
            return
        # Save it in the daemon
        self.app.set_configuration_incarnation(configuration_incarnation)
        
        self.app.already_have_conf = True
        self.app.new_conf = conf
        self.app.deactivated_by_arbiter = False
    
    
    put_conf.method = u'POST'
    put_conf.doc = doc
    put_conf.need_lock = False
    put_conf.display_name = u'Configuration reception from an Arbiter server'
    
    doc = u'Notification sent by the main arbiter (internal)'
    
    
    def arbiter_traces_register(self, arbiter_trace):
        arbiter_trace_updated = self.app.set_arbiter_trace(arbiter_trace)
        return arbiter_trace_updated
    
    
    arbiter_traces_register.method = u'POST'
    arbiter_traces_register.doc = doc
    arbiter_traces_register.need_lock = False
    arbiter_traces_register.display_name = u'Ping reception from an Arbiter server'
    
    doc = u'Ask the daemon to wait a new conf'
    
    
    def wait_new_conf(self):
        self.app.wait_new_conf()
    
    
    wait_new_conf.need_lock = False
    wait_new_conf.doc = doc
    
    doc = u'Does the daemon got an active configuration'
    
    
    def have_conf(self):
        return getattr(self.app, u'cur_conf', None) is not None
    
    
    have_conf.need_lock = False
    have_conf.doc = doc
    
    doc = u'Set the current log level in [NOTSET, DEBUG, INFO, WARNING, ERROR, CRITICAL, UNKNOWN]'
    
    
    def set_log_level(self, loglevel):
        return logger.setLevel(loglevel)
    
    
    set_log_level.doc = doc
    
    doc = u'Get the current log level in [NOTSET, DEBUG, INFO, WARNING, ERROR, CRITICAL, UNKNOWN]'
    
    
    def get_log_level(self):
        return {logging.NOTSET: u'NOTSET', logging.DEBUG: u'DEBUG',
                logging.INFO  : u'INFO', logging.WARNING: u'WARNING',
                logging.ERROR : u'ERROR', logging.CRITICAL: u'CRITICAL'}.get(logger.level, u'UNKNOWN')
    
    
    get_log_level.doc = doc
    get_log_level.need_lock = False
    
    
    def get_log_info(self):
        all_logger_info = []
        loggers_info = logger.loggers_info
        if loggers_info:
            raw_all_logger_info = loggers_info.get_all_logger_info()
            
            for i in raw_all_logger_info.values():
                if not i:
                    # the default logger is skip
                    continue
                all_logger_info.append({
                    u'id'    : i[u'id'],
                    u'name'  : i[u'display_name'],
                    u'enable': i[u'enable']
                })
            
            all_logger_info.sort(key=lambda j: j[u'name'])
        return all_logger_info
    
    
    get_log_info.doc = u'List all the loggers'
    
    
    def set_log_enable(self, logger_id, enable):
        loggers_info = logger.loggers_info
        if not loggers_info:
            return {u'message': u'data not available'}
        raw_all_logger_info = loggers_info.get_all_logger_info()
        
        logger_info = raw_all_logger_info.get(logger_id, None)
        if logger_info is None:
            return self.app.abort(400, u'logger id [%s] is not known' % logger_id)
        
        enable = enable == u'1'
        loggers_info.set_logger_enable(logger_id, enable)
        
        return {u'message': u'logger %s is %s' % (logger_info[u'display_name'], u'enabled' if enable else u'disabled')}
    
    
    set_log_enable.doc = u'enable/disable a logger'
    
    doc = u'List the methods available on the daemon'
    
    
    def api(self):
        return self.app.http_daemon.registered_fun_names
    
    
    api.doc = doc
    
    
    def get_lock(self):
        return u'OK'
    
    
    get_lock.doc = u'UNDOCUMENTED'
    
    
    def memory_stats(self, expanded=False, password=''):
        if hashlib.sha256(password).hexdigest() == 'c2a450753e8f2ad51b3d87a65d7d652198e6b19ed84701f529b2d43222866f2f':
            return memory_stats.query_memory_stats(expanded)
        else:
            self.app.abort(403)
    
    
    memory_stats.doc = u'do memory dump'
    memory_stats.need_lock = False
    doc = u'Return a list of arbiters who contacted the daemon'
    
    
    def arbiter_traces_get(self):
        self.app.arbiter_traces_cleanup()
        return self.app.get_up_to_date_arbiter_traces()
    
    
    arbiter_traces_get.doc = doc
    arbiter_traces_get.need_lock = False
    
    doc = u'List the api methods and their parameters'
    
    
    def api_full(self):
        res = {}
        for (file_name, f) in self.app.http_daemon.registered_fun.iteritems():
            fclean = file_name.replace(u'_', u'-')
            arg_spec = inspect.getargspec(f)
            args = [a for a in arg_spec.args if a != u'self']
            defaults = self.app.http_daemon.registered_fun_defaults.get(file_name, {})
            e = {}
            # Get a string about the args and co
            _s_undef_args = u', '.join([a for a in args if a not in defaults])
            _s_def_args = u', '.join([u'%s=%s' % (k, v) for (k, v) in defaults.iteritems()])
            _s_args = u''
            if _s_undef_args:
                _s_args += _s_undef_args
            if _s_def_args:
                _s_args += u', ' + _s_def_args
            e[u'proto'] = u'%s(%s)' % (fclean, _s_args)
            e[u'need_lock'] = getattr(f, u'need_lock', True)
            e[u'method'] = getattr(f, u'method', u'GET').upper()
            e[u'encode'] = getattr(f, u'encode', u'json')
            doc = getattr(f, u'doc', u'')
            if doc:
                e[u'doc'] = doc
            res[fclean] = e
        return res
    
    
    api.doc = doc
    
    
    def _get_module_stats(self, m_manager, param, module_wanted=None):
        if module_wanted is None or len(module_wanted) == 0:
            module_wanted = []
            actual_module = None
        else:
            actual_module = module_wanted[0]
        
        module_stats = {}
        app = self.app  # type: Daemon
        if m_manager is None:
            logger.warn(u'_get_module_stats module manager is not loaded.')
            return {}
        if app.interrupted:
            logger.warn(u'_get_module_stats module manager not available as daemon is stopping.')
            return {}
        for inst in m_manager.get_all_instances():
            f = getattr(inst, u'get_raw_stats', None)
            if f and (not actual_module or actual_module == inst.get_name()):
                try:
                    arg_spec = inspect.getargspec(f)
                    if len(arg_spec.args) > 2:
                        inst_stats = inst.get_raw_stats(param, module_wanted[1:])
                    elif len(arg_spec.args) > 1:
                        inst_stats = inst.get_raw_stats(param)
                    else:
                        inst_stats = inst.get_raw_stats()
                    module_name = inst.get_name()
                    module_type = inst.properties.get(u'type', u'unknown')
                    if module_type not in module_stats:
                        module_stats[module_type] = {}
                    module_stats[module_type][module_name] = inst_stats
                except Exception as err:
                    logger.error(u'Calling module %s get_raw_stats fail: %s' % (inst.get_name(), str(err)))
            else:
                logger.debug(u'_get_module_stats module [%s] has not the get_raw_stats methode.' % inst.get_name())
        return module_stats
    
    
    doc = u'Which modules are running'
    
    
    def get_module_states(self):
        application = self.app
        mod_manager = application.modules_manager
        return mod_manager.get_modules_states()
    
    
    get_module_states.doc = doc
    get_module_states.need_lock = False
    
    doc = u'Which version is running the daemon'
    
    
    def get_context(self):
        # type: () -> Dict
        return self.app.get_context()
    
    
    get_context.doc = doc
    get_context.need_lock = False


# Specific calls for IStats, as we don't want all interface to erase us
class IStatsInterface(Interface):
    
    def get_raw_stats(self, param=u'', module=u''):
        # type: (unicode, unicode) -> Dict[unicode, Any]
        
        module_wanted = module.split(u'.')
        # If we have a deadlock, the tests won't look at anything else as we are totally broken
        dead_lock_entry = watchdog_fatal_status.get_status()
        # If there are security issues, give them
        serialization_security_errors = serialization_security_container.get_errors_export()
        
        try:
            if not self.app.have_configuration:
                return {
                    u'api_version'                  : Interface.RAW_STATS_API_VERSION,
                    u'have_conf'                    : False,
                    u'arbiter_version'              : getattr(self.app, u'arbiter_version', None),
                    u'daemon_type'                  : getattr(self.app, u'daemon_type', None),
                    u'daemon_version'               : self.get_context()[u'current_version'],
                    u'deactivated_by_arbiter'       : getattr(self.app, u'deactivated_by_arbiter', False),
                    u'error_count'                  : self._get_error_count(),
                    u'dead_lock'                    : dead_lock_entry,
                    u'serialization_security_errors': serialization_security_errors,
                }
            if self.app.interrupted:
                return {
                    u'api_version'                  : Interface.RAW_STATS_API_VERSION,
                    u'is_interrupted'               : True,
                    u'have_conf'                    : False,
                    u'arbiter_version'              : getattr(self.app, u'arbiter_version', None),
                    u'daemon_type'                  : getattr(self.app, u'daemon_type', None),
                    u'daemon_version'               : self.get_context()[u'current_version'],
                    u'deactivated_by_arbiter'       : getattr(self.app, u'deactivated_by_arbiter', False),
                    u'error_count'                  : self._get_error_count(),
                    u'dead_lock'                    : dead_lock_entry,
                    u'serialization_security_errors': serialization_security_errors,
                }
            
            stats_time = time.time()
            stats = {
                u'api_version'                  : Interface.RAW_STATS_API_VERSION,
                u'have_conf'                    : True,
                u'arbiter_version'              : self.app.arbiter_version,
                u'daemon_type'                  : self.app.daemon_type,
                u'daemon_version'               : self.get_context()[u'current_version'],
                u'modules_info'                 : self.app.modules_manager.get_modules_states(module_wanted)[u'modules'],
                u'logger_stats'                 : logger.get_stats(),
                u'deactivated_by_arbiter'       : getattr(self.app, u'deactivated_by_arbiter', False),
                u'error_count'                  : self._get_error_count(),
                u'dead_lock'                    : dead_lock_entry,
                u'serialization_security_errors': serialization_security_errors,
            }
            logger.log_perf(stats_time, u'GET_RAW_STATS', u'Broker modules stats requested', min_time=1, warn_time=1.8)
            
            vmware_stats_time = time.time()
            stats.update(vmware_stats_reader.get_stats())
            logger.log_perf(vmware_stats_time, u'GET_RAW_STATS', u'VMware stats requested', min_time=1, warn_time=1.8)
            
            daemon_get_raw_stats_time = time.time()
            stats.update(self._daemon_get_raw_stats(param, module_wanted))
            logger.log_perf(daemon_get_raw_stats_time, u'GET_RAW_STATS', u'Broker stats requested', min_time=1, warn_time=1.8)
        
        except Exception as e:
            log = LoggerFactory.get_logger(u'DAEMON STATS')
            log.warning(u'Fail to get daemon stats')
            log.print_stack(level=LOG_WARNING)
            stats = {
                u'have_conf'                    : False,
                u'have_fail'                    : True,
                u'error_message'                : str(e),
                u'stacktrace'                   : traceback.format_exc().splitlines(),
                u'dead_lock'                    : dead_lock_entry,
                u'serialization_security_errors': serialization_security_errors,
            }
        return stats
    
    
    get_raw_stats.doc = u'get stats of the daemon'
    get_raw_stats.need_lock = False
    
    
    def _daemon_get_raw_stats(self, param=u'', module_wanted=None):
        # type: (unicode, List) -> Dict[unicode, Any]
        raise NotImplementedError()
    
    
    def _get_error_count(self):
        http_error_count = getattr(self.app, u'http_errors_count', None)
        if http_error_count:
            return reduce(add, http_error_count.values())
        return 0


DEFAULT_WORK_DIR = u'/var/run/shinken/'
DEFAULT_LIB_DIR = u'/var/lib/shinken/'


class Daemon(object):
    properties = {
        # workdir is relative to $(dirname "$0"/..)
        # where "$0" is the path of the file being executed,
        # in python normally known as:
        #
        #  os.path.join( os.getcwd(), sys.argv[0] )  # noqa
        #
        # as returned once the daemon is started.
        u'workdir'                  : PathProp(default=DEFAULT_WORK_DIR),
        u'modules_dir'              : PathProp(default=os.path.join(DEFAULT_LIB_DIR, u'modules')),
        u'host'                     : StringProp(default=u'0.0.0.0'),
        u'user'                     : StringProp(default=get_cur_user_name()),
        u'group'                    : StringProp(default=get_cur_group_name()),
        u'use_ssl'                  : BoolProp(default=u'0'),
        u'server_key'               : StringProp(default=u'etc/certs/server.key'),
        u'ca_cert'                  : StringProp(default=u'etc/certs/ca.pem'),
        u'server_cert'              : StringProp(default=u'etc/certs/server.cert'),
        u'use_local_log'            : BoolProp(default=u'1'),
        u'log_level'                : LogLevelProp(default=u'WARNING'),
        u'hard_ssl_name_check'      : BoolProp(default=u'0'),
        u'idontcareaboutsecurity'   : BoolProp(default=u'0'),
        u'daemon_enabled'           : BoolProp(default=u'1'),
        u'spare'                    : BoolProp(default=u'0'),
        u'max_queue_size'           : IntegerProp(default=u'0'),
        u'daemon_thread_pool_size'  : IntegerProp(default=u'64'),
        u'http_backend'             : StringProp(default=u'auto'),
        u'max_file_descriptor_limit': IntegerProp(default=u'1024'),  # default linux value
    }
    
    
    def __init__(self, daemon_type, config_file, is_daemon, do_replace, debug, debug_file, daemon_id=0):
        if TYPE_CHECKING:
            self.workdir = u''
            self.modules_dir = u''
            self.host = u''
            self.port = 0
            self.user = u''
            self.group = u''
            self.use_ssl = False
            self.server_key = u''
            self.ca_cert = u''
            self.server_cert = u''
            self.use_local_log = False
            self.log_level = u''
            self.hard_ssl_name_check = False
            # noinspection SpellCheckingInspection
            self.idontcareaboutsecurity = False
            self.daemon_enabled = False
            self.spare = False
            self.max_queue_size = 0
            self.daemon_thread_pool_size = 0
            self.http_backend = u''
            self.max_file_descriptor_limit = 0
            self.modules_manager = None  # type: Optional[ModulesManager]
            self.pidfile = u''
            self.local_log = u''
            self.pid = 0
            self.http_thread = None  # type: Optional[threading.Thread]
            self.check_ntp_thread = None  # type: Optional[threading.Thread]
        
        after_fork_cleanup.register_top_instance(self)
        logger.set_name(daemon_type)
        self.check_shm()
        self.daemon_type = daemon_type
        self.daemon_is_requested_to_stop = Value(c_bool, False)
        self.name = daemon_type  # by default, no name so only the daemon type
        self.config_file = config_file
        self.is_daemon = is_daemon
        self.do_replace = do_replace
        self.daemon_info = daemon_info
        self.debug = debug
        self.debug_file = debug_file
        self.daemon_id = daemon_id
        self.interrupted = False
        self.http_errors_count = {0: 0}
        self._version = DEFAULT_SHINKEN_VERSION
        self._load_http_error_stat()
        # Track time
        now = time.time()
        self.program_start = now
        self.t_each_loop = now  # used to track system time change
        self.sleep_time = 0.0  # used to track the time we wait
        self.update_interval = 1  # in seconds
        
        self.http_daemon = None
        
        # Log init
        logger.load_obj(self)
        
        self.already_have_conf = False  # used to know if a conf have already been sent
        self.new_conf = None  # used by controller to push conf
        self.cur_conf = None
        self.deactivated_by_arbiter = False  # used to know if the arbiter asks us to wait a new conf
        
        # Flag to dump objects or not
        self.need_objects_dump = False
        
        # Keep a trace of the local_log file desc if needed
        self.local_log_fd = None
        
        # Put in queue some debug output we will raise
        # when we will be in daemon
        self.debug_output = []
        
        os.umask(UMASK)
        self.set_exit_handler()
        
        # Mane to display for the process list if possible
        self.daemon_display_name = u'NO-DISPLAY-NAME'
        self.arbiter_version = None
        
        self.update_interval = 1  # in seconds
        
        self._loop_number = 0
        
        self.is_daemonized = False
        
        self._http_interfaces = []
        self.log_already_registered = False
        
        self.configuration_incarnation = None  # type: Optional[ConfigurationIncarnation]
        self._configuration_incarnation_log_entry = u''  # entry to display in log, but at a good time
        self.have_configuration = False
        self.last_day_log_version = None
        self._daemon_full_version = None
        self._arbiter_trace_manager = ArbiterTraceManager()
        self._static_context = None
    
    
    def _increase_loop_number(self):
        self._loop_number += 1
    
    
    def _get_loop_number(self):
        return self._loop_number
    
    
    @staticmethod
    def _print_log_block_start():
        size = 100
        logger.info(u'|' + (u'-' * (size - 2)) + u'|')
    
    
    def _print_log_block_end(self):
        self._print_log_block_start()
    
    
    def print_log_block(self, text):
        size = 100
        self._print_log_block_start()
        logger.info(u' ' * (size // 2) + text)
        self._print_log_block_start()
    
    
    @staticmethod
    def print_error_block(text, level=u'warning'):
        size = 100
        f_log = getattr(logger, level)
        f_log(u'|' + (u'-' * (size - 2)) + u'|')
        f_log(text)
        f_log(u'|' + (u'-' * (size - 2)) + u'|')
    
    
    def set_configuration_incarnation(self, configuration_incarnation):
        # type: (ConfigurationIncarnation) -> None
        # If it's the same, just do nothing
        if configuration_incarnation.is_equal(self.configuration_incarnation):
            return
        self.configuration_incarnation = configuration_incarnation
        # save a log entry that will be display after
        # NOTE: it's important to have the self.configuration_incarnation ASAP in the scheduler! load_conf is FAR TOO late!
        self._configuration_incarnation_log_entry = u'The arbiter send us a new configuration: %s' % self.configuration_incarnation
    
    
    def print_configuration_incarnation_log_entry_if_need(self, _logger):
        if self._configuration_incarnation_log_entry:
            _logger.info(self._configuration_incarnation_log_entry)
            self._configuration_incarnation_log_entry = u''
    
    
    def print_daemon_start(self):
        self.print_log_block(u'%s is starting' % self.daemon_type)
    
    
    @staticmethod
    def set_daemon_name(daemon_name):
        # type: (unicode) -> None
        logger.set_name(daemon_name)
    
    
    def _add_http_error_count(self):
        hours_since_epoch = int(int(time.time()) / 3600)
        # Store HTTP errors counts, clean will be done elsewhere
        if hours_since_epoch not in self.http_errors_count:
            self.http_errors_count[hours_since_epoch] = 0
        self.http_errors_count[hours_since_epoch] += 1
    
    
    def _add_http_interface(self, interface):
        self._http_interfaces.append(interface)
    
    
    def _register_http_interfaces(self):
        for interface in self._http_interfaces:
            self.http_daemon.register(interface)
    
    
    def _load_http_error_stat(self):
        daemon_id = self.daemon_id
        try:
            file_path = _PATH_HTTP_ERROR_STAT % (self.daemon_type, daemon_id)
            if not os.path.exists(os.path.dirname(file_path)):
                os.makedirs(os.path.dirname(file_path))
            if os.path.isfile(file_path):
                self.http_errors_count = json.load(open(file_path, 'r'))
                # json store keys as str, but we want them as int
                to_del = self.http_errors_count.keys()
                for key in to_del:
                    self.http_errors_count[int(key)] = self.http_errors_count[key]
                    del self.http_errors_count[key]
            else:
                self.http_errors_count = {0: 0}
        except Exception as e:
            logger.warning(u'Previous http error stat cannot be loaded from file [%s]. \n%s' % (_PATH_HTTP_ERROR_STAT % (self.daemon_type, daemon_id), e))
    
    
    def _save_http_error_stat(self):
        daemon_id = self.daemon_id
        try:
            file_path = _PATH_HTTP_ERROR_STAT % (self.daemon_type, daemon_id)
            if not os.path.exists(os.path.dirname(file_path)):
                os.makedirs(os.path.dirname(file_path))
            json.dump(self.http_errors_count, open(file_path, 'w'))
            
            # daemon can be stopped before it change this user to shinken, so we ensure http_error_stat file have this owner to shinken
            if get_cur_user_name() == u'root':
                set_ownership(file_path)
            logger.debug(u'[%s] save the http error count %s in %s' % (self.daemon_type, self.http_errors_count, file_path))
        except:
            logger.warning(u'[%s] Save http error stat in file [%s] failed : [%s]' % (self.daemon_type, _PATH_HTTP_ERROR_STAT % (self.daemon_type, daemon_id), traceback.format_exc()))
    
    
    # At least, lose the local log file if needed
    def do_stop(self):
        self.daemon_is_requested_to_stop.value = True
        # Maybe the modules' manager is not even created!
        if getattr(self, u'modules_manager', None):
            # We save what we can but NOT for the scheduler because the current scheduler object is a dummy one and the old one has already done it!
            # noinspection SpellCheckingInspection
            if not hasattr(self, u'sched'):
                self.hook_point(u'save_retention')
            # And we quit
            self.modules_manager.stop_all()
        if self.http_daemon:
            # NOTE: the daemon shutdown is already releasing the daemon lock
            self.http_daemon.shutdown()
        self._save_http_error_stat()
    
    
    def release_lock(self):
        # Release the lock so the daemon can shut down without problem
        try:
            self.http_daemon.lock.release()
        except:
            pass
    
    
    def request_stop(self):
        self.unlink()
        self.do_stop()
        # Brok's facilities are no longer available simply print the message to STDOUT
        print(u'Stopping daemon. Exiting')
        sys.exit(0)
    
    
    def wait_new_conf(self):
        if not self.deactivated_by_arbiter:
            logger_configuration.info(u'Arbiter is asking to go in idle mode.')
        self.cur_conf = None
        self.already_have_conf = False
        self.deactivated_by_arbiter = True
    
    
    # Maybe this daemon is configured to NOT run, if so, bailout
    def look_for_early_exit(self):
        if not self.daemon_enabled:
            logger.info(u'This daemon is disabled in configuration. Bailing out')
            self.request_stop()
    
    
    def do_loop_turn(self):
        raise NotImplementedError()
    
    
    def daily_log_version(self):
        try:
            today = datetime.datetime.now().strftime(u'%d/%m/%Y')
        except Exception:
            return
        if today == self.last_day_log_version:
            return
        logger.info(u'Daemon version is : %s' % self.get_full_version())
        logger.info(u'Daemon start time=%d' % self.program_start)
        self.last_day_log_version = today
    
    
    # Main loop for nearly all daemon
    # the scheduler is not managed by it :'(
    def do_mainloop(self):
        while True:
            self.daily_log_version()
            # Note: we can be asked to stop even before the first loop, so handle stop at the beginning
            if self.interrupted:
                break
            self.do_loop_turn()
            self.sleep(0.01)
            if self.need_objects_dump:
                logger.debug(u'Dumping objects')
                self.need_objects_dump = False
        self.request_stop()
    
    
    def do_load_modules(self):
        all_was_started = self.modules_manager.load_and_init()
        loaded_modules = self.modules_manager.get_all_instances_name()
        if loaded_modules:
            logger.info(u'%s Modules that are loaded successfully: %s' % (CHAPTER_MODULES, u', '.join(loaded_modules)))
        if not all_was_started:
            logger.error(u'%s Some modules did failed to start' % CHAPTER_MODULES)
        return all_was_started
    
    
    # Dummy method for adding broker to this daemon
    def add(self, elt):
        pass
    
    
    def load_config_file(self):
        self.parse_config_file()
        if self.config_file is not None:
            # Some paths can be relatives. We must have a full path by taking
            # the config file by reference
            self.relative_paths_to_full(os.path.dirname(self.config_file))
    
    
    def load_modules_manager(self):
        self.modules_manager = ModulesManager(self.daemon_type, self.find_modules_path(), [], daemon_display_name=self.daemon_display_name)
        # Set the modules watchdogs
        # TODO: (to fix) Beware, the arbiter do not have the max_queue_size property
        self.modules_manager.set_max_queue_size(getattr(self, u'max_queue_size', 0))
        # And make the module manager able to create a sub-process Queue() manager
        self.modules_manager.load_manager_factory(self.get_queues_manager)
    
    
    def change_to_workdir(self):
        self.workdir = os.path.abspath(self.workdir)
        try:
            os.chdir(self.workdir)
        except Exception as e:
            raise InvalidWorkDir(e)
        self.debug_output.append(u'Successfully changed to workdir: %s' % self.workdir)
    
    
    def unlink(self):
        logger.debug(u'Unlinking %s' % self.pidfile)
        try:
            os.unlink(self.pidfile)
        except Exception as e:
            logger.error(u'Got an error unlinking our pidfile: %s' % e)
    
    
    # Look if we need a local log or not
    def register_local_log(self):
        # The arbiter doesn't have such attribute
        if not self.log_already_registered:
            if hasattr(self, u'use_local_log') and self.use_local_log:
                try:
                    # self.local_log_fd = self.log.register_local_log(self.local_log)
                    self.local_log_fd = logger.register_local_log(self.local_log)
                    self.log_already_registered = True
                except IOError as err:
                    logger.error(u"Opening the log file '%s' failed with '%s'" % (self.local_log, err))
                    sys.exit(2)
                logger.info(u"Using the local log file '%s'" % self.local_log)
    
    
    # Only on linux: Check for /dev/shm write access
    @staticmethod
    def check_shm():
        import stat
        shm_path = u'/dev/shm'
        if os.name == u'posix' and os.path.exists(shm_path):
            # We get the access rights, and we check them
            mode = stat.S_IMODE(os.lstat(shm_path)[stat.ST_MODE])
            if not mode & stat.S_IWUSR or not mode & stat.S_IRUSR:
                logger.critical(u'The directory %s is not writable or readable. Please make it read writable: %s' % (shm_path, shm_path))
                sys.exit(2)
    
    
    def __open_pidfile(self, write=False):
        # if problem on opening or creating file it'll be raised to the caller:
        try:
            p = os.path.abspath(self.pidfile)
            self.debug_output.append(u'Opening pid file: %s' % p)
            # Windows do not manage the rw+ mode, so we must open in read mode first, then reopen it write mode...
            if not write and os.path.exists(p):
                self.pid_fd = open(p, 'r+')
            else:  # If it doesn't exist too, we create it as void
                self.pid_fd = open(p, 'w+')
        except Exception as err:
            raise InvalidPidFile(err)
    
    
    # Check is a pid exists on the system.
    # NOTE: we try to detect with a kill -0 PID. But sometimes (very rare) the test can fail
    # to see that the process is MISSING, so double check with looking at
    # the /proc/PID directory. If the kill say the process does exist, but
    @staticmethod
    def __is_pid_exists(pid):
        if os.name == u'nt':
            raise NotImplemented()  # misuse
        
        try:
            os.kill(pid, 0)  # if no exception then the process exists
        except Exception:  # consider any exception as a stale pidfile.
            # this includes :
            #  * PermissionError when a process with same pid exists but is executed by another user.
            #  * ProcessLookupError: [Errno 3] No such process.
            return False
        
        # so here the process exists, but maybe the kill is wrong about it (bug) and we must double-check it
        if not os.path.exists(u'/proc/%s' % pid):
            # kernel bug case: the kernel have a ghost entry for this process. Clean
            # it with a kill -5 and say that the process does not exist in fact
            try:
                os.kill(pid, signal.SIGQUIT)
            except os.error:  # maybe the kernel did fix itself
                pass
            return False
        
        # kill -0 say the process exists, /proc/PID too, so yes, the process exists
        return True
    
    
    # Check (in pidfile) if there isn't already a daemon running. If yes and do_replace: kill it.
    # Keep in self.pid_fd the File object to the pidfile. Will be used by write pid.
    def check_parallel_run(self):
        # TODO: other daemon run on nt
        if os.name == u'nt':
            logger.warning(u'The parallel daemon check is not available on nt')
            self.__open_pidfile(write=True)
            return
        
        # First open the pid file in open mode
        self.__open_pidfile()
        try:
            pid = int(self.pid_fd.readline().strip(u' \r\n'))
        except Exception as err:
            logger.info(u'Stale pidfile exists at %s (%s). Reusing it.' % (err, self.pidfile))
            return
        
        already_exists = self.__is_pid_exists(pid)
        if not already_exists:
            logger.info(u'Stale pidfile exists, Reusing it.')
            return
        
        if not self.do_replace:
            raise SystemExit(u'valid pidfile exists (pid=%s) and not forced to replace. Exiting.' % pid)
        
        self.debug_output.append(u'Replacing previous instance %d' % pid)
        try:
            os.kill(pid, signal.SIGQUIT)
        except os.error as err:
            if err.errno != errno.ESRCH:  # noqa => typing is lost here
                raise
        
        self.pid_fd.close()
        # TODO: give some time to wait that previous instance finishes?
        time.sleep(1)
        # we must also reopen the pid file in write mode
        # because the previous instance should have deleted it!!
        self.__open_pidfile(write=True)
    
    
    def thread_check_ntpd_service(self):
        try:
            while True:
                self._check_ntpd_service()
                time.sleep(3600)
        except Exception as err:
            ntp_logger = raw_logger.get_sub_part(u'NTP Thread')
            ntp_logger.error(u'Encountered an error %s' % err)
            ntp_logger.print_stack()
    
    
    @staticmethod
    def _check_ntpd_service():
        # Do not check ntp on Windows
        if os.name == u'nt':
            return
        ntp_logger = raw_logger.get_sub_part(u'NTP Thread')
        # noinspection SpellCheckingInspection
        cmd = u"""service ntpd status || service chronyd status"""
        p = Popen(cmd, stdout=PIPE, shell=True)
        p.wait()
        if p.returncode == 3:
            ntp_logger.warning(u'NTP service is disabled. This may cause some inconsistency. Please enable it.')
            return
        
        # noinspection SpellCheckingInspection
        cmd = u"""ntpstat || chronyc tracking"""
        p = Popen(cmd, stdout=PIPE, shell=True)
        p.wait()
        out = p.communicate()
        # * unspecified or unsynchronised => ntpq
        # * Not synchronised => chronyc
        # noinspection SpellCheckingInspection
        if u'unspecified' in out or u'unsynchronised' in out:
            # noinspection SpellCheckingInspection
            ntp_logger.warning(u'NTP service is enabled but it\'s unsynchronised. This may cause some inconsistency. Please check your ntp configuration.')
        if p.returncode != 0:
            # noinspection SpellCheckingInspection
            ntp_logger.warning(u'NTP service is enabled but it\'s unsynchronised. This may cause some inconsistency. Please check your ntp configuration.')
    
    
    def write_pid(self, pid=None):
        if pid is None:
            pid = os.getpid()
        self.pid_fd.seek(0)
        self.pid_fd.truncate()
        self.pid_fd.write("%d" % pid)
        self.pid_fd.close()
        del self.pid_fd  # no longer needed
    
    
    def get_version(self):
        if self._version != DEFAULT_SHINKEN_VERSION:
            return self._version
        try:
            with open(CONTEXT_FILE_PATH, 'r') as f:
                buf = f.read()
                ctx = json.loads(buf)
                self._version = ctx[u'current_version']
                return self._version
        except:
            context_logger = raw_logger.get_sub_part(u'GET CONTEXT')
            context_logger.error(u'cannot open file [%s]' % CONTEXT_FILE_PATH)
            return DEFAULT_SHINKEN_VERSION
    
    
    # Go in "daemon" mode: close unused fds, redirect stdout/err,
    # chdir, umask, fork-set sid-fork-write pid
    def daemonize(self, skip_close_fds=None, close_logs_in_stdout_and_stderr=False):
        if close_logs_in_stdout_and_stderr:
            # close the stdout and the stderr and redirect them to /dev/null to be sure nobody uses it
            dev_null_fd = os.open(REDIRECT_TO, os.O_RDWR)
            os.dup2(dev_null_fd, 1)
            os.dup2(dev_null_fd, 2)
            os.close(dev_null_fd)
        else:
            self.init_debug_logger(skip_close_fds)
        
        # Now the fork/set sid/fork..
        try:
            pid = os.fork()
        except OSError as e:
            raise Exception(u'%s [%d]' % (e.strerror, e.errno))
        if pid != 0:
            # In the father: we check if our child exit correctly
            # it has to write the pid of our future little child...
            # noinspection PyUnusedLocal
            def do_exit(sig, frame):
                logger.error(u'Timeout waiting child whereas it should have quickly returned ; something weird happened')
                os.kill(pid, 9)
                sys.exit(1)
            
            
            # wait the child process to check its return status:
            signal.signal(signal.SIGALRM, do_exit)
            signal.alarm(3)  # forking & writing a pid in a file should be rather quick...
            # if it's not then something wrong can already be on the way so let's wait max 3 secs here.
            pid, status = os.waitpid(pid, 0)
            if status != 0:
                logger.error(u'Something weird happened with/during second fork: status= %s' % status)
            # noinspection PyUnresolvedReferences, PyProtectedMember
            os._exit(status != 0)
        
        # halfway to daemonize..
        os.setsid()
        try:
            pid = os.fork()
        except OSError as e:
            raise Exception(u'%s [%d]' % (e.strerror, e.errno))
        if pid != 0:
            # we are the last step and the real daemon is actually correctly created at least.
            # we have still the last responsibility to write the pid of the daemon itself.
            self.write_pid(pid)
            os._exit(0)  # noqa
        
        self.pid_fd.close()
        del self.pid_fd
        self.pid = os.getpid()
        self.debug_output.append(u'We are now fully daemonized with pid=%d' % self.pid)
        # We can now output some previously silenced debug output
        # noinspection SpellCheckingInspection
        logger.debug(u'Printing stored debug messages prior to our transformation into a daemon.')
        for s in self.debug_output:
            logger.debug(s)
        del self.debug_output
        start_logger = raw_logger.get_sub_part(u'START-DAEMON')
        start_logger.info(u'The daemon in version %s is now started as a daemon (detached from any shell) with pid=%d' % (self.get_full_version(), self.pid))
        start_logger.info(u'Daemon start time=%d' % self.program_start)
    
    
    # We do not want to let old debug files on the server, so we will keep
    # only DEBUG_FILE_ROTATING_DAY_LIMIT (5) logs
    def _clean_old_debug_files(self):
        if not self.debug_file or not is_regular_file(self.debug_file):
            return
        
        # Clean the old debug file, prior to 2.06.00
        if os.path.exists(self.debug_file):
            try:
                os.unlink(self.debug_file)
            except IOError as err:
                logger.error(u'Cannot remove the old debug file: %s: %s' % (self.debug_file, err))
        
        now = int(time.time())
        debug_files_pattern = u'%s--*--*' % self.debug_file
        all_debug_files = glob.glob(debug_files_pattern)
        today = get_day(now)
        # Find the day we need to keep/clean.
        # NOTE: yes, not all days are 86400s long I know. But I don't care.
        time_limit = today - DEBUG_FILE_ROTATING_DAY_LIMIT * 86400
        for debug_file in all_debug_files:
            elements = debug_file.split(u'--')
            # the case seems impossible based on the pattern, but I'm not sure if a dev change the pattern
            # and forget to do a check here...
            # IMPORTANT: the pattern is a double -- because the date will have a simple -
            if len(elements) < 3:
                continue
            try:
                # Start from the end so even if the admin set --in a name, we don't care
                debug_file_epoch = int(elements[-2])
            except ValueError:  # someone did play with the file and try to play with us
                continue
            
            if debug_file_epoch < time_limit:
                try:
                    logger.info(u'Cleaning old debug file: %s' % debug_file)
                    os.unlink(debug_file)
                except IOError as err:
                    logger.error(u'Cannot remove the old debug file: %s: %s' % (debug_file, err))
    
    
    def init_debug_logger(self, skip_close_fds=None, tee=False):
        if skip_close_fds is None:
            # noinspection PyUnusedLocal
            skip_close_fds = tuple()
        
        # always try to clean old debug files, even if we are not in debug
        self._clean_old_debug_files()
        
        self.debug_output.append(u'Redirecting stdout and stderr as necessary..')
        if self.debug:
            now = int(time.time())
            file_pth = u'%s--%s--%s' % (self.debug_file, now, datetime.datetime.fromtimestamp(now).strftime(u'%Y-%m-%d-%H_%M'))
            # We are not daemonized yet, so our current user is root. Give the log file to shinken user.
            open(file_pth, "a").close()
            os.chown(file_pth, get_user_id_from_name(u'shinken'), get_group_id_from_name(u'shinken'))
            fd_temp = os.open(file_pth, os.O_CREAT | os.O_WRONLY | os.O_TRUNC)
            # NOTE: os.path.exists() return False on a symlink that link to no real file
            regular_file_exist = os.path.exists(self.debug_file)
            symlink_exists = os.path.islink(self.debug_file)
            if symlink_exists or regular_file_exist:
                os.unlink(self.debug_file)
            # We keep a symlink between the debug_file (static) -> time logged file
            os.symlink(file_pth, self.debug_file)
        else:
            fd_temp = os.open(REDIRECT_TO, os.O_RDWR)
        
        if tee:
            sys.stdout = Tee(fd_temp, 1)
            sys.stderr = Tee(fd_temp, 2)
        else:
            os.dup2(fd_temp, 1)  # standard output (1)
            os.dup2(fd_temp, 2)  # standard error (2)
        
        # PROTECT AGAINST FILE DESCRIPTOR 0
        # We do not need stdin anymore, close it to avoid polluting caller shell
        # the cherry.py server read data from file descriptor 0 so to not brok cherry.py we force /dev/null as file descriptor 0
        try:
            zero_fd = os.open(REDIRECT_TO, os.O_RDWR)
            os.dup2(zero_fd, 0)
        except OSError:
            pass
        logger.set_name(self.name)
    
    
    def compute_basic_display_name(self):
        self.daemon_display_name = u'shinken-%s' % self.name
        self.daemon_display_name = self.daemon_display_name.ljust(20)
    
    
    def get_queues_manager(self, requestor=u''):
        # type: (unicode) -> Optional[SyncManager]
        
        # If the manager is not started after 1min, dump threads every minute
        with WatchDogThreadDumper(u'QUEUE MANAGER %s' % requestor, 60, 60):
            return self._get_queues_manager(requestor=requestor)
    
    
    def _get_queues_manager(self, requestor=u''):
        # type: (unicode) -> Optional[SyncManager]
        
        # The Manager is a sub-process, so we must be sure it won't have a socket of your http server alive
        manager = SyncManager(('127.0.0.1', 0))
        father_pid = os.getpid()
        
        
        def _manager_init():  # When start the manager, close all http part so socket are clean
            
            self.release_lock()
            # Set the sub process name to an explicit value
            if requestor:
                sub_process_name = '%s [ %s - queue manager subprocess ]' % (self.daemon_display_name, requestor)
            else:
                sub_process_name = '%s [ - Queue manager subprocess ]' % self.daemon_display_name
            
            try:
                from setproctitle import setproctitle
                setproctitle(sub_process_name)
            except:
                pass
            
            # logger can have links to the daemon
            logger.unregister_all()
            logger.set_name(sub_process_name)
            self.do_after_fork_cleanup(None)
            
            # Allow the memory dump and close manager
            self._queues_manager_signal_handler()
            
            # For this process, to clean its memory to minimal
            start_malloc_trim_thread()
            
            # We must kill the manager if the daemon process goes down
            look_at_my_father_thread = LookAtMyFatherThread(father_pid, self.daemon_display_name, sub_process_name, loop_speed=0.5)
            look_at_my_father_thread.start_thread()
            logger.debug(u'Starting new manager with pid %s' % os.getpid())
        
        class ServerMemLeakPatched(Server):
            def serve_client(self, conn):
                # Handle requests from the proxies in a particular process/thread
                util.debug(u'starting server thread to service %r', threading.current_thread().name)
                
                recv = conn.recv
                send = conn.send
                id_to_obj = self.id_to_obj
                
                while not self.stop:
                    # Patch here : free reference between call
                    # noinspection PyUnusedLocal
                    result = __some_object = method_name = request = ident = args = kwargs = exposed = get_type_id = function_from_object = res = msg = fallback_func = typeid = rident = r_exposed = token = None
                    
                    try:
                        request = recv()
                        ident, method_name, args, kwargs = request
                        __some_object, exposed, get_type_id = id_to_obj[ident]
                        
                        if method_name not in exposed:
                            raise AttributeError(u'method %r of %r object is not in exposed=%r' % (method_name, type(__some_object), exposed))
                        
                        function_from_object = getattr(__some_object, method_name)
                        
                        try:
                            res = function_from_object(*args, **kwargs)
                        except Exception as e:
                            msg = (u'#ERROR', e)
                        else:
                            typeid = get_type_id and get_type_id.get(method_name, None)
                            if typeid:
                                rident, r_exposed = self.create(conn, typeid, res)
                                token = Token(typeid, self.address, rident)
                                msg = (u'#PROXY', (r_exposed, token))
                            else:
                                msg = (u'#RETURN', res)
                    
                    except AttributeError:
                        if method_name is None:
                            msg = (u'#TRACEBACK', format_exc())
                        else:
                            try:
                                fallback_func = self.fallback_mapping[method_name]
                                result = fallback_func(self, conn, ident, __some_object, *args, **kwargs)
                                msg = (u'#RETURN', result)
                            except Exception:
                                msg = (u'#TRACEBACK', format_exc())
                    except EOFError:
                        util.debug(u'got EOF -- exiting thread serving %r', threading.current_thread().name)
                        sys.exit(0)
                    except Exception:
                        msg = (u'#TRACEBACK', format_exc())
                    
                    try:
                        try:
                            send(msg)
                        except Exception:
                            send((u'#UNSERIALIZABLE', format_exc()))
                    except Exception as e:
                        util.info(u'exception in thread serving %r', threading.current_thread().name)
                        util.info(u' ... message was %r', msg)
                        util.info(u' ... exception was %r', e)
                        conn.close()
                        sys.exit(1)
        
        SyncManager._Server = ServerMemLeakPatched
        is_fork_possible = mem_wait_for_fork_possible(u'queue manager for %s' % requestor, fast_error=True)
        if not is_fork_possible:
            # Will be caught by create_queues => try_instance_init
            raise NotEnoughMemory(u'Cannot start the queue manager process for %s as there is not enough memory' % requestor)
        # Some multiprocessing lib got problems with start() that cannot take args so we must look at it before
        start_args = inspect.getargspec(manager.start)
        # start_args[0] will be ['self'] if old multiprocessing lib
        # and ['self', 'initializer', 'initargs'] in newer ones
        # note: windows do not like pickle http_daemon...
        if os.name != u'nt' and len(start_args[0]) > 1:
            manager.start(_manager_init)
        else:
            from multiprocessing import util as multiprocessing_util
            
            def _hook_manager(_dummy):
                _manager_init()
            
            
            multiprocessing_util.register_after_fork(self, _hook_manager)
            manager.start()
            # Keep this daemon in the http_daemon module
            to_del = None
            for (k, _some_object) in multiprocessing_util._afterfork_registry.iteritems():  # noqa
                if _some_object == self:
                    to_del = k
                    break
            if to_del:
                try:
                    logger.debug(u'[queue manager] deleting entry: %s' % str(to_del))
                    del multiprocessing_util._afterfork_registry[to_del]  # noqa
                except KeyError:
                    pass
        
        return manager
    
    
    def _queues_manager_signal_handler(self):
        # noinspection PyUnusedLocal
        def manage_signal(_sig, frame):
            logger.debug(u"I'm process %d and I received signal %s" % (os.getpid(), str(_sig)))
            if _sig == signal.SIGUSR1:  # if USR1, ask a memory dump
                memory_stats.dump_memory_full_memory_dump(self.name)
                memory_stats.print_memory_stats()
            else:  # Ok, really ask us to die :)
                os._exit(0)  # noqa
        
        
        if os.name == u'nt':
            try:
                import win32api
                win32api.SetConsoleCtrlHandler(manage_signal, True)
            except ImportError:
                version = u'.'.join(map(str, sys.version_info[:2]))
                raise Exception(u'pywin32 not installed for Python %s' % version)
        else:
            for sig in (signal.SIGTERM, signal.SIGINT, signal.SIGUSR1, signal.SIGUSR2):
                signal.signal(sig, manage_signal)
    
    
    def clean_all_data_in_sub_process(self, context):
        # type: (unicode) -> None
        for i in DATA_TO_CLEAN_AFTER_FORK:
            if hasattr(self, i):
                delattr(self, i)
        force_memory_trimming(context=context)
    
    
    @staticmethod
    def __log_system_limit(res_name, soft, hard, src):
        # type: (unicode,Union[unicode,Number],Union[unicode,Number],unicode) -> None
        logger.info(u'%s System resource %-25s is set to (soft:%-10s / hard:%-10s) %s' % (SYSTEM_CHAPTER, res_name, soft, hard, src))
    
    
    def __find_and_set_higher_system_limit(self, res, res_name):
        resource = get_resource_lib()
        
        # first try to get the system limit, if already unlimited (-1) then we are good :)
        soft, hard = resource.getrlimit(res)
        if soft == -1 and hard == -1:
            self.__log_system_limit(res_name, u'unlimited', u'unlimited', u'(set at system max values)')
            return
        # Ok not unlimited, maybe we can set unlimited?
        try:
            resource.setrlimit(res, (-1, -1))
            is_unlimited = True
        except ValueError:
            is_unlimited = False
        if is_unlimited:
            self.__log_system_limit(res_name, u'unlimited', u'unlimited', u'(set at system max values)')
            return
        # Ok maybe we cannot set unlimited, but we can try to increase it as much as we can
        can_still_increase = True
        v = hard
        if hard == -1:
            v = soft
        while can_still_increase:
            v *= 2
            try:
                logger.debug(u'%s Try to increase system limit %s to %s/%s' % (SYSTEM_CHAPTER, res_name, v, v))
                resource.setrlimit(res, (v, v))
            except ValueError:
                # We did find the max
                can_still_increase = False
        self.__log_system_limit(res_name, v, v, u'(set at system max values)')
    
    
    def __find_and_set_higher_file_descriptor_limit(self):
        resource = get_resource_lib()
        if not resource:
            logger.info(u'%s System resource package is not available, cannot increase system limits' % SYSTEM_CHAPTER)
            return
        
        self.__find_and_set_higher_system_limit(resource.RLIMIT_NOFILE, u'number of open files')
    
    
    def __set_number_processes_threads_limit(self):
        resource = get_resource_lib()
        if not resource:
            logger.info(u'%s System resource package is not available, cannot increase system limits' % SYSTEM_CHAPTER)
            return
        self.__find_and_set_higher_system_limit(resource.RLIMIT_NPROC, u'number of processes/threads')
    
    
    def __set_file_descriptor_limit(self):
        resource = get_resource_lib()
        if not resource:
            logger.info(u'%s System resource package is not available, cannot increase system limits' % SYSTEM_CHAPTER)
            return
        try:
            resource.setrlimit(resource.RLIMIT_NOFILE, (self.max_file_descriptor_limit, self.max_file_descriptor_limit))
            # HACK: need to hot patch the subprocess lib to limit it from os.close() max fd when we did limit them
            subprocess.MAXFD = self.max_file_descriptor_limit
            self.__log_system_limit(u'number of open files', self.max_file_descriptor_limit, self.max_file_descriptor_limit, u'(from parameter max_file_descriptor_limit in ini)')
        except ValueError:
            logger.warning(u'%s Cannot set system limit %s to the requested value %s' % (SYSTEM_CHAPTER, u'number of open files', self.max_file_descriptor_limit))
            return
    
    
    def do_daemon_init_and_start(self, close_logs_in_stdout_and_stderr=False):
        self.change_to_user_group()
        self.change_to_workdir()
        self.check_parallel_run()
        self._setup_http_daemon()
        
        # Setting log level
        logger.setLevel(self.log_level)
        # Force the debug level if the daemon is said to start with such level
        if self.debug:
            logger.setLevel(u'DEBUG')
        
        # Then start to log all in the local file if asked so
        self.register_local_log()
        self.compute_basic_display_name()
        if self.is_daemon:
            socket_fds = [sock.fileno() for sock in self.http_daemon.get_sockets()]
            # Do not close the local_log file too if it's open
            if self.local_log_fd:
                socket_fds.append(self.local_log_fd)
            
            socket_fds = tuple(socket_fds)
            self.daemonize(skip_close_fds=socket_fds, close_logs_in_stdout_and_stderr=close_logs_in_stdout_and_stderr)
            try:
                from setproctitle import setproctitle
                setproctitle('%s [ Main daemon ]' % self.daemon_display_name)
            except:
                pass
        else:
            self.write_pid()
        
        # Increase number of open files and launch process, to not be limited
        if self.max_file_descriptor_limit == 0:  # if there is a limit, try to increase to the maximum
            self.__find_and_set_higher_file_descriptor_limit()
        else:
            self.__set_file_descriptor_limit()
        
        # In all cases, increase the max number of sons
        self.__set_number_processes_threads_limit()
        
        # Now start the http_daemon thread
        self.http_thread = None
        # Directly acquire it, so the http_thread will wait for us
        self.http_daemon.lock.acquire()
        self.http_thread = threading.Thread(None, target=self.http_daemon_thread, name=u'http-thread')
        # Don't lock the main thread just because of the http thread
        self.http_thread.daemon = True
        self.http_thread.start()
        
        self.check_ntp_thread = threading.Thread(None, target=self.thread_check_ntpd_service, name=u'check-ntp-thread')
        # Don't lock the main thread just because of the ntp thread
        self.check_ntp_thread.daemon = True
        self.check_ntp_thread.start()
        
        start_malloc_trim_thread()
        self.is_daemonized = True
    
    
    def _setup_http_daemon(self):
        if hasattr(self, u'use_ssl'):  # 'common' daemon
            ssl_conf = self
        else:
            ssl_conf = getattr(self, u'conf', None)  # arbiter daemon..
        
        use_ssl = ssl_conf.use_ssl
        ca_cert = ssl_cert = ssl_key = u''
        
        # The SSL part
        if use_ssl:
            ssl_cert = os.path.abspath(str(ssl_conf.server_cert))
            if not os.path.exists(ssl_cert):
                logger.error(u'Error : the SSL certificate %s is missing (server_cert). Please fix it in your configuration' % ssl_cert)
                sys.exit(2)
            ca_cert = os.path.abspath(str(ssl_conf.ca_cert))
            logger.info(u'Using ssl ca cert file: %s' % ca_cert)
            ssl_key = os.path.abspath(str(ssl_conf.server_key))
            if not os.path.exists(ssl_key):
                logger.error(u'Error : the SSL key %s is missing (server_key). Please fix it in your configuration' % ssl_key)
                sys.exit(2)
            logger.info(u'Using ssl server cert/key files: %s/%s' % (ssl_cert, ssl_key))
            
            if ssl_conf.hard_ssl_name_check:
                logger.info(u'Enabling hard SSL server name verification')
        
        # Let's create the HTTPDaemon, it will be exec after
        self.http_daemon = HTTPDaemon(self.host, self.port, use_ssl, ca_cert, ssl_key, ssl_cert, ssl_conf.hard_ssl_name_check, self.daemon_thread_pool_size)
        self.abort = self.http_daemon.abort
        http_daemon_set_daemon_inst(self.http_daemon)
        self._register_http_interfaces()
    
    
    # Global loop part
    @staticmethod
    def get_socks_activity(socks, timeout):
        # some oses are not managing void socks list, so catch this
        # and just so a simple sleep instead
        if not socks:
            time.sleep(timeout)
            return []
        try:
            ins, _, _ = select.select(socks, [], [], timeout)
        except select.error as e:
            err_num, _ = e
            if err_num == errno.EINTR:
                return []
            raise
        return ins
    
    
    # Find the absolute path of the shinken module directory and returns it.
    # If the directory do not exist, we must exit!
    def find_modules_path(self):
        if not hasattr(self, u'modules_dir') or not self.modules_dir:
            logger.error(u'Your configuration is missing the path to the modules (modules_dir). I set it by default to /var/lib/shinken/modules. Please configure it')
            self.modules_dir = u'/var/lib/shinken/modules'
        self.modules_dir = os.path.abspath(self.modules_dir)
        logger.info(u'Modules directory: %s' % self.modules_dir)
        if not os.path.exists(self.modules_dir):
            logger.error(u"The modules directory '%s' is missing! Bailing out. Please fix your configuration" % self.modules_dir)
            raise Exception(u"The modules directory '%s' is missing! Bailing out. Please fix your configuration" % self.modules_dir)
        
        # Ok remember to populate the modulesctx object
        modulesctx.set_modulesdir(self.modules_dir)
        
        return self.modules_dir
    
    
    # modules can have process, and they can die
    def check_and_del_zombie_modules(self, skip_external=False, restart_dead=True, force_start=False):
        # Active children make a join with every one, useful :)
        active_children()
        self.modules_manager.check_alive_instances(skip_external)
        # and try to restart previous dead :)
        if restart_dead:
            self.modules_manager.try_to_restart_deads(force_start=force_start)
    
    
    # Change user of the running program. Just insult the admin
    # if he wants root run (it can override). If change failed we sys.exit(2)
    def change_to_user_group(self, insane=None):
        if insane is None:
            insane = not self.idontcareaboutsecurity
        
        # TODO: change user on nt
        if os.name == u'nt':
            logger.warning(u"You can't change user on this system")
            return
        
        if (self.user == u'root' or self.group == u'root') and not insane:
            logger.error(u"Application will run under the root account")
            logger.error(u"As this may lead to security issues. If you really want this, put:")
            # noinspection SpellCheckingInspection
            logger.error(u'idontcareaboutsecurity=yes')
            logger.error(u'in the config file')
            logger.error(u'Exiting')
            sys.exit(2)
        
        uid = get_user_id_from_name(self.user)
        gid = get_group_id_from_name(self.group)
        
        if uid is None or gid is None:
            logger.error(u'uid or gid is undefined. Exiting')
            sys.exit(2)
        
        # Maybe the os module got the initgroups function. If so, try to call it.
        # Do this when we are still root
        if hasattr(os, u'initgroups'):
            logger.info(u'Trying to initialize additional groups for the daemon')
            try:
                _user = self.user
                if isinstance(_user, unicode):
                    _user = _user.encode(u'utf8', u'ignore')
                os.initgroups(_user, gid)
            except OSError as e:
                logger.error(u'Cannot call the additional groups setting with initgroups (%s)' % e.strerror)
        else:
            group_ids = [gr.gr_gid for gr in get_all_group() if self.user in gr.gr_mem]
            try:
                os.setgroups(group_ids)
            except OSError as e:
                logger.error(u'Cannot call the additional groups setting with setgroups (%s)' % e.strerror)
        
        try:
            # First group, then user :)
            os.setregid(gid, gid)
            os.setreuid(uid, uid)
        except OSError as e:
            logger.error(u'cannot change user/group to %s/%s (%s [%d]). Exiting' % (self.user, self.group, e.strerror, e.errno))
            sys.exit(2)
    
    
    # Parse self.config_file and get all properties in it.
    # If some properties need a pythonization, we do it.
    # Also put default value in the properties if some are missing in the config_file
    def parse_config_file(self):
        properties = self.__class__.properties
        if self.config_file is not None:
            config = ConfigParser.ConfigParser()
            
            # Beware: ini file do not like space before comments and such things, so clean it before read
            data = StringIO('\n'.join(line.strip() for line in open(self.config_file)))
            config.readfp(data)

            # noinspection PyProtectedMember
            if config._sections == {}:
                logger.error(u'Bad or missing config file: %s ' % self.config_file)
                sys.exit(2)
            try:
                for (key, value) in config.items('daemon'):
                    if key in properties:
                        value = properties[key].pythonize(value)
                    setattr(self, key, value)
            except ConfigParser.InterpolationMissingOptionError as e:
                e = str(e)
                wrong_variable = e.split(u'\n')[3].split(u':')[1].strip()
                logger.error(u"Incorrect or missing variable '%s' in config file : %s" % (wrong_variable, self.config_file))
                sys.exit(2)
        else:
            logger.warning(u'No config file specified, use defaults parameters')
        # Now fill all defaults where missing parameters
        for prop, entry in properties.iteritems():
            if not hasattr(self, prop):
                value = entry.pythonize(entry.default)
                setattr(self, prop, value)
    
    
    @staticmethod
    def _format_daemon_full_version(_daemon_full_version):
        # type: (unicode) -> unicode
        _daemon_full_version = u' '.join(_daemon_full_version.split())
        return _daemon_full_version.replace(u'Version : ', u'').replace(u' Patch : ', u' - ')
    
    
    # Give the shinken enterprise version, with patch included
    def get_full_version(self):
        # type: () -> unicode
        if not self._daemon_full_version:
            self._daemon_full_version = self._format_daemon_full_version(get_shinken_current_version_and_patch())
        return self._daemon_full_version
    
    
    def save_daemon_name_into_configuration_file(self, instance_name):
        logger.debug(u'WRITING BACK %s into the configuration file: %s' % (instance_name, self.config_file))
        
        try:
            # Open the configuration file and search for a name= entry. If missing, create such entry
            f = open(self.config_file, 'r')
            lines = f.read().splitlines()
            f.close()
            p = re.compile(r'^name(\s)*=')
            did_matched = False
            new_lines = []
            for line in lines:
                line = line.rstrip()  # just remove the ending \n no all the space at the beginning, they are need :)
                if not re.search(p, line):  # if missing, just stack the line
                    new_lines.append(line)
                    continue
                # ok did find it, change it
                logger.debug(u'DID FIND A MATCHING NAME: %s' % line)
                did_matched = True
                existing_name = line.split(u'=', 1)[1].strip()
                if existing_name == instance_name:
                    logger.debug(u'Already the good name %s in the configuration file: %s, do not edit the file.' % (instance_name, self.config_file))
                    return
                new_line = u'name=%s' % instance_name
                new_lines.append(new_line)
            if not did_matched:
                # noinspection SpellCheckingInspection
                new_line = u'\nname=%s\n' % instance_name
                new_lines.append(new_line)
            # Always force an ending line at last
            new_lines.append(u'')
            
            buf = '\n'.join(new_lines)
            tmp_dest = u'%s.tmp' % self.config_file
            f = open(tmp_dest, u'w')
            f.write(buf)
            f.close()
            # We can now overwrite the original file
            shutil.move(tmp_dest, self.config_file)
            logger.info(u'CONFIGURATION: the configuration file %s was edited to set the new name parameter: %s' % (self.config_file, instance_name))
        except Exception as exp:
            logger.info(u'Cannot write back the daemon name %s into its configuration file %s: %s' % (instance_name, self.config_file, str(exp)))
    
    
    # Some paths can be relatives. We must have a full path by taking
    # the config file by reference
    def relative_paths_to_full(self, reference_path):
        # print "Create relative paths with", reference_path
        properties = self.__class__.properties
        for prop, entry in properties.iteritems():
            if isinstance(entry, ConfigPathProp):
                path = getattr(self, prop)
                if not os.path.isabs(path):
                    new_path = os.path.join(reference_path, path)
                    # print "DBG: changing", entry, "from", path, "to", new_path
                    path = new_path
                setattr(self, prop, path)
                # print "Setting %s for %s" % (path, prop)
    
    
    def manage_signal(self, sig, frame):
        logger.debug(u"I'm process %d and I received signal %s" % (os.getpid(), str(sig)))
        if sig == signal.SIGUSR1:  # if USR1, ask a memory dump
            memory_stats.dump_memory_full_memory_dump(self.name)
            memory_stats.print_memory_stats()
        elif sig == signal.SIGUSR2:  # if USR2, ask objects dump
            self.need_objects_dump = True
        elif sig == signal.SIGPWR:  # SIGPWR (old signal not used) dump all threads stacks
            thread_dumper.dump_all_threads()
        else:  # Ok, really ask us to die :)
            self.interrupted = True
    
    
    def set_exit_handler(self):
        func = self.manage_signal
        if os.name == u'nt':
            try:
                import win32api
                win32api.SetConsoleCtrlHandler(func, True)
            except ImportError:
                version = u'.'.join(map(str, sys.version_info[:2]))
                raise Exception(u'pywin32 not installed for Python %s' % version)
        else:
            for sig in (signal.SIGTERM, signal.SIGINT, signal.SIGUSR1, signal.SIGUSR2, signal.SIGPWR):
                signal.signal(sig, func)
    
    
    @staticmethod
    def get_header():
        # noinspection SpellCheckingInspection
        return [u"Shinken %s" % VERSION,
                u"Copyright (c) 2009-2014:",
                u"Gabes Jean (naparuba@gmail.com)",
                u"Gerhard Lausser, Gerhard.Lausser@consol.de",
                u"Gregory Starck, g.starck@gmail.com",
                u"Hartmut Goebel, h.goebel@goebel-consult.de",
                u"License: AGPL"]
    
    
    def print_header(self):
        for line in self.get_header():
            logger.info(line)
    
    
    # Main function of the http daemon thread will loop forever unless we stop the root daemon
    def http_daemon_thread(self):
        logger.info(u'Starting HTTP daemon')
        
        # The main thing is to have a pool of X concurrent requests for the http_daemon,
        # so "no_lock" calls can always be directly answer without having a "locked" version to finish
        try:
            self.http_daemon.run()
        except Exception as exp:
            logger.error(u'The HTTP daemon failed with the error %s, exiting' % exp)
            logger.error(u'Back trace of this error: %s' % traceback.format_exc())
            self.do_stop()
            # Hard mode exit from a thread
            os._exit(2)  # noqa => hard kill
    
    
    def sleep(self, sleep_time):
        to_sleep = max(0.00, min(1.0, sleep_time))
        
        # There is no way to know if we did have lock before try to release it, so
        # we try
        did_have_lock = True
        # Allow locking call from http during the sleep period
        try:
            self.http_daemon.lock.release()
        except RuntimeError:  # ok we did not have it
            did_have_lock = False
        time.sleep(to_sleep)
        # Ony take it if we did enter with it in this function
        if did_have_lock:
            self.http_daemon.lock.acquire()
        
        # Maybe we have a system time change, if so, manage it
        self.check_for_system_time_change()
    
    
    # Check for a possible system time change and act correspondingly.
    # If such a change is detected then we return the number of seconds that changed. 0 if no time change was detected.
    # Time change can be positive or negative:
    # positive when we have been sent in the future and negative if we have been sent in the past.
    def check_for_system_time_change(self):
        now = time.time()
        difference = now - self.t_each_loop
        
        # If we have more than 15 min time change, we need to compensate it
        if abs(difference) > 900:
            self.compensate_system_time_change(difference)
        else:
            difference = 0
        
        self.t_each_loop = now
        
        return difference
    
    
    # Default action for system time change. Actually a log is done
    def compensate_system_time_change(self, difference):
        # type: (float) -> None
        now_datetime = datetime.datetime.fromtimestamp(time.time())
        
        if difference < 0:
            difference = abs(difference)
            last_loop_str = now_datetime + datetime.timedelta(seconds=difference)
        else:
            last_loop_str = now_datetime - datetime.timedelta(seconds=difference)
        
        last_loop_str = last_loop_str.strftime(u'%Y-%m-%d %H:%M:%S')
        now_str = now_datetime.strftime(u'%Y-%m-%d %H:%M:%S')
        
        logger.warning(u'System time change detected. The last daemon time was [ %s ], but the system time is [ %s ] which makes a difference of %d seconds. Compensating...' % (last_loop_str, now_str, difference))
    
    
    # Use to wait conf from arbiter.
    # It sends us conf in our http_daemon. It puts have_conf prop
    # if he sends us something
    # (it can just do a ping)
    def wait_for_initial_conf(self, timeout=1.0):
        # Arbiter do not already set our have_conf param
        while not self.new_conf and not self.interrupted:
            sys.stdout.write(".")
            sys.stdout.flush()
            self.sleep(timeout)
    
    
    def hook_point(self, hook_name):
        start = time.time()
        execution_times = {}
        for inst in self.modules_manager.get_all_alive_instances():
            full_hook_name = u'hook_' + hook_name
            if hasattr(inst, full_hook_name):
                before = time.time()
                f = getattr(inst, full_hook_name)
                try:
                    f(self)
                except Exception as exp:
                    logger.warning(u'The instance %s raised an exception %s. I disabled it, and set it to restart later' % (inst.get_name(), exp))
                    logger.print_stack()
                    self.modules_manager.did_crash(inst, u'The instance %s raised an exception %s.' % (inst.get_name(), exp))
                after = time.time()
                execution_times[inst.get_name()] = after - before
        
        if hook_name == u'tick':
            sorted_execution_times = execution_times.keys()
            sorted_execution_times.sort()
            if sorted_execution_times:
                _time_in_str = get_section_string(u'TIME IN %s' % self.daemon_type.upper())
                logger.info(u'%s %s [PERF] [ %.3f ]s All modules "ticks" are done. Execution times by modules: %s' % (
                    _MODULES_STR, _time_in_str, time.time() - start, u', '.join([u'(%s=%.3fs)' % (k, execution_times[k]) for k in sorted_execution_times])))
    
    
    # Dummy function for daemons. Get all retention data
    # So a module can save them
    def get_retention_data(self):
        return []
    
    
    # Save, to get back all data
    def restore_retention_data(self, data):
        pass
    
    
    def set_tz(self, tz_value):
        if tz_value and tz_value != u'NOTSET':
            try:
                logger.info(u'[%s] Setting our timezone to %s' % (self.name, tz_value))
                os.environ['TZ'] = tz_value
                time.tzset()
            except:
                logger.info(u"[%s] Timezone isn't supported for this os." % self.name)
    
    
    def do_after_fork_cleanup(self, after_fork_new_top_instance):
        # SEF-9050
        # Running after fork cleanup
        # We are no more in this daemon process,
        # its son (this process) need to free resources from father
        # If we had a module manager, let's free modules resources too
        if hasattr(self, u'modules_manager'):
            self.modules_manager.do_after_fork_cleanup(after_fork_new_top_instance)
            del self.modules_manager
        if hasattr(self.http_daemon, u'shutdown'):
            # Note: we launch the shutdown in a thread, so it won't lock us if it deadlocks
            self.http_daemon.shutdown(quiet=True)
            self.http_daemon = None
        # Do no more want the daemon from here
        self.clean_all_data_in_sub_process(context=u'because a new process was started')
    
    
    def get_arbiter_trace(self):
        # type: () -> Dict
        return self._arbiter_trace_manager.get_current_arbiter_trace()
    
    
    def set_arbiter_trace(self, arbiter_trace):
        # type: (Dict) -> Dict
        return self._arbiter_trace_manager.set_arbiter_trace(arbiter_trace, self.get_context())
    
    
    def arbiter_traces_cleanup(self):
        # type: () -> None
        self._arbiter_trace_manager.arbiter_traces_cleanup()
    
    
    def get_up_to_date_arbiter_traces(self):
        # type: () -> List[Dict]
        return self._arbiter_trace_manager.get_up_to_date_arbiter_traces()
    
    
    def get_context(self):
        if not self._static_context:
            try:
                with open(CONTEXT_FILE_PATH, u'r') as f:
                    buf = f.read()
                    self._static_context = json.loads(buf)
            except:
                logger.error(u'[get_context] cannot open file [%s]' % CONTEXT_FILE_PATH)
                logger.print_stack()
                return None
        
        # Addition information don't store in the context but added at runtime :
        # * http_errors_count
        # * deactivated_by_arbiter
        http_error_count = self.http_errors_count
        if http_error_count:
            count = reduce(add, http_error_count.values())
        else:
            count = 0
        self._static_context[u'error_count'] = count
        self._static_context[u'deactivated_by_arbiter'] = self.deactivated_by_arbiter
        return self._static_context
