#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018
# This file is part of Shinken Enterprise, all rights reserved.
import json
import traceback
import uuid

import cPickle
import base64
from shinken.daemon import Interface, Daemon
from shinken.http_client import HTTPClient
from shinken.http_daemon import HTTPDaemonBusinessError
from shinken.log import LoggerFactory
from shinken.misc.type_hint import NoReturn, Str, Callable, List, Any, Dict, Tuple, TYPE_CHECKING

if TYPE_CHECKING:
    import types

logger = LoggerFactory.get_logger(u'remote_callable')

remote_callables = {}  # type: Dict[Tuple[str, str], Callable]


class RemoteCallException(Exception):
    def __init__(self, text):
        # type: (Str) -> NoReturn
        self.text = text
    
    
    def __repr__(self):
        # type: () -> Str
        return self.text
    
    
    def __str__(self):
        # type: () -> Str
        return u'RemoteCallException[text:%s]' % self.text


class RemoteCallBusinessError(HTTPDaemonBusinessError):
    def __init__(self, exception):
        # type: (Exception) -> NoReturn
        self.exception = exception
        self.stack = traceback.format_exc().splitlines()
    
    
    def __repr__(self):
        # type: () -> Str
        return json.dumps({
            u'exception': self.exception,
            u'stack'    : self.stack,
        })
    
    
    def __str__(self):
        # type: () -> Str
        return u'RemoteCallBusinessError[exception:%s]' % self.exception


def is_remote_callable(f):
    # type: (Callable) -> Callable
    f.is_remote_callable = True
    return f


class RemoteCallableObject(object):
    def __init__(self, remote_instance_id=None):
        if remote_instance_id is None:
            remote_instance_id = uuid.uuid4().hex
        self._remote_instance_id = remote_instance_id
        
        all_parents = []
        to_scan_parents = list(self.__class__.__bases__)
        watch_dog = 0
        while to_scan_parents:
            cursor = to_scan_parents.pop()
            if cursor is object:
                continue
            watch_dog += 1
            if watch_dog > 50:
                break
            all_parents.append(cursor)
            
            for next_to_scan_parents in cursor.__bases__:
                if next_to_scan_parents not in to_scan_parents and next_to_scan_parents not in all_parents:
                    to_scan_parents.append(next_to_scan_parents)
        
        for instance_member_name in dir(self):
            instance_member = getattr(self, instance_member_name)
            
            if callable(instance_member) and next((getattr(getattr(class_parent, instance_member_name, object), 'is_remote_callable', False) for class_parent in all_parents), False):
                remote_callables[(self._remote_instance_id, instance_member.__name__)] = instance_member


class RemoteCaller(object):
    def internal_call(self, function_to_call, *args, **kwargs):
        # type: (types.MethodType, List[Any], Dict[str, Any]) -> Any
        
        remote_instance_id = getattr(function_to_call.im_self, '_remote_instance_id', None)
        method_name = function_to_call.__name__
        
        if not remote_instance_id:
            raise RemoteCallException(u'function to call [%s] was not in a RemoteCallableObject' % function_to_call)
        
        return self.internal_call(remote_instance_id, method_name, *args, **kwargs)
    
    
    def internal_call_by_instance_id(self, remote_instance_id, method_name, *args, **kwargs):
        # type: (Str, Str, List[Any], Dict[str, Any]) -> Any
        raise NotImplementedError()


class HTTPRemoteCaller(RemoteCaller):
    
    def __init__(self, host='', port=0, use_ssl=False):
        # type: (Str, int, bool) -> NoReturn
        self.con = HTTPClient(address=host, port=port, use_ssl=use_ssl)
    
    
    def internal_call_by_instance_id(self, remote_instance_id, method_name, *args, **kwargs):
        # type: (Str, Str, List[Any], Dict[str, Any]) -> Any
        
        params = {
            u'remote_instance_id': remote_instance_id,
            u'method_name'       : method_name,
            u'args'              : args,
            u'kwargs'            : kwargs,
        }
        post = self.con.post(u'remote_call', params)
        return cPickle.loads(base64.b64decode(post))


class HTTPRemoteCallRoute(Interface):
    def __init__(self, app):
        # type: (Daemon) -> None
        super(HTTPRemoteCallRoute, self).__init__(app, default_lock=False)
    
    
    def remote_call(self, remote_instance_id, method_name, args, kwargs):
        # type: (Str, Str, List[Any], Dict[str, Any]) -> str
        remote_callable = remote_callables.get((remote_instance_id, method_name), None)
        if not remote_callable:
            raise RemoteCallException(u'function to call [%s] was not found' % method_name)
        
        try:
            ret_value = remote_callable(*args, **kwargs)
            ret_value = base64.b64encode(cPickle.dumps(ret_value))
            return ret_value
        except Exception as e:
            raise RemoteCallBusinessError(e)
    
    
    remote_call.method = u'post'
