#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2009-2022:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import json
import os
import threading
import time
import traceback

from shinken.configuration_incarnation import PartConfigurationIncarnationContainer
from shinken.ipc.share_item import ShareItem
from shinken.load import WindowsValue
from shinken.log import PART_INITIALISATION, LoggerFactory
from shinken.message import Message
from shinken.misc.os_utils import safe_write_file, clean_file_name
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.modules.base_module.basemodule import BaseModule, ModuleState
from shinken.util import DelayedForceMemoryTrimManagerByCounter
from shinkensolutions.lib_modules.configuration_reader_mixin import ConfigurationReaderMixin, SeparatorFormat, ConfigurationFormat, TypeConfiguration
from shinkensolutions.locking.fair_lock_ordonnancer import FairLockGroupOrdonnancer

if TYPE_CHECKING:
    from shinken.misc.monitoring_item_manager.regenerator import Regenerator
    from shinken.misc.type_hint import Optional, List, Callable, Any
    from shinken.log import PartLogger
    from shinken.misc.monitoring_item_manager.regenerator import Regenerator
    from shinken.configuration_incarnation import PartConfigurationIncarnation, ConfigurationIncarnation

BROKS_MANAGEMENT_EARLY_LOCK = False
BROKS_MANAGEMENT_ENABLE_CATCHUP = True
BROKS_MANAGEMENT_ALLOWED_LATE_SETS = 10
BROKS_MANAGEMENT_MAX_BROKS_IN_CATCHUP = 200000
BROKS_MANAGEMENT_CATCHUP_LOOPS = True

BROKS_MANAGEMENT_FORCE_MEMORY_TRIM_THRESHOLD_BEFORE_GC_COLLECT = 20
BROKS_MANAGEMENT_FORCE_MEMORY_TRIM_COUNTER_RESET_PERIOD = 10 * 60  # 10min

BROKS_GETTER__INCLUDE_DESERIALISATION_AND_CATCHUP_IN_LOCK = u'%s__broks_getter__include_deserialisation_and_catchup_in_lock'
BROKS_GETTER__ACTIVATE_LATE_SET_CATCHUP = u'%s__broks_getter__activate_late_set_catchup'
BROKS_GETTER__NB_LATE_SET_ALLOWED_BEFORE_CATCHUP = u'%s__broks_getter__nb_late_set_allowed_before_catchup'
BROKS_GETTER__CATCHUP_BROKS_MANAGED_BY_MODULE_IN_A_CATCHUP_LOOP = u'%s__broks_getter__catchup_broks_managed_by_module_in_a_catchup_loop'
BROKS_GETTER__CATCHUP_RUN_ENDLESS_UNTIL_NB_LATE_SET_ALLOWED_REACHED = u'%s__broks_getter__catchup_run_endless_until_nb_late_set_allowed_reached'

UNAVAILABILITY_MARGIN = 500  # in sec
BASE_RETENTION_FILE_PATH = u'/var/lib/shinken/broker_module_retention_stats_%s_%s.json'
MAX_UNAVAILABILITY_TIME_WHEN_NEW_CONFIGURATION_KEEP = 20


class ContextClass(object):
    def __init__(self, acquire=None, release=None):
        # type: (Optional[Callable[[], None]], Optional[Callable[[], None]]) -> None
        self.acquire = acquire
        self.release = release
    
    
    def __enter__(self):
        if self.acquire:
            self.acquire()
    
    
    def __exit__(self, exc_type, exc_val, exc_tb):
        if self.release:
            self.release()


class BrokerBaseModule(BaseModule, ConfigurationReaderMixin):
    def __init__(self, module_configuration):
        BaseModule.__init__(self, module_configuration)
        
        self._fair_lock_ordonnancer = None  # type: Optional[FairLockGroupOrdonnancer]
        self.consumer_lock = None  # type: Optional[ContextClass]
        self.producer_lock = None  # type: Optional[ContextClass]
        self.rg = None  # type: Optional[Regenerator]
        self._manage_broks_functions = []  # type: List[Callable[[Any], None]]
        self._writer_early_lock = BROKS_MANAGEMENT_EARLY_LOCK
        self._writer_catchup_enabled = BROKS_MANAGEMENT_ENABLE_CATCHUP
        self._writer_late_sets_allowed = BROKS_MANAGEMENT_ALLOWED_LATE_SETS
        self._writer_late_broks_max = BROKS_MANAGEMENT_MAX_BROKS_IN_CATCHUP
        self._writer_catchup_loops = BROKS_MANAGEMENT_CATCHUP_LOOPS
        self.unavailability_duration = WindowsValue(values_ttl=60)
        self.unavailability_time_when_new_configuration = 0
        self.unavailability_start_time_when_new_configuration = None
        self.unavailability_end_time_when_new_configuration = 0
        self.configuration_id_for_last_unavailability = None
        self.old_conf_uuid = None
        self.daemon_configuration = ShareItem(key_name=u'shinken_broker_daemon_configuration_for_modules', reinit=False)
        self.rg = None  # type: Optional[Regenerator]
        self.monitoring_configuration_part_process = PartConfigurationIncarnationContainer()
        self._last_unavailability_time_when_new_configuration = {}
        self._last_unavailability_time_when_new_configuration_lock = threading.RLock()
        self._unavailability_duration_lock = threading.RLock()
        
        self.time_before_warning_when_reloading_new_configuration = int(getattr(module_configuration, u'broker__module_webui__loading_new_configuration__time_before_warning_if_load_is_long', u'10'))
        self.is_configuration_fully_load = True
        self.is_configuration_loading = True
        self._nb_part_to_handle = 0
        self._nb_part_handle = 0
        
        if hasattr(self, u'module_type'):
            module_type = self.module_type
        else:
            module_type = getattr(self.myconf, u'module_type', None)
        
        log_init = self.logger.get_sub_part(PART_INITIALISATION, len(PART_INITIALISATION))
        
        if not module_type:
            log_init.warning(u'unknown module type, broks management will run with default values')
            return
        
        if module_type == u'broker_module_livedata':
            module_type_cfg_format = u'broker__module_livedata'
        else:
            module_type_cfg_format = u'broker__module_%s' % module_type
        
        configuration_format = [
            SeparatorFormat(u'BROKS GETTER'),
            ConfigurationFormat([BROKS_GETTER__INCLUDE_DESERIALISATION_AND_CATCHUP_IN_LOCK % module_type_cfg_format], BROKS_MANAGEMENT_EARLY_LOCK, TypeConfiguration.BOOL, u'_writer_early_lock'),
            ConfigurationFormat([BROKS_GETTER__ACTIVATE_LATE_SET_CATCHUP % module_type_cfg_format], BROKS_MANAGEMENT_ENABLE_CATCHUP, TypeConfiguration.BOOL, u'_writer_catchup_enabled'),
            ConfigurationFormat([BROKS_GETTER__NB_LATE_SET_ALLOWED_BEFORE_CATCHUP % module_type_cfg_format], BROKS_MANAGEMENT_ALLOWED_LATE_SETS, TypeConfiguration.INT, u'_writer_late_sets_allowed'),
            ConfigurationFormat([BROKS_GETTER__CATCHUP_BROKS_MANAGED_BY_MODULE_IN_A_CATCHUP_LOOP % module_type_cfg_format], BROKS_MANAGEMENT_MAX_BROKS_IN_CATCHUP, TypeConfiguration.INT, u'_writer_late_broks_max'),
            ConfigurationFormat([BROKS_GETTER__CATCHUP_RUN_ENDLESS_UNTIL_NB_LATE_SET_ALLOWED_REACHED % module_type_cfg_format], BROKS_MANAGEMENT_CATCHUP_LOOPS, TypeConfiguration.BOOL, u'_writer_catchup_loops'),
        ]
        ConfigurationReaderMixin.__init__(self, configuration_format, self.myconf, log_init)
        self.read_configuration()
        self.log_configuration(log_properties=True, show_values_as_in_conf_file=True)
    
    
    def loop_turn(self):
        pass
    
    
    def lock_init(
            self,
            broks_eater,
            fair_lock_ordonnancer_name=u'ITEMS ACCESS ORDONNANCER',
            consumer_name=u'HTTP requests',
            producer_name=u'Broks management',
            consumer_max_switch_time=1,
            producer_max_switch_time=1,
            error_log_time=30,
            logger=None
    ):
        # type: (List[Callable[[Any], None]], unicode, unicode, unicode, int, int, int, Optional[PartLogger]) -> None
        
        self._manage_broks_functions = broks_eater
        
        if not logger:
            logger = self.logger
        
        self._fair_lock_ordonnancer = FairLockGroupOrdonnancer(
            name=fair_lock_ordonnancer_name,
            consumer_name=consumer_name,
            consumer_max_wish_switch_time=consumer_max_switch_time,
            producer_name=producer_name,
            producer_max_wish_switch_time=producer_max_switch_time,
            logger=logger,
            error_log_time=error_log_time
        )
        
        self.consumer_lock = ContextClass(acquire=self._fair_lock_ordonnancer.consumer_acquire, release=self._fair_lock_ordonnancer.consumer_release)
        self.producer_lock = ContextClass(acquire=self._fair_lock_ordonnancer.producer_acquire, release=self._fair_lock_ordonnancer.producer_release)
    
    
    def do_loop_turn(self):
        raise NotImplementedError()
    
    
    def _die_with_strong_error(self, error, with_trace_back=True):
        crash_logger = LoggerFactory.get_logger('CRASH - INSIDE MODULE PROCESS')
        crash_logger.error(error)
        if isinstance(error, Exception):
            error = str(error)
        traceback_message = traceback.format_exc() if with_trace_back else error
        if with_trace_back:
            crash_logger.print_stack()
        msg = Message(id=0, type='ICrash', data={'name': self.get_name(), 'exception': error, 'trace': traceback_message})
        self.from_module_to_main_daemon_queue.put(msg)
        # wait 2 sec, so we know that the broker got our message, and die
        time.sleep(2)
        os._exit(2)  # noqa : forced exit
    
    
    # The manage brok thread will consume broks, but if it dies, broks will stack in the queue manager
    # This is not acceptable, and such error means a strong die with log in healthcheck
    def manage_brok_thread(self):
        try:
            self._manage_brok_thread()
        except Exception as exp:  # noqa: catch all to die
            self._die_with_strong_error(str(exp))
    
    
    @staticmethod
    def _make_memory_trim_manager_for_brok_management(minimum_ask_number_before_force_trim):
        # type: (int) -> DelayedForceMemoryTrimManagerByCounter
        return DelayedForceMemoryTrimManagerByCounter(
            minimum_ask_number_before_force_trim=minimum_ask_number_before_force_trim,
            counter_reset_period=BROKS_MANAGEMENT_FORCE_MEMORY_TRIM_COUNTER_RESET_PERIOD,
        )
    
    
    def _manage_brok_thread(self):
        have_lock = False
        logger_unavailability = LoggerFactory.get_unavailability_logger()
        logger_loading_configuration = LoggerFactory.get_loading_configuration_logger()
        logger_perf = self.logger.get_sub_part(u'MANAGE BROKS', len(u'MANAGE BROKS')).get_sub_part(u'PERF', len(u'PERF'))
        logger_late_broks_sets = self.logger.get_sub_part(u'MANAGE BROKS', len(u'MANAGE BROKS')).get_sub_part(u'LATE BROKS SETS', len(u'LATE BROKS SETS'))
        have_process_initial_broks = False
        unavailability_start_time = time.time()
        delayed_force_memory_trim_by_counter = self._make_memory_trim_manager_for_brok_management(BROKS_MANAGEMENT_FORCE_MEMORY_TRIM_THRESHOLD_BEFORE_GC_COLLECT)
        while True:
            start = time.time()
            broks = self.to_q.get()
            
            if self._writer_early_lock and not have_lock:
                self._fair_lock_ordonnancer.producer_acquire()
                have_lock = True
                unavailability_start_time = time.time()
            
            get_late_broks_set_time = time.time()
            nb_late_broks_sets_taken = 0
            if self._writer_catchup_enabled and (self.to_q.qsize() > self._writer_late_sets_allowed >= 0):
                t1 = time.time()
                while self.to_q.qsize() > 0:
                    t0 = time.time()
                    _late_broks_set = self.to_q.get()
                    _current_nb_late_broks = len(_late_broks_set)
                    broks.extend(_late_broks_set)
                    broks_to_process = len(broks)
                    logger_late_broks_sets.info(u'Getting brok set with %s broks in %.3fs [time for read queue size=%.3fs]. Total broks to process= %s/max:%s. Broks sets in queue: %s.' %
                                                (_current_nb_late_broks, time.time() - t0, t0 - t1, broks_to_process, self._writer_late_broks_max, self.to_q.qsize()))
                    nb_late_broks_sets_taken += 1
                    t1 = time.time()
                    
                    if broks_to_process > self._writer_late_broks_max:
                        logger_late_broks_sets.info(u'Late brok taken => limit reach : %s / limit: %s.' % (broks_to_process, self._writer_late_broks_max))
                        break
            
            after_get = time.time()
            # First unpickle broks, but outside the lock time
            for brok in broks:
                brok.prepare()
            after_prepare = time.time()
            
            # For updating, we cannot do it while answering queries, so wait for no readers
            if not self._writer_early_lock and not have_lock:
                self._fair_lock_ordonnancer.producer_acquire()
                have_lock = True
                unavailability_start_time = time.time()
            
            last_configuration_incarnation_received_by_arbiter = self.daemon_configuration.configuration_incarnation
            if last_configuration_incarnation_received_by_arbiter:
                config_incarnation_uuid = last_configuration_incarnation_received_by_arbiter.get_uuid()
                if self.old_conf_uuid != config_incarnation_uuid:
                    nb_part_to_handle = len(self.daemon_configuration.all_monitoring_configuration_part)
                    logger_loading_configuration.info(u'The configuration 〖%s〗 from the arbiter 〖%s〗 must be loaded with 〖%s〗 monitoring configuration parts.' %
                                                      (config_incarnation_uuid, last_configuration_incarnation_received_by_arbiter.get_author(), nb_part_to_handle))
                    logger_loading_configuration.debug(u'Monitoring configuration part id to load:〖%s〗' % u', '.join([u'%s-%s' % (i, config_incarnation_uuid) for i in self.daemon_configuration.all_monitoring_configuration_part]))
                    self.old_conf_uuid = config_incarnation_uuid
                    self.is_configuration_loading = True
                    self.is_configuration_fully_load = False
                    self._nb_part_to_handle = nb_part_to_handle
                    self._nb_part_handle = 0
                    if self.monitoring_configuration_part_process.get_configuration_handle_uuid() == config_incarnation_uuid:
                        self._nb_part_handle = len(self.monitoring_configuration_part_process.get_configuration_incarnation_part_from_handle_configuration())
                        self.is_configuration_fully_load = self._nb_part_to_handle == self._nb_part_handle
                        self.is_configuration_loading = not self.is_configuration_fully_load
                    
                    minimum_ask_number_before_force_trim = self._nb_part_to_handle - self._nb_part_handle
                    if minimum_ask_number_before_force_trim <= 0:
                        minimum_ask_number_before_force_trim = BROKS_MANAGEMENT_FORCE_MEMORY_TRIM_THRESHOLD_BEFORE_GC_COLLECT
                    else:
                        minimum_ask_number_before_force_trim = min(minimum_ask_number_before_force_trim, BROKS_MANAGEMENT_FORCE_MEMORY_TRIM_THRESHOLD_BEFORE_GC_COLLECT)
                    delayed_force_memory_trim_by_counter = self._make_memory_trim_manager_for_brok_management(minimum_ask_number_before_force_trim)
            
            after_wait_write_lock = time.time()
            broks_type_counter = {}
            broks_type_times = {}
            for brok in broks:
                brok_part_configuration_incarnation = brok.part_configuration_incarnation
                if brok.type == u'program_status':
                    part_log_message = brok_part_configuration_incarnation.build_log_message()
                    logger_loading_configuration.info(u'A new monitoring configuration part from the scheduler 〖%s〗 have been received. %s' % (brok_part_configuration_incarnation.scheduler_name, part_log_message))
                    have_process_initial_broks = True
                    logger_unavailability.info(u'The module may be unavailable while the new monitoring configuration part is being loaded. %s' % part_log_message)
                    self.is_configuration_loading = True
                
                try:
                    t0 = time.time()
                    for manage_broks_function in self._manage_broks_functions:
                        manage_broks_function(brok)
                    broks_type_counter[brok.type] = broks_type_counter.get(brok.type, 0) + 1
                    broks_type_times[brok.type] = broks_type_times.get(brok.type, 0.0) + (time.time() - t0)
                except Exception as exp:
                    self._die_with_strong_error(str(exp))  # no hope, cannot survive this
                
                if brok.type == u'initial_broks_done':
                    logger_loading_configuration.info(
                        u'The monitoring configuration part from the scheduler 〖%s〗 have been processed. %s' % (brok_part_configuration_incarnation.scheduler_name, brok_part_configuration_incarnation.build_log_message()))
                    have_process_initial_broks = True
                    self.monitoring_configuration_part_process.add(brok_part_configuration_incarnation)
                    delayed_force_memory_trim_by_counter.ask_for_memory_trim(context=u'because the module >%s< have received a new monitoring configuration' % self.name)
            
            end = time.time()
            logger_perf.info(u'[ %4d broks ] [ wait and get first set on queue=%.3fs ] [ get %s late sets on=%.3fs ] [ deserialization=%.3fs ] [ wait write lock=%.3fs ] [ manage broks=%.3fs ] [ total=%.3fs ]' % (
                len(broks), get_late_broks_set_time - start, nb_late_broks_sets_taken, after_get - get_late_broks_set_time, after_prepare - after_get, after_wait_write_lock - after_prepare, end - after_wait_write_lock, end - start))
            
            message = []
            for b_type, count in broks_type_counter.iteritems():
                message.append(u'[%s=%s]' % (b_type, count))
            logger_perf.info(u'                 => handled broks -> count by types : %s' % ' '.join(message))
            
            message = []
            for b_type, count in broks_type_times.iteritems():
                message.append(u'[%s=%0.3f]' % (b_type, count))
            logger_perf.info(u'                 => handled broks -> time by types : %s' % ' '.join(message))
            
            try:
                for _detail_timer_name, _detail_timer in self.rg.detail_timer.iteritems():
                    logger_detail_timer = logger_perf.get_sub_part(_detail_timer_name, register=False)
                    for sub_part_name, sub_part_time in _detail_timer.iteritems():
                        logger_detail_timer.debug(u'sub_part : time for %s [ %0.3f ]' % (sub_part_name, sub_part_time))
            except:
                # the broker module cannot have any regenerator
                pass
            
            # if we are late we don't release the lock, and we continue to process broks
            if self._writer_catchup_enabled and self._writer_catchup_loops:
                qsize = self.to_q.qsize()
                if qsize > self._writer_late_sets_allowed >= 0:
                    logger_perf.info(u'Number of Broks sets still in queue after managing broks is %s. We keep the lock and continue the brok managing.' % qsize)
                    continue
            
            if have_lock:
                # We can release the lock as a writer
                self._fair_lock_ordonnancer.producer_release()
                now = time.time()
                unavailability_duration = now - unavailability_start_time
                with self._unavailability_duration_lock:
                    self.unavailability_duration.push_value(unavailability_duration)
                
                if now - self.unavailability_end_time_when_new_configuration > UNAVAILABILITY_MARGIN:
                    self.unavailability_start_time_when_new_configuration = unavailability_start_time
                    self.unavailability_time_when_new_configuration = 0
                
                if self.configuration_id_for_last_unavailability != self.monitoring_configuration_part_process.get_configuration_handle_uuid():
                    self.unavailability_start_time_when_new_configuration = unavailability_start_time
                    self.unavailability_time_when_new_configuration = 0
                    self.configuration_id_for_last_unavailability = self.monitoring_configuration_part_process.get_configuration_handle_uuid()
                have_lock = False
                if have_process_initial_broks:
                    self.unavailability_time_when_new_configuration += unavailability_duration
                    have_process_initial_broks = False
                    
                    last_configuration_incarnation_received_by_arbiter = self.daemon_configuration.configuration_incarnation  # type: ConfigurationIncarnation
                    last_monitoring_configuration_part_received_by_scheduler = self.monitoring_configuration_part_process.get_last_part_configuration_incarnation()  # type: PartConfigurationIncarnation
                    monitoring_configuration_parts_id_to_handle_received_by_arbiter = self.daemon_configuration.all_monitoring_configuration_part
                    
                    if last_configuration_incarnation_received_by_arbiter and last_monitoring_configuration_part_received_by_scheduler and last_configuration_incarnation_received_by_arbiter.get_uuid() == last_monitoring_configuration_part_received_by_scheduler.get_uuid():
                        monitoring_configuration_part_to_handle = set([(i, last_configuration_incarnation_received_by_arbiter.get_uuid()) for i in monitoring_configuration_parts_id_to_handle_received_by_arbiter])
                    else:
                        monitoring_configuration_part_to_handle = set()
                    
                    monitoring_configuration_part_handle = set([(i.get_part_id(), i.get_uuid()) for i in self.monitoring_configuration_part_process.get_configuration_incarnation_part_from_handle_configuration()])
                    
                    conf_by_arbiter_log_message = last_configuration_incarnation_received_by_arbiter.build_log_message() if last_configuration_incarnation_received_by_arbiter else u'Configuration from Arbiter not received'
                    conf_by_scheduler_log_message = last_monitoring_configuration_part_received_by_scheduler.build_log_message()
                    logger_loading_configuration.debug(u'Last configuration incarnation received by Arbiter : %s' % conf_by_arbiter_log_message)
                    logger_loading_configuration.debug(u'Last monitoring configuration part received by Scheduler : %s' % conf_by_scheduler_log_message)
                    logger_loading_configuration.debug(u'Monitoring configuration part to handled:[%s]' % u', '.join([u'%s-%s' % (i[0], i[1]) for i in monitoring_configuration_part_to_handle]))
                    logger_loading_configuration.debug(u'Monitoring configuration part handled   :[%s]' % u', '.join([u'%s-%s' % (i[0], i[1]) for i in monitoring_configuration_part_handle]))
                    
                    nb_conf_handle = len(monitoring_configuration_part_handle)
                    nb_conf_to_handle = len(monitoring_configuration_part_to_handle)
                    self._nb_part_handle = nb_conf_handle
                    if monitoring_configuration_part_to_handle == monitoring_configuration_part_handle:
                        logger_loading_configuration.info(u'All 〖%s〗 monitoring configuration parts from schedulers have been handled' % nb_conf_handle)
                        self.is_configuration_fully_load = True
                        self.is_configuration_loading = False
                    elif monitoring_configuration_part_to_handle:
                        logger_loading_configuration.info(u'〖%s〗 / 〖%s〗 monitoring configuration parts from schedulers have been loaded' % (nb_conf_handle, nb_conf_to_handle))
                        self.is_configuration_fully_load = False
                        self.is_configuration_loading = False
                    else:
                        logger_loading_configuration.info(u'〖%s〗 monitoring configuration parts from schedulers have been handled' % nb_conf_handle)
                    self.save_last_unavailability(last_monitoring_configuration_part_received_by_scheduler, nb_conf_handle, nb_conf_to_handle, unavailability_start_time)
                    
                    conf_by_scheduler_no_part_log_message = last_monitoring_configuration_part_received_by_scheduler.get_configuration_incarnation().build_log_message()
                    part_message = u'〖%s〗' % nb_conf_handle
                    if monitoring_configuration_part_to_handle:
                        part_message = u'〖%s〗/〖%s〗' % (nb_conf_handle, nb_conf_to_handle)
                    
                    logger_unavailability.info(u'The module %s is now available. Unavailability started at 〖%s〗 took:〖%0.3fs〗 for load %s part of configuration %s.' % (
                        self.get_name(), logger_unavailability.format_time(self.unavailability_start_time_when_new_configuration), unavailability_duration, part_message, conf_by_scheduler_no_part_log_message))
                    self.unavailability_end_time_when_new_configuration = now
    
    
    def on_init(self):
        # type: () -> None
        with self._last_unavailability_time_when_new_configuration_lock:
            file_path = clean_file_name(BASE_RETENTION_FILE_PATH % (self.daemon_display_name, self.get_name()))
            if not os.path.exists(file_path):
                return
            with open(file_path, 'r') as fd:
                self._last_unavailability_time_when_new_configuration = json.load(fd)
    
    
    def save_last_unavailability(self, last_monitoring_configuration_part_received_by_scheduler, nb_conf_handle, nb_conf_to_handle, unavailability_start_time):
        # type: (PartConfigurationIncarnation, int, int, float) -> None
        with self._last_unavailability_time_when_new_configuration_lock:
            unavailability_id = u'%s-%s' % (last_monitoring_configuration_part_received_by_scheduler.get_uuid(), self.unavailability_start_time_when_new_configuration)
            # unavailability_id is a monitoring configuration id and a counter because we can have multiple unavailability with same monitoring configuration id
            # like when Arbiter push a new configuration when a scheduler have been reboot (see #SESUP-1671)
            info_unavailability_time_when_new_configuration = last_monitoring_configuration_part_received_by_scheduler.get_configuration_incarnation().dump_as_json()
            info_unavailability_time_when_new_configuration[u'unavailability_time_when_new_configuration'] = self.unavailability_time_when_new_configuration
            info_unavailability_time_when_new_configuration[u'nb_conf_handle'] = nb_conf_handle
            info_unavailability_time_when_new_configuration[u'nb_conf_to_handle'] = nb_conf_to_handle
            info_unavailability_time_when_new_configuration[u'unavailability_start_time'] = self._last_unavailability_time_when_new_configuration.get(unavailability_id, {}).get(u'unavailability_start_time', unavailability_start_time)
            self._last_unavailability_time_when_new_configuration[unavailability_id] = info_unavailability_time_when_new_configuration
            
            to_del = [(k, v[u'creation_date']) for k, v in self._last_unavailability_time_when_new_configuration.iteritems()]
            to_del.sort(key=lambda x: x[1], reverse=True)
            for uuid, _ in to_del[MAX_UNAVAILABILITY_TIME_WHEN_NEW_CONFIGURATION_KEEP:]:
                self._last_unavailability_time_when_new_configuration.pop(uuid)
            
            file_path = clean_file_name(BASE_RETENTION_FILE_PATH % (self.daemon_display_name, self.get_name()))
            safe_write_file(file_path, json.dumps(self._last_unavailability_time_when_new_configuration))
    
    
    def get_module_info(self):
        
        if not self.is_configuration_fully_load and not self.is_configuration_loading:
            with self._last_unavailability_time_when_new_configuration_lock:
                unavailability_id = u'%s-%s' % (self.monitoring_configuration_part_process.get_last_part_configuration_incarnation().get_uuid(), self.unavailability_start_time_when_new_configuration)
                unavailability_start_time = self._last_unavailability_time_when_new_configuration.get(unavailability_id, {}).get(u'unavailability_start_time', 0)
                if time.time() - unavailability_start_time > (self.time_before_warning_when_reloading_new_configuration * 60):
                    return {
                        u'status': ModuleState.WARNING,
                        u'output': u'The configuration is not fully handled, some parts are not loaded ( %s/%s )' % (self._nb_part_handle, self._nb_part_to_handle)
                    }
        return {
            u'status': ModuleState.OK,
            u'output': u'OK'
        }
    
    
    def _internal_get_raw_stats(self, param=u'', module_wanted=u''):
        result = super(BrokerBaseModule, self)._internal_get_raw_stats(param, module_wanted)
        result[u'unavailability_time_when_new_configuration'] = self.unavailability_time_when_new_configuration
        with self._unavailability_duration_lock:
            result[u'unavailability_duration'] = self.unavailability_duration.get_sum_value()
        with self._last_unavailability_time_when_new_configuration_lock:
            result[u'last_unavailability_time_when_new_configuration'] = self._last_unavailability_time_when_new_configuration
        if self.rg:
            result[u'item_counter'] = self.rg.item_counter
        
        return result
