#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.objects.itemsummary import ITEM_TYPE
from shinken.objects.schedulingitem import SchedulingItem


class VISUALISATION_STATUS(object):
    OK = 0
    WARNING = 1
    CRITICAL = 2
    UNKNOWN = 3


class BrokerItem(SchedulingItem):
    last_broker_data_update = None
    
    
    def get_type(self):
        # type: () -> unicode
        if self.got_business_rule:
            return ITEM_TYPE.CLUSTER
        return ITEM_TYPE.HOST
    
    
    def get_visualisation_name(self):
        if hasattr(self, u'visualisation_name'):
            return self.visualisation_name
        return None
    
    
    def get_context(self):
        context = u'NOTHING', u'NOTHING'
        
        # DISABLED is for hosts, service can't be like this because a disabled host has no service
        # Note: dt.ref.__repr__.__self__ will give the real object on weak ref proxy or itself on a real item
        if self.downtimes and next((dt for dt in self.downtimes if dt.ref.get_instance_uuid() == self.get_instance_uuid() and dt.is_in_effect), None):
            context = u'DOWNTIME', u'DOWNTIME'
        elif self.is_in_inherited_downtime():
            context = u'INHERITED-DOWNTIME', u'DOWNTIME'
        elif self.in_partial_downtime:
            context = u'PARTIAL-DOWNTIME', u'DOWNTIME'
        elif self.acknowledgement and not self.acknowledgement.automatic:
            context = u'ACKNOWLEDGED', u'ACKNOWLEDGED'
        elif self.is_partial_acknowledged:
            context = u'PARTIAL-ACKNOWLEDGED', u'ACKNOWLEDGED'
        elif self.is_flapping:
            context = u'FLAPPING', u'FLAPPING'
        elif self.got_business_rule and self.is_partial_flapping:
            context = u'PARTIAL-FLAPPING', u'FLAPPING'
        return context
    
    
    def get_ui_status(self):
        if self.state in [u'OK', u'UP']:
            status = VISUALISATION_STATUS.OK
        elif self.state == u'WARNING':
            status = VISUALISATION_STATUS.WARNING
        elif self.state in [u'CRITICAL', u'DOWN']:
            status = VISUALISATION_STATUS.CRITICAL
        else:
            status = VISUALISATION_STATUS.UNKNOWN
        
        return status
    
    
    def get_summary(self):
        return {u'status': self.get_ui_status(), u'summary': self.get_context()[0]}
    
    
    def get_element_visibility_on_not_authenticated_view(self):
        if hasattr(self, u'element_visibility_on_not_authenticated_view'):
            return self.element_visibility_on_not_authenticated_view
        return False
    
    
    def get_monitoring_start_time(self):
        return getattr(self, u'monitoring_start_time', None)
    
    
    def get_sla_critical_threshold(self):
        return getattr(self, u'sla_critical_threshold', None)
    
    
    def get_sla_warning_threshold(self):
        return getattr(self, u'sla_warning_threshold', None)
    
    
    def is_in_downtime(self):
        raise NotImplementedError()
    
    
    def is_in_inherited_downtime(self):
        raise NotImplementedError()
    
    
    def get_name(self):
        # type: ()-> str
        raise NotImplementedError()
    
    
    def notification_is_blocked_by_item(self, notification_type):
        raise NotImplementedError()
    
    
    def get_full_name(self):
        raise NotImplementedError()
    
    
    def last_time_non_ok_or_up(self):
        raise NotImplementedError()
