#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.


# The resultmodulation class is used for in scheduler modulation of results
# like the return code or the output.

import re
import time

from item import Item, Items
from shinken.property import StringProp, ListProp


class ModulationSyntaxError(RuntimeError):
    pass


# Rules factory
class ModulationParser(object):
    tokens = [
        (re.compile(r'/(.*?[^\\])?/->'), 'regex'),
        (re.compile(r'->\s*(WARNING|OK|CRITICAL|UNKNOWN)\s*\|?', re.IGNORECASE), 'out_state'),
        (re.compile(r'(WARNING|OK|CRITICAL|UNKNOWN)', re.IGNORECASE), 'in_state'),
        (re.compile(r'([^~\s]+)'), 'unknown_token')
    ]
    
    
    @staticmethod
    def _sorted_insert(tok, token_list):
        i = 0
        for prev_tok in token_list:
            prev_start = prev_tok.start
            if prev_start < tok.start:
                i += 1
            else:
                break
        token_list.insert(i, tok)
    
    
    @staticmethod
    def tokenize(rule_string):
        token_list = []
        for token_re, token_type in ModulationParser.tokens:
            while True:
                match = re.search(token_re, rule_string)
                if match:
                    mod_token = ModulationToken(token_type, match)
                    ModulationParser._sorted_insert(mod_token, token_list)
                    # The '~' is a neutral character that can't match any pattern.
                    # The idea is to fill what we previously matched to avoid pattern overlap and keep track of token order.
                    if token_type == 'regex':
                        rule_string = rule_string[:match.start()] + ('~' * (mod_token.len - 2)) + rule_string[match.end() - 2:]
                    else:
                        rule_string = rule_string[:match.start()] + ('~' * mod_token.len) + rule_string[match.end():]
                else:
                    break
        return token_list
    
    
    @staticmethod
    def parse(token_list):
        rules = []
        current_rule = ModulationRule()
        while token_list:
            tok = token_list.pop(0)
            if tok.type == 'in_state':
                current_rule.in_states.append(tok.val)
            elif tok.type == 'regex':
                current_rule.regexp = tok.val
            elif tok.type == 'out_state':  # rule ending token
                current_rule.out_state = tok.val
                rules.append(current_rule)
                current_rule = ModulationRule()
            else:
                raise ModulationSyntaxError('Invalid syntax for result modulation: %s' % tok.val)
        return rules
    
    
    @staticmethod
    def is_correct(rules_list):
        if not rules_list:
            return False
        for rule in rules_list:
            if rule.out_state is None:
                return False
        return True


# Tokens parsed in the initial complex rule
class ModulationToken(object):
    states_list = ['ok', 'warning', 'critical', 'unknown']
    
    
    def __init__(self, tok_type, match_object):
        self.type = tok_type
        self.start = match_object.start()
        self.len = match_object.end() - match_object.start()
        if tok_type in ('in_state', 'out_state'):
            self.val = self.states_list.index(match_object.group(1).lower())
        else:
            self.val = match_object.group(1).replace('\\/', '/')


# Rule generated from tokens, ready for evaluation.
class ModulationRule(object):
    def __init__(self):
        self.in_states = []
        self.out_state = None
        self.regexp = None
    
    
    def does_apply(self, state, check_output):
        if self.in_states and state not in self.in_states:
            return False
        return not self.regexp or re.search(self.regexp, check_output)

    def getInState(self):
        if self.in_states:
            return self.in_states[0]
        return ""

    def getRegex(self):
        if self.regexp is None:
            return ""
        return self.regexp


class Resultmodulation(Item):
    id = 1  # zero is always special in database, so we do not take risk here
    my_type = 'resultmodulation'
    
    properties = Item.properties.copy()
    properties.update({
        'resultmodulation_name': StringProp(),
        'exit_codes_match'     : ListProp(default=''),
        'exit_code_modulation' : StringProp(default=''),
        'modulation_period'    : StringProp(default=''),
        'output_rules'         : StringProp(default=''),
        'compiled_rules'       : ListProp(default=[])
    })
    
    
    # For debugging purpose only (nice name)
    def get_name(self):
        return self.resultmodulation_name
    
    
    # Make the return code modulation if need
    def module_return(self, return_code, output, long_output):
        output = output + '\n' + long_output
        # Only if in modulation_period of modulation_period == None
        if self.modulation_period is None or self.modulation_period.is_time_valid(time.time()):
            # Output based modulation
            if self.compiled_rules:
                for rule in self.compiled_rules:
                    if rule.does_apply(return_code, output):
                        return rule.out_state
            # Try to change the exit code only if a new one is defined
            elif self.exit_code_modulation is not None:
                # First with the exit_code_match
                if return_code in self.exit_codes_match:
                    return_code = self.exit_code_modulation
        
        return return_code
    
    
    def compile_rules(self):
        # Generate rules from text using the parser class
        parser = ModulationParser()
        try:
            toks = parser.tokenize(self.output_rules)
        except ModulationSyntaxError as e:
            self.configuration_errors.append(str(e))
            return
        self.compiled_rules = parser.parse(toks)
        # Optimisation step: pre-compile regular expressions
        for rule in self.compiled_rules:
            if rule.regexp:
                try:
                    rule.regexp = re.compile(rule.regexp)
                except:
                    self.configuration_errors.append('Invalid modulation rule regular expression %s.' % rule.regexp)
        if not parser.is_correct(self.compiled_rules):
            self.configuration_errors.append('Error while parsing modulation rules: Missing out state rule.')
    
    
    # We override the pythonize because we have special cases that we do not want
    # to be do at running
    def pythonize(self):
        # First apply Item pythonize
        super(self.__class__, self).pythonize()
        # Then very special cases
        # Intify the exit_codes_match, and make list
        self.exit_codes_match = [int(ec) for ec in getattr(self, 'exit_codes_match', [])]
        if getattr(self, 'output_rules', ''):
            self.compile_rules()
        if getattr(self, 'exit_code_modulation', ''):
            self.exit_code_modulation = int(self.exit_code_modulation)
        else:
            self.exit_code_modulation = None


class Resultmodulations(Items):
    name_property = "resultmodulation_name"
    inner_class = Resultmodulation
    
    
    def linkify(self, timeperiods):
        self.linkify_rm_by_tp(timeperiods)
    
    
    # We just search for each timeperiod the tp
    # and replace the name by the tp
    def linkify_rm_by_tp(self, timeperiods):
        for rm in self:
            # If we do nto have the property, then we are by default, and set None (always True)
            if rm.modulation_period == '':
                rm.modulation_period = None
                continue
            
            # We have the property, so now if we have a value, we must really check it
            mtp_name = rm.modulation_period.strip()
            
            # The new member list, in id
            mtp = timeperiods.find_by_name(mtp_name)
            
            if mtp_name and mtp is None:
                err = "Error: the modulation period '%s' for the result modulation '%s' is unknown or disabled" % (mtp_name, rm.get_name())
                rm.configuration_errors.append(err)
            
            rm.modulation_period = mtp
