#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#    Thibault Cohen, thibault.cohen@savoirfairelinux.com
#    Francois Mikus, fmikus@acktomic.com
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

""" This class is a common one for service/host. Here you
will find all scheduling related functions, like the schedule
or the consume_check. It's a very important class!
"""

import datetime
import logging
import os
import random
import re
import time
import traceback

from item import InheritableItem
from shinken.acknowledge import Acknowledge
from shinken.brok import BROK_VERSION_V020801P7, BROK_VERSION_V020801P15
from shinken.check import Check, CHECK_CAUSE, NO_END_VALIDITY, CHECK_STATUS, get_for_retention_callback
from shinken.checks_container import checks_container
from shinken.commandcall import UNSET_POLLER_REACTIONNER_TAG_VALUE
from shinken.dependencynode import DependencyNodeFactory, DependencyNode, DependencyNodeUninitializedException
from shinken.eventhandler import EventHandler
from shinken.log import logger, get_chapter_string, LoggerFactory
from shinken.macroresolver import MacroResolver
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.notification import Notification
from shinken.objects.proxyitem import proxyitemsmgr, proxyitemsgraph, root_problems_to_uuids
from shinken.objects.state_id import STATE_ID, STATE_TYPE_ID
from shinken.property import BoolProp, IntegerProp, FloatProp, StringProp, EpochProp, RawProp, VirtualProp, LinkToAnotherObjectProp, LinkToOthersObjectsProp
from shinken.rawdata import TYPE_RAW_DATA_CHECK_RESULT, TYPE_RAW_DATA_FULL
from shinken.toolbox.full_status_manipulator import FullStatusManipulator
from shinken.util import get_active_downtime_uuids
from shinken.util import to_svc_hst_distinct_lists, get_acknowledge_id, to_list_of_names, to_name_if_possible, object_to_deepcopy
from shinkensolutions.shinken_time_helper import print_human_readable_period, print_human_readable_date_time

if TYPE_CHECKING:
    from shinken.brok import Brok
    from shinken.commandcall import CommandCall
    from shinken.misc.type_hint import Dict, Union, List, Number, Optional, Set
    from shinken.misc.monitoring_item_manager.regenerator import Regenerator
    from shinken.objects.businessimpactmodulation import Businessimpactmodulations
    from shinken.objects.checkmodulation import CheckModulations
    from shinken.objects.command import Commands
    from shinken.objects.contact import Contact, Contacts
    from shinken.objects.contactgroup import Contactgroups
    from shinken.objects.escalation import Escalations
    from shinken.objects.host import Host
    from shinken.objects.macromodulation import MacroModulations
    from shinken.objects.proxyitem import ProxyItems
    from shinken.objects.service import Services
    from shinken.objects.timeperiod import Timeperiod

# on system time change just reevaluate the following attributes:
on_time_change_update = ('last_notification', 'last_state_change', 'last_hard_state_change')

MONITORING_CHECK_CONSUME_DEBUG_FLAG = os.environ.get('MONITORING_CHECK_CONSUME_DEBUG_FLAG', '0') == '1'
ENABLE_ESCALATION_DEBUG = os.environ.get('SHINKEN_SUPPORT_ENABLE_ESCALATION_DEBUG', '0') == '1'

# #SEF-6735: when downtimes are activated twice
DOWNTIME_INCOHERENCY = get_chapter_string('DOWNTIME-INCOHERENCY')

MISSING_DATA_OUTPUT = {
    u'en': u'''The status is set to UNKNOWN because last check by Shinken Entreprise on this element should have been done at %s ( waiting time has exceeded tolerance threshold of %s ).''',
    u'fr': u'''Le statut est positionné à UNKNOWN car la dernière vérification faite par Shinken Enterprise sur cet élément aurait dû avoir lieu à %s  ( l'attente a dépassé le seuil de tolérance de %s ).''',  # noqa : string in french
}

PENDING_OUTPUT = {
    u'en': u'Status unavailable since ( %s ), waiting for next check result%s',
    u'fr': u'Statut indisponible depuis ( %s ), en attente de la prochaine vérification%s'  # noqa : string in french
}

# When loading the retention, warn if too much checks objects on the elements (can be a bug)
_CHECKS_RETENTION_LOAD_ERROR_LIMIT = 100

PROPERTIES_RETRIEVABLE_FROM_COMMAND_TO_HOST = (u'check_running_timeout', u'warning_threshold_cpu_usage')
CUSTOM_DATA_FORMAT = re.compile(r'[^0-9a-zA-Z_-]')


class CHECK_LAUNCH_MODE(object):
    SCHEDULE = u'schedule'  # type: unicode
    PASSIVE = u'passive'  # type: unicode
    DEPENDENCY = u'dependency'  # type: unicode
    FRESHNESS = u'freshness'  # type: unicode


class SchedulingItem(InheritableItem):
    
    def get_name(self):
        # type: ()-> str
        raise NotImplementedError()
    
    
    def get_instance_uuid(self):
        # type: () -> unicode
        raise NotImplementedError()
    
    
    def raise_alert_log_entry(self):
        # type: () -> None
        raise NotImplementedError()
    
    
    def raise_flapping_stop_log_entry(self, change_ratio, threshold):
        # type: (Number, Number) -> None
        raise NotImplementedError()
    
    
    def raise_flapping_start_log_entry(self, change_ratio, threshold):
        # type: (Number, Number) -> None
        raise NotImplementedError()
    
    
    def raise_freshness_log_entry(self, t_stale_by, t_threshold):
        # type: (Number, Number) -> None
        raise NotImplementedError()
    
    
    def raise_event_handler_log_entry(self, command):
        # type: (CommandCall) -> None
        raise NotImplementedError()
    
    
    def raise_enter_downtime_log_entry(self):
        # type: () -> None
        raise NotImplementedError()
    
    
    def raise_exit_downtime_log_entry(self):
        # type: () -> None
        raise NotImplementedError()
    
    
    def raise_notification_log_entry(self, n):
        # type: (Notification) -> None
        raise NotImplementedError()
    
    
    def is_in_inherited_downtime(self):
        # type: () -> bool
        raise NotImplementedError()
    
    
    def is_in_inherited_acknowledged(self):
        # type: () -> bool
        raise NotImplementedError()
    
    
    def is_state(self, status):
        # type: (unicode) -> bool
        raise NotImplementedError()
    
    
    def set_impact_state(self):
        # type: () -> None
        raise NotImplementedError()
    
    
    def unset_impact_state(self):
        # type: () -> None
        raise NotImplementedError()
    
    
    def set_unreachable(self):
        # type: () -> None
        raise NotImplementedError()
    
    
    def is_in_downtime(self):
        # type: () -> bool
        raise NotImplementedError()
    
    
    def get_data_for_event_handler(self):
        # type: () -> List[SchedulingItem]
        raise NotImplementedError()
    
    
    def set_state_from_exit_status(self, return_code):
        # type: (int) -> None
        raise NotImplementedError()
    
    
    def get_data_for_notifications(self, contact, n):
        # type: (Contact, Notification) -> List[Union[SchedulingItem,Contact,Notification]]
        raise NotImplementedError()
    
    
    def notification_is_blocked_by_contact(self, n, contact):
        # type: (Notification, Contact) -> bool
        raise NotImplementedError()
    
    
    def get_data_for_checks(self):
        # type: () -> List[SchedulingItem]
        raise NotImplementedError()
    
    
    execution_timeout_display_type = ''
    custom_data_format = re.compile(r'[^0-9a-zA-Z_-]')
    if TYPE_CHECKING:
        sla_warning_threshold = 0.0
        sla_critical_threshold = 0.0
        monitoring_start_time = 0.0
    
    if TYPE_CHECKING:
        
        # From Config for Host / Service
        enable_notifications = False
        accept_passive_checks = False
        execute_checks = False
        enable_event_handlers = False
        missing_data_add_delay = 0
        log_notifications = False
        log_retries = False
        log_event_handlers = False
        log_initial_states = False
        global_event_handler = u''
        max_check_spread = 0
        interval_length = 0
        use_aggressive_host_checking = False
        enable_predictive_dependency_checks = False
        cached_check_horizon = 0
        enable_environment_macros = False
        enable_flap_detection = False
        global_low_flap_threshold = 0
        global_high_flap_threshold = 0
        warning_threshold_cpu_usage_default = 0
        event_handler_timeout = 0
        notification_timeout = 0
        ocsp_timeout = 0
        # noinspection SpellCheckingInspection
        ochp_timeout = 0
        # noinspection SpellCheckingInspection
        perfdata_timeout = 0
        ocsp_command = u''
        obsess_over = False
        # noinspection SpellCheckingInspection
        ochp_command = u''
        process_performance_data = False
        # noinspection SpellCheckingInspection
        perfdata_command = None  # type: Optional[CommandCall]
        # noinspection SpellCheckingInspection
        perfdata_file = u''
        # noinspection SpellCheckingInspection
        perfdata_file_template = u''
        # noinspection SpellCheckingInspection
        perfdata_file_mode = u''
        # noinspection SpellCheckingInspection
        perfdata_file_processing_command = u''
        check_for_orphaned = False
        global_check_freshness = False
        additional_freshness_latency = 0
        use_timezone = u''
        language = u''
        illegal_object_name_chars = u''
        illegal_macro_output_chars = u''
        
        flap_history = 0
        max_plugins_output_length = 0
        no_event_handlers_during_downtimes = False
        enable_problem_impacts_states_change = False
        
        # From Host / Service
        ok_up = u''
        
        
        # NOTE : we are in TYPE CHECKING definitions (only for IDE), to be reworked if __init__(...) is needed at runtime
        def __init__(self, params=None, skip_useless_in_configuration=False):
            super(SchedulingItem, self).__init__(params, skip_useless_in_configuration)
            
            # From Host / Service
            self.my_type = u'AbstractSchedulingItem'
            self.max_check_attempts = 0
            self.my_type_log_display = u''
            self.flap_detection_options = u''
            self.flap_detection_enabled = False
            self.retry_interval = 0
            self.check_interval = 0
            self.state_id = 0
            self.low_flap_threshold = 0
            self.high_flap_threshold = 0
            self.host_name = u''
            self.services = None  # type: Optional[Services]
            self.state_before_hard_unknown_reach_phase = u''
            self.check_command = None  # type: Optional[CommandCall]
            self.in_partial_downtime = False
            self.problem_has_been_acknowledged = False
            self.name = u''
            self.stalking_options = u''
            self.notification_options = u''
            self.display_name = u''
            self.visualisation_name = u''
            
            # From Host / Service in Broker (Regenerator)
            self.last_missing_data_start_in_broker = 0
            self.list_cache_var = {}
            
            # From Host
            self.parents = None  # type: Optional[List[Host]]
            
            # From Service
            self.host = None  # type: Optional[Host]
            
            # From SchedulingItem
            self.active_checks_enabled = False
            self.passive_checks_enabled = False
            self.check_freshness = False
            self.freshness_threshold = 0
            self.event_handler_enabled = False
            self.process_perf_data = False
            self.notification_interval = 0
            self.first_notification_delay = 0
            self.notifications_enabled = False
            self.time_to_orphanage = 0
            self.business_impact = 0
            self.check_running_timeout = 0
            self.warning_threshold_cpu_usage = 0
            self.is_cluster = False
            self.check_period = None  # type: Optional[Timeperiod]
            self.event_handler = None  # type: Optional[CommandCall]
            self.notification_period = None  # type: Optional[Timeperiod]
            self.contacts = None  # type: Optional[Contacts]
            self.contact_groups = None  # type: Optional[Contactgroups]
            self.view_contacts = None  # type: Optional[Contacts]
            self.view_contact_groups = None  # type: Optional[Contactgroups]
            self.notification_contacts = None  # type: Optional[Contacts]
            self.notification_contact_groups = None  # type: Optional[Contactgroups]
            self.edition_contacts = None  # type: Optional[Contacts]
            self.edition_contact_groups = None  # type: Optional[Contactgroups]
            self.notes_url = u''
            self.notes_multi_url = u''
            self.poller_tag = u''
            # noinspection SpellCheckingInspection
            self.reactionner_tag = u''
            # noinspection SpellCheckingInspection
            self.resultmodulations = u''
            self.business_impact_modulations = None  # type: Optional[Businessimpactmodulations]
            self.escalations = None  # type: Optional[Escalations]
            self.maintenance_period = None  # type: Optional[Timeperiod]
            # noinspection SpellCheckingInspection
            self.checkmodulations = None  # type: Optional[CheckModulations]
            # noinspection SpellCheckingInspection
            self.macromodulations = None  # type: Optional[MacroModulations]
            self.bp_rule = u''
            self.sla_warning_threshold = 0.0
            self.sla_critical_threshold = 0.0
            self.in_scheduled_downtime_during_last_check = False
            # noinspection SpellCheckingInspection
            self.must_respread = False
            self.last_broker_data_update = 0
            self.monitoring_start_time = 0.0
            self.modified_attributes = 0L
            self.last_chk = 0.0
            self.last_normal_chk = 0.0
            self.next_chk = 0.0
            self.in_checking = False
            self.latency = 0.0
            self.notification_latency = 0.0
            self.eventhandler_latency = 0.0
            self.attempt = 0
            self.state = u''
            self.state_type = u''
            self.state_type_id = 0
            self.context_id = 0
            self.state_validity_period = 0
            self.state_validity_end_time = 0.0
            self.missing_data_activation_time = 0.0
            self.last_state = u''
            self.last_state_id = 0
            self.last_state_type = u''
            self.last_state_change = 0.0
            self.last_hard_state_change = 0.0
            self.last_hard_state = u''
            self.last_hard_state_id = 0
            self.last_state_update = 0.0
            self.last_state_as_string = u''
            self.duration_sec = 0
            self.output = u''
            self.long_output = u''
            self.perf_data = u''
            self.last_perf_data = u''
            self.is_flapping = False
            self.is_partial_flapping = False
            self.inherited_flapping = False
            self.percent_state_change = 0.0
            self.is_partial_acknowledged = False
            self.acknowledgement_type = 0
            self.last_notification = 0.0
            self.current_notification_number = 0
            self.last_event_handler = 0.0
            self.check_type = 0
            self.execution_time = 0.0
            self.u_time_sum = 0.0
            self.s_time_sum = 0.0
            self.nb_executions = 0
            self.scheduled_downtime_depth = 0
            self.is_problem = False
            self.is_impact = False
            self.impacts = None  # type: Optional[List[SchedulingItem]]
            self.source_problems = None  # type: Optional[List[SchedulingItem]]
            self.flapping_changes = []  # type: List[bool]
            self.state_before_impact = u''
            self.state_id_before_impact = 0
            self.output_before_impact = u''
            self.state_changed_since_impact = False
            self.got_business_rule = False
            self.my_own_business_impact = 0
            self.bp_state = 0
            self.in_hard_unknown_reach_phase = False
            self.was_in_hard_unknown_reach_phase = False
            self.topology_change = False
            self.consecutive_ok = 0
            self.current_full_status = 0
            self.full_status_change_time = 0.0
            self.act_depend_of = []
            self.chk_depend_of = []
            self.act_depend_of_me = []
            self.chk_depend_of_me = []
            self.checks_in_progress = []  # type: List[Check]
            self.notifications_in_progress = {}
            self.notified_contacts = set()
            self.actions = []
            self.broks = []  # type: List[Brok]
            self.business_rule = None  # type: Optional[DependencyNode]
            self.notification_list = []
            self.in_maintenance = None
            self.downtimes = []
            self.comments = []
            self.active_downtime_uuids = []
            self.acknowledgement = None  # type: Optional[Acknowledge]
            self.partial_acknowledge = None  # type: Optional[Acknowledge]
            self.acknowledge_id = None  # type: Optional[unicode]
            self.parent_dependencies = set()  # type: Set[SchedulingItem]
            self.child_dependencies = set()  # type: Set[SchedulingItem]
            self.processed_business_rule = u''
            self.check_running_timeout_from = u''
            self.warning_threshold_cpu_usage_from = u''
            self.last_command_hash = u''
            self.last_command_name = u''
    
    properties = InheritableItem.properties.copy()
    # noinspection SpellCheckingInspection
    properties.update({
        
        # Simple properties, can be set into C structure
        'active_checks_enabled'      : BoolProp(default='1', fill_brok=['full_status']),
        'passive_checks_enabled'     : BoolProp(default='1', fill_brok=['full_status']),
        'check_freshness'            : BoolProp(default='0', fill_brok=['full_status']),
        'freshness_threshold'        : IntegerProp(default='0', fill_brok=['full_status']),
        'event_handler_enabled'      : BoolProp(default='0', fill_brok=['full_status']),
        'process_perf_data'          : BoolProp(default='1', fill_brok=['full_status']),
        'notification_interval'      : IntegerProp(default='1440', fill_brok=['full_status']),
        'first_notification_delay'   : IntegerProp(default='0', fill_brok=['full_status']),
        'notifications_enabled'      : BoolProp(default='1', fill_brok=['full_status']),
        'time_to_orphanage'          : IntegerProp(default="300", fill_brok=['full_status']),
        'business_impact'            : IntegerProp(default='2', fill_brok=['full_status']),
        'check_running_timeout'      : IntegerProp(default='-1'),
        'warning_threshold_cpu_usage': IntegerProp(default='-1'),
        'is_cluster'                 : BoolProp(default='0'),
        
        # Configuration Structure, like links to others elements
        'check_period'               : LinkToAnotherObjectProp(default='24x7', brok_transformation=to_name_if_possible, fill_brok=['full_status']),
        'event_handler'              : StringProp(default='', fill_brok=['full_status']),
        'notification_period'        : LinkToAnotherObjectProp(default='24x7', brok_transformation=to_name_if_possible, fill_brok=['full_status']),
        
        'contacts'                   : LinkToOthersObjectsProp(default='', brok_transformation=to_list_of_names, fill_brok=['full_status'], merging='join', handle_additive_inheritance=True),
        'contact_groups'             : LinkToOthersObjectsProp(default='', fill_brok=['full_status'], merging='join', handle_additive_inheritance=True),
        'view_contacts'              : LinkToOthersObjectsProp(default='', brok_transformation=to_list_of_names, fill_brok=['full_status'], merging='join', handle_additive_inheritance=True),
        'view_contact_groups'        : LinkToOthersObjectsProp(default='', fill_brok=['full_status'], merging='join', handle_additive_inheritance=True),
        'notification_contacts'      : LinkToOthersObjectsProp(default='', brok_transformation=to_list_of_names, fill_brok=['full_status'], merging='join', handle_additive_inheritance=True),
        'notification_contact_groups': LinkToOthersObjectsProp(default='', fill_brok=['full_status'], merging='join', handle_additive_inheritance=True),
        'edition_contacts'           : LinkToOthersObjectsProp(default='', brok_transformation=to_list_of_names, fill_brok=['full_status'], merging='join', handle_additive_inheritance=True),
        'edition_contact_groups'     : LinkToOthersObjectsProp(default='', fill_brok=['full_status'], merging='join', handle_additive_inheritance=True),
        
        'notes_url'                  : StringProp(default='', fill_brok=['full_status']),
        'notes_multi_url'            : StringProp(default='', fill_brok=['full_status']),
        
        'poller_tag'                 : StringProp(default=UNSET_POLLER_REACTIONNER_TAG_VALUE),
        'reactionner_tag'            : StringProp(default=UNSET_POLLER_REACTIONNER_TAG_VALUE),
        'resultmodulations'          : LinkToOthersObjectsProp(default='', merging='ordered', handle_additive_inheritance=True),
        'business_impact_modulations': LinkToOthersObjectsProp(default='', merging='ordered', handle_additive_inheritance=True),
        'escalations'                : LinkToOthersObjectsProp(default='', fill_brok=['full_status'], merging='join', handle_additive_inheritance=True),
        'maintenance_period'         : LinkToAnotherObjectProp(default='', brok_transformation=to_name_if_possible, fill_brok=['full_status']),
        'checkmodulations'           : LinkToOthersObjectsProp(default='', fill_brok=['full_status'], merging='ordered'),
        'macromodulations'           : LinkToOthersObjectsProp(default='', merging='ordered', handle_additive_inheritance=True),
        
        'bp_rule'                    : StringProp(default=''),
        
        'sla_warning_threshold'      : FloatProp(default='99', fill_brok=['full_status']),
        'sla_critical_threshold'     : FloatProp(default='97', fill_brok=['full_status']),
    })
    
    # properties used in the running state
    running_properties = InheritableItem.running_properties.copy()
    # noinspection SpellCheckingInspection
    running_properties.update({
        # Simple properties types
        'in_scheduled_downtime_during_last_check': BoolProp(default=False, retention=True, useless_in_configuration=True),
        'must_respread'                          : BoolProp(default=False, retention=True),
        'last_broker_data_update'                : IntegerProp(default=0),  # just used into the broker to know that the element state is not fresh anymore
        'monitoring_start_time'                  : EpochProp(default=-1, fill_brok=[u'full_status']),
        'modified_attributes'                    : IntegerProp(default=0, fill_brok=[u'full_status'], retention=True, useless_in_configuration=True),
        'last_chk'                               : EpochProp(default=0, fill_brok=[u'full_status', u'check_result'], fill_raw_data=[TYPE_RAW_DATA_CHECK_RESULT, TYPE_RAW_DATA_FULL], retention=True, useless_in_configuration=True),
        'last_normal_chk'                        : EpochProp(default=0, retention=True, useless_in_configuration=True),
        'next_chk'                               : EpochProp(default=0, fill_brok=[u'full_status', u'next_schedule'], retention=True, useless_in_configuration=True),
        'in_checking'                            : BoolProp(default=False, fill_brok=[u'full_status', u'check_result', u'next_schedule'], useless_in_configuration=True),
        'latency'                                : FloatProp(default=0, fill_brok=[u'full_status', u'check_result'], retention=True, useless_in_configuration=True),
        'notification_latency'                   : FloatProp(default=0, fill_brok=[u'full_status', u'check_result'], retention=True, useless_in_configuration=True),
        'eventhandler_latency'                   : FloatProp(default=0, fill_brok=[u'full_status', u'check_result'], retention=True, useless_in_configuration=True),
        'attempt'                                : IntegerProp(default=0, fill_brok=[u'full_status', u'check_result'], retention=True, useless_in_configuration=True),
        'state'                                  : StringProp(default=u'PENDING', fill_brok=[u'full_status', u'check_result'], retention=True, useless_in_configuration=True),
        'state_type'                             : StringProp(default=u'HARD', fill_brok=[u'full_status', u'check_result'], retention=True, useless_in_configuration=True),
        'state_type_id'                          : IntegerProp(default=0, fill_brok=[u'full_status', u'check_result'], fill_raw_data=[TYPE_RAW_DATA_CHECK_RESULT, TYPE_RAW_DATA_FULL], retention=True, useless_in_configuration=True),
        'context_id'                             : IntegerProp(default=0, fill_raw_data=[TYPE_RAW_DATA_CHECK_RESULT, TYPE_RAW_DATA_FULL], retention=True, useless_in_configuration=True),
        'state_validity_period'                  : IntegerProp(default=0, fill_brok=[u'full_status', u'check_result'], retention=True, useless_in_configuration=True),
        'state_validity_end_time'                : EpochProp(default=0, fill_brok=[u'full_status', u'check_result'], retention=True, useless_in_configuration=True),
        'missing_data_activation_time'           : EpochProp(default=0, fill_brok=[u'full_status', u'check_result'], retention=True, useless_in_configuration=True),
        
        'last_state'                             : StringProp(default=u'PENDING', fill_brok=[u'full_status', u'check_result'], retention=True, useless_in_configuration=True),
        'last_state_id'                          : IntegerProp(default=3, fill_brok=[u'full_status', u'check_result'], retention=True, useless_in_configuration=True),
        
        'last_state_type'                        : StringProp(default=u'HARD', fill_brok=[u'full_status', u'check_result'], retention=True, useless_in_configuration=True),
        'last_state_change'                      : EpochProp(default=0, fill_brok=[u'full_status', u'check_result'], retention=True, useless_in_configuration=True),
        'last_hard_state_change'                 : EpochProp(default=0, fill_brok=[u'full_status', u'check_result'], retention=True, useless_in_configuration=True),
        'last_hard_state'                        : StringProp(default=u'PENDING', fill_brok=[u'full_status'], retention=True, useless_in_configuration=True),
        'last_hard_state_id'                     : IntegerProp(default=0, fill_brok=[u'full_status'], retention=True, useless_in_configuration=True),
        'last_state_update'                      : EpochProp(default=0, fill_brok=[u'full_status'], retention=True, useless_in_configuration=True),
        'last_state_as_string'                   : StringProp(default=u'PENDING', fill_brok=[u'full_status', u'check_result'], retention=True),
        
        'duration_sec'                           : IntegerProp(default=0, fill_brok=[u'full_status'], retention=True, useless_in_configuration=True),
        
        'output'                                 : StringProp(default=u'', fill_brok=[u'full_status', u'check_result'], retention=True, useless_in_configuration=True),
        'long_output'                            : StringProp(default=u'', fill_brok=[u'full_status', u'check_result'], retention=True, useless_in_configuration=True),
        'perf_data'                              : StringProp(default=u'', fill_brok=[u'full_status', u'check_result'], fill_raw_data=[TYPE_RAW_DATA_CHECK_RESULT, TYPE_RAW_DATA_FULL], retention=True, useless_in_configuration=True),
        'last_perf_data'                         : StringProp(default=u'', retention=True, useless_in_configuration=True),
        
        'is_flapping'                            : BoolProp(default=False, fill_brok=[u'full_status'], retention=True, useless_in_configuration=True),
        'is_partial_flapping'                    : BoolProp(default=False, fill_brok=[u'full_status', u'check_result'], retention=True, useless_in_configuration=True),
        'inherited_flapping'                     : BoolProp(default=False, fill_brok=[u'full_status', u'check_result'], retention=True, useless_in_configuration=True),
        'percent_state_change'                   : FloatProp(default=0.0, fill_brok=[u'full_status', u'check_result'], retention=True, useless_in_configuration=True),
        
        'is_partial_acknowledged'                : BoolProp(default=False, fill_brok=[u'full_status', u'check_result'], retention=True, useless_in_configuration=True),
        'acknowledgement_type'                   : IntegerProp(default=1, fill_brok=[u'full_status', u'check_result'], retention=True, useless_in_configuration=True),
        
        'last_notification'                      : EpochProp(default=0, fill_brok=[u'full_status'], retention=True, useless_in_configuration=True),
        'current_notification_number'            : IntegerProp(default=0, fill_brok=[u'full_status'], retention=True, useless_in_configuration=True),
        'last_event_handler'                     : EpochProp(default=0, fill_brok=[u'full_status'], retention=True, useless_in_configuration=True),
        
        'check_type'                             : IntegerProp(default=0, fill_brok=[u'full_status', u'check_result'], retention=True, useless_in_configuration=True),
        
        'execution_time'                         : FloatProp(default=0.0, fill_brok=[u'full_status', u'check_result'], retention=True, useless_in_configuration=True),
        'u_time_sum'                             : FloatProp(default=0.0, useless_in_configuration=True),
        's_time_sum'                             : FloatProp(default=0.0, useless_in_configuration=True),
        'nb_executions'                          : IntegerProp(default=0, useless_in_configuration=True),
        
        'scheduled_downtime_depth'               : IntegerProp(default=0, fill_brok=[u'full_status'], retention=True, useless_in_configuration=True),
        
        'is_problem'                             : BoolProp(default=False, fill_brok=[u'full_status'], useless_in_configuration=True),
        'is_impact'                              : BoolProp(default=False, fill_brok=[u'full_status'], useless_in_configuration=True),
        # list of the impact I'm the cause of
        'impacts'                                : RawProp(default=[], useless_in_configuration=True, fill_brok=[u'full_status'], brok_transformation=to_svc_hst_distinct_lists),
        # list of problems that make us an impact
        'source_problems'                        : RawProp(default=[], fill_brok=[u'full_status'], brok_transformation=root_problems_to_uuids, useless_in_configuration=True),
        
        # Can be modified to be simple
        'flapping_changes'                       : RawProp(default=[], fill_brok=[u'full_status'], retention=True, useless_in_configuration=True),
        
        # keep a trace of the old state before being an impact
        'state_before_impact'                    : StringProp(default=u'PENDING', retention=True, useless_in_configuration=True),
        # keep a trace of the old state id before being an impact
        'state_id_before_impact'                 : IntegerProp(default=0, retention=True, useless_in_configuration=True),
        # Same for the output
        'output_before_impact'                   : StringProp(default=u'', retention=True, useless_in_configuration=True),
        # if the state change, we know, so we do not revert it
        'state_changed_since_impact'             : BoolProp(default=False, retention=True, useless_in_configuration=True),
        
        # Say if we are business based rule or not
        'got_business_rule'                      : BoolProp(default=False, fill_brok=[u'full_status', u'check_result'], fill_raw_data=[TYPE_RAW_DATA_FULL, TYPE_RAW_DATA_CHECK_RESULT]),
        
        # the save value of our business_impact for "problems"
        'my_own_business_impact'                 : IntegerProp(default=-1, fill_brok=[u'full_status'], retention=False, useless_in_configuration=True),
        
        # BPSTate hook
        'bp_state'                               : IntegerProp(default=3, fill_brok=[u'full_status', u'check_result'], retention=True, useless_in_configuration=True),
        
        # Manage the unknown/unreachable during hard state
        # From now it's not really used
        'in_hard_unknown_reach_phase'            : BoolProp(default=False, retention=True, useless_in_configuration=True),
        'was_in_hard_unknown_reach_phase'        : BoolProp(default=False, retention=True, useless_in_configuration=True),
        
        # Set if the element just change its father/son topology
        'topology_change'                        : BoolProp(default=False, fill_brok=[u'full_status'], useless_in_configuration=True),
        
        'consecutive_ok'                         : IntegerProp(default=0, fill_brok=[u'full_status'], useless_in_configuration=True),
        
        # Full status
        'current_full_status'                    : IntegerProp(default=-1, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        # Full status change time
        # first appeared in 02.08.01-Patched14 (DEV-1.6) named as update_time_full_status, and was renamed in 02.08.01-Patched15 (DEV-2.0)
        'full_status_change_time'                : EpochProp(default=0, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        
        # ##################################  Complex structures, like list of objects : Purely scheduler internal part
        
        # No broks for _depend_of because of too many links to hosts/services
        # dependencies for actions like notif of event handler, so AFTER check return
        'act_depend_of'                          : RawProp(default=[]),
        # dependencies for checks raise, so BEFORE checks
        'chk_depend_of'                          : RawProp(default=[]),
        # elements that depend on me, so the reverse than just upper
        'act_depend_of_me'                       : RawProp(default=[]),
        # elements that depend on me
        'chk_depend_of_me'                       : RawProp(default=[]),
        
        # No broks, it's just internal, and checks have too links
        'checks_in_progress'                     : RawProp(default=[], useless_in_configuration=True, retention=True, retention_preparation=get_for_retention_callback),
        'notifications_in_progress'              : RawProp(default={}, retention=True, useless_in_configuration=True, retention_preparation=object_to_deepcopy),
        
        # use for having all contacts we have notified
        # Warning: for the notified_contacts retention save, we save only the names of the contacts, and we should RELINK
        # them when we load it.
        'notified_contacts'                      : RawProp(default=set(), retention=True, retention_preparation=to_list_of_names, useless_in_configuration=True),
        
        # put here checks and notif raised
        'actions'                                : RawProp(default=[], useless_in_configuration=True),
        # and here broks raised
        'broks'                                  : RawProp(default=[], useless_in_configuration=True),
        
        'business_rule'                          : RawProp(default=None),  # Our Dependency node for the business rule
        
        # ##################################  Complex structures, like list of objects : must be brok in a way
        'notification_list'                      : RawProp(default=[], fill_brok=[u'full_status'], retention=True, useless_in_configuration=True),
        'in_maintenance'                         : RawProp(default=None, fill_brok=[u'full_status'], retention=True, useless_in_configuration=True),  # WARNING: this is not a bool
        'downtimes'                              : RawProp(default=[], fill_brok=[u'full_status'], retention=True, useless_in_configuration=True),
        'comments'                               : RawProp(default=[], useless_in_configuration=True),
        'active_downtime_uuids'                  : RawProp(default=[], fill_brok=['full_status'], retention=True, useless_in_configuration=True, brok_transformation=get_active_downtime_uuids),
        
        'acknowledgement'                        : RawProp(default=None, fill_brok=[u'full_status', u'check_result'], retention=True, useless_in_configuration=True),
        'partial_acknowledge'                    : RawProp(default=None, fill_brok=[u'full_status', u'check_result'], retention=True, useless_in_configuration=True),
        
        # Virtual because it's just for broks
        'acknowledge_id'                         : VirtualProp(default=None, fill_brok=[u'full_status', u'check_result'], retention=True, brok_transformation=get_acknowledge_id, useless_in_configuration=True),
        
        # Here are elements we depend on : our parents as network relation,  a host
        # we depend on in a host dependency or even if we are business based.
        'parent_dependencies'                    : RawProp(brok_transformation=to_svc_hst_distinct_lists, default=set(), fill_brok=[u'full_status']),
        # Here it's the guys that depend on us. So it's the total opposite of the parent_dependencies
        'child_dependencies'                     : RawProp(brok_transformation=to_svc_hst_distinct_lists, default=set(), fill_brok=[u'full_status']),
        
        # Previously processed business rule (with macro expanded)
        'processed_business_rule'                : StringProp(default=u'', fill_brok=[u'full_status']),
        
        'check_running_timeout_from'             : StringProp(default=u'', fill_brok=[u'full_status', u'check_result']),
        'warning_threshold_cpu_usage_from'       : StringProp(default=u'', fill_brok=[u'full_status', u'check_result']),
        # 'timeout'                             : IntegerProp(default=0, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        # 'start_time'                          : IntegerProp(default=0, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        # 'end_time'                            : IntegerProp(default=0, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        # 'early_timeout'                       : IntegerProp(default=0, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        # 'return_code'                         : IntegerProp(default=0, fill_brok=['full_status', 'check_result'], retention=True, useless_in_configuration=True),
        
        # Useful for check dumping to be able to find check's cpu_time average. No need to store in retention as they will be computed at each schedule (including first schedule at startup)
        'last_command_hash'                      : StringProp(default=''),
        'last_command_name'                      : StringProp(default=''),
    })
    
    
    def last_time_non_ok_or_up(self):
        # type: () -> int
        raise NotImplementedError()
    
    
    def notification_is_blocked_by_item(self, notification_type):
        # type: (unicode) -> bool
        raise NotImplementedError()
    
    
    # Call by pickle to data-ify the host
    # we do a dict because list are too dangerous for
    # retention save and co :( even if it's more
    # extensive
    # The setstate function do the inverse
    def __getstate__(self):
        cls = self.__class__
        # id is not in *_properties
        res = {'id': self.id}
        for prop in cls.properties:
            if hasattr(self, prop):
                res[prop] = getattr(self, prop)
        for prop in cls.running_properties:
            if hasattr(self, prop):
                res[prop] = getattr(self, prop)
        return res
    
    
    # Inverted function of __getstate__
    def __setstate__(self, state):
        cls = self.__class__
        self.id = state['id']
        for prop in cls.properties:
            if prop in state:
                setattr(self, prop, state[prop])
        for prop in cls.running_properties:
            if prop in state:
                setattr(self, prop, state[prop])
    
    
    # Register the son in my child_dependencies, and
    # myself in its parent_dependencies
    def register_son_in_parent_child_dependencies(self, son):
        # So we register it in our list
        self.child_dependencies.add(son)
        
        # and us to its parents
        son.parent_dependencies.add(self)
    
    
    def change_default_values_into_finals(self):
        tag_properties = ('poller_tag', 'reactionner_tag')
        for prop in tag_properties:
            current_value = getattr(self, prop, None)
            if current_value == UNSET_POLLER_REACTIONNER_TAG_VALUE:
                setattr(self, prop, 'None')
    
    
    def check_and_error_if_too_much_checks_in_progress(self, logger_):
        nb_checks_in_progress = len(self.checks_in_progress)
        if nb_checks_in_progress > _CHECKS_RETENTION_LOAD_ERROR_LIMIT:
            logger_.error(u'[ %s ] The %s have too much executions (%d > warning limit=%d). This can be a bug, please contact your support with a backup of your retention for verification. [ uuid=%s ] ' % (
                self.get_full_name(), self.my_type_log_display, nb_checks_in_progress, _CHECKS_RETENTION_LOAD_ERROR_LIMIT, self.get_instance_uuid()))
    
    
    def check_flapping_change(self, state_id, prev_state_id):
        if self.my_type == 'host' and not self.got_business_rule:
            state_to_state_code = {0: 'o', 1: 'd', 2: 'd', 3: 'u'}
        else:
            state_to_state_code = {0: 'o', 1: 'w', 2: 'c', 3: 'u'}
        state_code = state_to_state_code.get(state_id, 'u')
        
        if state_code in self.flap_detection_options:
            self.add_flapping_change(state_id != prev_state_id)
    
    
    # Add a flapping change, but no more than 20 states
    # Then update the self.is_flapping bool by calling update_flapping
    def add_flapping_change(self, state_have_change):
        cls = self.__class__
        
        # If this element is not in flapping check, or the flapping is globally disabled, bail out
        if not self.flap_detection_enabled or not cls.enable_flap_detection:
            if self.is_flapping or self.flapping_changes:
                self.flapping_changes = []
                self.is_flapping = False
                self.build_full_status(context_change_time=self.last_chk)
                self.broks.append(self.get_update_status_brok(context_change_time=self.last_chk))
                # Also update the proxy objects that we did change
                proxyitemsmgr.update_flapping(self.get_instance_uuid(), self.is_flapping, inherited_flapping=False)
            return
        
        self.flapping_changes.append(state_have_change)
        
        # Keep just 20 changes (global flap_history value)
        flap_history = cls.flap_history
        
        if len(self.flapping_changes) > flap_history:
            self.flapping_changes.pop(0)
        
        # Now we add a value, we update the is_flapping prop
        self.update_flapping()
        # Cluster case: can inherit flapping from sons
        if self.got_business_rule and not self.is_flapping:
            is_flapping, is_partial_flapping = self.business_rule.get_flapping_state()
            flapping_have_changed = False
            
            if self.is_flapping != is_flapping:
                flapping_have_changed = True
                self.is_flapping = is_flapping
                inherited_flapping = True if is_flapping else False
                # Update proxy object too
                proxyitemsmgr.update_flapping(self.get_instance_uuid(), is_flapping, inherited_flapping=inherited_flapping)
            if self.is_partial_flapping != is_partial_flapping:
                flapping_have_changed = True
                self.is_partial_flapping = is_partial_flapping
                proxyitemsmgr.update_partial_flapping(self.get_instance_uuid(), is_partial_flapping)
            
            if flapping_have_changed:
                self.broks.append(self.get_update_status_brok())
    
    
    # We update the is_flapping prop with value in self.flapping_states
    # Old values have less weight than new ones
    def update_flapping(self, automatic=False):
        flap_history = self.__class__.flap_history
        # We compute the flapping change in %
        r = 0.0
        i = 0
        for b in self.flapping_changes:
            if b:
                r += i * (1.2 - 0.8) / (flap_history - 1) + 0.8
            i += 1
        r = r / flap_history
        r *= 100
        
        # We can update our value
        self.percent_state_change = r
        
        # Now we get the low_flap_threshold and high_flap_threshold values
        # They can be from self, or class
        (low_flap_threshold, high_flap_threshold) = (self.low_flap_threshold, self.high_flap_threshold)
        if low_flap_threshold == -1:
            cls = self.__class__
            low_flap_threshold = cls.global_low_flap_threshold
        if high_flap_threshold == -1:
            cls = self.__class__
            high_flap_threshold = cls.global_high_flap_threshold
        
        # Now we check is flapping change, but only if we got enough
        # states to look at the value accuracy
        if self.is_flapping and r < low_flap_threshold:
            self.is_flapping = False
            # We also raise a log entry
            self.raise_flapping_stop_log_entry(r, low_flap_threshold)
            # and a notification
            self.create_notifications('FLAPPINGSTOP')
            # And update our status for modules
            self.broks.append(self.get_update_status_brok())
            # Also update the proxy objects that we did change
            proxyitemsmgr.update_flapping(self.get_instance_uuid(), self.is_flapping, inherited_flapping=self.inherited_flapping)
        
        if not self.is_flapping and r >= high_flap_threshold:
            self.is_flapping = True
            # We also raise a log entry
            self.inherited_flapping = False
            self.raise_flapping_start_log_entry(r, high_flap_threshold)
            # and a notification
            self.create_notifications('FLAPPINGSTART')
            # And update our status for modules
            self.broks.append(self.get_update_status_brok())
            # Also update the proxy objects that we did change
            proxyitemsmgr.update_flapping(self.get_instance_uuid(), self.is_flapping, inherited_flapping=self.inherited_flapping)
        
        if not self.is_flapping and automatic:
            self.is_flapping = True
            # We also raise a log entry
            self.inherited_flapping = True
            self.raise_flapping_start_log_entry(r, high_flap_threshold)
            # and a notification
            self.create_notifications('FLAPPINGSTART')
            # And update our status for modules
            self.broks.append(self.get_update_status_brok())
            # Also update the proxy objects that we did change
            proxyitemsmgr.update_flapping(self.get_instance_uuid(), self.is_flapping, inherited_flapping=self.inherited_flapping)
    
    
    # For a time, print something like
    # 10m 37s
    @staticmethod
    def _sprintf_duration(time_delta, x_elts=2):
        if time_delta is None:
            return 'N/A'
        # print "T", t
        # Get the difference between now and the time of the user
        seconds = time_delta
        
        # Now manage all case like in the past
        seconds = abs(seconds)
        # print "In future?", in_future
        
        # print "sec", seconds
        seconds = long(round(seconds))
        # print "Sec2", seconds
        minutes, seconds = divmod(seconds, 60)
        hours, minutes = divmod(minutes, 60)
        days, hours = divmod(hours, 24)
        weeks, days = divmod(days, 7)
        months, weeks = divmod(weeks, 4)
        years, months = divmod(months, 12)
        
        minutes = long(minutes)
        hours = long(hours)
        days = long(days)
        weeks = long(weeks)
        months = long(months)
        years = long(years)
        
        duration = []
        if years > 0:
            duration.append('%dy' % years)
        else:
            if months > 0:
                duration.append('%dM' % months)
            if weeks > 0:
                duration.append('%dw' % weeks)
            if days > 0:
                duration.append('%dd' % days)
            if hours > 0:
                duration.append('%dh' % hours)
            if minutes > 0:
                duration.append('%dm' % minutes)
            if seconds > 0:
                duration.append('%ds' % seconds)
        
        # print "Duration", duration
        # Now filter the number of printed elements if ask
        if x_elts >= 1:
            duration = duration[:x_elts]
        
        return ' '.join(duration)
    
    
    # IMPORTANT: this method should be used ONLY BY THE BROKER/regenerator so
    # do not send Broks or launch checks or whatever the scheduler is doing
    # If the state is still fresh we return this state expiration time
    def look_for_fake_unknown_state_due_to_missing_data_in_broker(self, regenerator, lang, now):
        # type: (Regenerator, unicode, float) -> Optional[float]
        if self.state_validity_period == NO_END_VALIDITY:
            return None
        
        freshness_threshold = self.state_validity_period
        # Old versions (before 2.8.1.p6) of scheduler don't know this field
        if getattr(self, u'brok_version', 0) >= BROK_VERSION_V020801P7:
            if not (0 < self.missing_data_activation_time <= now):
                return self.missing_data_activation_time if self.missing_data_activation_time > 0 else None
            if self.state_validity_end_time > 0:
                freshness_threshold = self.missing_data_activation_time - self.state_validity_end_time
        else:
            # Fallback with old computing
            data_age = now - self.last_broker_data_update
            if data_age < 0:  # last check in the future? skip this
                return None
            
            margin = 60
            if regenerator.minimal_time_before_an_element_become_missing_data:
                freshness_threshold = max(regenerator.minimal_time_before_an_element_become_missing_data, freshness_threshold + margin)
            
            # State is still fresh
            if data_age < (freshness_threshold + margin):
                return freshness_threshold + margin + now
        
        end_of_validity = int(getattr(self, u'state_validity_end_time', 0))
        if end_of_validity <= 0:
            end_of_validity = int(time.time())
        
        # Ok so end_of_validity fake the result
        # Save a change only if we really change state
        current_state_id = self.state_id if not self.got_business_rule else self.bp_state
        if current_state_id != STATE_ID.UNKNOWN or self.state != u'UNKNOWN':
            self.last_state_as_string = self.state
            self.last_state_id = current_state_id
            self.last_state_change = end_of_validity
            self.last_missing_data_start_in_broker = end_of_validity
            if getattr(self, u'brok_version', 0) >= BROK_VERSION_V020801P15:
                current_full_status = getattr(self, u'current_full_status', None)
                if current_full_status is not None:
                    self.current_full_status = FullStatusManipulator.set_missing_data_state_in_full_status(self.current_full_status)
                    self.full_status_change_time = end_of_validity
            self.current_full_status = (self.current_full_status / 10) * 10 + 4
            self.update_time_full_status = end_of_validity
        self.state_type = u'HARD'
        self.state_type_id = STATE_TYPE_ID.HARD
        self.state = u'UNKNOWN'
        if self.got_business_rule:
            self.bp_state = STATE_ID.UNKNOWN
        else:
            self.state_id = STATE_ID.UNKNOWN
        language = lang  # self.__class__.language is not available :(
        time_string = print_human_readable_date_time(end_of_validity, language)
        threshold_string = self._sprintf_duration(freshness_threshold)
        error_format = MISSING_DATA_OUTPUT.get(language, MISSING_DATA_OUTPUT[u'en'])
        error_text = error_format % (time_string, threshold_string)
        self.output = error_text
        self.long_output = ''
        
        # Invalidation of list cache (see webui-enterprise/plugins/tabular/tabular.py test_remote_tabular)
        if hasattr(self, 'list_cache_var'):
            # delattr(self, 'list_cache_var')
            self.list_cache_var = {}
        return None
    
    
    # Add an attempt but cannot be more than max_check_attempts
    def add_attempt(self):
        self.set_attempt(self.attempt + 1)
    
    
    def set_attempt(self, attempt):
        if attempt == 0:
            raise Exception('BAD set attempt for an element! %s  state=%s  last_state=%s  state_type=%s  max_check_attempts=%s. Stopping daemon to avoid data corruption.' % (
                self.get_full_name(), self.state, self.last_state, self.state_type, self.max_check_attempts))
        new_attempt = min(attempt, self.max_check_attempts)
        if self.attempt != new_attempt:
            if MONITORING_CHECK_CONSUME_DEBUG_FLAG:
                logger.info('[monitoring] The element %30s attempt did change %s => %s (max=%s)' % (self.get_full_name(), self.attempt, new_attempt, self.max_check_attempts))
            self.attempt = new_attempt
    
    
    # Return True if attempt is at max
    def is_max_attempts(self):
        return self.attempt >= self.max_check_attempts
    
    
    # Do we have parents/service dependencies that we must check when we are in error before we send a notification?
    def have_parents_like_dependencies(self):
        return len(self.act_depend_of) != 0
    
    
    # Call by scheduler to see if last state is older than
    # freshness_threshold if check_freshness, then raise a check
    # even if active check is disabled
    def do_check_freshness(self):
        now = time.time()
        # Before, check if class (host or service) have check_freshness OK
        # Then check if item want freshness, then check freshness
        cls = self.__class__
        if not self.in_checking and \
                cls.global_check_freshness and \
                self.check_freshness and \
                self.freshness_threshold != 0 and \
                self.last_state_update < now - (self.freshness_threshold + cls.additional_freshness_latency) and \
                self.passive_checks_enabled and \
                not self.active_checks_enabled:
            
            if self.check_period is None or self.check_period.is_time_valid(now):
                self.raise_freshness_log_entry(int(now - self.last_state_update), int(now - self.freshness_threshold))
                return self.launch_check(now, CHECK_LAUNCH_MODE.FRESHNESS)
            else:
                logger.debug("Should have checked freshness for passive only checked host:%s, but host is not in check period." % self.host_name)
        return None
    
    
    # Raise all impact from my error. I'm setting myself
    # as a problem, and I register myself as this in all
    # hosts/services that depend_on_me. So they are now my
    # impacts
    def set_myself_as_problem(self, enable_impact_state):
        now = time.time()
        
        # If I'm a bp_rule based, I cannot be a real problem, so don't set is_problem=True
        # but update business impact value if needed (business impact modulation) and
        # warn fathers about our business impact value
        if self.got_business_rule:
            # Update business impact value if needed
            self.update_business_impact_value()
            return
        
        was_pb = self.is_problem
        self.is_problem = True
        
        # we should warn potentials impact of our problem, and they should be cool to register them, so I've got my impacts list
        impacts = list(self.impacts)
        for (impact, status, dep_type, tp, inh_par) in self.act_depend_of_me:
            # Check if the status is ok for impact
            for s in status:
                if self.is_state(s):
                    # now check if we should bail out because of a not good timeperiod for dep
                    if tp is None or tp.is_time_valid(now):
                        new_impacts = impact.register_a_problem(self, enable_impact_state)
                        impacts.extend(new_impacts)
        
        # Only update impacts and create new brok if impacts changed, and you didn't change the is_problem bool
        s_impacts = set(impacts)
        if was_pb and s_impacts == set(self.impacts):
            return
        self.impacts = list(s_impacts)
        
        # We can update our business_impact value now
        self.update_business_impact_value()
        
        # And we register a new broks for update status
        self.broks.append(self.get_update_status_brok())
    
    
    # We update our 'business_impact' value with the max of
    # the impacts business_impact if we got impacts. And save our 'configuration'
    # business_impact if we do not have done it before
    # If we do not have impacts, we revert our value
    def update_business_impact_value(self):
        # We look at our crit modulations. If one apply, we take apply it, and it's done
        in_modulation = False
        for cm in self.business_impact_modulations:
            now = time.time()
            period = cm.modulation_period
            if period is None or period.is_time_valid(now):
                self.business_impact = cm.business_impact
                in_modulation = True
                # We apply the first available, that's all
                break
        
        # Save our updated business_impact if not already done
        if self.my_own_business_impact == -1:
            self.my_own_business_impact = self.business_impact
        
        # If we truly have impacts, we get the max business_impact
        # if it's huger than ourselves
        # also look at the cluster I'm in
        my_clusters_uuids = proxyitemsgraph.father_to_sons.get(self.get_instance_uuid(), [])
        my_clusters_business_impacts = [proxyitemsmgr[c_uuid].business_impact for c_uuid in my_clusters_uuids]
        
        if len(self.impacts) != 0 or len(my_clusters_business_impacts) != 0:
            if len(my_clusters_business_impacts) != 0:
                self.business_impact = max(self.business_impact, max(my_clusters_business_impacts))
            if len(self.impacts) != 0:
                self.business_impact = max(self.business_impact, max([e.business_impact for e in self.impacts]))
            # Update proxy item
            proxyitemsmgr.update_business_impact(self.get_instance_uuid(), self.business_impact)
            return
        
        # If we are not a problem, we set up our own_crit if we are not in a modulation period
        if self.my_own_business_impact != -1 and not in_modulation:
            self.business_impact = self.my_own_business_impact
            # update proxy item too
            proxyitemsmgr.update_business_impact(self.get_instance_uuid(), self.business_impact)
    
    
    # Look for my impacts, and remove me from theirs problems list
    def no_more_a_problem(self):
        was_pb = self.is_problem
        
        if self.is_problem:
            self.is_problem = False
            
            # we warn impacts that we are no more a problem
            for impact in self.impacts:
                impact.deregister_a_problem(self)
            
            # we can just drop our impacts list
            self.impacts = []
        
        # We update our business_impact value, it's not a huge thing :)
        self.update_business_impact_value()
        
        # If we were a problem, we say to everyone
        # our new status, with good business_impact value
        if was_pb:
            # And we register a new broks for update status
            self.broks.append(self.get_update_status_brok())
    
    
    # Call recursively by potentials impacts, so they
    # update their source_problems list. But do not
    # go below if the problem is not a real one for me
    # like If I've got multiple parents for examples
    def register_a_problem(self, pb, enable_impact_state):
        # Maybe we already have this problem? If so, bailout too
        if pb in self.source_problems:
            return []
        
        now = time.time()
        was_an_impact = self.is_impact
        # Our father already look of he impacts us.
        # So if we are here, it's that we really are impacted
        self.is_impact = True
        
        impacts = []
        # Ok, if we are impacted, we can add it in our  problem list
        
        # Maybe I was a problem myself, now I can say: not my fault!
        if self.is_problem:
            self.no_more_a_problem()
        
        # Ok, we are now an impact, we should take the good state but only when we just go in impact state
        # Cannot impact cluster, they really compute their states, they don't need a shortcut
        if enable_impact_state and not was_an_impact and not self.got_business_rule:
            self.set_impact_state()
            proxyitemsmgr.update_state(self.get_instance_uuid(), self.state_id)
        
        # Ok now we can be a simple impact
        impacts.append(self)
        if pb not in self.source_problems:
            self.source_problems.append(pb)
            # logger.debug('ROOT PROBLEM: %s got a new root problem: %s (%s)' % (self.get_name(), pb.get_name(), pb.get_instance_uuid()))
            proxyitemsmgr.update_root_problems(self.get_instance_uuid(), [src.get_instance_uuid() for src in self.source_problems])
        # we should send this problem to all potential impact that
        # depend on us
        for (impact, status, dep_type, tp, inh_par) in self.act_depend_of_me:
            # Check if the status is ok for impact
            for s in status:
                if self.is_state(s):
                    # now check if we should bail out because of a
                    # not good timeperiod for dep
                    if tp is None or tp.is_time_valid(now):
                        new_impacts = impact.register_a_problem(pb, enable_impact_state)
                        impacts.extend(new_impacts)
        
        # And we register a new broks for update status
        self.broks.append(self.get_update_status_brok())
        
        # now we return all impacts (can be void of course)
        return impacts
    
    
    # Just remove the problem from our problems list
    # and check if we are still 'impacted'. It's not recursive because problem
    # got the list of all its impacts
    def deregister_a_problem(self, pb):
        self.source_problems.remove(pb)
        proxyitemsmgr.update_root_problems(self.get_instance_uuid(), [src.get_instance_uuid() for src in self.source_problems])
        
        # For know if we are still an impact, maybe our dependencies
        # are not aware of the remove of the impact state because it's not ordered,
        # so we can just look at if we still have some problem in our list
        if len(self.source_problems) == 0:
            self.is_impact = False
            # No more an impact, we can unset the impact state
            self.unset_impact_state()
        
        # And we register a new broks for update status
        self.broks.append(self.get_update_status_brok())
    
    
    # When all dep are resolved, this function say if action can be raised or not by viewing dep
    # status network_dep have to be all raise to be no action
    # logic_dep: just one is enough
    def is_root_problem(self):
        # Use to know if notification is raise or not
        parent_is_down = []
        
        # If I have no dependency it is my fault
        if not self.have_parents_like_dependencies():
            return True
        
        # So if one logic is Raise, is dep is one network is no ok, is not dep at the end, raise no dep
        for (dep, status, _type, tp, inh_par) in self.act_depend_of:
            # For logic_dep, only one state raise put no action
            if _type == 'logic_dep':
                for s in status:
                    if dep.is_state(s):
                        return False
            # more complicated: if none of the states are match, the host is down so -> network_dep
            else:
                p_is_down = False
                dep_match = [dep.is_state(s) for s in status]
                # check if the parent match a case, so he is down
                if True in dep_match:
                    p_is_down = True
                parent_is_down.append(p_is_down)
        # if a parent is not down, no dep can explain the pb
        if False in parent_is_down:
            return True
        else:  # every parents are dead, so... It's not my fault :)
            return False
    
    
    # We check if we are no action just because of ours parents (or host for service)
    # TODO: factorize with previous check?
    def check_and_set_unreachability(self):
        # We must have all parents dead to be unreachable
        all_parents_are_down = self.are_all_parents_down()
        
        if all_parents_are_down and self.state != 'UNREACHABLE':
            self.set_unreachable()
    
    
    def are_all_parents_down(self):
        # We have no parents? They are not DOWN :)
        if not self.have_parents_like_dependencies():
            return False
        # We have at least one parent, so if one it not down, we have our answer
        for (dep, status, _type, _, _) in self.act_depend_of:
            if _type == 'network_dep':
                is_down = any((dep.is_state(s) for s in status))
                if not is_down:
                    return False
        return True
    
    
    # Use to know if I raise dependency for someone else (with status)
    # If I do not raise dep, maybe my dep raise me. If so, I raise dep.
    # So it's a recursive function
    def do_i_raise_dependency(self, status, inherit_parents):
        # Do I raise dep ?
        for s in status:
            if self.is_state(s):
                return True
        
        # If we do not inherit parent, we have no reason to be blocking
        if not inherit_parents:
            return False
        
        # Ok, I do not raise dep, but my dep maybe raise me
        now = time.time()
        for (dep, status, _type, tp, inh_parent) in self.chk_depend_of:
            if dep.do_i_raise_dependency(status, inh_parent):
                if tp is None or tp.is_time_valid(now):
                    return True
        
        # No, I really do not raise...
        return False
    
    
    # Use to know if my dep force me not to be checked
    # So check the chk_depend_of if they raise me
    def is_no_check_dependent(self):
        now = time.time()
        for (dep, status, _type, tp, inh_parent) in self.chk_depend_of:
            if tp is None or tp.is_time_valid(now):
                if dep.do_i_raise_dependency(status, inh_parent):
                    return True
        return False
    
    
    # call by a bad consume check where item see that he have dep,
    # and maybe he is not in real fault.
    # Returns: if there were dep checks created
    def raise_dependencies_check(self, my_bad_check):
        now = time.time()
        cls = self.__class__
        raised_dep_checks_uuids = []
        for (dep, status, _type, tp, inh_par) in self.act_depend_of:
            # If the dep timeperiod is not valid, do not raise the dep,
            # None=everytime
            if tp is not None and not tp.is_time_valid(now):
                continue
            
            # if the update is 'fresh', do not raise dep,
            # cached_check_horizon = cached_service_check_horizon for service
            if dep.last_state_update >= now - cls.cached_check_horizon:
                continue
            
            # If the parent do not have active checks, then it's useless to ask for a command
            # launch
            if not dep.active_checks_enabled:
                continue
            
            if self.should_ask_dependency_check(dep):
                raised_check_uuid = dep.launch_check(now, CHECK_LAUNCH_MODE.DEPENDENCY, depend_on_me=my_bad_check)
                if raised_check_uuid is not None:
                    raised_dep_checks_uuids.append(raised_check_uuid)
                    if MONITORING_CHECK_CONSUME_DEBUG_FLAG:
                        logger.info(u'[monitoring] [%s] will now wait for check from %s (check id=%s)' % (self.get_full_name(), dep.get_full_name(), raised_check_uuid))
        
        if len(raised_dep_checks_uuids) == 0:
            return False  # cannot create dep checks, so our own check will be able to be fully consumed
        
        # Let the check know that we are waiting for parent checks
        my_bad_check.register_dependency_checks(raised_dep_checks_uuids)  # -> my bad check goes WAIT DEP
        
        # When we are unreachable we need to wait until we get results for all dependent checks
        # before we send the brok ; if one is back up, the result should not be "UNREACHABLE" anymore
        if self.state != 'UNREACHABLE':
            self.broks.append(self.get_check_result_brok())
        if MONITORING_CHECK_CONSUME_DEBUG_FLAG:
            logger.info('[monitoring] The element %-20s check analysis is postpone until its parents are checked:  state=%s  last_state=%s  state_type=%s attempt=%s max_check_attempts=%s  because of %s parents' % (
                self.get_full_name(), self.state, self.last_state, self.state_type, self.attempt, self.max_check_attempts, len(self.act_depend_of)))
        
        return True  # we did create dep checks so our check will have to wait
    
    
    # Ask for a dep check only if the dependence is UP/OK and the no UP/OK state is soft and the last call was not too fresh
    # See SEF-9343
    def should_ask_dependency_check(self, dependency):
        # type: (SchedulingItem, SchedulingItem) -> bool
        requester = self
        dependency_ok_up = dependency.__class__.ok_up
        requester_ok_up = self.__class__.ok_up
        logger_monitoring = LoggerFactory.get_logger(u'SCHEDULING').get_sub_part(u'STATUS DEPENDENCY UPDATE')
        
        # dep on pending are asking a check for display a state in ui-interface
        if dependency.state == u'PENDING':
            logger_monitoring.debug(u'〖 %s 〗 ask check of 〖 %s 〗 because 〖 %s 〗 is in state %s' % (requester.get_full_name(), dependency.get_full_name(), dependency.get_full_name(), dependency.state))
            return True
        
        # the no UP/OK state is hard, so we trust this state
        if dependency.state != dependency_ok_up and dependency.state_type == u'HARD':
            logger_monitoring.debug(u'〖 %s 〗 => no check needed 〖 %s 〗 because it is in state %s and %s' % (requester.get_full_name(), dependency.get_full_name(), dependency.state, dependency.state_type))
            return False
        
        # If the requester is in soft not OK and dependency is not OK we don't ask a dependency check
        if dependency.state != dependency_ok_up and requester.state != requester_ok_up and requester.state_type == u'SOFT':
            logger_monitoring.debug(u'〖 %s 〗 => no check needed 〖 %s 〗 because 〖 %s 〗 is in SOFT' % (requester.get_full_name(), dependency.get_full_name(), requester.get_full_name()))
            return False
        
        # the last check is fresh enough : valid period is retry_interval / max_check_attempts for active check for other it is 60 by default (passive host will not launch this check)
        if dependency.retry_interval == 0 or dependency.max_check_attempts == 0 or not dependency.active_checks_enabled or not dependency.__class__.execute_checks:
            last_check_state_valid_period_before_new_recheck_for_dependency = dependency.__class__.interval_length
        else:
            last_check_state_valid_period_before_new_recheck_for_dependency = dependency.retry_interval * dependency.__class__.interval_length / dependency.max_check_attempts
        now = time.time()
        state_period = abs(now - dependency.last_chk)  # abs : cause time change can happen
        if state_period < last_check_state_valid_period_before_new_recheck_for_dependency:
            logger_monitoring.debug(u'〖 %s 〗 => no check needed 〖 %s 〗 because last check is still valid ( asked at %ss and valid during %ss )' % (
                requester.get_full_name(), dependency.get_full_name(), state_period, last_check_state_valid_period_before_new_recheck_for_dependency))
            return False
        
        logger_monitoring.debug(u'〖 %s 〗 ask check of 〖 %s 〗 because status expired (  The last check was done %ss ago and validity was %ss )' % (
            requester.get_full_name(), dependency.get_full_name(), state_period, last_check_state_valid_period_before_new_recheck_for_dependency))
        return True
    
    
    # def _log_schedule(self, now, msg):
    #     _logger = LoggerFactory.get_logger(u'SCHEDULE ITEM')
    #     if _logger.is_enable():
    #         _logger.debug(u'''〖%s〗〖%s〗 %s''' % (self.get_full_name(), PartLogger.format_time(now), msg))
    
    # Main scheduling function
    # If a check is in progress, or active check are disabled, do not schedule a check.
    # The check interval change with HARD state or not:
    # * SOFT: retry_interval
    # * HARD: check_interval
    # The first scheduling is evenly distributed, so all checks are not launched at the same time.
    # IMPORTANT: we do all we can to keep this initial spreading, to not synchronize checks because of forced checks (dep) or from uis
    def compute_next_chk(self, force=False, force_time=None, startup_minimal_time_before_missing=None, update_validity_end_time=False, force_check_spread_out=False):
        # type: (bool, int, int, bool, bool) -> (bool, bool)
        
        cls = self.__class__
        now = int(time.time())
        scheduler_is_starting = startup_minimal_time_before_missing is not None
        scheduler_startup_forced_spread = False
        
        # used to create the check with the 'retry' cause
        is_a_retry = False
        
        if force:
            # self._log_schedule(now, u'Schedule start with params ( force:〖%s〗 force_time:〖%s〗 startup_minimal_time_before_missing:〖%s〗 )' % (force, force_time, startup_minimal_time_before_missing))
            if force_time is None:
                force_time = now
            
            # self._log_schedule(now, u'schedule is forced at 〖%s〗' % PartLogger.format_hours(force_time))
            self.next_chk = int(force_time)
        
        else:
            
            # if the check is in checking (on poller) we only reschedule if we force time
            if self.in_checking and not force_time:
                # self._log_schedule(now, u'Schedule ends because check is in_checking')
                return is_a_retry, False
            
            # self._log_schedule(now, u'Schedule starts with params ( force:〖%s〗 force_time:〖%s〗 startup_minimal_time_before_missing:〖%s〗 )' % (force, force_time, startup_minimal_time_before_missing))
            # if active check is not enabled we skip schedule
            if not self.active_checks_enabled or not cls.execute_checks:
                # self._log_schedule(now, u'Schedule ends because active check is disabled. self.active_checks_enabled:〖%s〗 cls.execute_checks:〖%s〗' % (self.active_checks_enabled, cls.execute_checks))
                return is_a_retry, False
            
            # If check_interval is 0, we should not add it for a service but suppose a 5min sched for hosts
            if self.check_interval == 0:
                if cls.my_type == u'service':
                    # self._log_schedule(now, u'Schedule ends because self.check_interval:〖%s〗 is 0 on service' % self.check_interval)
                    return is_a_retry, False
                else:  # host
                    self.check_interval = 300 / cls.interval_length
                    # self._log_schedule(now, u'Schedule forces self.check_interval to 〖%s〗 on host because old value was 0.' % self.check_interval)
            
            # Interval change is in a HARD state or not. If the retry is 0, take the normal value
            if self.state_type == u'HARD' or self.retry_interval == 0:
                interval = self.check_interval * cls.interval_length
                # self._log_schedule(now, u'Schedule will use the self.check_interval:〖%s〗.' % interval)
            else:
                interval = self.retry_interval * cls.interval_length
                # self._log_schedule(now, u'Schedule will use the self.retry_interval:〖%s〗.' % interval)
                is_a_retry = True
            
            time_add = interval
            
            # If the scheduler daemon was call with force_check_spread_out argument, we are wishing to DROP all previous scheduling
            # and so to simulate this we are setting the next_chk to its default value 0
            if force_check_spread_out or not self.next_chk:
                self.next_chk = 0
                self.last_normal_chk = 0
            
            # Maybe we load next_chk from retention and the value of the next_chk is still the past
            # Other case: we load a next_chk that is in distant future because it leaped a time period.
            # Then, evaluate check_period again, it may have changed to a closer one.
            have_a_check_period = self.check_period and not getattr(self.check_period, u'time_is_always_valid',
                                                                    False)  # IMPORTANT: getattr instead if self. => time_is_always_valid was added in 02.07.07/02.08.02, so with a new scheduler/old arbiter, we can miss the property. DELETE the getattr in the next version
            time_is_invalid_in_timeperiod = have_a_check_period and not self.check_period.is_time_valid(self.next_chk)
            must_reschedule = not (now < self.next_chk < now + interval) or time_is_invalid_in_timeperiod
            
            # self._log_schedule(now, u'〖Last schedule〗 is in check period = have_a_check_period:〖%s〗 time_is_invalid_in_timeperiod:〖%s〗' % (have_a_check_period, time_is_invalid_in_timeperiod))
            # self._log_schedule(now, u'〖Last schedule〗 is in interval = 〖%s〗<〖%s〗<〖%s〗' % (PartLogger.format_time(now), PartLogger.format_time(self.next_chk), PartLogger.format_time(now + interval)))
            # self._log_schedule(now, u'〖Last schedule〗 is valid must_reschedule:〖%s〗' % must_reschedule)
            
            # If we must compute a new next_chk we verify the checks spreading.
            if must_reschedule:
                # self._log_schedule(now, u'A schedule is needed. must_respread:〖%s〗 scheduler_is_starting:〖%s〗' % (self.must_respread, scheduler_is_starting))
                # Determine when a new check (randomize and distribute next check time) or recurring check should happen.
                if self.next_chk == 0:
                    # At the start, we cannot have an interval more than cls.max_check_spread
                    # But after the first schedule we must reschedule for spread check with interval > cls.max_check_spread * cls.interval_length
                    # This check are identify with must_respread
                    # self._log_schedule(now, u'〖spreading check〗 schedule self.next_chk value is 0. We spread check.')
                    if interval > cls.max_check_spread * cls.interval_length:
                        interval = cls.max_check_spread * cls.interval_length
                        self.must_respread = True
                        
                        # self._log_schedule(now, u'〖spreading check〗 schedule interval > max_check_spread. Using max_check_spread for interval and must_respread')
                    time_add = interval * random.uniform(0.0, 1.0)
                
                elif self.must_respread and force_time is None and self.state_type == u'HARD':  # Spread must occur on a normal schedule on check_interval to be recorded
                    time_add = interval * random.uniform(0.0, 1.0)
                    self.must_respread = False
                    # self._log_schedule(now, u'〖spreading check〗 We spread check because must_respread.')
                
                elif scheduler_is_starting:
                    # outdated check on startup or conf reload, force a fast schedule with max_check_spread
                    interval = min(interval, cls.max_check_spread * cls.interval_length)
                    time_add = interval * random.uniform(0.0, 1.0)
                    scheduler_startup_forced_spread = True
                    # self._log_schedule(now, u'〖spreading check〗 We spread check because scheduler_is_starting.')
                
                # Be sure we are going at least 1s step, and not bellow
                time_add = max(1.0, time_add)
                
                starting_point = self.next_chk
                # If we are hard, use the last hard time for computing, instead of the current self.next_chk that is the last retry one
                # Because if a lot of equipment recover at same time we want to keep spread as hard
                if (self.state_type == u'HARD' or starting_point > now) and self.last_normal_chk:
                    # starting time has already been randomized, no need to had extra randomness ( #SEF-10906: (random + random) = not so random )
                    starting_point = self.last_normal_chk
                    time_add = interval
                    if starting_point > now:
                        # Get back to previous last_normal_chk (before now) after a forced schedule
                        nb_interval = int(float(starting_point - now) / interval)
                        starting_point = starting_point - (nb_interval + 1) * interval

                    # self._log_schedule(now, u'〖starting point〗 state_type is HARD, we use last_normal_chk:〖%s〗 as starting point.' % PartLogger.format_time(starting_point))
                
                if not starting_point or starting_point > now:
                    starting_point = now
                    # self._log_schedule(now, u'〖starting point〗 No last schedule found, use now:〖%s〗 as starting point.' % PartLogger.format_time(starting_point))
                
                next_supposed_time = starting_point
                
                nb_time_interval = float(now - starting_point) / time_add
                if nb_time_interval > 1.0:
                    # We set next_supposed_time juste before now with a padding of time_add to keep spreading
                    next_supposed_time = starting_point + (int(nb_time_interval) * time_add)
                next_supposed_time = int(next_supposed_time + time_add)
                
                # self._log_schedule(now, u'〖starting point〗 starting point:〖%s〗 time_add:〖%s〗 next_supposed_time:〖%s〗' % (PartLogger.format_time(starting_point), time_add, PartLogger.format_time(next_supposed_time)))
                
                if have_a_check_period:
                    next_valid_time = self.check_period.get_next_valid_time_from_t(next_supposed_time)
                    # self._log_schedule(now, u'〖check_period:%s〗 Check have a check period. next_supposed_time:〖%s〗 next_valid_time:〖%s〗' % (self.check_period.get_name(), PartLogger.format_time(next_supposed_time), PartLogger.format_time(next_valid_time)))
                    if next_valid_time:
                        # We switched to next valid time range in timeperiod
                        if next_supposed_time != next_valid_time:
                            # So we must re-spread check from the beginning of the time range
                            time_add = interval * random.uniform(0.0, 1.0)
                            next_supposed_time = next_valid_time + time_add
                            
                            # self._log_schedule(now, u'〖check_period:%s〗 next_supposed_time was not in check_period, we re-spread next check with interval. New time_add:〖%s〗 new next_supposed_time:〖%s〗' % (self.check_period.get_name(), time_add, PartLogger.format_time(next_supposed_time)))
                            
                            # We check the re-spread is in timeperiod if not
                            next_invalid_time = self.check_period.get_next_invalid_time_from_t(next_valid_time, max_period_to_search=next_supposed_time - next_valid_time)
                            # self._log_schedule(now, u'〖check_period:%s〗 next_invalid_time:〖%s〗 for check_period from:〖%s〗 to:〖%s〗 ' % (self.check_period.get_name(), PartLogger.format_time(next_invalid_time), PartLogger.format_time(next_valid_time), PartLogger.format_time(next_supposed_time)))
                            if next_invalid_time:
                                time_add = (next_invalid_time - next_valid_time) * random.uniform(0.0, 1.0)  # Spread this check on the small valid range
                                next_supposed_time = next_valid_time + time_add
                                # self._log_schedule(now, u'〖check_period:%s〗 New next_supposed_time:〖%s〗 because old value was in invalid range' % (self.check_period.get_name(), PartLogger.format_time(next_supposed_time)))
                    
                    else:
                        next_supposed_time = None
                        # self._log_schedule(now, u'〖check_period:%s〗 No valid time found in timerange' % self.check_period.get_name())
                
                self.next_chk = next_supposed_time
                # self._log_schedule(now, u'next_chk:〖%s〗 ' % PartLogger.format_time(self.next_chk))
                
                if self.next_chk is None:
                    self.state_validity_period = NO_END_VALIDITY
                    self.missing_data_activation_time = 0
                    self.state_validity_end_time = 0
                else:
                    self.state_validity_period = self.next_chk - now
                    if update_validity_end_time:
                        self.state_validity_end_time = self.next_chk
                    if update_validity_end_time or startup_minimal_time_before_missing:
                        command_timeout = self.check_running_timeout
                        if startup_minimal_time_before_missing is not None:
                            missing_data_add_delay = startup_minimal_time_before_missing
                        else:
                            missing_data_add_delay = cls.missing_data_add_delay
                        if self.state_validity_end_time <= 0:
                            validity_time = self.next_chk
                        else:
                            validity_time = self.state_validity_end_time
                        self.missing_data_activation_time = validity_time + missing_data_add_delay + command_timeout

                # self._log_schedule(now, u'state_validity_period:〖%s〗 ' % self.state_validity_period)
                
                # if we are hard, save the previous scheduling, but not for forced check
                must_set_last_normal_chk = self.state_type == u'HARD' and not scheduler_startup_forced_spread and self.next_chk
                # self._log_schedule(now, u'must_set_last_normal_chk:〖%s〗 ( self.state_type:〖%s〗 scheduler_startup_forced_spread:〖%s〗 self.next_chk:〖%s〗 )' % (must_set_last_normal_chk, self.state_type, scheduler_startup_forced_spread, self.next_chk))
                if must_set_last_normal_chk:
                    self.last_normal_chk = int(self.next_chk)
                    # self._log_schedule(now, u'Setting last_normal_chk at :〖%s〗 ' % self.last_normal_chk)
            
            # If next time is None, do not go
            if self.next_chk is None:
                # self._log_schedule(now, u'Schedule end with self.next_chk is None.')
                # Nagios do not raise it, I'm wondering if we should
                return is_a_retry, False
                
        return is_a_retry, True


    # Main scheduling function
    # If a check is in progress, or active check are disabled, do not schedule a check.
    # - Allow allow_next_schedule_brok = on special cases where next_chk is recomputed, generate a next_schedule brok
    def schedule(self, force=False, force_time=None, allow_next_schedule_brok=False, startup_minimal_time_before_missing=None, force_check_spread_out=False):
        # next_chk il already set, do not change
        # unless we force the check or the time
        if self.in_checking and not (force or force_time):
            return None

        is_a_retry, have_to_check = self.compute_next_chk(force=force, force_time=force_time, startup_minimal_time_before_missing=startup_minimal_time_before_missing, force_check_spread_out=force_check_spread_out)
        # logger.debug(u'schedule: item[%s] compute_next_check -> [%s,%s]' % (self.get_full_name(), is_a_retry, have_to_check))
        if not have_to_check:
            return None

        # If next time is None, do not go
        if self.next_chk is None:
            # Nagios do not raise it, I'm wondering if we should
            return None

        # Get the command to launch, and put it in queue
        self.launch_check(self.next_chk, CHECK_LAUNCH_MODE.SCHEDULE, force=force, retry=is_a_retry, allow_next_schedule_brok=allow_next_schedule_brok)
        # logger.debug(u'schedule: item[%s] : check will be at [%s]' % (self.get_full_name(), time.strftime('%F %T', time.localtime(self.next_chk))))


    
    
    # If we've got a system time change, we need to compensate it
    # If modify all past value. For active one like next_chk, it's the current
    # checks that will give us the new value
    def compensate_system_time_change(self, difference):
        # We only need to change some value
        for p in on_time_change_update:
            val = getattr(self, p)  # current value
            # Do not go below 1970 :)
            val = max(0, val + difference)  # diff may be negative
            setattr(self, p, val)
    
    
    # For disabling active checks, we need to set active_checks_enabled
    # to false, but also make a dummy current checks attempts so the
    # effect is immediate.
    def disable_active_checks(self):
        self.active_checks_enabled = False
        for c in self.checks_in_progress:
            c.status = CHECK_STATUS.WAITCONSUME
            c.exit_status = self.state_id
            c.output = self.output
            c.check_time = time.time()
            c.execution_time = 0
            c.perf_data = self.perf_data
    
    
    def remove_in_progress_check(self, c):
        # The check is consumed, update the in_checking properties
        if c in self.checks_in_progress:
            self.checks_in_progress.remove(c)
        self.update_in_checking()
    
    
    # Is in checking if and only if there are still checks not consumed
    def update_in_checking(self):
        self.in_checking = (len(self.checks_in_progress) != 0)
    
    
    # Del just a notification that is returned
    def remove_in_progress_notification(self, notification):
        self.last_notification = notification.check_time
        self.notification_latency = max(0, notification.check_time - notification.original_t_to_go)
        notification_uuid = notification.get_uuid()
        notification.status = 'zombie'
        if notification_uuid in self.notifications_in_progress:
            del self.notifications_in_progress[notification_uuid]
    
    
    # We zombify problems and recovery because a fresher state notification is available
    def remove_in_progress_status_notifications(self):
        for n in self.notifications_in_progress.values():
            if n.type in ('PROBLEM', 'RECOVERY'):
                self.remove_in_progress_notification(n)
    
    
    # Get an event handler if item got an event handler
    # command. It must be enabled locally and globally
    def get_event_handlers(self, externalcmd=False):
        cls = self.__class__
        
        # The external command always pass
        # if not, only if we enable them (auto launch)
        if self.event_handler is None or ((not self.event_handler_enabled or not cls.enable_event_handlers) and not externalcmd):
            return
        
        # If we do not force, and we are in downtime, bail out
        # if the no_event_handlers_during_downtimes is 1 in conf
        if cls.no_event_handlers_during_downtimes and not externalcmd and self.is_in_downtime():
            return
        
        m = MacroResolver()
        data = self.get_data_for_event_handler()
        cmd = m.resolve_command(self.event_handler, data)
        rt = self.event_handler.reactionner_tag
        shell_execution = self.event_handler.shell_execution
        command_name = getattr(getattr(self.event_handler, 'command', {}), 'command_name', 'MISSING_NAME')
        
        e = EventHandler(cmd, timeout=cls.event_handler_timeout, ref=self, reactionner_tag=rt, shell_execution=shell_execution, command_name=command_name)
        # print "DBG: Event handler call created"
        # print "DBG: ",e.__dict__
        self.raise_event_handler_log_entry(self.event_handler)
        logger.debug('[EVENTHANDLER] a new event handler %s is created (on the object %s local queue currently)' % (e.get_uuid(), self.get_full_name()))
        
        # ok we can put it in our temp action queue
        self.actions.append(e)
    
    
    def consume_event_result(self, event):
        self.last_event_handler = event.check_time
        self.eventhandler_latency = max(0, event.check_time - event.original_t_to_go)
    
    
    ##########################################################################################################
    #                                     DOWNTIME                                                           #
    ##########################################################################################################
    
    def _update_in_scheduled_downtime(self):
        if self.scheduled_downtime_depth >= 1:
            self.in_scheduled_downtime = True
        else:  # 0
            self.in_scheduled_downtime = False
    
    
    # A new downtime just enter() so w must start to be in downtime if needed
    def increase_downtime_depth(self):
        if self.scheduled_downtime_depth == 0:
            self.raise_enter_downtime_log_entry()
            self.create_notifications('DOWNTIMESTART')
        
        self.scheduled_downtime_depth += 1
        self._update_in_scheduled_downtime()
        # Update our proxy to allow other elements (like custer) to change themselves
        proxyitemsmgr.update_downtime(self.get_instance_uuid(), self.in_scheduled_downtime)
        
        services = getattr(self, 'services', [])
        for s in services:
            proxyitemsmgr.update_in_inherited_downtime(s.get_instance_uuid(), s.is_in_inherited_downtime())
    
    
    # A downtime just exit, decrease the level of downtime, and maybe we are no more in downtime
    def decrease_downtime_depth(self, reason='END'):
        self.scheduled_downtime_depth -= 1
        
        # Manage sub-zero case because old versions did have a cancel bug that make the
        # -1 -2 values possible
        if self.scheduled_downtime_depth <= 0:
            self.scheduled_downtime_depth = 0
            self.raise_exit_downtime_log_entry()
            self.create_notifications('DOWNTIME%s' % reason)
            self._update_in_scheduled_downtime()
            # Update our proxy to allow other elements (like custer) to change themselves
            proxyitemsmgr.update_downtime(self.get_instance_uuid(), self.in_scheduled_downtime)
            
            # when a downtime ends and the host/service was critical a notification is sent with the next critical check
            # So we should set a flag here which signals consume_result to send a notification
            self.in_scheduled_downtime_during_last_check = True
            
            services = getattr(self, 'services', [])
            for s in services:
                proxyitemsmgr.update_in_inherited_downtime(s.get_instance_uuid(), s.is_in_inherited_downtime())
                # Let the service know it was in downtime-inherited and so it is needed, it can send new notification now
                s.in_scheduled_downtime_during_last_check = True
    
    
    # Whenever a non-ok hard state is reached, we must check whether this
    # host/service has a flexible downtime waiting to be activated
    def check_for_flexible_downtime(self):
        status_updated = False
        for dt in self.downtimes:
            # activate flexible downtimes (do not activate triggered downtimes)
            if dt.fixed is False and dt.is_in_effect is False and dt.start_time <= self.last_chk and self.state_id != 0 and dt.trigger_id == 0:
                n = dt.enter()  # returns downtime start notifications
                if n is not None:
                    self.actions.append(n)
                status_updated = True
        if status_updated is True:
            self.broks.append(self.get_update_status_brok())
    
    
    # #SEF-6735: we can have downtime incoherency where a downtime can be entered() twice (I think because of trigger)
    # but I'm not sure about this, so at retention reload, try to detect it and fix it
    def detect_and_repair_downtime_incoherency(self):
        # If no downtime depth, no problem detected
        if self.scheduled_downtime_depth == 0:
            return
        # Bad case is sure when the depth is 1 or more, the count is not
        # the same as the number of active downtime
        nb_active_downtimes = len([dt for dt in self.downtimes if dt.in_scheduled_downtime()])
        if nb_active_downtimes != self.scheduled_downtime_depth:
            logger.error('%s The %s %s has bad downtime values (saved number of downtime=%s, actual=%s). We are fixing the values. Please report it to the support.' % (
                DOWNTIME_INCOHERENCY, self.my_type, self.get_full_name(), self.scheduled_downtime_depth, nb_active_downtimes))
            self.scheduled_downtime_depth = nb_active_downtimes
            self._update_in_scheduled_downtime()  # recompute if we are in downtime or not
    
    
    # UNKNOWN during a HARD state are not so important, and they should
    # not raise notif about it
    def update_hard_unknown_phase_state(self):
        self.was_in_hard_unknown_reach_phase = self.in_hard_unknown_reach_phase
        
        # We do not care about SOFT state at all,
        # and we are sure we are no more in such a phase
        if self.state_type != 'HARD' or self.last_state_type != 'HARD':
            self.in_hard_unknown_reach_phase = False
        
        # So if we are not in already in such a phase, we check for
        # a start or not. So here we are sure to be in a HARD/HARD following
        # state
        if not self.in_hard_unknown_reach_phase:
            if self.state == 'UNKNOWN' and self.last_state != 'UNKNOWN' \
                    or self.state == 'UNREACHABLE' and self.last_state != 'UNREACHABLE':
                self.in_hard_unknown_reach_phase = True
                # We also back up with which state we were before enter this phase
                self.state_before_hard_unknown_reach_phase = self.last_state
                return
        else:
            # if we were already in such a phase, look for its end
            if self.state != 'UNKNOWN' and self.state != 'UNREACHABLE':
                self.in_hard_unknown_reach_phase = False
        
        # If we just exit the phase, look if we exit with a different state
        # than we enter or not. If so, lie and say we were not in such phase
        # because we need so to raise a new notif
        if not self.in_hard_unknown_reach_phase and self.was_in_hard_unknown_reach_phase:
            if self.state != self.state_before_hard_unknown_reach_phase:
                self.was_in_hard_unknown_reach_phase = False
    
    
    # consume a check return and send action in return
    # main function of reaction of checks like raise notifications
    # Special case:
    # is_flapping: immediate notif when problem
    # is_in_scheduled_downtime: no notification
    # is_volatile: notif immediately (service only)
    def consume_result(self, check_to_consume):
        # type: (Check) -> None
        if MONITORING_CHECK_CONSUME_DEBUG_FLAG:
            logger.info(u'[monitoring] We are consuming the check (%s) from %-20s exit_code=%s check.status=%s => element state=%s  last_state=%s  state_type=%s attempt=%s max_check_attempts=%s' % (
                check_to_consume.get_printable_name(), self.get_full_name(), check_to_consume.exit_status, check_to_consume.status, self.state, self.last_state, self.state_type, self.attempt, self.max_check_attempts))
        cls = self.__class__
        ok_up = cls.ok_up  # OK for service, UP for host
        
        # Just be sure we are not here with an invalid check object
        if not check_to_consume.can_be_consume():
            raise Exception('ERROR: a check execution was trying to be consumed but without a valid state (%s). In order to protect against data integrity damage, the daemon is going to do an emergency stop' % check_to_consume.status)
        
        # if self.attempt == 0 and self.state != 'PENDING' and check_to_consume.status != CHECK_STATUS.ALL_DEP_ARE_FINISH:
        #     raise Exception('consume_result:: was bad since the beginning ! %-40s  state=%s  last_state=%s  state_type=%s  max_check_attempts=%s' % (self.get_dbg_name(), self.state, self.last_state, self.state_type, self.max_check_attempts))
        
        # We can make multiple consume_result if the check need to wait a dep (see wait dep state)
        first_consume_result = check_to_consume.status == CHECK_STATUS.WAITCONSUME
        if first_consume_result:
            # A check is waiting for dependency checks, freeze other checks on this element
            # Do not apply this freeze on ALL_DEP_ARE_FINISH check #SEF-9396
            blocking_check_waiting_for_its_dependencies = next((c for c in self.checks_in_progress if (c.get_uuid() != check_to_consume.get_uuid() and c.consume_is_waiting_for_dependencies())), None)
            if blocking_check_waiting_for_its_dependencies:
                # check_to_consume will come back here at each scheduler loop turn
                # this behaviour has been chosen to reduce code changes about checks and retention #SEF-8930
                if MONITORING_CHECK_CONSUME_DEBUG_FLAG:
                    logger.info(u'[ monitoring ] consume of check[%s] has already started, postponing check[%s]' % (blocking_check_waiting_for_its_dependencies, check_to_consume))
                return
            
            self._consume_result_data_from_check(check_to_consume)
        
        # if check_to_consume.status == CHECK_STATUS.ALL_DEP_ARE_FINISH:
        #     logger.info('DEP DEP DEP genius: %s is having a re-consume' % self.get_dbg_name())
        
        # If we got a bad result on a normal check, and we have dep, we raise dep checks,
        # put the actual check in wait dep, and we return all new checks
        if self.have_parents_like_dependencies() and check_to_consume.need_to_launch_dependency_checks():
            dep_checks_are_created = self.raise_dependencies_check(check_to_consume)
            if dep_checks_are_created:  # our check is now WAIT DEP, so we need to wait for dep checks to be
                return  # finish before consume again
        
        # Here all our parents are up-to-date, we can continue to consume our check now :)
        # Use to know if notification/event handlers are raise or not
        is_root_problem = self.is_root_problem()
        # logger.info('MIDDLE CONSUME: %s is a root problem at this point? => %s' % (self.get_dbg_name(), is_root_problem))
        # we change the state, do whatever we are or not in an impact mode, we can put it
        self.state_changed_since_impact = True
        
        self.duration_sec = time.time() - self.last_state_change
        
        # We recheck just for network_dep. Maybe we are just unreachable, and we need to override the state_id
        if check_to_consume.status == CHECK_STATUS.ALL_DEP_ARE_FINISH:
            self.check_and_set_unreachability()
        
        # Finish with this check object, take interesting data and clear it
        exit_status = check_to_consume.exit_status
        # We are done with this check state and dependencies, we will now only look at exit_status
        check_to_consume.set_fully_consumed()
        # The check is consumed, we do not need anymore in our host/service object
        self.remove_in_progress_check(check_to_consume)
        
        # noinspection PyUnusedLocal
        check_to_consume = None  # do no more touch this check from here!
        
        if exit_status == 0:
            # #SEF-8955 state_type fix, code moved in _consume_result_data_from_check with state update
            # keep this code commented to investigate potential behaviour change
            
            # Only root problem/impact management and notification stay here
            #
            # # OK following a previous OK. perfect if we were not in SOFT
            # if self.last_state in (ok_up, 'PENDING'):
            #     # print "Case 1 (OK following a previous OK): code:%s last_state:%s" % (c.exit_status, self.last_state)
            #     self.unacknowledge_problem()
            #
            #     # If first run, then allow to launch the event_handler
            #     if self.last_state == 'PENDING':
            #         self.get_event_handlers()
            #
            #     # action in return can be notification or other checks (dependencies)
            #     if (self.state_type == 'SOFT') and self.last_state != 'PENDING':
            #         if self.is_max_attempts():
            #             self.state_type = 'HARD'
            #     else:
            #         self.set_attempt(1)  # reset attempts
            #         self.state_type = 'HARD'
            # # OK following a NON-OK.
            # else:
            #     self.unacknowledge_problem()
            #     # print "Case 2 (OK following a NON-OK): code:%s last_state:%s" % (c.exit_status, self.last_state)
            #     if self.state_type == 'SOFT':
            #         # OK following a NON-OK still in SOFT state
            #         if not check_was_a_forced_check_by_a_son:
            #             self.add_attempt()
            #         self.raise_alert_log_entry()
            #         # Eventhandler gets OK;SOFT;++attempt, no notification needed
            #         self.get_event_handlers()
            #         # Internally it is a hard OK
            #         self.state_type = 'HARD'
            #         self.set_attempt(1)  # reset attempts
            #     elif self.state_type == 'HARD':
            #         # OK following a HARD NON-OK
            #         self.raise_alert_log_entry()
            #         # Eventhandler and notifications get OK;HARD;max_attempts
            #         # Ok, so current notifications are not needed, we 'zombie' them
            #         self.remove_in_progress_status_notifications()
            #         if is_root_problem:
            #             self.create_notifications('RECOVERY')
            #         self.get_event_handlers()
            #         # Internally it is a hard OK
            #         self.state_type = 'HARD'
            #         self.set_attempt(1)  # reset attempts
            #
            #         # self.update_hard_unknown_phase_state()
            #         # I'm no more a problem if I was one
            #         self.no_more_a_problem()
            if self.last_state not in (ok_up, 'PENDING'):
                if self.last_state_type == 'HARD':
                    # OK following a HARD NON-OK
                    # Eventhandler and notifications get OK;HARD;max_attempts
                    # Ok, so current notifications are not needed, we 'zombie' them
                    self.remove_in_progress_status_notifications()
                    if is_root_problem:
                        self.create_notifications('RECOVERY')
                    
                    # I'm no more a problem if I was one
                    self.no_more_a_problem()
        
        # Bad state
        elif exit_status != 0:
            # #SEF-8955 state_type fix, code moved in _consume_result_data_from_check with state update
            # keep this code commented to investigate potential behaviour change
            #
            # # Volatile part, Only for service, print "Case 3 (volatile only)"
            # if getattr(self, 'is_volatile', False):
            #     # There are no repeated attempts, so the first non-ok results are in a hard state
            #     self.set_attempt(1)  # reset attempts
            #     self.state_type = 'HARD'
            #     # status != 0 so add a log entry (before actions that can also raise log
            #     # it is smarter to log error before notification)
            #     self.raise_alert_log_entry()
            #     self.check_for_flexible_downtime()
            #     self.remove_in_progress_status_notifications()
            #     if is_root_problem:
            #         self.create_notifications('PROBLEM')
            #     # Ok, event handlers here too
            #     self.get_event_handlers()
            #
            #     # PROBLEM/IMPACT
            #     # I'm a problem only if I'm the root problem
            #     if is_root_problem:
            #         self.set_myself_as_problem(enable_impact_state=True)
            #
            # # NON-OK follows OK. Everything was fine, but now trouble is ahead
            # elif self.last_state in (ok_up, 'PENDING'):
            #     # print "Case 4: NON-OK follows OK: code:%s last_state:%s" % (c.exit_status, self.last_state)
            #     if self.is_max_attempts():
            #         # if max_attempts == 1 we're already in deep trouble
            #         self.state_type = 'HARD'
            #         self.raise_alert_log_entry()
            #         self.remove_in_progress_status_notifications()
            #         self.check_for_flexible_downtime()
            #         if is_root_problem:
            #             self.create_notifications('PROBLEM')
            #         # Oh? This is the typical go for an event handler :)
            #         self.get_event_handlers()
            #
            #         # PROBLEM/IMPACT
            #         # I'm a nuisance only if I'm the root_problem,
            #         # so not no_action:
            #         if is_root_problem:
            #             self.set_myself_as_problem(enable_impact_state=True)
            #
            #     else:
            #         # This is the first NON-OK result. Initiate the SOFT-sequence
            #         # Also launch the event handler, he might fix it.
            #         self.set_attempt(1)  # reset attempts
            #         self.state_type = 'SOFT'
            #         self.raise_alert_log_entry()
            #         self.get_event_handlers()
            # If no OK in a no OK: if hard, still hard, if soft, check at self.max_check_attempts
            # when we go in hard, we send notification
            # elif self.last_state != ok_up:
            #     # print "Case 5 (no OK in a no OK): code:%s last_state:%s state_type:%s" % (c.exit_status, self.last_state,self.state_type)
            #     if self.state_type == 'SOFT':
            #         if not check_was_a_forced_check_by_a_son:
            #             self.add_attempt()
            #         if self.is_max_attempts():  # GOING HARD
            #             # Ok here is when we just go to the hard state
            #             self.state_type = 'HARD'
            #             self.raise_alert_log_entry()
            #             self.remove_in_progress_status_notifications()
            #             # There is a request in the Nagios trac to enter downtimes
            #             # on soft states which does make sense. If this becomes
            #             # the default behavior, just move the following line
            #             # into the else-branch below.
            #             self.check_for_flexible_downtime()
            #             if is_root_problem:
            #                 self.create_notifications('PROBLEM')
            #             # So event handlers here too
            #             self.get_event_handlers()
            #
            #             # PROBLEM/IMPACT
            #             # I'm a problem only if I'm the root problem,
            #             # so not no_action:
            #             if is_root_problem:
            #                 self.set_myself_as_problem(enable_impact_state=True)
            #
            #         else:  # STAY SOFT
            #             self.raise_alert_log_entry()
            #             # eventhandler is launched each time during the soft state
            #             self.get_event_handlers()
            #     # Bad following a bad/HARD
            #     else:
            #         # print "Case 6 (else):", self.get_full_name()
            #         # Send notifications whenever the state has changed. (W -> C)
            #         # but not if the current state is UNKNOWN (hard C-> hard U -> hard C should not restart notifications)
            #         if self.state != self.last_state:
            #             self.update_hard_unknown_phase_state()
            #             # print self.last_state, self.last_state_type, self.state_type, self.state
            #             if not self.in_hard_unknown_reach_phase and not self.was_in_hard_unknown_reach_phase:
            #                 self.unacknowledge_problem_if_not_sticky()
            #                 self.raise_alert_log_entry()
            #                 self.remove_in_progress_status_notifications()
            #                 if is_root_problem:
            #                     self.create_notifications('PROBLEM')
            #
            #         elif self.in_scheduled_downtime_during_last_check is True:
            #             # during the last check I was in a downtime.
            #             # but now the status is still critical and notifications are possible again. send an alert immediately
            #             self.remove_in_progress_status_notifications()
            #             if is_root_problem:
            #                 self.create_notifications('PROBLEM')
            #
            #         # PROBLEM/IMPACT
            #         # Forces problem/impact registration even if no state change
            #         # was detected as we may have a non-OK state restored from
            #         # retention data. This way, we rebuild problem/impact hierarchy.
            #         # I'm a problem only if I'm the root problem,
            #         # so not no_action:
            #         if is_root_problem:
            #             self.set_myself_as_problem(enable_impact_state=True)
            
            # looking for :
            # 1. volatile element
            # 2. previously (OK or newly scheduled) element in (non-OK) HARD state now
            # 3. previously non-OK SOFT state element entering (non-OK) HARD state now
            # to manage notifications and impact
            if (getattr(self, 'is_volatile', False) or
                    (self.last_state in (ok_up, 'PENDING') and self.state_type == 'HARD') or
                    (self.last_state != ok_up and (self.last_state_type == 'SOFT' and self.state_type == 'HARD'))):
                self.remove_in_progress_status_notifications()
                if is_root_problem:
                    self.create_notifications('PROBLEM')
                    self.set_myself_as_problem(enable_impact_state=True)
            
            # looking for
            # previously non-OK HARD state element (so still HARD non-OK state)
            # to manage UNKNOWN / non-UNKNOWN (but still non OK) status switch + notification + impact
            elif self.last_state not in (ok_up, 'PENDING') and self.last_state_type == 'HARD':
                # print "Case 6 (else):", self.get_full_name()
                # Send notifications whenever the state has changed. (W -> C)
                # but not if the current state is UNKNOWN (hard C-> hard U -> hard C should not restart notifications)
                if self.state != self.last_state:
                    self.update_hard_unknown_phase_state()
                    # print self.last_state, self.last_state_type, self.state_type, self.state
                    if not self.in_hard_unknown_reach_phase and not self.was_in_hard_unknown_reach_phase:
                        self.unacknowledge_problem_if_not_sticky()
                        self.raise_alert_log_entry()
                        self.remove_in_progress_status_notifications()
                        if is_root_problem:
                            self.create_notifications('PROBLEM')
                
                elif self.in_scheduled_downtime_during_last_check is True:
                    # during the last check I was in a downtime.
                    # but now the status is still critical and notifications are possible again. send an alert immediately
                    self.remove_in_progress_status_notifications()
                    if is_root_problem:
                        self.create_notifications('PROBLEM')
                
                # PROBLEM/IMPACT
                # Forces problem/impact registration even if no state change
                # was detected as we may have a non-OK state restored from
                # retention data. This way, we rebuild problem/impact hierarchy.
                # I'm a problem only if I'm the root problem,
                # so not no_action:
                if is_root_problem:
                    self.set_myself_as_problem(enable_impact_state=True)
        
        self.update_hard_unknown_phase_state()
        # Reset this flag. If it was true, actions were already taken
        self.in_scheduled_downtime_during_last_check = False
        
        self.compute_next_chk(update_validity_end_time=True)
        
        # now is the time to update state_type_id and our last_hard_state
        if self.state_type == 'HARD':
            self.state_type_id = STATE_TYPE_ID.HARD
            self.last_hard_state = self.state
            self.last_hard_state_id = self.state_id
        else:
            self.state_type_id = STATE_TYPE_ID.SOFT
        
        self.broks.append(self.get_check_result_brok())
        
        if MONITORING_CHECK_CONSUME_DEBUG_FLAG:
            logger.info(u'[monitoring] Check return analysis is done. The element %-20s is now state=%s  last_state=%s  state_type=%s attempt=%s max_check_attempts=%s broks generated=%s' % (
                self.get_full_name(), self.state, self.last_state, self.state_type, self.attempt, self.max_check_attempts, len(self.broks)))
    
    
    def _consume_result_data_from_check(self, check_to_consume):
        ok_up = self.__class__.ok_up  # OK for service, UP for host
        launch_event_handler = False
        
        # Protect against bad type output/long_output and perf_data: if str, go in unicode
        check_to_consume.assert_unicode_strings()
        # Latency can be <0 is we get a check from the retention file so if <0, set 0
        try:
            self.latency = max(0, check_to_consume.check_time - check_to_consume.original_t_to_go)
        except TypeError:
            pass
        # Now get data from check
        self.last_chk = check_to_consume.check_time + check_to_consume.execution_time
        # Get output and forgot bad UTF8 values for simple str ones (we can get already unicode with external commands)
        self.output = check_to_consume.output
        self.long_output = check_to_consume.long_output
        # Set the check result type also in the host/service
        # 0 = result came from an active check
        # 1 = result came from a passive check
        self.check_type = check_to_consume.check_type
        # Get the perf_data only if we want it in the configuration
        if self.__class__.process_performance_data and self.process_perf_data:
            self.last_perf_data = self.perf_data
            self.perf_data = check_to_consume.perf_data
        elif not self.process_perf_data:
            self.last_perf_data = self.perf_data
            self.perf_data = u''
        # Let the check modulate based on our own result modulations. Will be done only once
        check_to_consume.apply_result_modulations(self.resultmodulations)
        # remember how we were before this check
        self.last_state_type = self.state_type
        # keep a trace of the old state_id
        old_state_id = self.state_id
        old_state_as_string = self.state
        if self.got_business_rule:
            old_state_id = self.bp_state
        if self.state == 'PENDING':
            old_state_id = STATE_ID.UNKNOWN
        self.set_state_from_exit_status(check_to_consume.exit_status)
        check_was_a_forced_check_by_a_son = check_to_consume.is_dependent()
        # #SEF-8955 state_type fix, this code has been moved from consume_result (post impact dependencies solving)
        # to here (at state update, before impact dependencies solving)
        if check_to_consume.exit_status == 0:
            # OK following a previous OK. perfect if we were not in SOFT
            if self.last_state in (ok_up, 'PENDING'):
                # print "Case 1 (OK following a previous OK): code:%s last_state:%s" % (c.exit_status, self.last_state)
                self.unacknowledge_problem()
                
                # If first run, then allow to launch the event_handler
                if self.last_state == 'PENDING':
                    launch_event_handler = True
                
                # action in return can be notification or other checks (dependencies)
                if (self.state_type == 'SOFT') and self.last_state != 'PENDING':
                    if self.is_max_attempts():
                        self.state_type = 'HARD'
                else:
                    self.set_attempt(1)  # reset attempts
                    self.state_type = 'HARD'
            # OK following a NON-OK.
            else:
                self.unacknowledge_problem()
                # print "Case 2 (OK following a NON-OK): code:%s last_state:%s" % (c.exit_status, self.last_state)
                if self.state_type == 'SOFT':
                    # OK following a NON-OK still in SOFT state
                    if not check_was_a_forced_check_by_a_son:
                        self.add_attempt()
                    self.raise_alert_log_entry()
                    # Eventhandler gets OK;SOFT;++attempt, no notification needed
                elif self.state_type == 'HARD':
                    # OK following a HARD NON-OK
                    self.raise_alert_log_entry()
                    # Eventhandler and notifications get OK;HARD;max_attempts
                    # #Ok, so current notifications are not needed, we 'zombie' them
                    # self.remove_in_progress_status_notifications()
                    # if is_root_problem:
                    #     self.create_notifications('RECOVERY')
                launch_event_handler = True
                # Internally it is a hard OK
                self.state_type = 'HARD'
                self.set_attempt(1)  # reset attempts
        
        # Bad state
        elif check_to_consume.exit_status != 0:
            # Volatile part, Only for service, print "Case 3 (volatile only)"
            if getattr(self, 'is_volatile', False):
                # There are no repeated attempts, so the first non-ok results are in a hard state
                self.set_attempt(1)  # reset attempts
                self.state_type = 'HARD'
                # status != 0 so add a log entry (before actions that can also raise log
                # it is smarter to log error before notification)
                self.raise_alert_log_entry()
                self.check_for_flexible_downtime()
                # Ok, event handlers here too
                launch_event_handler = True
            
            # NON-OK follows OK. Everything was fine, but now trouble is ahead
            elif self.last_state in (ok_up, 'PENDING'):
                # print "Case 4: NON-OK follows OK: code:%s last_state:%s" % (c.exit_status, self.last_state)
                if self.last_state == 'PENDING':
                    self.set_attempt(1)  # non initialized value default to 0, we count starting at 1 ...
                
                if self.is_max_attempts():
                    # if max_attempts == 1 we're already in deep trouble
                    self.state_type = 'HARD'
                    self.raise_alert_log_entry()
                    # self.remove_in_progress_status_notifications()
                    self.check_for_flexible_downtime()
                    # if is_root_problem:
                    #     self.create_notifications('PROBLEM')
                    # Oh? This is the typical go for an event handler :)
                    launch_event_handler = True
                    
                    # PROBLEM/IMPACT
                    # I'm a problem only if I'm the root problem,
                    # so not no_action:
                    # if is_root_problem:
                    #     self.set_myself_as_problem(enable_impact_state=True)
                
                else:
                    # This is the first NON-OK result. Initiate the SOFT-sequence
                    # Also launch the event handler, he might fix it.
                    self.set_attempt(1)  # reset attempts
                    self.state_type = 'SOFT'
                    self.raise_alert_log_entry()
                    launch_event_handler = True
            
            # If no OK in a no OK: if hard, still hard, if soft, check at self.max_check_attempts
            # when we go in hard, we send notification
            elif self.last_state != ok_up:
                # print "Case 5 (no OK in a no OK): code:%s last_state:%s state_type:%s" % (c.exit_status, self.last_state,self.state_type)
                if self.state_type == 'SOFT':
                    if not check_was_a_forced_check_by_a_son:
                        self.add_attempt()
                    if self.is_max_attempts():  # GOING HARD
                        # Ok here is when we just go to the hard state
                        self.state_type = 'HARD'
                        self.raise_alert_log_entry()
                        # self.remove_in_progress_status_notifications()
                        # There is a request in the Nagios trac to enter downtimes
                        # on soft states which does make sense. If this becomes
                        # the default behavior, just move the following line
                        # into the else-branch below.
                        self.check_for_flexible_downtime()
                        # if is_root_problem:
                        #     self.create_notifications('PROBLEM')
                        # So event handlers here too
                        launch_event_handler = True
                        
                        # PROBLEM/IMPACT
                        # I'm a problem only if I'm the root problem,
                        # so not no_action:
                        # if is_root_problem:
                        #     self.set_myself_as_problem(enable_impact_state=True)
                    
                    else:  # STAY SOFT
                        self.raise_alert_log_entry()
                        # eventhandler is launched each time during the soft state
                        launch_event_handler = True
        
        # Fill last_hard_state_change to now
        # if we just change from SOFT->HARD or
        # in HARD we change of state (Warning->critical, or critical->ok, etc etc)
        if self.state_type == 'HARD' and (self.last_state_type == 'SOFT' or self.last_state != self.state):
            self.last_hard_state_change = self.last_chk
        
        state_id = self.state_id
        if self.got_business_rule:
            state_id = self.bp_state
        if state_id != old_state_id or self.last_state == 'PENDING' or self.state != old_state_as_string:
            self.last_state_change = self.last_state_update
            self.last_state_id = old_state_id
            self.last_state_as_string = old_state_as_string
        
        self.check_flapping_change(state_id, old_state_id)
        # logger.debug("[scheduler] [%s] phase:set_state_from_exit_status new item state [%s-%s] last_state_change[%s] " % (self.get_full_name(), self.state_id, self.state, self.last_state_change))
        self.consecutive_ok = self.consecutive_ok + 1 if self.state == ok_up else 0
        # Now get real stats from our check, as we can really consume it once and only once here
        self.execution_time = check_to_consume.execution_time
        # Get the user/system execution ime for this
        # execution, and sum all
        self.u_time_sum += check_to_consume.get_cpu_time()
        self.s_time_sum += check_to_consume.s_time
        self.nb_executions += 1
        
        if launch_event_handler:
            self.get_event_handlers()
    
    
    # Just update the notification command by resolving Macros
    # And because we are just launching the notification, we can say
    # that this contact have been notified
    def update_notification_command(self, n):
        cls = self.__class__
        m = MacroResolver()
        data = self.get_data_for_notifications(n.contact, n)
        n.command = m.resolve_command(n.command_call, data, allow_cache=False)
        if cls.enable_environment_macros or n.enable_environment_macros:
            n.env = m.get_env_macros(data)
    
    
    # See if an escalation is eligible at t and notif nb=n
    # noinspection SpellCheckingInspection
    def is_escalable(self, n):
        cls = self.__class__
        
        # We search since when we are in notification for escalations
        # that are based on time
        in_notif_time = time.time() - n.creation_time
        
        # Check is an escalation match the current_notification_number
        for es in self.escalations:
            if es.is_eligible(n.t_to_go, self.state, n.notif_nb, in_notif_time, cls.interval_length):
                return True
        
        return False
    
    
    # Give for a notification the next notification time
    # by taking the standard notification_interval or ask for
    # our escalation if one of them need a smaller value to escalade
    def get_next_notification_time(self, notification, from_time=None):
        now = time.time()
        cls = self.__class__
        
        if from_time is None:
            from_time = notification.t_to_go
        
        creation_time = notification.creation_time
        
        # Look at the minimum notification interval
        notification_interval = self.notification_interval
        # and then look for currently active notifications, and take notification_interval
        # if filled and less than the self value
        in_notif_time = time.time() - creation_time
        for escalation in self.escalations:
            if escalation.is_eligible(from_time, self.state, notification.notif_nb, in_notif_time, cls.interval_length):
                if escalation.notification_interval != -1 and escalation.notification_interval != 0 and escalation.notification_interval < notification_interval:
                    if ENABLE_ESCALATION_DEBUG:
                        logger.debug('ESCALATION: [%s] The escalation %s is eligible and with a short notification interval, so we are using its interval: %s.' % (self.get_full_name(), escalation.get_name(), escalation.notification_interval))
                    notification_interval = escalation.notification_interval
        
        # So take the by default time
        std_time = from_time + notification_interval * cls.interval_length
        
        # Maybe the notification comes from retention data and next notification alert is in the past
        # if so let use the now value instead
        if std_time < now:
            std_time = now + notification_interval * cls.interval_length
        
        # standard time is a good one
        next_notification_time = std_time
        
        for escalation in self.escalations:
            # If the escalation was already raised, we do not look for a new "early start"
            if escalation.get_name() not in notification.already_start_escalations:
                r = escalation.get_next_notif_time(std_time, self.state, creation_time, cls.interval_length)
                # If we got a real result (time base escalation), and earlier that we previously have, take it
                if r is not None and now < r < next_notification_time:
                    if ENABLE_ESCALATION_DEBUG:
                        logger.debug('ESCALATION: [%s] The escalation %s give us a valid start time (%s). Looking if other escalations are giving us earlier time.' % (self.get_full_name(), escalation.get_name(), time.asctime(time.localtime(r))))
                    next_notification_time = r
            # if notification_interval of escalation is set to 0 we set next notification to infinite
            elif escalation.notification_interval == 0:
                if ENABLE_ESCALATION_DEBUG:
                    logger.debug('ESCALATION: [%s] The escalation %s has no notification interval, so disabling next notification.' % (self.get_full_name(), escalation.get_name()))
                next_notification_time = now + 525600 * 60 * 100
        
        # And we take the minimum of this result. Can be standard or escalation asked
        return next_notification_time
    
    
    # Get all contacts (uniq) from eligible escalations
    # noinspection SpellCheckingInspection
    def get_escalable_contacts(self, n):
        cls = self.__class__
        
        # We search since when we are in notification for escalations
        # that are based on this time
        in_notif_time = time.time() - n.creation_time
        
        contacts = set()
        for es in self.escalations:
            if es.is_eligible(n.t_to_go, self.state, n.notif_nb, in_notif_time, cls.interval_length):
                contacts.update(es.contacts)
                # And we tag these escalations as started now
                n.already_start_escalations.add(es.get_name())
        
        return list(contacts)
    
    
    def is_notification_time_allowed(self, time_to_send_notification):
        # type: (float) -> bool
        
        # Does the notification period allow sending out this notification?
        if self.notification_period is None or not self.notification_period.is_time_valid(time_to_send_notification):
            return False
        
        return True
    
    
    # Create a "master" notification here,
    # which will later (immediately before the reactionner gets it)
    # be split up in many "child" notifications, one for each contact.
    def create_notifications(self, notification_type, ack_data=u'', ack_author=u''):
        # type: (unicode, unicode, unicode) -> None
        cls = self.__class__
        
        if self.notification_is_blocked_by_item(notification_type):
            # If notifications are blocked on the host/service level somehow
            # and repeated notifications are not configured,
            # we can silently drop this one
            return
        
        now = time.time()
        t_wished = now
        # if first notification, we must add first_notification_delay
        if self.current_notification_number == 0 and notification_type == u'PROBLEM':
            last_time_non_ok_or_up = self.last_time_non_ok_or_up()
            if last_time_non_ok_or_up == 0:
                # this happens at initial
                t_wished = now + self.first_notification_delay * cls.interval_length
            else:
                t_wished = last_time_non_ok_or_up + self.first_notification_delay * cls.interval_length
        if self.notification_period is None:
            # This case should not happen because «notification_period» is forced to be 24x7 when not defined or forced to null
            # We keep it just in case -> SEF-9455
            time_to_send_notification = int(t_wished)
        else:
            time_to_send_notification = self.notification_period.get_next_valid_time_from_t(t_wished)
        
        # SEF-9414 - if notification_period do not valid we can get a None time
        if time_to_send_notification is None:
            today = datetime.date.today()
            last_print_notification_period = getattr(self, 'last_print_notification_period', None)
            # We log this message only once a day
            if today != last_print_notification_period:
                logger_notification = LoggerFactory.get_logger(u'NOTIFICATION')
                if self.notification_period:
                    logger_notification.warning(u'The notification for %s was not send because notification_period %s do not provide a date in the next 366 days ( either in the past or nothing is defined, or days are excluded ).' % (
                        self.get_full_name(), self.notification_period.get_name()))
                else:
                    logger_notification.error(u'The notification for %s was not send because time_to_send_notification is None.' % self.get_full_name())
                    logger_notification.print_stack()
                setattr(self, 'last_print_notification_period', today)
            return
        
        # The current_notification_number of the item itself will only be change when this notification (or its children) have actually been sent.
        if notification_type == u'PROBLEM':
            # Problem notification incremented notification_number.
            next_notif_nb = self.current_notification_number + 1
        
        elif notification_type == u'RECOVERY':
            # Recovery resets the notification counter to zero
            self.current_notification_number = 0
            next_notif_nb = self.current_notification_number
        else:
            # downtime/flap/etc. do not change the notification number
            next_notif_nb = self.current_notification_number
        
        notification = Notification(type=notification_type,
                                    status=u'scheduled',
                                    command=u'VOID',
                                    command_call=None,
                                    ref=self,
                                    contact=None,
                                    t_to_go=time_to_send_notification,
                                    timeout=cls.notification_timeout,
                                    notif_nb=next_notif_nb,
                                    ack_author=ack_author,
                                    ack_data=ack_data,
                                    command_name=u'MASTER_NOTIFICATION')
        
        # Keep a trace in our notifications queue
        self.notifications_in_progress[notification.get_uuid()] = notification
        # and put it in the temp queue for scheduler
        self.actions.append(notification)
    
    
    # In create_notifications we created a notification "template" => master_notification.
    # When it's time to hand it over to the reactionner, this master notification needs
    # to be split in several child notifications, one for each contact
    # To be more exact, one for each contact who is willing to accept
    # notifications of this type and at this time
    def scatter_notification(self, master_notification):
        cls = self.__class__
        child_notifications = []
        
        if master_notification.contact:
            # only master notifications can be split up
            return []
        if master_notification.type == 'RECOVERY':
            if self.first_notification_delay != 0 and len(self.notified_contacts) == 0:
                # Recovered during first_notification_delay. No notifications
                # have been sent yet, so we keep quiet
                contacts = []
            else:
                # The old way.
                # Only send recover notifications to those contacts who also got problem notifications
                contacts = list(self.notified_contacts)
            self.notified_contacts.clear()
        
        else:
            # Check is an escalation match. If yes, get all contacts from escalations
            if self.is_escalable(master_notification):
                contacts = self.get_escalable_contacts(master_notification)
            # else take normal contacts
            else:
                contacts = self.contacts
        
        for contact in contacts:
            # We do not want to notify again a contact with notification interval == 0 that has been already notified.
            # Can happen when a service exit a downtime and still in crit/warn (and not ack)
            if master_notification.type == "PROBLEM" and self.notification_interval == 0 and contact in self.notified_contacts:
                continue
            # Get the property name for notif commands, like service_notification_commands for service
            notif_commands = contact.get_notification_commands(cls.my_type)
            
            for cmd in notif_commands:
                rt = cmd.reactionner_tag
                shell_execution = cmd.shell_execution
                command_name = getattr(getattr(cmd, 'command', {}), 'command_name', 'MISSING_NAME')
                child_n = Notification(type=master_notification.type,
                                       status='scheduled',
                                       command='VOID',
                                       command_call=cmd,
                                       ref=self,
                                       contact=contact,
                                       t_to_go=master_notification.t_to_go,
                                       timeout=cls.notification_timeout,
                                       notif_nb=master_notification.notif_nb,
                                       reactionner_tag=rt,
                                       module_type=cmd.module_type,
                                       enable_environment_macros=cmd.enable_environment_macros,
                                       shell_execution=shell_execution,
                                       ack_data=master_notification.ack_data,
                                       ack_author=master_notification.ack_author,
                                       command_name=command_name)
                if not self.notification_is_blocked_by_contact(child_n, contact):
                    # Update the notification with fresh status information
                    # of the item. Example: during the notification_delay
                    # the status of a service may have changed from WARNING to CRITICAL
                    self.update_notification_command(child_n)
                    self.raise_notification_log_entry(child_n)
                    self.notifications_in_progress[child_n.get_uuid()] = child_n
                    child_notifications.append(child_n)
                    
                    # New contact in displayed notification
                    if master_notification.type == 'PROBLEM':
                        # Remember the contacts. We might need them later in the
                        # recovery code some lines above
                        self.notified_contacts.add(contact)
        if child_notifications:
            # After a non-recovery notification, we need 20 checks to close the incident. Reset the counter if not a recovery.
            if master_notification.type != 'RECOVERY':
                self.consecutive_ok = 0
            # New incident bloc
            if not self.notification_list or 'bloc_end' in self.notification_list[-1]:
                self.notification_list.append({'bloc_start': time.time(), 'bloc_content': []})
            last_bloc = self.notification_list[-1]
            # New notification in bloc
            last_bloc['bloc_content'].append({'type': master_notification.type, 'time': time.time(), 'contacts': list(set(c.contact.contact_name for c in child_notifications)), 'status': self.state})
        return child_notifications
    
    
    # Filter to drop incidents that are older than a week
    # but keep not close one
    @staticmethod
    def _is_incident_ok_to_keep(incident):
        return ('bloc_end' not in incident) or ('bloc_end' in incident and incident['bloc_end'] >= time.time() - (7 * 86400))
    
    
    def clean_incidents(self):
        self.notification_list = filter(self._is_incident_ok_to_keep, self.notification_list)
        # End the incident with the newest notification time
        if self.consecutive_ok >= 20 and self.notification_list:
            self.notification_list[-1]['bloc_end'] = max([notif['time'] for notif in self.notification_list[-1]['bloc_content']])
    
    
    # someone wants to hook in an existing check,
    def _find_in_progress_check_we_can_hot_hook(self):
        return next((check for check in self.checks_in_progress if check.can_be_hot_hook_for_dependency()))
    
    
    # Add a check to check the host/service and return id of the check
    # * allow_next_schedule_brok = we can skip creating if we are called on the very first scheduling (initial broks will do the job)
    def launch_check(self, t_to_go, check_launch_mode, depend_on_me=None, force=False, retry=False, allow_next_schedule_brok=True):
        check = None
        cls = self.__class__
        
        # Look if we are in check or not
        self.update_in_checking()
        
        now = int(time.time())
        
        # the check is being forced, so we just replace next_chk time by now
        if force and self.in_checking:
            check_in_progress = self.checks_in_progress[0]  # type: Check
            check_in_progress.force_time_change(now)
            # Force it to run now, so change both computed and current time to go
            checks_container.do_index_job_execution(check_in_progress)
            return check_in_progress.get_uuid()
        
        # Dependency check, we have to create a new check that will be launched only once (now)
        # Otherwise it will delay the next real check. this can lead to an infinite SOFT state.
        if check_launch_mode == CHECK_LAUNCH_MODE.DEPENDENCY and not force and self.in_checking:
            hot_hookable_in_progress_check = self._find_in_progress_check_we_can_hot_hook()  # type: Check
            if hot_hookable_in_progress_check:
                was_rescheduled = hot_hookable_in_progress_check.hot_hook_for_dependency(depend_on_me)
                if was_rescheduled:
                    # Let the scheduler know we did change the parent check time, so it can fast index it
                    checks_container.do_index_job_execution(hot_hookable_in_progress_check)
                return hot_hookable_in_progress_check.get_uuid()
            else:  # no current checks are hookable, so we must recreate them by copy it
                check_in_progress = self.checks_in_progress[0]  # type: Check #0 is OK because in_checking is True
                # c_in_progress has almost everything we need, but we can't copy.deepcopy() it we need another c.id
                command_line = check_in_progress.command
                timeout = check_in_progress.timeout
                timeout_from = check_in_progress.timeout_from
                poller_tag = check_in_progress.poller_tag
                env = check_in_progress.env
                module_type = check_in_progress.module_type
                shell_execution = check_in_progress.shell_execution
                warning_threshold_cpu_usage = check_in_progress.warning_threshold_cpu_usage
                
                if hasattr(self, 'host') and hasattr(self, 'service_description') and self.host and hasattr(self.host, 'host_name'):
                    command_name = "%s-//-%s" % (self.host.host_name, self.service_description)
                else:
                    command_name = self.host_name
                command_name += "-//-%s" % check_in_progress.command_name
                
                if self.check_command is not None:
                    is_internal = getattr(self.check_command, 'internal', None)
                else:
                    command_line = u'shinken_no_command'
                    is_internal = True
                
                check = Check(
                    CHECK_STATUS.SCHEDULED, command_line, self, t_to_go,
                    depend_on_me=depend_on_me,
                    timeout=timeout,
                    timeout_from=timeout_from,
                    poller_tag=poller_tag,
                    env=env,
                    module_type=module_type,
                    dependency_check=True,
                    shell_execution=shell_execution,
                    command_name=command_name,
                    cause=CHECK_CAUSE.DEPENDENCY,
                    warning_threshold_cpu_usage=warning_threshold_cpu_usage,
                    is_internal=is_internal
                )
                
                self.checks_in_progress.append(check)
                self.last_command_hash = check.get_hash()
                self.last_command_name = command_name
                self.actions.append(check)
                # A new check means the host/service changes its next_check need to be refreshed
                if allow_next_schedule_brok:
                    self.broks.append(self.get_next_schedule_brok())
                # print "Creating new check with new id : %d, old id : %d" % (c.id, c_in_progress.id)
                return check.get_uuid()
        
        if not self.is_no_check_dependent() or force:
            if check_launch_mode == CHECK_LAUNCH_MODE.DEPENDENCY and self.my_type == 'host' and self.passive_checks_enabled and not self.active_checks_enabled:
                logger.debug("Host check is for an host that is only passively checked (%s), do not launch the check !" % self.host_name)
                return None
            
            # By default, we will use our default check_command
            check_command = self.check_command
            # But if a check modulation is available, use this one instead.
            # Take the first available
            for checkmodulation in self.checkmodulations:
                command_checkmodulation = checkmodulation.get_check_command(t_to_go)
                if command_checkmodulation:
                    check_command = command_checkmodulation
                    break
            
            # Get the command to launch
            macro_resolver = MacroResolver()
            data = self.get_data_for_checks()
            
            # By default env is void
            env = {}
            # Maybe we are in a forced passive checks, then there can be no command here
            if check_command is not None:
                # The macro resolver can return error and use shinken_echo_with_exit_code_and_skip_macro for handle it
                command_line = macro_resolver.resolve_command(check_command, data)
                is_internal = getattr(check_command, 'internal', None)
                if command_line.startswith(u'shinken_echo_with_exit_code_and_skip_macro'):
                    is_internal = True
                # And get all environment variables only if needed
                if cls.enable_environment_macros or check_command.enable_environment_macros:
                    env = macro_resolver.get_env_macros(data)
            else:
                command_line = u'shinken_no_command'
                is_internal = True
            
            if hasattr(self, 'host') and hasattr(self, 'service_description') and self.host and hasattr(self.host, 'host_name'):
                command_name = "%s-//-%s" % (self.host.host_name, self.service_description)
            else:
                command_name = self.host_name
            
            if self.check_command:
                command_name += "-//-%s" % self.check_command.command.command_name
            
            # the cause is a RETRY id retry is passed as param
            default_check_cause = CHECK_CAUSE.RETRY if retry else CHECK_CAUSE.SCHEDULE
            # if the check was force, set the cause to force, else use the default cause (schedule or retry)
            check_cause = CHECK_CAUSE.FORCE if force else default_check_cause
            
            # if force we don't need the computed t, we set the time.time() so the check will be done ASAP
            if force:
                t_to_go = now
            poller_tag = check_command.poller_tag if check_command else 'None'
            module_type = check_command.module_type if check_command else 'fork'
            shell_execution = check_command.shell_execution if check_command else False
            # Make the Check object and put the service in checking
            # Make the check inherit poller_tag from the command
            # And reactionner_tag too
            
            check = Check(
                CHECK_STATUS.SCHEDULED, command_line, self, t_to_go,
                depend_on_me=depend_on_me,
                timeout=self.check_running_timeout,
                timeout_from=self.check_running_timeout_from,
                poller_tag=poller_tag,
                env=env,
                module_type=module_type,
                shell_execution=shell_execution,
                command_name=command_name,
                cause=check_cause,
                warning_threshold_cpu_usage=self.warning_threshold_cpu_usage,
                is_internal=is_internal
            )
            
            if check_launch_mode == CHECK_LAUNCH_MODE.PASSIVE:
                check.set_type_passive()
                have_freshness = cls.global_check_freshness and self.check_freshness and self.freshness_threshold != 0
                check.state_validity_period = self.freshness_threshold if have_freshness else NO_END_VALIDITY
            elif check_launch_mode in (CHECK_LAUNCH_MODE.SCHEDULE, CHECK_LAUNCH_MODE.DEPENDENCY):
                check.state_validity_period = self.check_interval * cls.interval_length
            elif check_launch_mode == CHECK_LAUNCH_MODE.FRESHNESS:
                check.state_validity_period = NO_END_VALIDITY
            
            # If the check we will launch change the state of our item, the next check will be done with the retry_interval if the check is schedule, so we must update the state_validity_period
            if check_launch_mode in (CHECK_LAUNCH_MODE.SCHEDULE, CHECK_LAUNCH_MODE.DEPENDENCY):
                check.state_validity_period_when_go_in_soft = self.retry_interval * cls.interval_length
            else:
                check.state_validity_period_when_go_in_soft = None
            
            # We keep a trace of all checks in progress
            # to know if we are in checking_or not
            self.checks_in_progress.append(check)
            self.last_command_hash = check.get_hash()
            self.last_command_name = command_name
        
        self.update_in_checking()
        
        # We need to put this new check in our actions queue
        # so scheduler can take it
        if check is not None:
            self.actions.append(check)
            # A new check means the host/service changes its next_check need to be refreshed
            if allow_next_schedule_brok:
                self.broks.append(self.get_next_schedule_brok())
            return check.get_uuid()
        # None mean I already take it into account
        return None
    
    
    # returns either 0 or a positive number
    # 0 == don't check for orphans
    # non-zero == number of secs that can pass before
    #             marking the check an orphan.
    def get_time_to_orphanage(self):
        # if disabled program-wide, disable it
        if not self.check_for_orphaned:
            return 0
        # otherwise, check what my local conf says
        if self.time_to_orphanage <= 0:
            return 0
        return self.time_to_orphanage
    
    
    # Get the perf data command with macro resolved for this
    def get_perfdata_command(self):
        cls = self.__class__
        if not cls.process_performance_data or not self.process_perf_data:
            return
        
        if cls.perfdata_command is not None:
            m = MacroResolver()
            data = self.get_data_for_event_handler()
            cmd = m.resolve_command(cls.perfdata_command, data)
            reactionner_tag = cls.perfdata_command.reactionner_tag
            shell_execution = cls.perfdata_command.shell_execution
            command_name = getattr(getattr(self.perfdata_command, 'command', {}), 'command_name', 'MISSING_NAME')
            e = EventHandler(cmd, timeout=cls.perfdata_timeout,
                             ref=self, reactionner_tag=reactionner_tag,
                             shell_execution=shell_execution,
                             command_name=command_name)
            
            # ok we can put it in our temp action queue
            self.actions.append(e)
    
    
    # Create the whole business rule tree
    # if we need it
    def create_business_rules(self, proxy_items):
        # type: (ProxyItems) -> None
        cmd_call = getattr(self, 'check_command', None)
        
        # If we do not have a command, we bail out
        if cmd_call is None:
            return
        
        # we get our based command, like
        # check_tcp!80 -> check_tcp
        cmd = cmd_call.call
        elements = cmd.split('!')
        base_cmd = elements[0]
        # If it's bp_rule, we got a rule :)
        if base_cmd == 'bp_rule':
            self.got_business_rule = True
            # print "Got rule", elements, cmd
            rule = ''
            if len(elements) >= 2:
                rule = '!'.join(elements[1:])
                if len(rule) == 0:
                    self.business_rule = DependencyNode()
            # Only (re-)evaluate the business rule if it has never been
            # evaluated before, or it contains a macro.
            if rule != "" and (re.match(r"\$[\w\d_-]+\$", rule) or self.business_rule is None):
                data = self.get_data_for_checks()
                m = MacroResolver()
                rule = m.resolve_simple_macros_in_string(rule, data)
                prev = getattr(self, "processed_business_rule", None)
                
                if rule == prev:
                    # Business rule did not change (no macro was modulated)
                    return
                
                fact = DependencyNodeFactory(self)
                logger.debug('create_business_rules:: "%s" creating final rule: "%s"' % (self.get_name(), rule))
                node = fact.eval_cor_pattern(rule, proxy_items, ref=self)
                self.processed_business_rule = rule
                if node is None:
                    self.business_rule = DependencyNode()
                else:
                    self.business_rule = node
    
    
    # Expands format string macros with item attributes
    def expand_business_rule_item_macros(self, template_string, item):
        output = template_string
        status = getattr(item, "state", "")
        output = re.sub(r"\$STATUS\$", status, output, flags=re.I)
        short_status = self.status_to_short_status(status)
        output = re.sub(r"\$SHORT_STATUS\$", short_status, output, flags=re.I)
        service_description = getattr(item, "service_description", "")
        output = re.sub(r"\$SERVICE_DESCRIPTION\$", service_description, output, flags=re.I)
        host_name = getattr(item, "host_name", "")
        output = re.sub(r"\$HOST_NAME\$", host_name, output, flags=re.I)
        full_name = item.get_full_name()
        output = re.sub(r"\$FULL_NAME\$", full_name, output, flags=re.I)
        return output
    
    
    # Returns status string shorten name.
    @staticmethod
    def status_to_short_status(status):
        # type: (str) -> str
        mapping = {
            "OK"      : "O",
            "WARNING" : "W",
            "CRITICAL": "C",
            "UNKNOWN" : "U",
            "UP"      : "U",
            "DOWN"    : "D"
        }
        return mapping.get(status, status)
    
    
    # We ask us to manage our own internal check, like a bp_rule based one
    def manage_internal_check(self, check, lang=u'en'):
        # type: (Check, unicode) -> None
        _logger = LoggerFactory.get_logger(u'INTERNAL-COMMAND').get_sub_part(check.command_name, register=False)
        if check.command.startswith(u'bp_'):
            try:
                # Re-evaluate the business rule to take into account macro modulation.
                # Caution: We consider the that the macro modulation did not change business rule dependency tree.
                # Only Xof: values should be modified by modulation.
                self.create_business_rules(proxyitemsmgr)
                check.simulate_execution_return(self.business_rule.get_state(self), self.business_rule.output, time.time(), check.execution_time, long_output=self.business_rule.long_output)
            except DependencyNodeUninitializedException:
                # #SEF-8702 when data from other schedulers is not available, keep previous state, indefinitely
                logger.debug(u'manage_internal_check: %s is waiting for ProxyItems to be initialized' % self.get_full_name())
                check.simulate_execution_return(self.bp_state, self.output, time.time(), check.execution_time, long_output=self.long_output)
            except Exception as exp:
                # Notifies the error, and return an UNKNOWN state.
                output = u'Error while re-evaluating business rule: %s' % exp
                if lang == u'fr':
                    output = u'Error while re-evaluating business rule: %s' % exp  # TODO fr
                
                check.simulate_execution_return(3, output, time.time(), check.execution_time, long_output='')
                _logger.debug(u'Error while re-evaluating business rule')
                _logger.print_stack(level=logging.DEBUG)
        # _internal_host_up is for putting host as UP
        elif check.command == u'_internal_host_up':
            output = u'Host assumed to be UP'
            if lang == u'fr':
                output = u'L\'hôte est considéré comme vivant'
            
            check.simulate_execution_return(0, output, time.time(), 0, long_output='')
        # Echo is just putting the same state again
        elif check.command == u'_echo':
            check.simulate_execution_return(0, self.output, time.time(), 0, long_output='')
        elif check.command == u'shinken_no_command':
            output = u'There is no command for this check'
            if lang == u'fr':
                output = u'Il n\'y a pas de commande pour ce check'
            
            check.simulate_execution_return(0, output, time.time(), 0, long_output='')
        elif check.command.startswith(u'shinken_echo_with_exit_code_and_skip_macro'):
            try:
                _, exit_code, output = check.command.split(u';', 2)
                outputs = output.split(u'\n', 1)
                check.simulate_execution_return(int(exit_code), outputs[0], time.time(), 0, long_output=outputs[1] if len(outputs) > 1 else u'')
            except:
                try:
                    _, exit_code, output = check.command.split(u'⋕', 2)
                    outputs = output.split(u'\n', 1)
                    check.simulate_execution_return(int(exit_code), outputs[0], time.time(), 0, long_output=outputs[1] if len(outputs) > 1 else u'')
                except:
                    check.simulate_execution_return(3, u'The internal command shinken_echo_with_exit_code_and_skip_macro has a invalid format', time.time(), 0, long_output='')
                    _logger.error(u'The internal command shinken_echo_with_exit_code_and_skip_macro has a invalid format')
        _logger.debug(u'getting return code:[%s] and output:[%s]' % (check.exit_status, check.output))
    
    
    # If I'm a business rule service/host, I register myself to the
    # elements I will depend on, so They will have ME as an impact
    def create_business_rules_dependencies(self, hosts, services):
        if self.got_business_rule and self.business_rule is not None:
            # print "DBG: ask me to register me in my dependencies", self.get_name()
            all_proxies = self.business_rule.list_all_elements()
            elements = []
            for p in all_proxies:
                if p.type == 'check':
                    e = services.find_srv_by_name_and_hostname(p.host_name, p.name)
                    elements.append(e)
                else:  # host
                    e = hosts.find_by_name(p.name)
                    elements.append(e)
            # I will register myself in this
            for e in elements:
                # print "I register to the element", e.get_name()
                # all states, every timeperiod, and inherit parents
                e.add_business_rule_act_dependency(self, ['d', 'u', 's', 'f', 'c', 'w'], None, True)
    
    
    # Rebuild the possible reference a SchedulingItem can have
    def rebuild_ref(self):
        for o in self.downtimes:
            o.ref = self
    
    
    # Go launch all our triggers
    def eval_triggers(self):
        return
    
    
    # We update the proxy objects with current values
    def update_proxy(self):
        my_uuid = self.get_instance_uuid()
        # state
        
        proxyitemsmgr.update_state(my_uuid, self.state_id)
        
        # flapping
        proxyitemsmgr.update_flapping(my_uuid, self.is_flapping, inherited_flapping=self.inherited_flapping)
        
        # ack
        author = ''
        comment = ''
        proxyitemsmgr.update_acknowledge(my_uuid, self.problem_has_been_acknowledged)
        if self.acknowledgement:
            proxyitemsmgr.update_acknowledge_author_and_comment_auto(my_uuid, self.acknowledgement.author, self.acknowledgement.comment, self.acknowledgement.automatic)
        
        if self.my_type == 'service' and self.is_in_inherited_acknowledged():  # try host config
            author = self.host.acknowledgement.author
            comment = self.host.acknowledgement.comment
        proxyitemsmgr.update_in_inherited_acknowledged(my_uuid, self.is_in_inherited_acknowledged(), author=author, comment=comment)
        
        proxyitemsmgr.update_partial_acknowledge(my_uuid, self.is_partial_acknowledged)
        if self.partial_acknowledge:
            proxyitemsmgr.update_partial_acknowledge_author_and_comment(my_uuid, self.partial_acknowledge.author, self.partial_acknowledge.comment)
        
        # downtime
        proxyitemsmgr.update_downtime(my_uuid, self.in_scheduled_downtime)
        proxyitemsmgr.update_in_inherited_downtime(my_uuid, self.is_in_inherited_downtime())
        
        # business impact
        proxyitemsmgr.update_business_impact(my_uuid, self.business_impact)
    
    
    ##########################################################################################################
    #                                     ACKNOWLEDGEMENT                                                    #
    ##########################################################################################################
    
    # Create an acknowledgement for this element
    def acknowledge_problem(self, sticky, notify, persistent, author, comment, end_time=0, automatic=False):
        if self.state != self.ok_up:
            if notify:
                self.create_notifications('ACKNOWLEDGEMENT', ack_data=comment, ack_author=author)
            self.problem_has_been_acknowledged = True
            if sticky == 2:
                sticky = True
            else:
                sticky = False
            self.acknowledgement = Acknowledge(self, sticky, notify, persistent, author, comment, end_time=end_time, automatic=automatic)
            if self.got_business_rule:
                try:
                    self.business_rule.get_state(self)
                    self.output = self.business_rule.output
                    self.long_output = self.business_rule.long_output
                except DependencyNodeUninitializedException:
                    logger.debug(u'acknowledge_problem: %s is waiting for ProxyItems to be initialized' % self.get_full_name())
                except Exception as e:
                    # Notifies the error, and return an UNKNOWN state.
                    self.output = "Error while re-evaluating business rule: %s" % e
                    self.long_output = ""
                    logger.debug("[%s] Error while re-evaluating business rule:\n%s" % (self.get_name(), traceback.format_exc()))
            self.build_full_status(context_change_time=self.acknowledgement.start_time)
            self.broks.append(self.get_update_status_brok(context_change_time=self.acknowledgement.start_time))
            is_host = self.__class__.my_type == 'host'
            # for hosts, also update info about our services as they can be unacknowledged too
            if is_host:
                for s in self.services:
                    s.build_full_status(context_change_time=self.acknowledgement.start_time)
                    s.broks.append(s.get_update_status_brok(context_change_time=self.acknowledgement.start_time))
            # also update proxy objects
            proxyitemsmgr.update_acknowledge(self.get_instance_uuid(), self.problem_has_been_acknowledged)
            proxyitemsmgr.update_acknowledge_author_and_comment_auto(self.get_instance_uuid(), author, comment, automatic)
            if is_host:
                for s in self.services:
                    proxyitemsmgr.update_in_inherited_acknowledged(s.get_instance_uuid(), s.is_in_inherited_acknowledged(), author=self.acknowledgement.author, comment=self.acknowledgement.comment)
    
    
    # Create a partial acknowledge for this element
    def partial_acknowledge_problem(self, sticky, persistent, author, comment, end_time=0):
        if self.state != self.ok_up:
            if sticky == 2:
                sticky = True
            else:
                sticky = False
            self.partial_acknowledge = Acknowledge(self, sticky, False, persistent, author, comment, end_time=end_time, automatic=True)
            self.build_full_status(context_change_time=self.partial_acknowledge.start_time)
            self.broks.append(self.get_update_status_brok(context_change_time=self.partial_acknowledge.start_time))
            
            # also update proxy objects
            proxyitemsmgr.update_partial_acknowledge(self.get_instance_uuid(), self.is_partial_acknowledged)
            proxyitemsmgr.update_partial_acknowledge_author_and_comment(self.get_instance_uuid(), author, comment)
    
    
    # Look if we have an ack that is too old with an expiration date and should be deleted
    def check_for_expire_acknowledge(self):
        if self.acknowledgement and self.acknowledgement.end_time != 0 and self.acknowledgement.end_time < time.time():
            self.unacknowledge_problem()
    
    
    # Delete the acknowledgement object and reset the flag
    #  but do not remove the associated comment.
    def unacknowledge_problem(self):
        if self.problem_has_been_acknowledged:
            self.problem_has_been_acknowledged = False
            # Should not be deleted, a None is Good
            self.acknowledgement = None
            if self.got_business_rule:
                try:
                    self.business_rule.get_state(self)
                    self.output = self.business_rule.output
                    self.long_output = self.business_rule.long_output
                except DependencyNodeUninitializedException:
                    logger.debug(u'unacknowledge_problem: %s is waiting for ProxyItems to be initialized' % self.get_full_name())
                except Exception as e:
                    # Notifies the error, and return an UNKNOWN state.
                    self.output = "Error while re-evaluating business rule: %s" % e
                    self.long_output = ""
                    logger.debug("[%s] Error while re-evaluating business rule:\n%s" % (self.get_name(), traceback.format_exc()))
            
            self.broks.append(self.get_update_status_brok())
            is_host = self.__class__.my_type == 'host'
            # for hosts, also update info about our services as they can be un-acknowledgement too
            if is_host:
                for s in self.services:
                    s.broks.append(s.get_update_status_brok())
            
            # also update proxy objects
            proxyitemsmgr.update_acknowledge(self.get_instance_uuid(), self.problem_has_been_acknowledged)
            proxyitemsmgr.update_acknowledge_author_and_comment_auto(self.get_instance_uuid(), '', '', False)
            if is_host:
                for s in self.services:
                    proxyitemsmgr.update_in_inherited_acknowledged(s.get_instance_uuid(), s.is_in_inherited_acknowledged())
    
    
    # Check if we have an acknowledgement and if this is marked as sticky.
    # This is needed when a non-ok state changes
    def unacknowledge_problem_if_not_sticky(self):
        if hasattr(self, 'acknowledgement') and self.acknowledgement is not None:
            if not self.acknowledgement.sticky:
                self.unacknowledge_problem()
    
    
    # Get my direct fathers, so only level +1 one
    # * cluster: what are we computing
    # * hosts: parents
    def __get_my_direct_fathers(self):
        if self.got_business_rule:
            fathers = self.business_rule.list_all_elements()
        else:  # hosts
            # Get my parents, but as proxy elements
            fathers = [proxyitemsmgr[p.uuid] for p in self.parents]
        return fathers
    
    
    # return if all our fathers (source problems and direct cluster elements) are in an acknowledgement context
    def __are_all_my_fathers_acknowledged(self):
        # Now look at bp rule elements
        fathers = self.__get_my_direct_fathers()
        nb_acknowledged = 0
        nb_partial_acknowledged = 0
        nb_bad_state = 0
        # Expands child items format string macros.
        for father in fathers:
            if father.is_ok_up():
                continue
            nb_bad_state += 1
            if father.is_acknowledge():
                logger.debug(u'%s.__are_all_my_fathers_acknowledged:: %s is ack ' % (self.get_full_name(), father.name))
                nb_acknowledged += 1
                continue
            # Maybe our father is just in partial, so count it, so we will be in partial too
            if father.is_partial_acknowledge():
                logger.debug(u'%s.__are_all_my_fathers_acknowledged:: %s is partial ack ' % (self.get_full_name(), father.name))
                nb_partial_acknowledged += 1
            
            # maybe it's a service and its host is ack
            if father.type == 'service':
                father_host = proxyitemsmgr[father.host_uuid]
                if father_host.is_acknowledge():
                    logger.debug(u'%s.__are_all_my_fathers_acknowledged:: %s host is ack ' % (self.get_full_name(), father.name))
                    nb_acknowledged += 1
                    continue
                if father_host.is_partial_acknowledge():
                    logger.debug(u'%s__are_all_my_fathers_acknowledged:: %s host is partial ack ' % (self.get_full_name(), father.name))
                    nb_partial_acknowledged += 1
        # partial can be because of other ack or been partial ack
        partial = (nb_acknowledged != 0) or (nb_partial_acknowledged != 0)
        # but for full we need to be with all in acknowledgements
        full_ack = (nb_bad_state != 0) and (nb_acknowledged == nb_bad_state)
        # if we have both partial and full ack, then it means we are full ack only
        if partial and full_ack:
            partial = False
        return partial, full_ack
    
    
    def __are_all_my_fathers_flapping(self):
        # Now look at bp rule elements
        fathers = self.__get_my_direct_fathers()
        nb_fathers = len(fathers)
        nb_flapping = 0
        nb_partial_flapping = 0
        # Expands child items format string macros.
        
        for father in fathers:
            if father.in_flapping:
                logger.debug('[__are_all_my_fathers_flapping]:: %s is flap ' % father.name)
                nb_flapping += 1
                continue
            # Maybe our father is just in partial, count it, so we will be in partial too
            if father.in_partial_flapping:
                logger.debug('[__are_all_my_fathers_flapping]:: %s is partial flap ' % father.name)
                nb_partial_flapping += 1
            
            # maybe it's a service and its host is ack
            if father.type == 'service':
                father_host = proxyitemsmgr[father.host_uuid]
                if father_host.in_flapping:
                    logger.debug('[__are_all_my_fathers_flapping]:: %s host is flap ' % father.name)
                    nb_flapping += 1
                    continue
                if father_host.in_partial_flapping:
                    logger.debug('[__are_all_my_fathers_flapping]:: %s host is partial flap ' % father.name)
                    nb_partial_flapping += 1
        # partial can be because of other ack or een partial ack
        partial = (nb_flapping != 0) or (nb_partial_flapping != 0)
        # but for full we need to be with all in acknowledgements
        full_flap = (nb_fathers != 0) and (nb_flapping == nb_fathers)
        # if we have both partial and full flapping, then it means we are full ack only
        if partial and full_flap:
            partial = False
        return partial, full_flap
    
    
    # update partial acknowledge, and if needed, create a brok from it
    def __update_partial_ack(self, partial):
        was_partial = self.is_partial_acknowledged
        if partial == was_partial:
            return
        self.is_partial_acknowledged = partial
        if not partial:
            self.partial_acknowledge = None
        # just a check result is enough here
        self.broks.append(self.get_check_result_brok())
        
        # Update proxy object too
        proxyitemsmgr.update_partial_acknowledge(self.get_instance_uuid(), self.is_partial_acknowledged)
        
        if self.partial_acknowledge:
            proxyitemsmgr.update_partial_acknowledge_author_and_comment(self.get_instance_uuid(), self.partial_acknowledge.author, self.partial_acknowledge.comment)
        else:
            proxyitemsmgr.update_partial_acknowledge_author_and_comment(self.get_instance_uuid(), '', '')
    
    
    # update partial flapping, and if needed, create a brok from it
    def __update_partial_flap(self, partial):
        was_partial = self.is_partial_flapping
        if partial == was_partial:
            return
        self.is_partial_flapping = partial
        # just a check result is enough here
        self.broks.append(self.get_check_result_brok())
        
        # Update proxy object too
        proxyitemsmgr.update_partial_flapping(self.get_instance_uuid(), self.is_partial_flapping)
    
    
    # update partial downtime, and if needed, create a brok from it
    def update_partial_dt(self, partial):
        was_partial = self.in_partial_downtime
        if partial == was_partial:
            return
        self.in_partial_downtime = partial
        # just a check result is enough here
        self.broks.append(self.get_check_result_brok())
        
        # Update proxy object too
        proxyitemsmgr.update_partial_downtime(self.get_instance_uuid(), self.in_partial_downtime)
    
    
    @staticmethod
    def _ack_get_authors_origins_and_comments(authors, author, item_name, comment):
        # type: (Dict, basestring, basestring, Union[str,unicode]) -> None
        if author == u'by_multiple_authors' or comment.startswith(u'shinken_multiple'):
            # logger.debug(u' ACK COMMENT SPLIT %s item:%s author:%s comm:%s' % (self.get_full_name(), item_name, author, comment))
            # In this case, authors can be found in comments
            _authors = comment.split('\n')
            for _, _author, _host_name, _comment in [a.split('<->')[0:4] for a in _authors if a.startswith(u'shinken_multiple')]:
                if _author not in authors:
                    authors[_author] = set()
                authors[_author].add((_host_name, _comment))
        elif author:
            # logger.debug(u' ACK COMMENT no SPLIT %s item:%s author:%s comm:%s' % (self.get_full_name(), item_name, author, comment))
            if author not in authors:
                authors[author] = set()
            authors[author].add((item_name, comment))
    
    
    # We look for possible automatic ack creation if ALL our sources are in acknowledge too.
    # * CREATION: all our source problems are ack, we are going ack
    # * REMOVING: we remove our ack if one of our source pb is not ack,
    #             BUT only if the ack was an automatic one
    def compute_automatic_acknowledge(self):
        state_id = self.bp_state if self.got_business_rule else self.state_id
        
        # If the item is OK we remove the partial ack
        if state_id == 0:
            self.__update_partial_ack(False)
            return
        
        # We are an impact, so we must look if we already have a NOT automatic ack (because we won't remove it and exit)
        if self.problem_has_been_acknowledged:  # REMOVING
            if self.acknowledgement.automatic:  # but only for automatic
                partial_ack, full_ack = self.__are_all_my_fathers_acknowledged()
                # first look if we need to update partial ack or not
                self.__update_partial_ack(partial_ack)
                if not full_ack:
                    self.unacknowledge_problem()
                    return
        else:  # CREATION
            # It may have some call to __update_partial_ack set to True, so we have some item with is_partial_acknowledged to True without partial_acknowledged set
            # There partial_acknowledged will remove in next iteration
            # See SEF-5075 for a sample case
            partial_ack, full_ack = self.__are_all_my_fathers_acknowledged()
            # first look if we need to update partial ack or not
            self.__update_partial_ack(partial_ack)
            # no ack and my pbs are ack? go in automatic ack
            if full_ack or partial_ack:
                fathers = [p for p in self.__get_my_direct_fathers() if p.is_acknowledge() or p.is_partial_acknowledge()]
                authors = {}
                
                # NOTE: beware, fathers are PROXYS!!!
                for p in fathers:
                    if p.ack_author:
                        # logger.debug(u' ACK: %s item:%s author:%s, comm:%s' % (self.get_full_name(), p.get_full_name(), p.ack_author, p.ack_comment))
                        self._ack_get_authors_origins_and_comments(authors, p.ack_author, p.get_full_name(), p.ack_comment)
                    elif p.partial_ack_authors:
                        # logger.debug(u' PARTIAL ACK: %s item:%s author:%s, comm:%s' % (self.get_full_name(), p.get_full_name(), p.partial_ack_authors, p.partial_ack_comments))
                        self._ack_get_authors_origins_and_comments(authors, p.partial_ack_authors, p.get_full_name(), p.partial_ack_comments)
                
                if len(authors) > 1:
                    author = u'by_multiple_authors'
                elif len(authors) == 1:
                    author = authors.keys()[0]
                else:  # Case in  which an old retention without author was loaded
                    author = ''
                comments = []
                for (_author, coms) in authors.iteritems():
                    for c in coms:
                        # #SEF-9539 Safety break for infinite "comments add" loop, shinken_multiple should never be there !
                        if c[1].startswith(u'shinken_multiple'):
                            logger.error(u'''SCHEDULER BUG wrong acknowledgment's comment management on %s DISCARDING COMMENT FROM author:%s on item:%s with comment:(%s)''' % (self.get_full_name(), _author, c[0], c[1]))
                            comment = u'Shinken truncated acknowledge comments to prevent loop analysis (refer to support)'
                        else:
                            comment = c[1]
                        comments.append(u'shinken_multiple<->%s<->%s<->%s' % (_author, c[0], comment))
                comment = '\n'.join(comments)
                
                if full_ack:
                    # create an automatic ack with notification
                    self.acknowledge_problem(2, True, None, author, comment, automatic=True)
                if partial_ack:
                    if self.partial_acknowledge and self.partial_acknowledge.author == author and self.partial_acknowledge.comment == comment:
                        return
                    if not author and not comment:
                        self.__update_partial_ack(False)
                    else:
                        # create an automatic ack with notification
                        self.partial_acknowledge_problem(2, None, author, comment)
    
    
    def compute_automatic_flapping(self):
        if self.is_flapping:
            if self.inherited_flapping:
                partial_flapping, full_flapping = self.__are_all_my_fathers_flapping()
                self.__update_partial_flap(partial_flapping)
                if not full_flapping:
                    self.update_flapping()
        else:  # CREATION
            partial_flapping, full_flapping = self.__are_all_my_fathers_flapping()
            # first look if we need to update partial ack or not
            self.__update_partial_flap(partial_flapping)
            # no flapping and my pbs are flapping? go in automatic flapping
            if full_flapping:
                # create an automatic flapping
                self.update_flapping(automatic=True)
    
    
    def get_active_downtime_uuids(self):
        return [dt.uuid for dt in self.downtimes if dt.is_in_effect]
    
    
    def is_in_downtime(self):
        raise NotImplementedError
    
    
    def is_in_inherited_downtime(self):
        raise NotImplementedError
    
    
    def build_full_status(self, context_change_time=None):
        # type: (Optional[Number]) -> None
        now = int(time.time())
        missing_data_state = STATE_ID.MISSING_DATA
        status_end_time = self.state_validity_end_time
        
        if self.got_business_rule:
            state_id = self.bp_state
        else:
            state_id = self.state_id
        
        if (self.current_full_status % 10) != missing_data_state and 0 < self.last_chk < self.missing_data_activation_time < now:
            # logger.debug(u' build_full_status activating missing data for element [%s] (state:[%s] last_chk:[%s] mdat:[%s] now:[%s] fs:[%s] state_id:[%s])' % (self.get_full_name(), state_id, self.last_chk, self.missing_data_activation_time, now, self.current_full_status, self.state_id))
            state_id = missing_data_state  # MISSING DATA, previous status has expired
            lang = self.__class__.language
            self.output = MISSING_DATA_OUTPUT[lang] % (print_human_readable_date_time(status_end_time, lang), print_human_readable_period(self.missing_data_activation_time - self.last_chk))
            self.long_output = u''
            
            # WARNING: following data will not be reliable, as compute_next_chk will overwrite it (at startup/conf reload)
            # MISSING DATA state should never expire (only a real status should replace it)
            # Broker's modules must reset these values when full status state is MISSING DATA
            self.missing_data_activation_time = 0
            self.state_validity_end_time = 0
            self.state_validity_period = NO_END_VALIDITY
        
        # Do not overwrite previously set MISSING_DATA state
        if self.last_chk < self.full_status_change_time and (self.current_full_status % 10) == missing_data_state:
            state_id = missing_data_state
        
        current_full_status = \
            len(self.get_active_downtime_uuids()) * 10000000000 + \
            self.inherited_flapping * 1000000000 + \
            self.is_in_inherited_acknowledged() * 100000000 + \
            self.is_in_inherited_downtime() * 10000000 + \
            self.is_partial_flapping * 1000000 + \
            self.is_partial_acknowledged * 100000 + \
            self.in_partial_downtime * 10000 + \
            self.is_flapping * 1000 + \
            self.problem_has_been_acknowledged * 100 + \
            self.in_scheduled_downtime * 10 + \
            state_id
        
        if self.current_full_status != current_full_status:
            # logger.debug(u' build_full_status changing full_status for element [%s] (state:[%s] last_chk:[%s] mdat:[%s] now:[%s] fs:[%s]>[%s] state_id:[%s])' % (self.get_full_name(), state_id, self.last_chk, self.missing_data_activation_time, now, self.current_full_status, current_full_status, self.state_id))
            if state_id != missing_data_state and (current_full_status % 10) != (self.current_full_status % 10):
                # We have a real state change, starts from check result
                new_full_status_time = self.last_chk
            elif state_id == missing_data_state and (current_full_status / 10) == (self.current_full_status / 10):
                new_full_status_time = status_end_time if 0 < self.full_status_change_time < status_end_time else self.full_status_change_time
            else:
                # We have a context change, decide when it started (default: now)
                new_full_status_time = now
                if context_change_time and context_change_time > self.full_status_change_time and context_change_time >= (now - 60):
                    # Use the same ref as context start time if not too old (loop time of Scheduler should be less than 60 sec), and if previous full status time is older
                    new_full_status_time = context_change_time
            
            self.current_full_status = current_full_status
            self.full_status_change_time = new_full_status_time
        # else:
        #     logger.debug(u' build_full_status keeping full_status for element [%s] (state:[%s] last_chk:[%s] mdat:[%s] now:[%s] fs:[%s] state_id:[%s])' % (self.get_full_name(), state_id, self.last_chk, self.missing_data_activation_time, now, self.current_full_status, self.state_id))
    
    
    def get_state_from_full_status(self):
        return FullStatusManipulator.get_state_from_full_status(self.current_full_status, self.full_status_change_time)
    
    
    def get_full_name(self):
        raise NotImplementedError()
    
    
    @staticmethod
    def apply_inheritance_from_command(item, commands, property_name):
        # type:(SchedulingItem, Commands, unicode) -> bool
        value_set = False
        if hasattr(item, u'check_command'):
            command_properties_name = {
                u'check_running_timeout'      : u'timeout',
                u'warning_threshold_cpu_usage': u'warning_threshold_cpu_usage'
            }
            command = commands.find_by_name(item.check_command)
            if command is not None and getattr(command, command_properties_name[property_name], None) not in (u'-1', -1):
                setattr(item, property_name, getattr(command, command_properties_name[property_name]))
                setattr(item, u'%s_from' % property_name, u'command')
                value_set = True
        return value_set
    
    
    def load_value_from_global_conf(self):
        # type: () -> None
        # Two step to get check_running_timeout default value from cfg because
        # when executing apply_inheritance_from_command default value from shinken.cfg are not computed
        if getattr(self, u'check_running_timeout_from', None) == u'global':
            setattr(self, u'check_running_timeout', getattr(self.__class__, u'check_timeout'))
        if getattr(self, u'warning_threshold_cpu_usage_from', None) == u'global':
            setattr(self, u'warning_threshold_cpu_usage', getattr(self.__class__, u'warning_threshold_cpu_usage_default'))
    
    
    def empty_perf_data(self):
        # type: () -> None
        if self.process_perf_data:
            return
        self.last_perf_data = self.perf_data
        self.perf_data = u''
