#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2009-2022:
#     Gabes Jean, naparuba@gmail.com
#     Gerhard Lausser, Gerhard.Lausser@consol.de
#     Gregory Starck, g.starck@gmail.com
#     Hartmut Goebel, h.goebel@goebel-consult.de
#     Martin Benjamin, b.martin@shinken-solutions.com
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import base64
import cPickle
import copy
import logging
import threading
import time
import zlib

from .daemon import Daemon, Interface
from .http_client import HTTPClient, HTTPExceptions
from .inter_daemon_message import InterDaemonMessage
from .log import LoggerFactory
from .misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from .log import PartLogger
    from .daemons.brokerdaemon import Broker
    from .misc.type_hint import NoReturn, Optional, List, Dict, Union


logger = LoggerFactory.get_logger()
logger_configuration = logger.get_sub_part(u'CONFIGURATION')
logger_configuration_update = logger.get_sub_part(u'CONFIGURATION').get_sub_part(u'UPDATE')
logger_connection = logger.get_sub_part(u'CONNECTION')

RECENT_CONFIGURATION_CHANGE_THRESHOLD = 120  # under 120s we consider a configuration change 'recent'


# Interface for Arbiter, our big MASTER It gives us our conf
class IForArbiter(Interface):
    app = None  # type: Union[BaseSatellite, Broker]
    doc = u'Remove a scheduler connection (internal)'
    
    
    # Arbiter ask us to stop managing a scheduler_id anymore, I do it and don't ask why
    def remove_from_conf(self, sched_id, reason):
        try:
            sched_id = int(sched_id)
        except ValueError:
            logger_configuration_update.error(u'Arbiter did ask to remove a shard %s but the format is invalid' % sched_id)
            return
        self.app.reset_configuration_change()  # arbiter change our configuration, we note this
        self.app.delete_shard_id(sched_id, reason)
        return
    
    
    remove_from_conf.doc = doc
    
    doc = u'DISABLED'
    
    
    # Old call, disabled
    def what_i_managed(self):
        return None
    
    
    what_i_managed.need_lock = False
    what_i_managed.doc = doc
    
    doc = u'Return the managed configuration ids (internal)'
    
    
    # Give to arbiter which configuration we are managing (with configuration incarnation, and schedulers)
    def get_currently_managed_configuration(self):
        manage_configuration = self.app.get_currently_managed_configuration()
        logger.debug(u'The arbiter asked me what I manage. It\'s %s' % manage_configuration)
        return manage_configuration
    
    
    get_currently_managed_configuration.need_lock = False
    get_currently_managed_configuration.doc = doc
    
    doc = u'Ask the daemon to drop its configuration and wait for a new one'
    
    
    # Call by arbiter if it thinks we are running, but we must not (like
    # if I was a spare that take a conf but the master returns, I must die
    # and wait a new conf)
    # Us: No please...
    # Arbiter: I don't care, hasta la vista baby!
    # Us: ... <- Nothing! We are dead! you don't get it or what??
    # Reading code is not a job for eyes only...
    def wait_new_conf(self):
        logger_configuration.info(u'Arbiter wants me to wait for a new configuration')
        super(IForArbiter, self).wait_new_conf()
        self.app.schedulers.clear()
    
    
    wait_new_conf.doc = doc
    
    doc = u'Push broks objects to the daemon (internal)'
    
    
    # NB: following methods are only used by broker
    # Used by the Arbiter to push broks to broker
    def push_broks(self, broks):
        with self.app.arbiter_broks_lock:
            self.app.arbiter_broks.extend(broks.values())
    
    
    push_broks.method = u'POST'
    # We are using a Lock just for NOT lock this call from the arbiter :)
    push_broks.need_lock = False
    push_broks.doc = doc
    push_broks.display_name = u'Arbiter Architecture informations push to Brokers'
    
    doc = u'Get the shinken internal commands (like recheck, set acknowledge, etc) from the daemon (internal)'
    
    
    # The arbiter ask us our external commands in queue
    # Same than push_broks, we will not use Global lock here,
    # and only lock for external_commands
    def get_external_commands(self):
        with self.app.external_commands_lock:
            cmds = self.app.get_external_commands()
            raw = cPickle.dumps(cmds)
            raw = base64.b64encode(zlib.compress(raw))
        return raw
    
    
    get_external_commands.need_lock = False
    get_external_commands.doc = doc
    
    doc = u'Does the daemon got configuration (receiver)'
    
    
    # NB: only useful for receiver
    def got_conf(self):
        return self.app.cur_conf is not None
    
    
    got_conf.need_lock = False
    got_conf.doc = doc
    
    
    def push_messages(self, messages):
        # type: (List[InterDaemonMessage]) -> None
        with self.app.push_messages_by_arbiter_lock:
            logger_message = LoggerFactory.get_logger(u'MANAGE INTER DAEMON MESSAGE')
            self.app.push_messages_by_arbiter.extend(messages)
            if logger_message.is_debug():
                logger_message.debug(u'new 〖%s〗 messages receive from arbiter' % len(messages))
                for m in messages:
                    logger_message.debug(u'message receive : 〖%s〗' % m.get_message_info())
    
    
    push_messages.doc = u'Send messages from Arbiter to other daemon (internal)'
    push_messages.method = u'POST'
    push_messages.need_lock = False
    
    
    def get_messages(self):
        # type: () -> str
        with self.app.messages_to_send_to_arbiter_lock:
            ret = self.app.messages_to_send_to_arbiter
            self.app.messages_to_send_to_arbiter = []
            get_messages_buffer = cPickle.dumps(ret)
            get_messages_buffer = zlib.compress(get_messages_buffer)
            get_messages_buffer = base64.b64encode(get_messages_buffer)
            return get_messages_buffer
    
    
    get_messages.doc = u'Get messages from other daemon to Arbiter (internal)'
    get_messages.need_lock = False


class BaseSatellite(Daemon):
    u"""Super class for Broker, scheduler, reactionner, poller"""
    
    # Should we look at passive property for external connection
    is_using_passive_connection_information = False
    
    
    def __init__(self, name, config_file, is_daemon, do_replace, debug, debug_file, daemon_id=0):
        super(BaseSatellite, self).__init__(name, config_file, is_daemon, do_replace, debug, debug_file, daemon_id)
        # Ours schedulers
        self.schedulers = {}
        # Our arbiters
        self.arbiters = {}
        
        # Our pollers and reactionners, maybe void for some daemons
        self.pollers = {}
        self.reactionners = {}
        self.receivers = {}
        # Now we create the interfaces
        self._add_http_interface(IForArbiter(self))
        
        # Can have a queue of external_commands given by modules will be taken by arbiter to process
        self.external_commands = []
        self.external_commands_lock = threading.RLock()
        
        self.push_messages_by_arbiter = []  # type: List[InterDaemonMessage]
        self.push_messages_by_arbiter_lock = threading.RLock()
        
        self.messages_to_send_to_arbiter = []  # type: List[InterDaemonMessage]
        self.messages_to_send_to_arbiter_lock = threading.RLock()
        # message to send at arbiter made by the daemon (the messages will be add to messages_to_send_to_arbiter in manage_inter_daemon_messages)
        self.messages_to_send_to_arbiter_internal = []  # type: List[InterDaemonMessage]
        self.messages_to_send_to_arbiter_internal_lock = threading.RLock()
        
        self.activated = True
        self.spare = False
        # We will have a tread by distant satellites, so we must protect our access
        self.satellite_lock = threading.RLock()
        self.satellite_threads = {}
        
        # Keep broks so they can be eaten by a broker
        self.broks = {}
        
        # Last configuration change
        self._last_configuration_change = 0  # epoch of the last change from the arbiter
    
    
    def do_loop_turn(self):
        raise NotImplementedError
    
    
    def manage_inter_daemon_messages(self):
        # Step process message send by arbiter
        self._process_message_from_arbiter()
        
        # Step get all message to send at arbiter
        self._get_messages_to_send_at_arbiter()
    
    
    def _get_messages_to_send_at_arbiter(self):
        logger_message = LoggerFactory.get_logger(u'MANAGE INTER DAEMON MESSAGE')
        
        messages_to_send_to_arbiter = []
        for module in self.modules_manager.get_all_alive_instances():
            messages_to_send_to_arbiter.extend(module.get_messages_to_send_at_arbiter())
        
        with self.messages_to_send_to_arbiter_internal_lock:
            tmp = self.messages_to_send_to_arbiter_internal
            self.messages_to_send_to_arbiter_internal = []
            messages_to_send_to_arbiter.extend(tmp)
        
        if messages_to_send_to_arbiter:
            logger_message.info(u'Message to send at arbiter:〖%s〗' % len(messages_to_send_to_arbiter))
        
        if logger_message.is_debug():
            for message_to_send_to_arbiter in messages_to_send_to_arbiter:
                logger_message.debug(u'Message to send at arbiter:〖%s〗' % message_to_send_to_arbiter.get_message_info())
        
        with self.messages_to_send_to_arbiter_lock:
            self.messages_to_send_to_arbiter.extend(messages_to_send_to_arbiter)
    
    
    def _process_message_from_arbiter(self):
        logger_message = LoggerFactory.get_logger(u'MANAGE INTER DAEMON MESSAGE')
        
        with self.push_messages_by_arbiter_lock:
            if not self.push_messages_by_arbiter:
                return
            push_message_by_arbiter = self.push_messages_by_arbiter
            self.push_messages_by_arbiter = []
        
        nb_message_todo = len(push_message_by_arbiter)
        process_messages = []
        for inter_daemon_message in push_message_by_arbiter:
            if inter_daemon_message.message_to.daemon_name != self.name:
                continue
            if inter_daemon_message.message_to.module_name == u'':
                # Message is for our daemon, not its modules
                self.handle_push_messages_by_arbiter(inter_daemon_message)
                process_messages.append(inter_daemon_message)
                continue
            
            # OK : it's a message for one type of module, we give it to all the modules concerned
            for module in self.modules_manager.get_all_alive_instances():
                if inter_daemon_message.message_to.module_name == module.get_name():
                    module.handle_push_messages_by_arbiter(inter_daemon_message)
                    process_messages.append(inter_daemon_message)
        
        for process_message in process_messages:
            logger_message.debug(u'Processed message from arbiter:〖%s〗' % process_message.get_message_info())
            push_message_by_arbiter.remove(process_message)
        logger_message.info(u'Message from arbiter:〖%s〗 processed:〖%s〗 remaining:〖%s〗' % (nb_message_todo, len(process_messages), len(push_message_by_arbiter)))
        if logger_message.is_debug():
            for remaining_message in push_message_by_arbiter:
                logger_message.debug(u'Remaining message from arbiter:〖%s〗' % remaining_message.get_message_info())
        
        with self.push_messages_by_arbiter_lock:
            self.push_messages_by_arbiter.extend(push_message_by_arbiter)
    
    
    def send_message_to_arbiter(self, inter_daemon_message):
        # type: (InterDaemonMessage) -> None
        logger_message = LoggerFactory.get_logger(u'MANAGE INTER DAEMON MESSAGE')
        logger_message.debug(u'%s queue message to send at arbiter 〖%s〗' % (self.daemon_type, inter_daemon_message.get_message_info()))
        with self.messages_to_send_to_arbiter_internal_lock:
            self.messages_to_send_to_arbiter_internal.append(inter_daemon_message)
    
    
    def handle_push_messages_by_arbiter(self, inter_daemon_message):
        # type: (InterDaemonMessage) -> None
        if inter_daemon_message.message_type == u'ping':
            self.send_message_to_arbiter(InterDaemonMessage(u'pong', message_to=inter_daemon_message.message_from, message_from=inter_daemon_message.message_to, data={}))
    
    
    def reset_configuration_change(self):
        self._last_configuration_change = int(time.time())
    
    
    def is_configuration_change_recent(self):
        now = int(time.time())
        elapsed_time = abs(now - self._last_configuration_change)  # note: abs= manage time get back
        return elapsed_time <= 120
    
    
    def is_activated(self):
        return self.activated
    
    
    # Someone asks us our broks. We send them, and clean the queue
    def get_broks(self):
        res = copy.copy(self.broks)
        self.broks.clear()
        return res
    
    
    # The arbiter can resend us new conf in the pyro_daemon port.
    # We do not want to lose time about it, so it's not a blocking  wait, timeout = 0s
    # If it sends us a new conf, we re init the connections of all schedulers
    def watch_for_new_conf(self, timeout):
        self.sleep(timeout)
    
    
    # by default, do nothing
    def clean_previous_run(self):
        return
    
    
    def _print_new_update_conf_received(self):
        # type: () -> PartLogger
        
        if self.cur_conf is None:  # First configuration
            _logger = logger_configuration
            logger_configuration.info(u'----- Loading the new configuration from the arbiter')
        else:  # configuration update
            _logger = logger_configuration_update
            logger_configuration_update.info(u'----- Loading a configuration update from the arbiter')
        return _logger
    
    
    def _set_is_activated(self, activated, _logger):
        # type: (bool, PartLogger) -> bool
        was_activated = self.activated
        self.activated = activated
        if self.activated != was_activated:
            if self.activated:
                _logger.info(u'Switching from sleeping to active mode')
            else:
                _logger.info(u'Switching from active to sleeping mode')
        return was_activated
    
    
    def _set_spare(self, spare, _logger):
        # type: (bool, PartLogger) -> bool
        was_spare = self.spare
        self.spare = spare
        if self.spare != was_spare:
            if self.spare:
                _logger.info(u'Going into SPARE')
            else:
                _logger.info(u'Going from SPARE to MASTER')
        return was_spare
    
    
    # When arbiter deactivated us because we are a spare we need to clean some data.
    def _go_as_not_active(self, was_activated, _logger):
        # type: (bool, PartLogger) -> NoReturn
        # I'm not activated AKA spare, make some clean and log that
        self.clean_previous_run()
        self.arbiters.clear()
        self._clean_known_daemons()
        
        if was_activated:
            _logger.info(u'Stopping modules as we are going in sleep mode')
        self.modules_manager.stop_all()
        self.have_modules = False
    
    
    # When arbiter deactivated us we need to clear known daemons because shinken-healthcheck talk to info.
    def _clean_known_daemons(self):
        for daemon_list in (u'receivers', u'pollers', u'reactionners', u'schedulers'):
            if hasattr(self, daemon_list):
                getattr(self, daemon_list).clear()
    
    
    @staticmethod
    def __print_added_daemon_line(daemon_type, daemon_name, daemon_id, uri, _logger):
        # type: (unicode, unicode, Optional[unicode], unicode, PartLogger) -> NoReturn
        
        shard_string = u'[shard_id=%4d]' % daemon_id if daemon_id is not None else '               '
        _logger.info(u'  + ADDED     %-13s : [name=%-20s] %s [uri=%s]' % (daemon_type, daemon_name, shard_string, uri))
    
    
    @staticmethod
    def __print_removed_daemon_line(daemon_type, daemon_name, daemon_id, uri, _logger):
        # type: (unicode, unicode, Optional[unicode], unicode, PartLogger) -> NoReturn
        
        shard_string = u'[shard_id=%4d]' % daemon_id if daemon_id is not None else '               '
        _logger.info(u'  - REMOVED   %-13s : [name=%-20s] %s [uri=%s]' % (daemon_type, daemon_name, shard_string, uri))
    
    
    def _print_new_and_deleted_daemons(self, new_schedulers=None, deleted_schedulers=None, new_arbiters=None, new_pollers=None, deleted_pollers=None, new_reactionners=None, deleted_reactionners=None, _logger=None):
        # type: (List, List, List, List, List, List, List, PartLogger) -> NoReturn
        # Print new daemons, in a bloc
        
        if deleted_reactionners is None:
            deleted_reactionners = []
        if new_reactionners is None:
            new_reactionners = []
        if deleted_pollers is None:
            deleted_pollers = []
        if new_pollers is None:
            new_pollers = []
        if new_arbiters is None:
            new_arbiters = []
        if deleted_schedulers is None:
            deleted_schedulers = []
        if new_schedulers is None:
            new_schedulers = []
        
        if deleted_schedulers or deleted_pollers or deleted_reactionners:
            _logger.info(u'The arbiter asked us to remove daemons:')
            for (daemon_name, daemon_id, uri) in deleted_schedulers:
                self.__print_removed_daemon_line(u'scheduler', daemon_name, daemon_id, uri, _logger)
            for (daemon_name, uri) in deleted_pollers:
                self.__print_removed_daemon_line(u'poller', daemon_name, None, uri, _logger)
            for (daemon_name, uri) in deleted_reactionners:
                self.__print_removed_daemon_line(u'reactionner', daemon_name, None, uri, _logger)
        
        if new_schedulers or new_arbiters or new_pollers or new_reactionners:
            _logger.info(u'The arbiter send us new daemons:')
            for (daemon_name, daemon_id, uri) in new_schedulers:
                self.__print_added_daemon_line(u'scheduler', daemon_name, daemon_id, uri, _logger)
            for (daemon_name, uri) in new_arbiters:
                self.__print_added_daemon_line(u'arbiter', daemon_name, None, uri, _logger)
            for (daemon_name, uri) in new_pollers:
                self.__print_added_daemon_line(u'poller', daemon_name, None, uri, _logger)
            for (daemon_name, uri) in new_reactionners:
                self.__print_added_daemon_line(u'reactionner', daemon_name, None, uri, _logger)
    
    
    def delete_shard_id(self, shard_id, reason):
        scheduler = self.schedulers.get(shard_id, None)
        if scheduler is None:
            logger_configuration_update.info(u'Arbiter asked me to remove the shard %s I do not have it in my keys: %s' % (shard_id, self.schedulers.keys()))
            return
        scheduler_name = scheduler[u'name']
        uri = scheduler[u'uri']
        reason_string = u'' if not reason else u' because %s' % reason
        logger_configuration_update.info(u'----- Loading a configuration update from the arbiter')
        logger_configuration_update.info(u'The arbiter ask us to remove daemons %s:' % reason_string)
        self.__print_removed_daemon_line(u'scheduler', scheduler_name, shard_id, uri, logger_configuration_update)
        
        # Do the job
        del self.schedulers[shard_id]
    
    
    @staticmethod
    def _find_previous_daemon_by_uri(into, uri):
        for (_id, daemon_entry) in into.iteritems():
            if daemon_entry[u'uri'] == uri:
                return _id
        return None
    
    
    @staticmethod
    def _get_daemon_uri(daemon):
        proto = u'https' if daemon[u'use_ssl'] else u'http'
        uri = u'%s://%s:%s/' % (proto, daemon[u'address'], daemon[u'port'])
        return uri
    
    
    @staticmethod
    def _update_daemon_addr_port_with_satellitemap(daemon, satellite_map, _logger):
        # type: (Dict, Dict, PartLogger) -> NoReturn
        
        daemon_name = daemon[u'name']
        map_entry = satellite_map.get(daemon_name, None)
        if map_entry:
            # MAP ENTRY will looks like   {'port': 7768, 'address': u'51.15.255.102'}
            old_address = daemon.get(u'address', u'')
            old_port = daemon.get(u'port', u'')
            new_address = map_entry.get(u'address', u'')
            new_port = map_entry.get(u'port', u'')
            _logger.get_sub_part(u'SATELLITEMAP').info(u'Replacing the daemon %-15s to address:port from %s:%s => %s:%s as defined in our daemon .cfg file (satellitemap property)' % (daemon_name, old_address, old_port, new_address, new_port))
            daemon.update(map_entry)
    
    
    def _set_default_values_to_scheduler_entry(self, entry):
        raise NotImplementedError()
    
    
    def _set_daemon_id_of_scheduler(self, daemon, daemon_id):
        raise NotImplementedError()
    
    
    # Simple function to stack an entry to new/delete schedulers/pollers/reactionners/etc
    @staticmethod
    def __stack_into_daemons_list(daemon_name, daemon_id, daemon_uri, lst, do_stack_id):
        daemon_id_entry = daemon_id if do_stack_id else None
        lst.append((daemon_name, daemon_id_entry, daemon_uri))
    
    
    def _set_or_update_scheduler_from_configuration(self, new_daemon, new_daemon_id, global_conf, new_schedulers, deleted_schedulers, _logger, do_stack_daemon_id=True):
        daemon_name = new_daemon[u'name']
        
        # replacing poller address and port by those defined in satellitemap
        # IMPORTANT: do this BEFORE get uri
        self._update_daemon_addr_port_with_satellitemap(new_daemon, global_conf[u'satellitemap'], _logger)
        uri = self._get_daemon_uri(new_daemon)
        new_daemon[u'uri'] = uri
        
        # There are 5 cases:
        # * 1=> a new daemon with new id  => just create it
        # * 2=> a new daemon with old id  (one replace another) => delete old, create new entry
        # * 3=> old daemon with new id  (daemon change) => no log DELETE but just move the entry
        # * 4=> another daemon on old id  (daemon change, and replace another) => log the DELETE, move the entry
        # * 5=> same daemon on same id  => touch nothing
        
        previous_id = self._find_previous_daemon_by_uri(self.schedulers, uri)
        
        # But maybe we need to keep them from the past (if same uri => aka same daemon)
        if previous_id is not None:  # Daemon was already exiting (case 3->5)
            
            if previous_id == new_daemon_id:  # CASE 5: same daemon on same id
                _logger.debug(u'The daemon %s did not change id (%s) skipping it' % (daemon_name, new_daemon_id))
                return
            
            # => CASE 3 or 4 (daemon change)
            previous_daemon = self.schedulers[previous_id]  # must exist as previous_id is not None
            
            _logger.info(u'The daemon %s did switch id %s -> %s' % (daemon_name, previous_id, new_daemon_id))
            # We log it is deleted from this old place
            self.__stack_into_daemons_list(previous_daemon[u'name'], previous_id, previous_daemon[u'uri'], deleted_schedulers, do_stack_daemon_id)
            
            # Maybe it can overwrite another daemon, if so, show it
            new_place_daemon_that_will_be_overlap = self.schedulers.get(new_daemon_id, None)
            if new_place_daemon_that_will_be_overlap is not None:  # CASE 4
                self.__stack_into_daemons_list(new_place_daemon_that_will_be_overlap[u'name'], new_daemon_id, new_place_daemon_that_will_be_overlap[u'uri'], deleted_schedulers, do_stack_daemon_id)
            else:  # CASE 3: old daemon with new id  (daemon change)
                pass
            # In all cases, we are logging this new daemon +ADDED (to have its new number)
            self.__stack_into_daemons_list(daemon_name, new_daemon_id, uri, new_schedulers, do_stack_daemon_id)
            
            # The scheduler is no more at this entry
            del self.schedulers[previous_id]
            # Let the old object know it did change
            self._set_daemon_id_of_scheduler(previous_daemon, new_daemon_id)
            
            self.schedulers[new_daemon_id] = previous_daemon  # do not lose the old object, just moving it
        else:  # New daemon, CASE 1 or 2
            _logger.debug(u'The daemon %s is a new one (will be set at %s)' % (daemon_name, new_daemon_id))
            # Create the new structure based on what the arbiter did send us
            self._set_default_values_to_scheduler_entry(new_daemon)
            self._set_daemon_id_of_scheduler(new_daemon, new_daemon_id)
            
            # Now check if we didn't overlap an old scheduler in this place
            deleted_scheduler = self.schedulers.get(new_daemon_id, None)
            if deleted_scheduler is not None:  # CASE 2: we are replacing this one
                self.__stack_into_daemons_list(deleted_scheduler[u'name'], new_daemon_id, deleted_scheduler[u'uri'], deleted_schedulers, do_stack_daemon_id)
            else:  # CASE 1: new one
                pass
            # In both cases, we are logging this new daemon
            self.__stack_into_daemons_list(daemon_name, new_daemon_id, uri, new_schedulers, do_stack_daemon_id)
            # Now save the object
            self.schedulers[new_daemon_id] = new_daemon
    
    
    # Give the arbiter the data about what I manage as shards/schedulers to know if we should
    # be updated or not. We will return the shards ids or them
    # Note: maybe the arbiter did just send us a new_conf, but we did not consume it, so
    # if we have one, we should look in it instead of the running conf that will be changed at
    # the end of our turn
    def get_currently_managed_configuration(self):
        configuration_incarnation_dump = {}
        if self.configuration_incarnation is not None:  # at start, we are void
            configuration_incarnation_dump = self.configuration_incarnation.dump_as_json()
        r = {u'configuration_incarnation_dump': configuration_incarnation_dump, u'schedulers': {}, u'activated': self.activated}
        
        # Maybe we did receive a new configuration but did not consume it
        with self.satellite_lock:
            if self.new_conf:  # got a new conf, use it instead of the running one
                for (shard_id, scheduler) in self.new_conf[u'schedulers'].iteritems():
                    r[u'schedulers'][shard_id] = scheduler[u'name']
                return r
        
        # no new conf, and no conf at all? we are just void
        if not self.already_have_conf:
            return None
        
        # ok look in the running configuration
        for (shard_id, scheduler) in self.schedulers.iteritems():
            # Maybe it's a synchronizer for try check that is here, if so, skip it
            if scheduler.get(u'unmanaged_by_arbiter', False):
                continue
            r[u'schedulers'][shard_id] = scheduler[u'name']
        
        logger.debug(u'The arbiter ask us what we are managing, it is %s ' % r)
        
        return r
    
    
    # Call by arbiter to get our external commands
    def get_external_commands(self):
        res = self.external_commands
        self.external_commands = []
        return res
    
    
    def get_satellite_connections(self):
        res = []
        for (daemon_type, daemon_list) in [(u'receiver', getattr(self, u'receivers', None)), (u'poller', getattr(self, u'pollers', None)), (u'reactionner', getattr(self, u'reactionners', None)), (u'scheduler', getattr(self, u'schedulers', None))]:
            if not daemon_list:
                continue
            
            for daemon_conf in daemon_list.values():
                # scheduler can have directly scheduler object here, skip it
                if not isinstance(daemon_conf, dict):
                    continue
                # if synchronizer dummy entry, skip it
                if daemon_conf.get(u'unmanaged_by_arbiter', False):
                    continue
                # skip passive pollers on schedulers
                if self.is_using_passive_connection_information and u'passive' in daemon_conf and not daemon_conf[u'passive']:
                    continue
                
                proto = u'https' if daemon_conf[u'use_ssl'] else u'http'
                con_info = {
                    u'name'   : daemon_conf[u'name'],
                    u'type'   : daemon_type,
                    u'address': daemon_conf[u'address'],
                    u'proto'  : proto,
                    u'uri'    : daemon_conf[u'uri'],
                    u'port'   : daemon_conf[u'port']
                }
                if u'passive' in daemon_conf:
                    con_info[u'passive'] = daemon_conf[u'passive']
                if u'timeout' in daemon_conf:
                    con_info[u'timeout'] = daemon_conf[u'timeout']
                res.append(con_info)
        return res
    
    
    # Get the good tabs for links by the kind. If unknown, return None
    def get_link_from_type(self, daemon_type, daemon_id):
        t = {
            u'scheduler': self.schedulers,
            u'arbiter'  : self.arbiters,
        }
        with self.satellite_lock:
            return t.get(daemon_type, {}).get(daemon_id, None)
    
    
    def get_any_link_from_type(self, daemon_type):
        t = {
            u'scheduler': self.schedulers,
            u'arbiter'  : self.arbiters,
        }
        with self.satellite_lock:
            return len(t.get(daemon_type, {})) != 0
    
    
    # Check if we do not connect too often to this
    def is_connection_try_too_close(self, elt):
        now = time.time()
        last_connection = elt[u'last_connection']
        if now - last_connection < 5:
            return True
        return False
    
    
    # initialize or re-initialize connection with scheduler
    def pynag_con_init(self, sat_entry):
        daemon_type = sat_entry[u'type']
        
        if daemon_type == u'scheduler':
            # If sched is not active, I do not try to init
            # it is just useless
            is_active = sat_entry[u'active']
            if not is_active:
                return
        
        # If we try to connect too much, we slow down our tests
        if self.is_connection_try_too_close(sat_entry):
            return
        
        # Ok, we can now update it
        sat_entry[u'last_connection'] = time.time()
        
        # DBG: print 'Running id before connection', daemon_incarnation
        uri = sat_entry[u'uri']
        timeout = sat_entry.get(u'timeout', 3)
        data_timeout = sat_entry.get(u'data_timeout', 120)
        name = sat_entry[u'name']
        try:
            sat_entry[u'con'] = HTTPClient(uri=uri, strong_ssl=sat_entry[u'hard_ssl_name_check'], timeout=timeout, data_timeout=data_timeout)
        except HTTPExceptions as exp:
            # But the multiprocessing module is not compatible with it!
            # so we must disable it immediately after
            logger_connection.info(u'Connection problem to the %s %s (uri="%s"): %s' % (daemon_type, name, uri, str(exp)))
            sat_entry[u'con'] = None
            return
        
        before_ = time.time()
        did_connect = self.ping_and_check_distant_daemon(sat_entry)
        elapsed = time.time() - before_
        
        if did_connect:
            logger_connection.info(u'Connection OK to the %s %s in %.3fs (uri="%s", ping_timeout=%ss, data_timeout=%ss)' % (daemon_type, name, elapsed, uri, timeout, data_timeout))
    
    
    # For a new distant daemon, if it is a scheduler, ask for a new full broks generation
    def _manage_new_distant_daemon_incarnation(self, entry, old_incar, new_incar):
        raise NotImplemented(u'_manage_new_distant_daemon_incarnation')
    
    
    def ping_and_check_distant_daemon(self, sat_entry):
        con = sat_entry[u'con']
        daemon_type = sat_entry[u'type']
        if con is None:
            self.pynag_con_init(sat_entry)
            con = sat_entry['con']
            if con is None:
                return False
        try:
            # initial ping must be quick
            con.get(u'ping')
            new_incar = con.get(u'get_daemon_incarnation')
            new_incarnation = False
            # protect daemon_incarnation from modification
            with self.satellite_lock:
                # data transfer can be longer
                daemon_incarnation = sat_entry[u'daemon_incarnation']
                # logger.debug('type[%s] daemon_incarnation old[%s]/new[%s]' % (daemon_type, daemon_incarnation, new_incar))
                
                # The schedulers have been restarted: it has a new run_id.
                # So we clear all verifs, they are obsolete now.
                
                if new_incar != daemon_incarnation:
                    # Only log schedulers currently, that's the only one that matter currently here
                    if daemon_type == u'scheduler':
                        if new_incar == {}:
                            logger_configuration.info(u'The %s has not yet received any configuration from Arbiter.' % (sat_entry[u'name']))
                        elif daemon_incarnation:
                            logger_configuration.info(u'The %s change it\'s configuration from %s to %s' % (sat_entry[u'name'], new_incar[u'configuration_incarnation_uuid'], daemon_incarnation[u'configuration_incarnation_uuid']))
                        else:  # it was without any configuration
                            logger_configuration.info(u'The %s received a new configuration (uuid=%s)' % (sat_entry[u'name'], new_incar[u'configuration_incarnation_uuid']))
                    new_incarnation = True
                # Ok all is done, we can save this new incarnation
                sat_entry[u'daemon_incarnation'] = new_incar
            if new_incarnation:
                self._manage_new_distant_daemon_incarnation(sat_entry, daemon_incarnation, new_incar)
            return True
        except HTTPExceptions as exp:
            logger.error(u'Connection problem to the %s %s: %s' % (daemon_type, sat_entry[u'name'], str(exp)))
            sat_entry[u'con'] = None
            return False
    
    
    # Daemon get something from distant, should be implemented!
    def get_jobs_from_distant(self, e):
        raise NotImplementedError(u'get_jobs_from_distant')
    
    
    # By default we are connecting to everyone
    # only the scheduler will skip some pollers/reactionners because it need to
    # connect only to passive one
    def should_connect_to_distant_satellite(self, satellite_type, distant_link):
        return True
    
    
    def _do_satellite_thread(self, s_type, s_id, t_name):
        try:
            logger.debug(u'SATELLITE THREAD: Starting thread to exchange with %s [%s]' % (s_type, t_name))
            self.do_satellite_thread(s_type, s_id, t_name)
            logger.debug(u'SATELLITE THREAD: Exiting thread to exchange with %s [%s]' % (s_type, t_name))
        except Exception:  # we need ALL the exceptions here
            logger.warning(u'The thread for the %s %s is in error: but it will restart. Your monitoring is still working. Please fill a bug with your log.' % (s_type, t_name))
            logger.print_stack(level=logging.WARNING)
    
    
    def do_satellite_thread(self, s_type, s_id, t_name):
        with self.satellite_lock:
            distant_link = self.get_link_from_type(s_type, s_id)
            if distant_link is None:  # already down?
                return
        
        uri = distant_link[u'uri']
        while not self.interrupted:
            with self.satellite_lock:
                # first look if we are still need or not
                distant_link = self.get_link_from_type(s_type, s_id)
            
            if distant_link is None or (uri != distant_link[u'uri']):  # no more present, or uri did change? EXIT!!!
                logger.info(u'SATELLITE THREAD: The connection thread to the %s with the id %s (%s) is not longer need as this daemon is no more that same as before. Restarting it.' % (s_type, s_id, t_name))
                return
            
            # If the Arbiter did deactivate us, we can stop all our running threads, they will be restarted
            # when we will have a new configuration
            if self.deactivated_by_arbiter:
                logger.info(u'SATELLITE THREAD: The connection thread to the %s with the id %s (%s) is not longer need as this daemon is now idle.' % (s_type, s_id, t_name))
                return
            
            # For some distant satellite, we have a thread but currently the distant daemon is not
            # interesting (not passive, etc.). we need to have a thread, so it's already running, and ready as soon as
            # the configuration will change
            if self.should_connect_to_distant_satellite(s_type, distant_link):
                # we have an entry, so we can ping it
                if self.ping_and_check_distant_daemon(distant_link):
                    # so now if it changes, we can know that we must drop this thread
                    # logger.debug("[Broks] SATELLITE THREAD: get broks from daemon:[%s] id:[%s] push_flavor:[%s] name:[%s]" % (s_type, s_id, e['daemon_incarnation'], t_name))
                    self.get_jobs_from_distant(distant_link)
            
            time.sleep(distant_link.get(u'loop_delay', 1))
    
    
    # We will look for satellites, and if we don't have a thread or a dead one, start a new one
    def assert_valid_satellite_threads(self):
        with self.satellite_lock:
            in_conf_satellites = set()
            types = {u'scheduler': self.schedulers, u'poller': self.pollers, u'reactionner': self.reactionners, u'receiver': self.receivers}
            for (satellite_type, satellite_definition) in types.iteritems():
                for (satellite_id, satellite_entry) in satellite_definition.iteritems():
                    self._assert_one_satellite_thread(satellite_type, satellite_id, satellite_entry)
                    in_conf_satellites.add(satellite_entry[u'name'])
            
            # Cleanup of disabled thread entries
            launched_satellites = set([daemon_name for daemon_name in self.satellite_threads.iterkeys()])
            rogue_satellites = launched_satellites - in_conf_satellites
            for satellite_name in rogue_satellites:
                thread = self.satellite_threads[satellite_name]  # type: threading.Thread
                if not thread.is_alive():
                    thread.join()
                    del self.satellite_threads[satellite_name]
    
    
    def _assert_one_satellite_thread(self, satellite_type, satellite_id, satellite_entry):
        satellite_name = satellite_entry[u'name']
        thread = self.satellite_threads.get(satellite_name, None)
        restart = False
        
        if thread is None:
            restart = True
        elif not thread.is_alive():
            thread.join(1)
            restart = True
        
        if restart:
            thread_name = u'SAT:%.3s:%s' % (satellite_type, (u'%s%s' % (satellite_name[:5], satellite_name[-2:])) if (len(satellite_name) > 7 and u'0' <= satellite_name[-1] <= u'9') else satellite_name)
            thread = threading.Thread(None, target=self._do_satellite_thread, name=thread_name, args=(satellite_type, satellite_id, thread_name))
            thread.daemon = True
            thread.start()
            self.satellite_threads[satellite_name] = thread
