#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#     Gabes Jean, naparuba@gmail.com
#     Gerhard Lausser, Gerhard.Lausser@consol.de
#     Gregory Starck, g.starck@gmail.com
#     Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import copy
import ctypes
import gc
import json
import math
import os
import re
import sys
import threading
import time
from collections import deque
from weakref import proxy

import psutil

from shinken.log import logger
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.date_helper import get_now

if TYPE_CHECKING:
    from shinken.misc.type_hint import Any, Dict, List, Optional, Tuple, Union

try:
    from ClusterShell.NodeSet import NodeSet, NodeSetParseRangeError
except ImportError:
    NodeSet = None
    NodeSetParseRangeError = NodeSet

try:
    if hasattr(sys.stdout, 'encoding'):
        stdout_encoding = sys.stdout.encoding
        safe_stdout = (stdout_encoding == 'UTF-8')
    else:
        safe_stdout = False
except Exception as exp:
    logger.error('Encoding detection error= %s' % (exp))
    safe_stdout = False

PAGESIZE = 0


# NOTE: This object is NOT meant to be a singleton (because of possible fork issues)
# It must be a local object attached to a module, a daemon, or whatever.
class DelayedForceMemoryTrimManagerByPeriod(object):
    __slots__ = ('_last_force_trim_time', '_minimum_time_before_next_force_trim')
    
    
    def __init__(self, minimum_time_before_next_force_trim):
        # type: (float) -> None
        self._last_force_trim_time = 0  # type: float
        self._minimum_time_before_next_force_trim = minimum_time_before_next_force_trim  # type: float
    
    
    def ask_for_memory_trim(self, context):
        # type: (unicode) -> None
        now = get_now()
        if now - self._last_force_trim_time < self._minimum_time_before_next_force_trim:
            return
        self._last_force_trim_time = now
        force_memory_trimming(context)


# NOTE: This object is NOT meant to be a singleton (because of possible fork issues)
# It must be a local object attached to a module, a daemon, or whatever.
class DelayedForceMemoryTrimManagerByCounter(object):
    __slots__ = ('_last_force_trim_time', '_force_trim_asked_counter', '_minimum_ask_number_before_force_trim', '_counter_reset_period')
    
    
    def __init__(self, minimum_ask_number_before_force_trim, counter_reset_period):
        # type: (int, float) -> None
        self._last_force_trim_time = 0  # type: float
        self._force_trim_asked_counter = 0  # type: int
        self._minimum_ask_number_before_force_trim = minimum_ask_number_before_force_trim  # type: int
        self._counter_reset_period = counter_reset_period  # type: float
    
    
    def ask_for_memory_trim(self, context):
        # type: (unicode) -> None
        now = get_now()
        if self._counter_reset_period > 0:
            if now - self._last_force_trim_time >= self._counter_reset_period:
                self._last_force_trim_time = now
                self._force_trim_asked_counter = 0
        
        self._force_trim_asked_counter += 1
        if self._force_trim_asked_counter >= self._minimum_ask_number_before_force_trim:
            self._last_force_trim_time = now
            self._force_trim_asked_counter = 0
            force_memory_trimming(context)


def start_malloc_trim_thread():
    memory_trimming_enable_gc_collect_2()
    t = threading.Thread(None, target=malloc_trim, name='malloc-trim')
    t.daemon = True
    t.start()


def force_malloc_trim():
    global is_malloc_trim_forced
    is_malloc_trim_forced = True  # ask the malloc thread to do the job


def force_memory_trimming(context):
    # type: (unicode) -> None
    gen = 2
    pid = os.getpid()
    before_collect = time.time()
    gc.collect(gen)
    logger.log_perf(before_collect, u'MEMORY', u'REQUESTED python garbage collection ( %s ) for memory area ( generation %d ) in process with pid %d' % (context, gen, pid), min_time=0)
    force_malloc_trim()


def memory_trimming_disable_gc_collect_2():
    global malloc_trim_limit_nb_call_to_gc_collect_2
    malloc_trim_limit_nb_call_to_gc_collect_2 = 0


def memory_trimming_enable_gc_collect_2():
    global malloc_trim_limit_nb_call_to_gc_collect_2
    malloc_trim_limit_nb_call_to_gc_collect_2 = None


def memory_trimming_allow_some_calls_to_gc_collect_2(times):
    # type: (int) -> None
    global malloc_trim_limit_nb_call_to_gc_collect_2
    if malloc_trim_limit_nb_call_to_gc_collect_2 is not None:
        malloc_trim_limit_nb_call_to_gc_collect_2 = int(times)


# Boolean used to ask for libc6.malloc_trim(0) on the malloc_trim thread
# NOTE: only ONE thread must do the call, NEVER call from multiple threads
is_malloc_trim_forced = False
malloc_trim_limit_nb_call_to_gc_collect_2 = None  # type: Optional[int]


def malloc_trim(do_forever=True):
    global is_malloc_trim_forced
    global malloc_trim_limit_nb_call_to_gc_collect_2
    gc.disable()
    gen = 0
    _i = 0
    try:
        libc6 = ctypes.CDLL('libc.so.6')
        pid = os.getpid()
        while True:
            _i += 1
            gen += 1
            gen %= 2
            before_collect = time.time()
            if not do_forever or (_i % 10 == 0 and (malloc_trim_limit_nb_call_to_gc_collect_2 is None or malloc_trim_limit_nb_call_to_gc_collect_2 > 0)):  # every 10min, do a huge collect
                if malloc_trim_limit_nb_call_to_gc_collect_2 is not None and malloc_trim_limit_nb_call_to_gc_collect_2 > 0:
                    malloc_trim_limit_nb_call_to_gc_collect_2 = malloc_trim_limit_nb_call_to_gc_collect_2 - 1
                gen = 2
            gc.collect(gen)
            logger.log_perf(before_collect, u'MEMORY', u'PERIODIC python garbage collection for memory area ( generation %d ) in process with pid %d' % (gen, pid), min_time=0)
            
            time_start = time.time()
            libc6.malloc_trim(0)
            logger.log_perf(time_start, u'MEMORY', u'Return of freed memory space to the OS ( malloc_trim ) in process with pid %d' % pid)
            if not do_forever:
                return
            # Sleep 60s, but maybe launch malloc_trim is ask by another thread
            for _ in xrange(60):
                time.sleep(1)
                # Maybe someone ask us a malloc_trim now, do it
                # IMPORTANT: only call in THIS thread, NEVER in another
                if is_malloc_trim_forced:
                    time_start = time.time()
                    libc6.malloc_trim(0)
                    logger.log_perf(time_start, u'MEMORY', u'Return of freed memory space to the OS ( malloc_trim ) in process with pid %d' % pid)
                    is_malloc_trim_forced = False
    except Exception:  # noqa
        logger.info(u'malloc_trim unsupported by this OS.')


def get_memory_consumption():
    global PAGESIZE
    if os.name == 'nt':
        p = psutil.Process()
        memory_info = p.memory_info()
        return memory_info.wset / (1024 * 1024), memory_info.vms / (1024 * 1024)
    # not linux?
    if not os.path.exists('/proc/self/statm'):
        return 0
    if PAGESIZE == 0:
        PAGESIZE = os.sysconf('SC_PAGESIZE')
    with open('/proc/self/statm') as f:
        _statm = f.read().split()
    
    return (int(_statm[1]) * PAGESIZE) / (1024 * 1024), (int(_statm[2]) * PAGESIZE) / (1024 * 1024)


_memory_usage_regexp = re.compile(r'([0-9]+)')  # compile once


def get_global_memory_consumption():
    # TODO: manage windows
    if os.name == 'nt':
        virtual_memory = psutil.virtual_memory()
        meminfo = {
            'MemTotal'    : virtual_memory.total / 1024,
            'MemFree'     : virtual_memory.available / 1024,
            'MemAvailable': virtual_memory.available / 1024,
        }
        return meminfo
    
    else:
        try:
            with open('/proc/meminfo', 'r') as meminfoProc:
                lines = meminfoProc.readlines()
        except IOError as e:
            logger.error('getMemoryUsage: exception = %s', e)
            return False
    
    # logger.debug('getMemoryUsage: open success, parsing')
    regexp = _memory_usage_regexp  # get the object once
    
    meminfo = {}
    # Loop through and extract the numerical values
    for line in lines:
        values = line.split(':')
        try:
            # Picks out the key (values[0]) and makes a list with the value as the meminfo value (values[1])
            # We are only interested in the KB data so regexp that out
            match = re.search(regexp, values[1])
            
            if match is not None:
                meminfo[str(values[0])] = int(match.group(0))
        except IndexError:
            break
    
    return meminfo


def mem_usage():
    consumption = get_global_memory_consumption()
    return (consumption['MemTotal'] - consumption['MemFree']) / 1024


def get_cpu_running_procs():
    if os.name == 'nt':
        return -1
    with open("/proc/stat") as stats:
        try:
            line = stats.readline()
            while not line.startswith('procs_running'):
                line = stats.readline()
            nb_running = int(line.split(' ')[-1])
            line = stats.readline()
            # nb_running = procs_running + procs_blocked (the next line)
            nb_running += int(line.split(' ')[-1])
        except EOFError:
            nb_running = -1
        except Exception:
            nb_running = -1
    return nb_running


class WeakMethod(object):
    """A callable object. Takes one argument to init: 'object.method'.
    Once created, call this object -- MyWeakMethod() --
    and pass args/kwargs as you normally would.
    """
    
    
    def __init__(self, object_dot_method):
        self.target = proxy(object_dot_method.__self__)
        self.method = proxy(object_dot_method.__func__)
        ###Older versions of Python can use 'im_self' and 'im_func' in place of '__self__' and '__func__' respectively
    
    
    def __call__(self, *args, **kwargs):
        """Call the method with args and kwargs as needed."""
        return self.method(self.target, *args, **kwargs)


def split_list_by_pack(_list, divider):
    list_splitted = []
    len_of_list = len(_list)
    divided_number = len_of_list / divider
    idx = 0
    for x in xrange(divided_number + 1):
        if x == divided_number:
            list_splitted.append(_list[idx:])
        else:
            new_idx = idx + divider
            list_splitted.append(_list[idx:new_idx])
            idx = new_idx
    return list_splitted


def set_process_name(full_name):
    try:
        from setproctitle import setproctitle
        setproctitle(full_name)
    except Exception as _exp:
        logger.error(u'Cannot set process name as %s: %s' % (full_name, _exp))


# For a !SORTED! list we want to take only a part of it (a windows),
# by a windows page number and a number of windows
def get_window_of_a_sorted_list(sorted_list, window_page_number_from_zero, number_of_windows):
    shard_size = len(sorted_list) // number_of_windows  # beware, can be smaller so shard_size * number_of_windows < len()
    begin = window_page_number_from_zero * shard_size
    end = begin + shard_size
    # For the last element, we should not limit as we can be smaller that the end, but explicitly take all until the end
    if window_page_number_from_zero == number_of_windows - 1:
        end = None
    return sorted_list[begin:end]


# Weight round robin distribution for a list
# Important: as we round robin it, if the list change, a lot of position will change too
#            so do not use this algorithm for the WHOLE distribution list but only a small one
def _get_weighted_indexes(weights_distribution):
    current_weight = 0
    max_weight = max(weights_distribution)
    while current_weight <= max_weight:
        for idx, weight in enumerate(weights_distribution):
            if current_weight >= weight:  # the wiehgt is too small, cannot take more
                continue
            yield idx
        current_weight += 1


# For a sorted list, we want to take a evenly distribution of elements that follow at maximum
# the weights
def _get_rr_distribution_from_short_list(sorted_list, weights_distribution):
    r = {}
    nb_weights = len(weights_distribution)
    for i in xrange(nb_weights):
        r[i] = []
    weight_index_generator = _get_weighted_indexes(weights_distribution)
    for element in sorted_list:
        idx = weight_index_generator.next()
        r[idx].append(element)
    return r


# For a !SORTED! list we want to take only a part of it (a windows),
# with a weight base input
# * the list to get windows from
# * the sum of previous windows weight (so we know which shard we should get)
# * current shard weight
# * sum of all shards weights
# Algorithm: we will split the list into 2 parts (shardsize = int(len / sum_of_weight))
# [                                      |      ]
#    static=shardsize*sum_of_weights       rest
# Rest is the floating part of len()/shardsize, so is lower than shardsize
# and so we will recall the windows computing into this rest part and so the result can have elements like:
# [        XXXX                          |  X   ]  (so rest is a low size version of the static dispatching)
# in the rest part, we will ask for a weight round robin distribution, to respect more the weight
# on small size
# For example for a 10 elements inputs with receiver weights 1 & 2
# 1st call: (prev)0, (current)1, (total)3 => [0, 1, 2, 9]
# 2nd call: (prev)1, (current)2, (total)3 => [3, 4, 5, 6, 7, 8]
# IMPORTANT: before change a single line, ask JEAN for it
def get_window_of_a_sorted_list_with_weights(sorted_list, previous_shards_weight_sum, shard_weight, sum_of_shards_weights, shard_index, shard_weights):
    list_len = len(sorted_list)
    # As the math.floor will set more weigh on the last element, we prefer for smaller internal NOT
    # have this (the last receiver will have ALL small nodes)
    if sum_of_shards_weights >= list_len:
        # IMPORTANT: we can call _get_distribution_from_short_list because we are sure to NOT consume ALL the generator inside
        shard_distribution = _get_rr_distribution_from_short_list(sorted_list, shard_weights)
        return shard_distribution[shard_index]
    else:
        # But for large distribution, let's take an across way
        shard_size = int(math.floor(float(len(sorted_list)) / sum_of_shards_weights))  # beware, can be smaller so shard_size * sum_of_shards_weights < len()
    
    # We will have 2 parts to get:
    # * static part from 0 => shard_size * sum_of_shards_weights
    # * the rest from shard_size * sum_of_shards_weights + 1 => end :: will be smaller than shard_size!
    # and will be dispatch based on shard_weigths
    
    # STATIC
    # We start at the end of the previous shards
    begin = previous_shards_weight_sum * shard_size
    
    # and end at our weight * shard_size
    end = begin + shard_size * shard_weight
    
    static_part = sorted_list[begin:end]
    
    start_of_rest = shard_size * sum_of_shards_weights
    
    # REST (for the float part of len()/shard_size
    rest = sorted_list[start_of_rest:]
    if len(rest) == 0:  # no rest? no float part, so we are already good
        return static_part
    
    # Ask for a windows for this small part, as the part is small, it will return with only a static part as shard_size will be 1
    this_shard_part_of_the_rest = get_window_of_a_sorted_list_with_weights(rest, previous_shards_weight_sum, shard_weight, sum_of_shards_weights, shard_index, shard_weights)
    
    # Give back both part
    return static_part + this_shard_part_of_the_rest


def get_memory_total_and_available_in_mo():
    consumption = get_global_memory_consumption()
    total = consumption['MemTotal'] / 1024
    
    # centos 6 / 7 => MemAvailable is only on newer linux
    if 'MemAvailable' in consumption:
        available = consumption['MemAvailable'] / 1024
    else:
        available = (consumption['MemFree'] + consumption['Cached'] + consumption['SReclaimable']) / 1024
    return total, available


def get_memory_used_percent():
    # On windows we have to go with psutil as /proc is not available (maybe one day?)
    if os.name == 'nt':
        return psutil.virtual_memory().percent / 100.0
    total, available = get_memory_total_and_available_in_mo()
    available_percent = available / float(total)
    used_percent = 1 - available_percent
    return used_percent


def mem_get_total_and_available_and_available_for_us_and_reserved_memory_in_mo(reserved_memory=0):
    total, available = get_memory_total_and_available_in_mo()
    reserved_memory_in_mo = mem_get_reserved_memory(total, reserved_memory=reserved_memory)
    available_for_us = max(0, available - reserved_memory_in_mo)
    return total, available, available_for_us, reserved_memory_in_mo


# We should not go over the 90% memory usage (with a 10% reserved memory), so maybe there will be 0
# available memory
def mem_get_available_memory(reserved_memory=0):
    total, available, available_for_us, reserved_memory_in_mo = mem_get_total_and_available_and_available_for_us_and_reserved_memory_in_mo(reserved_memory=reserved_memory)
    logger.debug('[MEMORY] * Available for daemon:%dMo  =  ( Available on the system:%dMo )  -  ( Reserved Memory:%dMo = [ Total:%dMo / Memory percent reservation:%d%% ] )  ' % (
        available_for_us, available, reserved_memory_in_mo, total, reserved_memory))
    return available_for_us


def mem_get_reserved_memory(total_memory, reserved_memory=0):
    reserved_memory_percent = reserved_memory / 100.0
    reserved_memory = total_memory * reserved_memory_percent if reserved_memory != 0 else 0
    return int(reserved_memory)


# We will fork, so we take our own memory consumption and see if the current available memory
# is enough
def mem_is_a_fork_possible(reserved_memory=0):
    available_memory = mem_get_available_memory(reserved_memory=reserved_memory)
    mymemory_consumption = get_memory_consumption()[0]
    is_fork_possible = available_memory > mymemory_consumption
    logger.debug("[MEMORY]   => Process (%d) memory consumption=%dMo. Fork is possible currently:%s" % (os.getpid(), mymemory_consumption, is_fork_possible))
    return is_fork_possible


def mem_wait_for_fork_possible(process_name, fast_error=False, retry_time=5, reserved_memory=0):
    from .log import logger  # lazy load to avoid circular imports
    start = time.time()
    last_warn = 0.0
    while not mem_is_a_fork_possible(reserved_memory=reserved_memory):
        total, available, available_for_us, reserved_memory_in_mo = mem_get_total_and_available_and_available_for_us_and_reserved_memory_in_mo(reserved_memory=reserved_memory)
        mymemory_consumption = get_memory_consumption()[0]
        err_string = '[MEMORY] Cannot start a new process/worker for "%s" as there is not enough memory to start a new process:  Current process (%d) memory=%dMo > Available for forking:%dMo  =  ( Available on the system:%dMo )  -  ( Reserved Memory:%dMo = [ Total:%dMo / Memory percent reservation:%d%% ] )' % \
                     (process_name, os.getpid(), mymemory_consumption, available_for_us, available, reserved_memory_in_mo, total, reserved_memory)
        if fast_error:  # cannot wait too long, so we have only one turn.
            logger.error('%s. Exiting.' % (err_string))
            return False
        time.sleep(0.1)
        now = time.time()
        if now > start + 1 and now > last_warn + 1:
            logger.warning('%s. Waiting since %ds for memory to be available.' % (err_string, now - start))
            last_warn = now
        if now > start + retry_time:
            logger.error('%s. We did wait %ds. Retry time exceeded. Exiting.' % (err_string, now - start))
            return False
    return True


########### Strings #############
# Try to print strings, but if there is an utf8 error, go in simple ascii mode
# (Like if the terminal do not have en_US.UTF8 as LANG for example)
def safe_print(*args):
    l = []
    for e in args:
        # If we got an str, go in unicode, and if we cannot print
        # utf8, go in ascii mode
        if isinstance(e, str):
            if safe_stdout:
                s = unicode(e, 'utf8', errors='ignore')
            else:
                s = e.decode('ascii', 'replace').encode('ascii', 'replace').decode('ascii', 'replace')
            l.append(s)
        # Same for unicode, but skip the unicode pass
        elif isinstance(e, unicode):
            if safe_stdout:
                s = e
            else:
                s = e.encode('ascii', 'replace')
            l.append(s)
        # Other types can be directly convert in unicode
        else:
            l.append(unicode(e))
    # Ok, now print it :)
    print(u' '.join(l))


def split_semicolon(line, maxsplit=None):
    """Split a line on semicolons characters but not on the escaped semicolons
    """
    # Split on ';' character
    splitted_line = line.split(';')
    
    splitted_line_size = len(splitted_line)
    
    # if maxsplit is not specified, we set it to the number of part
    if maxsplit is None or 0 > maxsplit:
        maxsplit = splitted_line_size
    
    # Join parts  to the next one, if ends with a '\'
    # because we mustn't split if the semicolon is escaped
    i = 0
    while i < splitted_line_size - 1:
        
        # for each part, check if its ends with a '\'
        ends = splitted_line[i].endswith('\\')
        
        if ends:
            # remove the last character '\'
            splitted_line[i] = splitted_line[i][:-1]
        
        # append the next part to the current if it is not the last and the current
        # ends with '\' or if there is more than maxsplit parts
        if (ends or i >= maxsplit) and i < splitted_line_size - 1:
            
            splitted_line[i] = ";".join([splitted_line[i], splitted_line[i + 1]])
            
            # delete the next part
            del splitted_line[i + 1]
            splitted_line_size -= 1
        
        # increase i only if we don't have append because after append the new
        # string can end with '\'
        else:
            i += 1
    
    return splitted_line


# Json-ify the objects
def jsonify_r(obj):
    res = {}
    cls = obj.__class__
    if not hasattr(cls, 'properties'):
        try:
            json.dumps(obj)
            return obj
        except Exception as exp:
            return None
    properties = cls.properties.keys()
    if hasattr(cls, 'running_properties'):
        properties += cls.running_properties.keys()
    for prop in properties:
        if not hasattr(obj, prop):
            continue
        v = getattr(obj, prop)
        # Maybe the property is not jsonable
        try:
            if isinstance(v, set):
                v = list(v)
            json.dumps(v)
            res[prop] = v
        except Exception as exp:
            if isinstance(v, list):
                lst = []
                for _t in v:
                    t = getattr(_t.__class__, 'my_type', '')
                    if t == 'CommandCall':
                        try:
                            lst.append(_t.call)
                        except:
                            pass
                        continue
                    if t and hasattr(_t, t + '_name'):
                        lst.append(getattr(_t, t + '_name'))
                    else:
                        print("CANNOT MANAGE OBJECT", _t, type(_t), t)
                res[prop] = lst
            else:
                t = getattr(v.__class__, 'my_type', '')
                if t == 'CommandCall':
                    try:
                        res[prop] = v.call
                    except:
                        pass
                    continue
                if t and hasattr(v, t + '_name'):
                    res[prop] = getattr(v, t + '_name')
                else:
                    print("CANNOT MANAGE OBJECT", v, type(v), t)
    return res


################################### TIME ##################################
# @memoized
def get_end_of_day(year, month_id, day):
    end_time = (year, month_id, day, 23, 59, 59, 0, 0, -1)
    if None in end_time:
        return None
    
    end_time_epoch = time.mktime(end_time)
    return end_time_epoch


# @memoized
def print_date(t):
    return time.asctime(time.localtime(t))


# @memoized
def get_day(t):
    return int(t - get_sec_from_morning(t))


# Same but for week day
def get_wday(t):
    t_lt = time.localtime(t)
    return t_lt.tm_wday


# @memoized
def get_sec_from_morning(t):
    t_lt = time.localtime(t)
    h = t_lt.tm_hour
    m = t_lt.tm_min
    s = t_lt.tm_sec
    return h * 3600 + m * 60 + s


# @memoized
def get_start_of_day(year, month_id, day):
    start_time = (year, month_id, day, 00, 00, 00, 0, 0, -1)
    if None in start_time:
        return None
    
    try:
        start_time_epoch = time.mktime(start_time)
    except OverflowError:
        # Windows mktime sometimes crashes on (1970, 1, 1, ...)
        start_time_epoch = 0.0
    
    return start_time_epoch


# change a time in seconds like 3600 into a format: 0d 1h 0m 0s
def format_t_into_dhms_format(t):
    s = t
    m, s = divmod(s, 60)
    h, m = divmod(m, 60)
    d, h = divmod(h, 24)
    return '%sd %sh %sm %ss' % (d, h, m, s)


################################# Pythonization ###########################
# first change to float so manage for example 25.0 to 25
def to_int(val):
    return int(float(val))


def to_float(val):
    return float(val)


def to_char(val):
    return val[0]


def __clean_and_uniq(lst):
    new_list = deque()
    for elt in lst:
        stripped_elt = elt.strip()
        if stripped_elt != '' and stripped_elt not in new_list:
            new_list.append(stripped_elt)
    
    return list(new_list)


def to_split(val, split_on_coma=True):
    if isinstance(val, (list, set)):
        return __clean_and_uniq(val)
    if not split_on_coma:
        return __clean_and_uniq([val])
    val = __clean_and_uniq(val.split(','))
    return val


def to_best_int_float(val):
    i = int(float(val))
    f = float(val)
    # If the f is a .0 value,
    # best match is int
    if i == f:
        return i
    return f


# bool('0') = true, so...
def to_bool(val, with_strict_check=False):
    # type: (unicode, bool) -> bool
    if val in ('1', 'on', 'true', 'True', True):
        return True
    else:
        if with_strict_check and val not in ('0', 'off', 'false', 'False', False):
            raise ValueError(u'[%s] is not a boolean value' % val)
        return False


def from_bool_to_string(b):
    if b:
        return '1'
    else:
        return '0'


def from_bool_to_int(b):
    if b:
        return 1
    else:
        return 0


def from_list_to_split(val):
    val = ','.join(['%s' % v for v in val])
    return val


def from_float_to_int(val):
    val = int(val)
    return val


def to_mb_size(size_int):
    return float(size_int) / (1024 * 1024)


### Functions for brok_transformations
### They take 2 parameters: ref, and a value
### ref is the item like a service, and value
### if the value to preprocess

# Just a string list of all names, with ,
def to_list_string_of_names(ref, tab):
    return ",".join([e.get_name() for e in tab])


# Just a list of names
def to_list_of_names(ref, tab):
    return [e.get_name() for e in tab]


# This will give a string if the value exists
# or '' if not
def to_name_if_possible(ref, value):
    if value:
        return value.get_name()
    return ''


# take a list of hosts and return a list
# of all host_names
def get_real_state_id(host, value):
    return host.bp_state if host.got_business_rule else host.state_id


# Sums up all context flags in a boolean mask
def get_context_id(item, value=None):
    return item.in_scheduled_downtime << 5 | item.in_partial_downtime << 4 | item.is_flapping << 3 | item.is_partial_flapping << 2 | item.problem_has_been_acknowledged << 1 | item.is_partial_acknowledged


# take a list of hosts and return a list
# of all host_names
def to_hostnames_list(ref, tab):
    r = []
    for h in tab:
        if hasattr(h, 'host_name'):
            r.append(h.host_name)
    return r


# For raw data, we need to get the list of the cluster sons uuid, and store them in the cluster if we are asking it again
def to_cluster_members_uuid_list_and_do_cache_result(h, _):
    # If nto a cluster, bail out
    if not h.got_business_rule:
        return []
    # maybe it's already computed?
    cache_r = getattr(h, '__cache_cluster_members_uuids', None)
    if cache_r is not None:
        return cache_r
    r = []
    mbers = h.business_rule.list_all_elements()
    for m in mbers:
        r.append((m.uuid, m.my_type))
    # cache the result in our element
    setattr(h, '__cache_cluster_members_uuids', r)
    return r


def to_list_of_host_name_and_host_service(ref, tab):
    r = {'hosts': [], 'services': []}
    if tab:
        for element in tab:
            cls = element.__class__
            name = element.get_dbg_name()
            if cls.my_type == 'service':
                name = name.split('/')
                host_service = (name[0], name[1])
                r['services'].append(host_service)
            else:
                r['hosts'].append(name)
    
    if r == {'hosts': [], 'services': []}:
        r = None
    return r


def object_to_deepcopy(ref, tab):
    new_tab = copy.deepcopy(tab)
    return new_tab


# Will create a dict with 2 lists:
# *services: all services of the tab
# *hosts: all hosts of the tab
def to_svc_hst_distinct_lists(ref, tab):
    r = {'hosts': [], 'services': []}
    if tab:
        for e in tab:
            cls = e.__class__
            if cls.my_type == 'service':
                name = e.get_dbg_name()
                r['services'].append(name)
            else:
                name = e.get_dbg_name()
                r['hosts'].append(name)
    return r


def get_in_scheduled_downtime(ref, value):
    return ref.is_in_downtime()


def get_in_inherited_downtime(ref, value):
    return ref.is_in_inherited_downtime()


def get_problem_has_been_acknowledged(ref, value):
    return ref.is_acknowledged()


def get_in_inherited_acknowledged(ref, value):
    return ref.is_in_inherited_acknowledged()


def get_acknowledge_id(ref, value):
    return ref.get_acknowkledge_id()


def get_active_downtime_uuids(ref, value):
    return ref.get_active_downtime_uuids()


def get_brok_instance_uuid(ref, value):
    return ref.get_instance_uuid()


# Just get the string name of the object
# (like for realm)
def get_obj_name(obj):
    # Maybe we do not have a real object but already a string. If so
    # return the string
    if isinstance(obj, basestring):
        return obj
    return obj.get_name()


# Just get the string name of the object but this one
# can be None => ''
def get_obj_name_possibly_none(obj):
    # Maybe we do not have a real object but already a string. If so
    # return the string
    if obj is None:
        return ''
    return obj.get_name()


# Same as before, but call with object,prop instead of just value
# But if we got an attribute error, return ''
def get_obj_name_two_args_and_void(obj, value):
    try:
        return value.get_name()
    except AttributeError:
        return ''


# Get the full name if there is one
def get_obj_full_name(obj):
    try:
        return obj.get_full_name()
    except Exception:
        return obj.get_name()


# Get the service->host realm
def get_service_realm(obj, value):
    r = obj.get_realm()
    if r is None:
        return ''
    return get_obj_name(r)


# return the list of keys of the custom dict
# but without the _ before
def get_customs_keys(d):
    return [k[1:] for k in d.keys()]


# return the values of the dict
def get_customs_values(d):
    return d.values()


# Checks that a parameter has an unique value. If it's a list, the last
# value set wins.
def unique_value(val):
    if isinstance(val, list):
        if val:
            return val[-1]
        else:
            return ''
    else:
        return val


def make_unicode(change_input):
    change_output = change_input
    if isinstance(change_input, unicode):
        return change_output
    elif hasattr(change_input, 'decode') and type(change_input) != unicode:
        change_output = change_input.decode('utf8', 'ignore')
    elif isinstance(change_input, dict):
        change_output = {}
        for prop, value in change_input.iteritems():
            change_output[make_unicode(prop)] = make_unicode(value)
    elif isinstance(change_input, list):
        change_output = [make_unicode(i) for i in change_input]
    elif isinstance(change_input, tuple):
        change_output = tuple([make_unicode(i) for i in change_input])
    return change_output


###################### Sorting ################
def scheduler_no_spare_first(x, y):
    if x.spare and not y.spare:
        return 1
    elif x.spare and y.spare:
        return 0
    else:
        return -1


# -1 is x first, 0 equal, 1 is y first
def alive_then_spare_then_deads(x, y):
    # First are alive
    if x.alive and not y.alive:
        return -1
    if y.alive and not x.alive:
        return 0
    # if not alive both, I really don't care...
    if not x.alive and not y.alive:
        return -1
    # Ok, both are alive... now spare after no spare
    if not x.spare and y.spare:
        return -1
    # x is a spare, so y must be before, even if
    # y is a spare
    if not y.spare and x.spare:
        return 1
    # They are the same, get by name
    return cmp(x.get_name(), y.get_name())


# -1 is x first, 0 equal, 1 is y first
def sort_by_ids(x, y):
    if x.id < y.id:
        return -1
    if x.id > y.id:
        return 1
    # So is equal
    return 0


# From a tab, get the avg, min, max
# for the tab values, but not the lower ones
# and higher ones that are too distinct
# than major ones
def nighty_five_percent(t):
    t2 = copy.copy(t)
    t2.sort()
    
    l = len(t)
    
    # If void tab, wtf??
    if l == 0:
        return (None, None, None)
    
    t_reduce = t2
    # only take a part if we got more than 100 elements, or it's a non sense
    if l > 100:
        offset = int(l * 0.05)
        t_reduce = t_reduce[offset:-offset]
    
    reduce_len = len(t_reduce)
    reduce_sum = sum(t_reduce)
    
    reduce_avg = float(reduce_sum) / reduce_len
    reduce_max = max(t_reduce)
    reduce_min = min(t_reduce)
    
    return (reduce_avg, reduce_min, reduce_max)


##################### Cleaning ##############
def strip_and_uniq(tab):
    new_list = []
    for elt in tab:
        stripped_elt = elt.strip()
        if stripped_elt != '' and stripped_elt not in new_list:
            new_list.append(stripped_elt)
    
    return new_list


#################### Pattern change application (mainly for host) #######


def expand_xy_pattern(pattern):
    ns = NodeSet(str(pattern))
    if len(ns) > 1:
        for elem in ns:
            for a in expand_xy_pattern(elem):
                yield a
    else:
        yield pattern


# This function is used to generate all pattern change as
# recursive list.
# for example, for a [(1,3),(1,4),(1,5)] xy_couples,
# it will generate a 60 item list with:
# Rule: [1, '[1-5]', [1, '[1-4]', [1, '[1-3]', []]]]
# Rule: [1, '[1-5]', [1, '[1-4]', [2, '[1-3]', []]]]
# ...
def got_generation_rule_pattern_change(xy_couples):
    res = []
    xy_cpl = xy_couples
    if xy_couples == []:
        return []
    (x, y) = xy_cpl[0]
    for i in xrange(x, y + 1):
        n = got_generation_rule_pattern_change(xy_cpl[1:])
        if n != []:
            for e in n:
                res.append([i, '[%d-%d]' % (x, y), e])
        else:
            res.append([i, '[%d-%d]' % (x, y), []])
    return res


# this function apply a recursive pattern change
# generate by the got_generation_rule_pattern_change
# function.
# It take one entry of this list, and apply
# recursively the change to s like:
# s = "Unit [1-3] Port [1-4] Admin [1-5]"
# rule = [1, '[1-5]', [2, '[1-4]', [3, '[1-3]', []]]]
# output = Unit 3 Port 2 Admin 1
def apply_change_recursive_pattern_change(s, rule):
    # print "Try to change %s" % s, 'with', rule
    # new_s = s
    (i, m, t) = rule
    # print "replace %s by %s" % (r'%s' % m, str(i)), 'in', s
    s = s.replace(r'%s' % m, str(i))
    # print "And got", s
    if t == []:
        return s
    return apply_change_recursive_pattern_change(s, t)


def safe_add_to_dict(dictionary, key, value, as_set=False):
    # type: (Dict, unicode, Any, bool) -> None
    if key not in dictionary:
        dictionary[key] = set([value]) if as_set else [value]
    else:
        if as_set:
            dictionary[key].add(value)
        else:
            dictionary[key].append(value)


# For service generator, get dict from a _custom properties
# as _disks   C$(80%!90%),D$(80%!90%)$,E$(80%!90%)$
# return {'C': '80%!90%', 'D': '80%!90%', 'E': '80%!90%'}
# And if we have a key that look like [X-Y] we will expand it
# into Y-X+1 keys
GET_KEY_VALUE_SEQUENCE_ERROR_NOERROR = 0
GET_KEY_VALUE_SEQUENCE_ERROR_SYNTAX = 1
GET_KEY_VALUE_SEQUENCE_ERROR_NODE = 3


def get_key_value_sequence(entry, default_value=None):
    array1 = []
    array2 = []
    conf_entry = entry
    
    # match a key$(value1..n)$
    keyval_pattern_txt = r"""\s*(?P<key>[^,]+?)(?P<values>(\$\(.*?\)\$)*)(?:[,]|$)"""
    keyval_pattern = re.compile('(?x)' + keyval_pattern_txt)
    # match a whole sequence of key$(value1..n)$
    all_keyval_pattern = re.compile('(?x)^(' + keyval_pattern_txt + ')+$')
    # match a single value
    value_pattern = re.compile('(?:\$\((?P<val>.*?)\)\$)')
    # match a sequence of values
    all_value_pattern = re.compile('^(?:\$\(.*?\)\$)+$')
    
    if all_keyval_pattern.match(conf_entry):
        for mat in re.finditer(keyval_pattern, conf_entry):
            r = {'KEY': mat.group('key').strip(), 'VALUE1': None}
            # now fill the empty values with the default value
            if default_value is not None:
                if all_value_pattern.match(default_value):
                    valnum = 1
                    for val in re.finditer(value_pattern, default_value):
                        r['VALUE' + str(valnum)] = val.group('val')
                        valnum += 1
                else:
                    r['VALUE1'] = default_value
            # The key is in mat.group('key')
            # If there are also value(s)...
            if mat.group('values'):
                if all_value_pattern.match(mat.group('values')):
                    # If there are multiple values, loop over them
                    valnum = 1
                    for val in re.finditer(value_pattern, mat.group('values')):
                        r['VALUE' + str(valnum)] = val.group('val')
                        valnum += 1
                else:
                    # Value syntax error
                    return (None, GET_KEY_VALUE_SEQUENCE_ERROR_SYNTAX)
            r['VALUE'] = r['VALUE1']
            array1.append(r)
    else:
        # Something is wrong with the values. (Maybe unbalanced '$(')
        # TODO: count opening and closing brackets in the pattern
        return (None, GET_KEY_VALUE_SEQUENCE_ERROR_SYNTAX)
    
    # Now create new one but for [X-Y] matchs
    #  array1 holds the original entries. Some of the keys may contain wildcards
    #  array2 is filled with originals and inflated wildcards
    
    if NodeSet is None:
        # The pattern that will say if we have a [X-Y] key.
        pat = re.compile('\[(\d*)-(\d*)\]')
    
    for r in array1:
        
        key = r['KEY']
        orig_key = r['KEY']
        
        # We have no choice, we cannot use NodeSet, so we use the
        # simple regexp
        if NodeSet is None:
            m = pat.search(key)
            got_xy = (m is not None)
        else:  # Try to look with a nodeset check directly
            try:
                ns = NodeSet(str(key))
                # If we have more than 1 element, we have a xy thing
                got_xy = (len(ns) != 1)
            except NodeSetParseRangeError:
                return (None, GET_KEY_VALUE_SEQUENCE_ERROR_NODE)
                pass  # go in the next key
        
        # Now we've got our couples of X-Y. If no void,
        # we were with a "key generator"
        
        if got_xy:
            # Ok 2 cases: we have the NodeSet lib or not.
            # if not, we use the dumb algo (quick, but manage less
            # cases like /N or , in patterns)
            if NodeSet is None:  # us the old algo
                still_loop = True
                xy_couples = []  # will get all X-Y couples
                while still_loop:
                    m = pat.search(key)
                    if m is not None:  # we've find one X-Y
                        (x, y) = m.groups()
                        (x, y) = (int(x), int(y))
                        xy_couples.append((x, y))
                        # We must search if we've gotother X-Y, so
                        # we delete this one, and loop
                        _old_key = key
                        key = key.replace('[%d-%d]' % (x, y), 'Z' * 10)
                        if key == _old_key:  # Critical error, abort for now
                            return None, GET_KEY_VALUE_SEQUENCE_ERROR_NODE
                        del _old_key
                    else:  # no more X-Y in it
                        still_loop = False
                
                # Now we have our xy_couples, we can manage them
                
                # We search all pattern change rules
                rules = got_generation_rule_pattern_change(xy_couples)
                
                # Then we apply them all to get ours final keys
                for rule in rules:
                    res = apply_change_recursive_pattern_change(orig_key, rule)
                    new_r = {}
                    for key in r:
                        new_r[key] = r[key]
                    new_r['KEY'] = res
                    array2.append(new_r)
            
            else:
                # The key was just a generator, we can remove it
                # keys_to_del.append(orig_key)
                
                # We search all pattern change rules
                # rules = got_generation_rule_pattern_change(xy_couples)
                nodes_set = expand_xy_pattern(orig_key)
                new_keys = list(nodes_set)
                
                # Then we apply them all to get ours final keys
                for new_key in new_keys:
                    # res = apply_change_recursive_pattern_change(orig_key, rule)
                    new_r = {}
                    for key in r:
                        new_r[key] = r[key]
                    new_r['KEY'] = new_key
                    array2.append(new_r)
        else:
            # There were no wildcards
            array2.append(r)
    # t1 = time.time()
    # print "***********Diff", t1 -t0
    
    return array2, GET_KEY_VALUE_SEQUENCE_ERROR_NOERROR


# ------------ Files management ------------
# We got a file like /tmp/toto/toto2/bob.png And we want to be sure the dir
# /tmp/toto/toto2/ will really exists so we can copy it. Try to make if if need
# and return True/False if succeed
def expect_file_dirs(root, path):
    dirs = os.path.normpath(path).split('/')
    dirs = [d for d in dirs if d != '']
    # We will create all directory until the last one
    # so we are doing a mkdir -p .....
    # TODO: and windows????
    tmp_dir = root
    for d in dirs:
        _d = os.path.join(tmp_dir, d)
        logger.info('Verify the existence of file %s' % (_d))
        if not os.path.exists(_d):
            try:
                os.mkdir(_d)
            except:
                return False
        tmp_dir = _d
    
    return True


class DuplicateForEachStatus:
    NOERROR = 0
    SYNTAX_ERROR = 1
    NODE_ERROR = 3


class DuplicateForEachParser(object):
    SEPARATOR = ','
    START_VALUE = '$('
    END_VALUE = ')$'
    REGEX_NODE = r'\[(\d*)-(\d*)\]'
    
    
    @staticmethod
    def parse(data_value, default_value=None):
        # type: (str, Union[str, None]) -> Tuple[Union[List, None], int]
        if data_value in ('null', '__DEFAULT_NO_TEMPLATE__'):
            return [{'VALUE': None, 'VALUE1': None, 'KEY': data_value}], DuplicateForEachStatus.NOERROR
        
        _indexes_comma = DuplicateForEachParser._get_all_indexes(data_value, DuplicateForEachParser.SEPARATOR)
        _size_comma_indexes = len(_indexes_comma)
        _indexes_openning_values = DuplicateForEachParser._get_all_indexes(data_value, DuplicateForEachParser.START_VALUE)
        _indexes_closing_values = DuplicateForEachParser._get_all_indexes(data_value, DuplicateForEachParser.END_VALUE)
        
        if len(_indexes_openning_values) != len(_indexes_closing_values):
            return None, DuplicateForEachStatus.SYNTAX_ERROR
        
        _, status = DuplicateForEachParser._remove_comma_in_value(_indexes_openning_values, _indexes_closing_values, _indexes_comma)
        if status != DuplicateForEachStatus.NOERROR:
            return None, status
        
        _dfe_elements = DuplicateForEachParser._split_with_indexes(data_value, _indexes_comma)
        
        checks_list, status = DuplicateForEachParser._split_key_and_value(_dfe_elements, default_value)
        
        return checks_list, status
    
    
    @staticmethod
    def unparse(formatted_dfe, default_value=None):
        # type: (List[dict], Union[None, str]) -> str
        final_list = []
        DuplicateForEachParser.get_default_value_to_list(default_value)
        
        for definition in formatted_dfe:
            sorted_values = [v for k, v in sorted(definition.iteritems()) if k.startswith('VALUE') and k != 'VALUE' and v is not None]
            tmp = ['%s' % (definition['KEY'])]
            if sorted_values:
                tmp.append('$(%s)$' % ')$$('.join(sorted_values))
            final_list.append(''.join(tmp))
        return ','.join(final_list)
    
    
    @staticmethod
    def _get_all_indexes(string, key):
        # type: (str, str) -> List
        indexes = []
        last_index = -1
        while True:
            try:
                last_index = string.index(key, last_index + 1)
                indexes.append(last_index)
            except ValueError:
                return indexes
    
    
    @staticmethod
    def _remove_comma_in_value(indexes_openning_values, indexes_closing_values, indexes_comma):
        i = 0
        while i < len(indexes_openning_values):
            _index_openning = indexes_openning_values[i]
            _index_closing = indexes_closing_values[i]
            if i > 0 and _index_openning - indexes_closing_values[i - 1] < 2:
                return None, DuplicateForEachStatus.SYNTAX_ERROR
            
            if _index_openning > _index_closing:
                return None, DuplicateForEachStatus.SYNTAX_ERROR
            
            if (i + 1) < len(indexes_openning_values) and indexes_openning_values[i + 1] < _index_closing:
                return None, DuplicateForEachStatus.SYNTAX_ERROR
            
            for _index_to_remove in [com for com in indexes_comma if _index_openning < com < _index_closing]:
                indexes_comma.remove(_index_to_remove)
            
            i += 1
        
        return None, DuplicateForEachStatus.NOERROR
    
    
    @staticmethod
    def _split_with_indexes(string, indexes, size_split_label=1):
        _to_return = []
        if len(indexes) == 0:
            return [string]
        
        _size_indexes = len(indexes)
        i = 0
        while i <= _size_indexes:
            if i == 0 and indexes[i] == 0:
                pass
            elif i == 0:
                _to_return.append(string[:indexes[i]])
            elif i == _size_indexes:
                _to_return.append(string[indexes[i - 1] + size_split_label:])
            else:
                _to_return.append(string[indexes[i - 1] + size_split_label:indexes[i]])
            i += 1
        
        return [f for f in _to_return if f != '']
    
    
    @staticmethod
    def _split_key_and_value(dfe_elements, default_value):
        # type: (List, str) -> Tuple[Union[List, None], int]
        
        raw_check_list = []
        for flat_dfe in dfe_elements:
            
            _indexes_openning = DuplicateForEachParser._get_all_indexes(flat_dfe, DuplicateForEachParser.START_VALUE)
            
            _splitted_key_and_values = DuplicateForEachParser._split_with_indexes(flat_dfe, _indexes_openning, len(DuplicateForEachParser.START_VALUE))
            _splitted_values = _splitted_key_and_values[1:]
            
            key = _splitted_key_and_values[0].strip()
            
            check = {
                'KEY'   : key,
                'VALUE' : None,
                'VALUE1': None,
            }
            
            _default_values = DuplicateForEachParser.get_default_value_to_list(default_value)
            max_value = max(len(_default_values), len(_splitted_values))
            
            if max_value > 0:
                index = 1
                while index <= max_value:
                    _value = _splitted_values[index - 1] if len(_splitted_values) >= index else _default_values[index - 1]
                    
                    check['VALUE%s' % index] = _value.split(DuplicateForEachParser.END_VALUE)[0]
                    
                    index += 1
                
                if check['VALUE1'] is not None:
                    check['VALUE'] = check['VALUE1']
            
            raw_check_list.append(check)
        
        # Ok we have a list of check, now we check if som keys contains an x-range [1-9]
        return DuplicateForEachParser._split_from_xrange_format(raw_check_list)
    
    
    @staticmethod
    def _split_from_xrange_format(check_list):
        # type: (List) -> Tuple[Union[List, None], int]
        to_return = []
        node_pattern = re.compile(DuplicateForEachParser.REGEX_NODE)
        for check in check_list:
            
            key = check['KEY']
            orig_key = check['KEY']
            
            matched = node_pattern.search(key)
            
            if matched is None:
                to_return.append(check)
            else:
                still_loop = True
                xy_couples = []  # will get all X-Y couples
                while still_loop:
                    matched = node_pattern.search(key)
                    if matched is not None:  # we've find one X-Y
                        (x, y) = matched.groups()
                        (x, y) = (int(x), int(y))
                        xy_couples.append((x, y))
                        # We must search if we've got other X-Y, so
                        # we delete this one, and loop
                        _old_key = key
                        key = key.replace('[%d-%d]' % (x, y), 'Z' * 10)
                        if key == _old_key:  # Critical error, abort for now
                            return None, DuplicateForEachStatus.NODE_ERROR
                        del _old_key
                    else:  # no more X-Y in it
                        still_loop = False
                
                # Now we have our xy_couples, we can manage them
                
                # We search all pattern change rules
                rules = got_generation_rule_pattern_change(xy_couples)
                
                # Then we apply them all to get ours final keys
                for rule in rules:
                    res = apply_change_recursive_pattern_change(orig_key, rule)
                    new_r = {}
                    for key in check:
                        new_r[key] = check[key]
                    new_r['KEY'] = res
                    to_return.append(new_r)
        return to_return, DuplicateForEachStatus.NOERROR
    
    
    @staticmethod
    def get_default_value_to_list(default_value):
        if default_value is None:
            return []
        
        _default_indexes = DuplicateForEachParser._get_all_indexes(default_value, DuplicateForEachParser.START_VALUE)
        return DuplicateForEachParser._split_with_indexes(default_value, _default_indexes, len(DuplicateForEachParser.START_VALUE))


# ------------ For  human display ------------
def to_float_kb(size):
    return size / 1024.0


# ------------ MISC ------------
def transform_str_key_dict_into_int_dict(d):
    # type: (Dict) -> Dict
    new_d = {}
    for (str_k, v) in d.iteritems():
        try:
            int_k = int(str_k)
            new_d[int_k] = v
        except ValueError:
            return None
    return new_d


# A blank item to replace lock
class BlankContext(object):
    def __enter__(self):
        pass
    
    
    def __exit__(self, exc_type, exc_val, exc_tb):
        pass


blank_context = BlankContext()


# Use these functions if you need to make a counter with time.
# Because if you add and subtract float the counter start to drift.
def from_float_sec_to_int_micro_sec(t_in_sec):
    # type: (float) -> int
    return int(t_in_sec * 1000000.0)


# Use these functions if you need to make a counter with time.
# Because if you add and subtract float the counter start to drift.
def from_int_micro_sec_to_float_sec(t_in_us):
    # type: (int) -> float
    return t_in_us / 1000000.0
