#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2021:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.
import json

from .http_client import HTTPExceptions
from .log import logger, get_chapter_string
from .misc.type_hint import TYPE_CHECKING
from .satellitelink import SatelliteLink

if TYPE_CHECKING:
    from .misc.type_hint import Dict
    from .objects.inventory import ElementsInventory


class WithInventorySatelliteLink(SatelliteLink):
    properties = SatelliteLink.properties.copy()
    
    
    def __init__(self, *args, **kwargs):
        super(WithInventorySatelliteLink, self).__init__(*args, **kwargs)
        # We will have the list of realms that this receiver can be contacted with
        self._accessible_realms = []
    
    
    def __getstate__(self):
        _link_as_dict = super(WithInventorySatelliteLink, self).__getstate__()
        _link_as_dict['_accessible_realms'] = self._accessible_realms
        return _link_as_dict
    
    
    def __setstate__(self, state):
        super(WithInventorySatelliteLink, self).__setstate__(state)
        self._accessible_realms = state['_accessible_realms']
    
    
    def get_name(self):
        raise NotImplementedError()
    
    
    # A realm is giving its name to know that this receiver can be contacted by it
    def set_accessible_realm(self, realm_name):
        self._accessible_realms.append(realm_name)
    
    
    # The receiver need to have a new entry in the configuration: 'known_realms'
    # that was prepared by the realms just before
    def prepare_for_conf(self):
        super(WithInventorySatelliteLink, self).prepare_for_conf()
        self.cfg['known_realms'] = self._accessible_realms
    
    
    # For a specific realm, we want to know if the receiver have the inventory up-to-date
    # * ask it for its current inventory hash. If matching, also check that the shard index&number of shards are OK too
    # * if not matching, push the new one (inventory + index + shard_size)
    def assert_have_realm_inventory(self, realm_name, realm_inventory):
        try:
            if self.con is None:
                self.create_connection()
            
            # If the connection failed to initialize, bail out
            if self.con is None:
                return
            
            if not self.is_alive() or not self.reachable:
                return
            
            self.con.get('ping')
        except HTTPExceptions as exp:  # It's just a ping error, not a big dead, will be caught by the real .ping()
            logger.info('%s Failed to check elements inventory. Will retry as soon as connection is back. Current error: %s' % (get_chapter_string(self.get_name()), str(exp)))
            # No other actions, ping will take care of bad pings and DEAD things
            return
        
        inventory_hash = realm_inventory.get_hash()
        args = {
            'realm_name'    : realm_name,
            'inventory_hash': inventory_hash
        }
        _logger = self.logger_configuration.get_sub_part(realm_name)
        url = 'already_have_realm_inventory'
        try:
            already_have = self.con.get(url, args=args)
            if already_have:
                _logger.debug('The %s does already have a valid inventory (%s elements)' % (self.my_type, realm_inventory.get_len()))
            else:
                _logger.info('Starting to send realm inventory to the %s ' % self.my_type)
                payload = {
                    'realm_name'     : realm_name,
                    'realm_inventory': realm_inventory
                }
                url = 'push_realm_inventory'
                self.con.post(url, payload, wait='long')
                _logger.info('The %s is now up-to-date with inventory %s (%s elements)' % (self.my_type, inventory_hash, realm_inventory.get_len()))
                # We did push both inventory and shard_index, it's certain to be OK
        except HTTPExceptions as exp:
            _logger.warning('Cannot check/push elements inventory (url=/%s): %s. We will retry later' % (url, str(exp)))
            # No other actions, ping will take care of bad pings and DEAD things
    
    
    def assert_have_inventories(self, inventories):
        # type: (Dict[unicode, ElementsInventory]) -> None
        _logger = self.logger_dispatch.get_sub_part(u'INVENTORIES')
        
        try:
            if not self.is_alive() or not self.reachable:
                return
            
            if self.con is None:
                self.create_connection()
            
            # If daemon connection fail, bail out
            if self.con is None:
                return
            
            self.con.get(u'ping')
        except HTTPExceptions as exp:  # It's just a ping error, not a big dead, will be caught by the real .ping()
            _logger.info(u'Unable to connect. We will retry later. Cause : %s' % str(exp))
            # No other actions, ping will take care of bad pings and DEAD things
            return
        
        inventories_hash = {}
        for realm_name, inventory in inventories.iteritems():
            inventories_hash[realm_name] = inventory.get_hash()
        
        try:
            str_already_have_inventories = self.con.post(u'already_have_inventories', args={u'inventories_hash': inventories_hash})
            # post query result are not json load
            already_have_inventories = json.loads(str_already_have_inventories)  # type: Dict[unicode,bool]
            _logger.debug(u'''assert_have_inventories inventories up-to-date status:[%s]''' % already_have_inventories)
            to_update_realms = [realm_name for realm_name, already_up_to_date in already_have_inventories.iteritems() if not already_up_to_date]
            if not to_update_realms:
                _logger.debug(u'Inventories are already up-to-date.')
                return
        except HTTPExceptions as exp:
            _logger.warning(u'Failed to check inventories up-to-date status. We will retry later. Cause : %s. ' % str(exp))
            return
        
        try:
            push_inventories = {}
            for to_update_realm in to_update_realms:
                push_inventories[to_update_realm] = inventories[to_update_realm]
            _logger.info(u'''Starting to send realms' inventories to the %s %s''' % (self.my_type, self.get_name()))
            self.con.post(u'push_inventories', {u'push_inventories': push_inventories, u'configuration_incarnation': self.configuration_incarnation}, wait=u'long')
            _logger.info(u'The %s %s has now up-to-date inventories ' % (self.my_type, self.get_name()))
        except HTTPExceptions as exp:
            _logger.warning(u'''Failed to send elements' inventories of %s %s. We will retry later. Cause : %s. ''' % (self.my_type, self.get_name(), str(exp)))
            return
