#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2020:
# This file is part of Shinken Enterprise, all rights reserved.
import inspect
import os

from shinken.modules.base_module.basemodule import BaseModule, SOURCE_STATE
from shinken.ipc.remote_call import is_remote_callable, RemoteCallableObject
from shinken.misc.type_hint import TYPE_CHECKING, cast
from shinken.objects.module import Module as ModuleConfiguration
from shinkensolutions.api.synchronizer import ITEM_TYPE, ComponentManagerSyncui, ComponentManagerSynchronizer, PROPERTIES_USED_AS, ValidationState, SOURCE_PROPERTY_NAME
from shinkensolutions.api.synchronizer.source.api_item_properties import ApiItemProperties
from shinkensolutions.api.synchronizer.source.callback.callback_downtime import CallbackDowntime, RouteTestCallbackDowntimeConnection
from shinkensolutions.api.synchronizer.source.callback.callback_interface import SourceCallbackOnDelete
from shinkensolutions.api.synchronizer.source.compute_sync_keys import SyncKeysManager
from shinkensolutions.api.synchronizer.source.display_origin_item.display_origin_item_format import DisplayOriginItemFormat
from shinkensolutions.api.synchronizer.source.file_loader import FileLoader
from shinkensolutions.api.synchronizer.source.internal_tab.tab_configuration import RouteReloadSourceConfigurationProperties, RouteConfirmConfigurationProperties, TabConfiguration
from shinkensolutions.api.synchronizer.source.item.item_container import ItemsContainers
from shinkensolutions.api.synchronizer.source.mapping.origin_to_source.mapper import MapperOriginToSource, ProxyDisableMapper, GenericDictMapperOriginToSource, ReloadMapperEvent
from shinkensolutions.api.synchronizer.source.mapping.source_to_conf.mapper import MapperSourceToConf
from shinkensolutions.api.synchronizer.source.origin_item_description import OriginItemDescription
from shinkensolutions.api.synchronizer.source.route.route_container import RouteContainer
from shinkensolutions.api.synchronizer.source.rules_application_template import SourceRulesManager, ReloadSourceRulesManagerEvent
from shinkensolutions.api.synchronizer.source.source_configuration_value import SERVICE_MODE, ServiceMode
from shinkensolutions.api.synchronizer.source.source_exception import SourceException
from shinkensolutions.api.synchronizer.source.source_import_progression import SourceImportProgression
from shinkensolutions.api.synchronizer.source.source_setup import SourceSetup
from shinkensolutions.api.synchronizer.source.tab.tab_container import TabContainer
from shinkensolutions.api.synchronizer.source.tab.tab_host_template_binding_rules import RouteReloadTemplateBindingRules
from shinkensolutions.api.synchronizer.source.tab.tab_host_template_binding_rules import TabHostTemplateBindingRules
from shinkensolutions.api.synchronizer.source.tab.tab_mapping_origin_to_source import RouteReloadMappingOriginToSource
from shinkensolutions.api.synchronizer.source.tab.tab_mapping_origin_to_source import TabMappingOriginToSource

if TYPE_CHECKING:
    from shinkensolutions.api.synchronizer import SourceConfiguration, SourceTranslatePart, ItemType
    from shinkensolutions.api.synchronizer.source.tab.tab import AbstractTab
    from shinken.misc.type_hint import NoReturn, Optional, Dict, List, Tuple


class SourceModule(BaseModule, RemoteCallableObject):
    
    def __init__(self, source_configuration, source_setup=None):
        # type: (SourceConfiguration, Optional[SourceSetup]) -> NoReturn
        BaseModule.__init__(self, cast(ModuleConfiguration, source_configuration))
        RemoteCallableObject.__init__(self, source_configuration.get_name())
        self.source_name = source_configuration.get_name()
        self.module_type = source_configuration.module_type
        
        self.source_configuration = source_configuration
        self.route_container = RouteContainer()
        self.tab_container = TabContainer(self.route_container)
        self.type_to_import = []
        self.source_setup = source_setup or SourceSetup()
        self.translator = None  # type: Optional[SourceTranslatePart]
        self.sync_keys_manager = None  # type: Optional[SyncKeysManager]
        
        self.mapper_origin_to_source = None  # type: Optional[MapperOriginToSource]
        
        self._state_of_display_origin_item = u''  # type: ServiceMode
        
        self._state_of_mapping_origin_to_source = u''  # type: ServiceMode
        self._display_origin_item_format = None  # type: Optional[DisplayOriginItemFormat]
        self._mapper_source_to_conf = MapperSourceToConf(self.logger)
        self._reload_mapper = None  # type: Optional[ReloadMapperEvent]
        
        self._load_error_message = u''
        self._on_delete_callbacks = []  # type: List[SourceCallbackOnDelete]
        self._internal_tabs = {}  # type: Dict[unicode, AbstractTab]
        self.source_import_progression = SourceImportProgression()
        self.file_loader = None  # type: Optional[FileLoader]
        
        self._state_of_host_template_binding_rule = u''  # type: ServiceMode
        self.host_template_binding_rule_manager = None  # type: Optional[SourceRulesManager]
        self._reload_source_rules_manager = None  # type: Optional[ReloadSourceRulesManagerEvent]
        
        self._state_of_origin_item_properties_description = u''  # type: ServiceMode
        self.origin_item_description = None  # type: Optional[OriginItemDescription]
        
        self._state_of_api_item_properties = u''  # type: ServiceMode
        self.api_item_properties = None  # type: Optional[ApiItemProperties]
        
        self.tab_configuration = None  # type: Optional[TabConfiguration]
        
        self._source_services = {}
    
    
    def get_remote_instance_id(self):
        return self.source_name
    
    
    def callback_on_delete(self, item, item_type):
        # type: (Dict, ItemType) -> NoReturn
        for on_delete_callback in self._on_delete_callbacks:
            on_delete_callback.callback_on_delete(item, item_type)
    
    
    def get_source_import_progression(self):
        return self.source_import_progression
    
    
    def reload(self, source_configuration):
        self.logger.info(u'reloading source')
        
        self.module_type = source_configuration.module_type
        self.source_configuration = source_configuration
        
        self.type_to_import = getattr(self.source_configuration, SOURCE_PROPERTY_NAME.TYPE_TO_IMPORT, [])
        if not self.type_to_import and self.source_setup.private_type_to_import:
            self.type_to_import = self.source_setup.private_type_to_import
        self.translator = ComponentManagerSynchronizer.get_translate_component().source_translator(self.module_type)
        self.file_loader = FileLoader(self.module_type, self.source_name)
        
        self._reload_api_item_properties_filter()
        self._reload_origin_item_description()
        self._reload_sync_keys_manager()
        self._reload_host_template_binding_rule()
        self._reload_mapping_origin_to_source()
        self._reload_display_origin_item()
        self._reload_internal_tabs()
        self._reload_reload_route_configuration()
        self._reload_callbacks()
        self._reload_test_callback_downtime_connection()
        
        self.load_configuration(self.source_configuration)
    
    
    def load(self, _):
        try:
            self.logger.info('loading source')
            
            self.type_to_import = getattr(self.source_configuration, SOURCE_PROPERTY_NAME.TYPE_TO_IMPORT, [])
            if not self.type_to_import and self.source_setup.private_type_to_import:
                self.type_to_import = self.source_setup.private_type_to_import
            self.translator = ComponentManagerSynchronizer.get_translate_component().source_translator(self.module_type)
            self.file_loader = FileLoader(self.module_type, self.source_name)
            
            self._load_api_item_properties_filter()
            self._load_origin_item_description()
            self._load_sync_keys_manager()
            self._load_host_template_binding_rule()
            self._load_mapping_origin_to_source()
            self._load_display_origin_item()
            self._load_internal_tabs()
            self._load_reload_route_configuration()
            self._load_callbacks()
            self._load_test_callback_downtime_connection()
            
            self.load_configuration(self.source_configuration)
        except Exception as e:
            fail_to_load_source_message = u'Fail to load source with error : [%s]' % e
            self.logger.error(fail_to_load_source_message)
            self.logger.print_stack()
            self._load_error_message = fail_to_load_source_message
    
    
    def _reload_display_origin_item(self):
        self._state_of_display_origin_item = self.source_setup.private_state_of_display_origin_item
        self._display_origin_item_format = self.source_setup.private_display_origin_item_format
        if SERVICE_MODE.is_enable(self._state_of_display_origin_item) and self._display_origin_item_format:
            self._display_origin_item_format.load(self)
    
    
    def _load_display_origin_item(self):
        self._reload_display_origin_item()
    
    
    def _reload_mapping_origin_to_source(self):
        self._state_of_mapping_origin_to_source = self.source_setup.private_state_of_mapping_origin_to_source
        if SERVICE_MODE.is_enable(self._state_of_mapping_origin_to_source):
            self.mapper_origin_to_source.reload_mapper()
    
    
    def _load_mapping_origin_to_source(self):
        self._state_of_mapping_origin_to_source = self.source_setup.private_state_of_mapping_origin_to_source
        
        if SERVICE_MODE.is_enable(self._state_of_mapping_origin_to_source):
            rule_component = ComponentManagerSynchronizer.get_rule_component()
            _origin_to_source_type_mapper = self.source_setup.private_origin_to_source_type_mapper or GenericDictMapperOriginToSource
            _origin_to_source_mappers_name = list(self.source_setup.private_origin_to_source_mappers_name)
            
            self.mapper_origin_to_source = MapperOriginToSource(
                self.logger,
                self.translator,
                rule_component,
                self.file_loader,
                self._state_of_mapping_origin_to_source,
                _origin_to_source_type_mapper,
                self.origin_item_description,
                self.api_item_properties,
                _origin_to_source_mappers_name
            )
            
            self.tab_container.add(u'tab_mapping_origin_to_source', TabMappingOriginToSource(self.logger, self.translator, self.source_name, self.mapper_origin_to_source, self.type_to_import))
            self._reload_mapper = ReloadMapperEvent(self.mapper_origin_to_source)
            self.route_container.add(u'route_reload_mapping_origin_to_source', RouteReloadMappingOriginToSource(self.logger, self.source_name, self.mapper_origin_to_source, self._reload_mapper))
        else:
            self.mapper_origin_to_source = ProxyDisableMapper()
    
    
    def _reload_host_template_binding_rule(self):
        self.host_template_binding_rule_manager.reload_rule()
    
    
    def _load_host_template_binding_rule(self):
        self._state_of_host_template_binding_rule = self.source_setup.private_state_of_host_template_binding_rule
        if SERVICE_MODE.is_enable(self._state_of_host_template_binding_rule):
            self.host_template_binding_rule_manager = SourceRulesManager(
                self.logger,
                self.file_loader,
                self._state_of_host_template_binding_rule,
                self.translator,
                self.api_item_properties
            )
            
            tab_host_template_binding_rules = TabHostTemplateBindingRules(
                self.logger,
                self.translator,
                self.source_name,
                self.host_template_binding_rule_manager,
            )
            self.tab_container.add(u'tab_host_template_binding_rules', tab_host_template_binding_rules)
            
            self._reload_source_rules_manager = ReloadSourceRulesManagerEvent(self.host_template_binding_rule_manager)
            route_reload_template_binding_rules = RouteReloadTemplateBindingRules(
                self.logger,
                self.source_name,
                self.host_template_binding_rule_manager,
                self._reload_source_rules_manager,
            )
            self.route_container.add(u'route_reload_template_binding_rules', route_reload_template_binding_rules)
    
    
    def _reload_sync_keys_manager(self):
        properties_used_as_synckey = {}
        for item_type in ITEM_TYPE.ALL_TYPES:
            properties_used_as_synckey[item_type] = getattr(self.source_configuration, u'%s%s' % (PROPERTIES_USED_AS, item_type), [])
        self.sync_keys_manager = SyncKeysManager(self.logger, properties_used_as_synckey)
    
    
    def _load_sync_keys_manager(self):
        self._reload_sync_keys_manager()
    
    
    def _reload_origin_item_description(self):
        self._state_of_origin_item_properties_description = self.source_setup.private_state_of_origin_item_properties_description
        origin_item_description_class = self.source_setup.private_origin_item_description_class or OriginItemDescription
        self.origin_item_description = origin_item_description_class(
            self.logger,
            self.translator,
            self.file_loader,
            ComponentManagerSynchronizer.get_configuration_component().lang,
            self._state_of_origin_item_properties_description
        )
    
    
    def _load_origin_item_description(self):
        self._reload_origin_item_description()
    
    
    def _reload_api_item_properties_filter(self):
        self._state_of_api_item_properties = self.source_setup.private_state_of_api_item_properties
        api_item_properties_class = self.source_setup.private_api_item_properties_class or ApiItemProperties
        self.api_item_properties = api_item_properties_class(self.logger, self.translator, self.file_loader, self._state_of_api_item_properties)
    
    
    def _load_api_item_properties_filter(self):
        self._reload_api_item_properties_filter()
    
    
    def _reload_reload_route_configuration(self):
        pass
    
    
    def _load_reload_route_configuration(self):
        route_reload_source_configuration_properties = RouteReloadSourceConfigurationProperties(self.logger, self.source_name, self.tab_configuration, self.host_template_binding_rule_manager, self.mapper_origin_to_source)
        self.route_container.add(u'route_reload_source_configuration_properties', route_reload_source_configuration_properties)
        route_confirm_configuration_properties = RouteConfirmConfigurationProperties(self.logger, self.source_name)
        self.route_container.add(u'route_confirm_configuration_properties', route_confirm_configuration_properties)
    
    
    def _reload_test_callback_downtime_connection(self):
        pass
    
    
    def _load_test_callback_downtime_connection(self):
        callback_downtime = self.get_callback_downtime()
        if callback_downtime:
            route_test_callback_downtime_connection = RouteTestCallbackDowntimeConnection(self.logger, self.source_name, callback_downtime)
            self.route_container.add(u'route_test_callback_downtime_connection', route_test_callback_downtime_connection)
    
    
    def _reload_internal_tabs(self):
        pass
    
    
    def _load_internal_tabs(self):
        tab_configuration = TabConfiguration(self.logger, self.translator, self.source_name)
        self._internal_tabs[tab_configuration.name] = tab_configuration
        self.tab_configuration = tab_configuration
    
    
    def _reload_callbacks(self):
        self._on_delete_callbacks = self.source_setup.private_on_delete_callbacks
        for callback in self._on_delete_callbacks:
            callback.init_callback(self.source_configuration, self.logger, self.source_name, self.translator)
    
    
    def _load_callbacks(self):
        self._reload_callbacks()
    
    
    def get_callback_downtime(self):
        # type: () -> Optional[CallbackDowntime]
        return next((c for c in self._on_delete_callbacks if isinstance(c, CallbackDowntime)), None)
    
    
    # Call by Synchronizer after fork. Use for init source thread.
    def source_start(self, source_is_enabled):
        if self._reload_mapper:
            self._reload_mapper.start_thread()
        if self._reload_source_rules_manager:
            self._reload_source_rules_manager.start_thread()
    
    
    def build_items_containers(self):
        # type: () -> ItemsContainers
        containers = ItemsContainers()
        containers.set_sync_key_service(self.sync_keys_manager)
        if SERVICE_MODE.is_enable(self._state_of_display_origin_item) and self._display_origin_item_format:
            containers.set_display_origin_item_format(self._display_origin_item_format)
        if self._state_of_host_template_binding_rule in (SERVICE_MODE.ON, SERVICE_MODE.NOT_OVERLOAD_BY_USER):
            containers.set_source_rules_manager(self.host_template_binding_rule_manager)
        
        return containers
    
    
    def get_display_origin_item_format(self):
        return self._display_origin_item_format
    
    
    def load_configuration(self, configuration):
        raise NotImplementedError()
    
    
    def do_loop_turn(self):
        pass
    
    
    def loop_turn(self):
        pass
    
    
    @staticmethod
    def prepare_error_exit(message, level=SOURCE_STATE.CRITICAL):
        result = {
            u'output'  : message,
            u'state'   : level,
            u'objects' : {},
            u'errors'  : [message] if level == SOURCE_STATE.CRITICAL else [],
            u'warnings': [message] if level == SOURCE_STATE.WARNING else [],
        }
        return result
    
    
    def get_objects(self):
        ret = self._get_objects()
        return ret
    
    
    def _get_objects(self):
        errors = []
        
        if self.source_configuration.validation_state.has_error():
            errors.extend(self.source_configuration.validation_state.get_errors())
        
        if SERVICE_MODE.is_enable(self._state_of_mapping_origin_to_source) and self.mapper_origin_to_source.validation_state.has_error():
            errors.extend(self.mapper_origin_to_source.validation_state.get_errors())
        
        if SERVICE_MODE.is_enable(self._state_of_api_item_properties) and self.api_item_properties.validation_state.has_error():
            errors.extend(self.api_item_properties.validation_state.get_errors())
        
        if SERVICE_MODE.is_enable(self._state_of_origin_item_properties_description) and self.origin_item_description.validation_state.has_error():
            errors.extend(self.origin_item_description.validation_state.get_errors())
        
        if SERVICE_MODE.is_enable(self._state_of_host_template_binding_rule) and self.host_template_binding_rule_manager.validation_state.has_error():
            errors.extend(self.host_template_binding_rule_manager.validation_state.get_errors())
        
        if self._load_error_message:
            errors.append(self._load_error_message)
        
        if errors:
            result = self.prepare_error_exit(u',<br>'.join(errors))
            return result
        try:
            return self.import_source_items()
        except SourceException as e:
            return self.prepare_error_exit(e.message, e.level)
    
    
    def get_validation_state(self):
        # type: () -> ValidationState
        validation_state = ValidationState()
        
        validation_state.update_from_validation_state(self.source_configuration.validation_state)
        
        if SERVICE_MODE.is_enable(self._state_of_mapping_origin_to_source):
            validation_state.update_from_validation_state(self.mapper_origin_to_source.validation_state)
        
        if SERVICE_MODE.is_enable(self._state_of_api_item_properties):
            validation_state.update_from_validation_state(self.api_item_properties.validation_state)
        
        if SERVICE_MODE.is_enable(self._state_of_origin_item_properties_description):
            validation_state.update_from_validation_state(self.origin_item_description.validation_state)
        
        if SERVICE_MODE.is_enable(self._state_of_host_template_binding_rule):
            validation_state.update_from_validation_state(self.host_template_binding_rule_manager.validation_state)
        
        return validation_state
    
    
    def import_source_items(self):
        raise NotImplementedError()
    
    
    def get_translation_js(self):
        return_js_import = [
            u'<script type="text/javascript" src="/source/translation/static/%d/en.js"></script>' % ComponentManagerSyncui.get_configuration_component().http_start_time,
            u'<script type="text/javascript" src="/source/translation/static/%d/fr.js"></script>' % ComponentManagerSyncui.get_configuration_component().http_start_time
        ]
        
        module_folder = os.path.dirname(inspect.getfile(self.__class__))
        translation_file_en = os.path.join(module_folder, u'translation', u'en.js')
        translation_file_fr = os.path.join(module_folder, u'translation', u'fr.js')
        
        if os.path.exists(translation_file_en):
            return_js_import.append(u'<script type="text/javascript" src="/source/translation/%s/%d/en.js"></script>' % (self.module_type, ComponentManagerSyncui.get_configuration_component().http_start_time))
        if os.path.exists(translation_file_fr):
            return_js_import.append(u'<script type="text/javascript" src="/source/translation/%s/%d/fr.js"></script>' % (self.module_type, ComponentManagerSyncui.get_configuration_component().http_start_time))
        
        return '\n'.join(return_js_import)
    
    
    def get_configuration_fields(self):
        pass
    
    
    @is_remote_callable
    def get_confirmation_configuration_html(self):
        # type: () -> unicode
        return u'''<div class="shinken-confirm-message-container">%s</div>''' % self.translator.translate(u'source.default_confirm_message')
    
    
    def need_confirmation(self):
        # type: () -> Tuple[bool, unicode]
        need_confirmation = self.source_configuration.need_confirmation()
        message = u'%s : %s' % (self.translator.translate(u'tab_configuration.confirm_changes.button_label'), u','.join(self.source_configuration.get_properties_confirmation_needed()))
        return need_confirmation, message
    
    
    def confirm_configuration(self):
        # type: () -> NoReturn
        self.source_configuration.confirm_configuration()
