#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2020:
# This file is part of Shinken Enterprise, all rights reserved.
import base64
import httplib
import socket
import ssl
import time
import urllib

from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.api.synchronizer import ITEM_TYPE, DEF_ITEMS, SourcePropertyDefinition, SourcePropertyDefinitionBind, PYTHON_TYPE, PROPERTY_TYPE, HTML_CATEGORY, COMPUTE_DELETED_ELEMENT_NOT_IN_SOURCE, PUT_IN, ComponentManagerSyncui, \
    DaemonIsRequestedToStopException
from shinkensolutions.api.synchronizer.source.callback.callback_interface import SourceCallbackOnDelete
from shinkensolutions.api.synchronizer.source.route.route import AbstractRoute

if TYPE_CHECKING:
    from shinken.misc.type_hint import NoReturn, Union, Dict, List
    from shinken.log import PartLogger
    from shinkensolutions.api.synchronizer import SourceConfiguration, SourceTranslatePart
    from shinkensolutions.api.synchronizer.source.abstract_module.source_module import SourceModule

DOWNTIME_PROPERTIES_DEFINITIONS = [
    SourcePropertyDefinition(u'downtime_on_delete_element_activated', default_value=u'0', python_type=PYTHON_TYPE.BOOLEAN, save_in_last_execution=True, html_type=PROPERTY_TYPE.CHECKBOX, html_category=HTML_CATEGORY.DOWNTIME_ON_DELETE, order=1,
                             linked_definition=SourcePropertyDefinitionBind([u'put_in', u'compute_deleted_element_not_in_source'], inactive_value=[PUT_IN.SOURCE_SPACE, COMPUTE_DELETED_ELEMENT_NOT_IN_SOURCE.DISABLED], show_help=[u'1', u'0']),
                             tabulated=1),
    SourcePropertyDefinition(u'downtime_on_delete_element_duration', default_value=u'1440', python_type=PYTHON_TYPE.INTEGER, html_category=HTML_CATEGORY.DOWNTIME_ON_DELETE, order=2,
                             linked_definition=SourcePropertyDefinitionBind([u'put_in', u'compute_deleted_element_not_in_source', u'downtime_on_delete_element_activated'],
                                                                            inactive_value=[PUT_IN.SOURCE_SPACE, COMPUTE_DELETED_ELEMENT_NOT_IN_SOURCE.DISABLED, u'0'], show_help=[u'1', u'0', u'0']), tabulated=2),
    SourcePropertyDefinition(u'downtime_on_delete_element_comment', default_value=u'', python_type=PYTHON_TYPE.STRING, html_type=PROPERTY_TYPE.TEXTAREA, html_category=HTML_CATEGORY.DOWNTIME_ON_DELETE, order=3,
                             linked_definition=SourcePropertyDefinitionBind([u'put_in', u'compute_deleted_element_not_in_source', u'downtime_on_delete_element_activated'],
                                                                            inactive_value=[PUT_IN.SOURCE_SPACE, COMPUTE_DELETED_ELEMENT_NOT_IN_SOURCE.DISABLED, u'0'], show_help=[u'1', u'0', u'0']), tabulated=2),
    SourcePropertyDefinition(u'downtime_on_delete_element_author', default_value=u'', python_type=PYTHON_TYPE.USER, html_category=HTML_CATEGORY.DOWNTIME_ON_DELETE, order=4,
                             linked_definition=SourcePropertyDefinitionBind([u'put_in', u'compute_deleted_element_not_in_source', u'downtime_on_delete_element_activated'],
                                                                            inactive_value=[PUT_IN.SOURCE_SPACE, COMPUTE_DELETED_ELEMENT_NOT_IN_SOURCE.DISABLED, u'0'], show_help=[u'1', u'0', u'0']), tabulated=2),
    SourcePropertyDefinition(u'downtime_on_delete_element_api_url', default_value=u'127.0.0.1', python_type=PYTHON_TYPE.STRING, html_category=HTML_CATEGORY.DOWNTIME_ON_DELETE, order=5,
                             linked_definition=SourcePropertyDefinitionBind([u'put_in', u'compute_deleted_element_not_in_source', u'downtime_on_delete_element_activated'],
                                                                            inactive_value=[PUT_IN.SOURCE_SPACE, COMPUTE_DELETED_ELEMENT_NOT_IN_SOURCE.DISABLED, u'0'], show_help=[u'1', u'0', u'0']), tabulated=2),
    SourcePropertyDefinition(u'downtime_on_delete_element_api_port', default_value=u'7760', python_type=PYTHON_TYPE.STRING, html_category=HTML_CATEGORY.DOWNTIME_ON_DELETE, order=6,
                             linked_definition=SourcePropertyDefinitionBind([u'put_in', u'compute_deleted_element_not_in_source', u'downtime_on_delete_element_activated'],
                                                                            inactive_value=[PUT_IN.SOURCE_SPACE, COMPUTE_DELETED_ELEMENT_NOT_IN_SOURCE.DISABLED, u'0'], show_help=[u'1', u'0', u'0']), tabulated=2),
    SourcePropertyDefinition(u'downtime_on_delete_element_api_use_ssl', default_value=u'0', python_type=PYTHON_TYPE.BOOLEAN, html_type=PROPERTY_TYPE.CHECKBOX, html_category=HTML_CATEGORY.DOWNTIME_ON_DELETE, order=7,
                             linked_definition=SourcePropertyDefinitionBind([u'put_in', u'compute_deleted_element_not_in_source', u'downtime_on_delete_element_activated'],
                                                                            inactive_value=[PUT_IN.SOURCE_SPACE, COMPUTE_DELETED_ELEMENT_NOT_IN_SOURCE.DISABLED, u'0'], show_help=[u'1', u'0', u'0']), tabulated=2),
    SourcePropertyDefinition(u'downtime_on_delete_element_api_user', default_value=u'admin', python_type=PYTHON_TYPE.STRING, html_category=HTML_CATEGORY.DOWNTIME_ON_DELETE, order=8,
                             linked_definition=SourcePropertyDefinitionBind([u'put_in', u'compute_deleted_element_not_in_source', u'downtime_on_delete_element_activated'],
                                                                            inactive_value=[PUT_IN.SOURCE_SPACE, COMPUTE_DELETED_ELEMENT_NOT_IN_SOURCE.DISABLED, u'0'], show_help=[u'1', u'0', u'0']), tabulated=2),
    SourcePropertyDefinition(u'downtime_on_delete_element_api_password', default_value=u'admin', python_type=PYTHON_TYPE.STRING, html_type=PROPERTY_TYPE.INPUT_PASSWORD, html_category=HTML_CATEGORY.DOWNTIME_ON_DELETE, order=9,
                             linked_definition=SourcePropertyDefinitionBind([u'put_in', u'compute_deleted_element_not_in_source', u'downtime_on_delete_element_activated'],
                                                                            inactive_value=[PUT_IN.SOURCE_SPACE, COMPUTE_DELETED_ELEMENT_NOT_IN_SOURCE.DISABLED, u'0'], show_help=[u'1', u'0', u'0']), tabulated=2)
]  # type: List[SourcePropertyDefinition]


class CallbackDowntime(SourceCallbackOnDelete):
    
    def __init__(self):
        super(CallbackDowntime, self).__init__()
        self.downtime_on_delete_element_activated = False
        self.downtime_on_delete_element_duration = u''
        self.downtime_on_delete_element_comment = u''
        self.downtime_on_delete_element_author = u''
        self.downtime_on_delete_element_api_url = u''
        self.downtime_on_delete_element_api_port = u''
        self.downtime_on_delete_element_api_use_ssl = False
        
        self.downtime_on_delete_element_api_user = u''
        self.downtime_on_delete_element_api_password = u''
        self._auth_string = u''
        self.PROPERTIES_DEFINITIONS = DOWNTIME_PROPERTIES_DEFINITIONS
    
    
    def init_callback(self, conf, logger, source_name, translator):
        # type: (SourceConfiguration, PartLogger, unicode, SourceTranslatePart) -> NoReturn
        super(CallbackDowntime, self).init_callback(conf, logger, source_name, translator)
        self.logger = logger.get_sub_part('CallbackDowntime')
        
        self.downtime_on_delete_element_activated = getattr(conf, u'downtime_on_delete_element_activated')
        self.downtime_on_delete_element_duration = getattr(conf, u'downtime_on_delete_element_duration')
        self.downtime_on_delete_element_comment = getattr(conf, u'downtime_on_delete_element_comment')
        self.downtime_on_delete_element_author = getattr(conf, u'downtime_on_delete_element_author')
        self.downtime_on_delete_element_api_url = getattr(conf, u'downtime_on_delete_element_api_url')
        self.downtime_on_delete_element_api_port = getattr(conf, u'downtime_on_delete_element_api_port')
        self.downtime_on_delete_element_api_use_ssl = getattr(conf, u'downtime_on_delete_element_api_use_ssl')
        
        self.downtime_on_delete_element_api_user = getattr(conf, u'downtime_on_delete_element_api_user')
        self.downtime_on_delete_element_api_password = getattr(conf, u'downtime_on_delete_element_api_password')
        self._auth_string = base64.encodestring('%s:%s' % (self.downtime_on_delete_element_api_user.encode(u'utf-8'), self.downtime_on_delete_element_api_password.encode(u'utf-8'))).replace(u'\n', u'')
    
    
    def callback_on_delete(self, item, item_type):
        # type: (Union[SourceModule,CallbackDowntime], Dict, unicode) -> NoReturn
        
        if not self.downtime_on_delete_element_activated:
            return
        
        if item_type not in ITEM_TYPE.ALL_HOST_CLASS:
            self.logger.debug(u'Canceling CallbackDowntime, the item type of %s is not HOST !' % item[DEF_ITEMS[item_type].get(u'key_name')])
            return
        
        # Be sure that the item_name is in unicode (maybe the source give us str)
        item_name = item[u'host_name']
        if isinstance(item_name, str):
            item_name = item_name.decode(u'utf-8', u'ignore')
        
        receiver_con = self.get_receiver_syncui_connection((u'%s:%s' % (self.downtime_on_delete_element_api_url, self.downtime_on_delete_element_api_port)), self.downtime_on_delete_element_api_use_ssl)
        
        downtime_start = u'%s' % str(int(time.time()))
        downtime_end = u'%s' % str(int(time.time() + int(self.downtime_on_delete_element_duration) * 60))
        
        self.logger.info(
            u'[DOWNTIME] Creating a downtime on host %s, by author %s, ending in %sm to receiver api %s:%s.' % (
                item_name, self.downtime_on_delete_element_author, self.downtime_on_delete_element_duration, self.downtime_on_delete_element_api_url, self.downtime_on_delete_element_api_port))
        
        params = urllib.urlencode({
            u'host_name' : item_name.encode(u'utf8'),
            u'start_time': downtime_start,
            u'end_time'  : downtime_end,
            u'author'    : self.downtime_on_delete_element_author.encode(u'utf8'),  # NOTE: url encode MUST be str if with utf8
            u'comment'   : self.downtime_on_delete_element_comment.encode(u'utf8'),
        })
        
        headers = {
            u'Content-type' : u'application/x-www-form-urlencoded',
            u'Accept'       : u'text/plain',
            u'Authorization': u'Basic %s' % self._auth_string
        }
        
        try:
            receiver_con.request(u'POST', u'/downtime', params, headers)
            resp = receiver_con.getresponse()
            if resp.status != 200 and resp.reason != u'OK':
                msg = u'Callback error: Status=%d, reason=%s' % (resp.status, resp.reason)
                self.logger.warning(msg)
                raise Exception(msg)
        except socket.error:
            msg = u'No connection could be established to %s:%s because the target computer expressly refused it' % (self.downtime_on_delete_element_api_url, self.downtime_on_delete_element_api_port)
            self.logger.warning(msg)
            raise Exception(msg)
    
    
    @staticmethod
    def get_receiver_syncui_connection(url, http_use_ssl=False):
        if http_use_ssl:
            # If we are in SSL mode, do not look at certificate too much
            # NOTE: ssl.SSLContext is only available on last python 2.7 versions
            if hasattr(ssl, u'SSLContext'):
                ssl_context = ssl.SSLContext(ssl.PROTOCOL_TLSv1)
                ssl_context.check_hostname = False
                ssl_context.verify_mode = ssl.CERT_NONE
            else:
                ssl_context = None
            
            args = {}
            if ssl_context:
                args[u'context'] = ssl_context
            conn = httplib.HTTPSConnection(url, **args)
        else:
            conn = httplib.HTTPConnection(url)
        return conn
    
    
    def test_connection(self, import_in_progress):
        # type: (bool) -> Dict[unicode,unicode]
        
        if import_in_progress:
            _return = {
                u'state' : RouteTestCallbackDowntimeConnection.CONNECTION_STATE.KO,
                u'output': self.translator.translate(u'callback_downtime.test_connection.source_is_running'),
            }
        
        if not self.downtime_on_delete_element_activated:
            return {
                u'state' : RouteTestCallbackDowntimeConnection.CONNECTION_STATE.KO,
                u'output': self.translator.translate(u'callback_downtime.test_connection.callback_is_disable'),
            }
        
        receiver_url = (u'%s:%s' % (self.downtime_on_delete_element_api_url, self.downtime_on_delete_element_api_port))
        receiver_con = self.get_receiver_syncui_connection(receiver_url, self.downtime_on_delete_element_api_use_ssl)
        
        params = u''
        headers = {
            u'Content-type' : u'application/x-www-form-urlencoded',
            u'Accept'       : u'text/plain',
            u'Authorization': u'Basic %s' % self._auth_string
        }
        
        try:
            receiver_con.request(u'GET', u'/ping', params, headers)
            resp = receiver_con.getresponse()
            if resp.status == 401:
                return {
                    u'state' : RouteTestCallbackDowntimeConnection.CONNECTION_STATE.KO,
                    u'output': self.translator.translate(u'callback_downtime.test_connection.connection_refused_missing_password') % receiver_url
                }
            elif resp.status == 403:
                return {
                    u'state' : RouteTestCallbackDowntimeConnection.CONNECTION_STATE.KO,
                    u'output': self.translator.translate(u'callback_downtime.test_connection.connection_refused_incorrect_password') % receiver_url
                }
            elif resp.status != 200:
                return {
                    u'state' : RouteTestCallbackDowntimeConnection.CONNECTION_STATE.KO,
                    u'output': self.translator.translate(u'callback_downtime.test_connection.unexpected_return') % receiver_url
                }
            else:
                _read = resp.read()
                if _read != u'pong:ws arbiter':
                    return {
                        u'state' : RouteTestCallbackDowntimeConnection.CONNECTION_STATE.KO,
                        u'output': self.translator.translate(u'callback_downtime.test_connection.unexpected_return') % receiver_url
                    }
        except socket.error as e:
            return {
                u'state' : RouteTestCallbackDowntimeConnection.CONNECTION_STATE.KO,
                u'output': self.translator.translate(u'callback_downtime.test_connection.connection_refused') % (receiver_url, e.strerror)
            }
        except Exception as e:
            return {
                u'state' : RouteTestCallbackDowntimeConnection.CONNECTION_STATE.KO,
                u'output': self.translator.translate(u'callback_downtime.test_connection.connection_refused') % (receiver_url, str(e))
            }
        
        _return = {
            u'state' : RouteTestCallbackDowntimeConnection.CONNECTION_STATE.OK,
            u'output': self.translator.translate(u'callback_downtime.test_connection.test_is_ok') % (u'%s:%s' % (self.downtime_on_delete_element_api_url, self.downtime_on_delete_element_api_port))
        }
        return _return


class RouteTestCallbackDowntimeConnection(AbstractRoute):
    class CONNECTION_STATE(object):
        OK = u'ok'
        KO = u'ko'
    
    def __init__(self, logger, source_name, callback_downtime):
        # type: (PartLogger, unicode, CallbackDowntime) -> None
        super(RouteTestCallbackDowntimeConnection, self).__init__(logger, source_name, u'callback_downtime/test_connection')
        self.callback_downtime = callback_downtime
    
    
    def controller(self):
        # type: () -> Dict[unicode,unicode]
        _source_info = ComponentManagerSyncui.get_source_info_component().get_source_info(self.source_name)
        
        try:
            _return = ComponentManagerSyncui.get_internal_api_synchronizer_component().test_callback_downtime_connection(self.source_name, _source_info.is_import_in_progress())
        except DaemonIsRequestedToStopException:
            daemon_is_requested_to_stop = ComponentManagerSyncui.get_translate_component().translator().translate(u'common.backend.exceptions.daemon_is_requested_to_stop')
            _return = {
                u'state' : RouteTestCallbackDowntimeConnection.CONNECTION_STATE.KO,
                u'output': daemon_is_requested_to_stop
            }
        return _return
