#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2020:
# This file is part of Shinken Enterprise, all rights reserved


import inspect
import os

from shinken.basemodule import SOURCE_STATE
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.api.synchronizer import ComponentManagerSyncui, HTML_CATEGORY, BusinessException, HTTPSyncUIBusinessError, FrontEndPropertySource
from shinkensolutions.api.synchronizer.http_lib_external.v01_00 import HelpZone, HtmlDistributor, HTML_TYPE, LabelHtml, instance_manager_distributor, ORIGIN_DISTRIBUTOR_RESET, Counter
from shinkensolutions.api.synchronizer.http_lib_external.v01_00.html.template_html import template_format
from shinkensolutions.api.synchronizer.source.route.route import AbstractRoute
from shinkensolutions.api.synchronizer.source.tab.tab import AbstractTab

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict, List
    from shinken.log import PartLogger
    from shinkensolutions.api.synchronizer import SourceTranslatePart, ProxySourceInfo, ValidationState
    from shinkensolutions.api.synchronizer.source.rules_application_template import SourceRulesManager
    from shinkensolutions.api.synchronizer.source.mapping.origin_to_source.mapper import MapperOriginToSource


class TabConfiguration(AbstractTab):
    
    def __init__(self, logger, translator, source_name):
        # type: (PartLogger, SourceTranslatePart, unicode) -> None
        super(TabConfiguration, self).__init__(logger, translator, source_name, u'configuration', translator.translate(u'tab.configuration.title'))
    
    
    def controller(self):
        # type: () -> Dict
        _source_info = ComponentManagerSyncui.get_source_info_component().get_source_info(self.source_name)
        html_distributor, list_sync_keys_keys, list_specific_keys, list_downtime_keys = self.init_html_object(_source_info)
        
        validation_state = _source_info.validation_state  # type:ValidationState
        validation_state._translator = ComponentManagerSyncui.get_translate_component().translator()
        
        return_data = {
            u'http_start_time'              : self.http_start_time,
            u'html_distributor'             : html_distributor,
            u'source_name'                  : self.source_name,
            u'source_state'                 : _source_info.state,
            u'translate'                    : self.translator.translate,
            u'synckey_keys_properties_keys' : list_sync_keys_keys,
            u'specific_properties_keys'     : list_specific_keys,
            u'downtime_link_properties_keys': list_downtime_keys,
            u'files_configuration'          : [{u'path': _source_info.get_load_from_file(), u'status': u'ok'}],
            u'validation_state'             : validation_state.as_html()
        }
        return return_data
    
    
    def init_html_object(self, source_info):
        # type: (ProxySourceInfo) -> (HtmlDistributor, List, List, List)
        html_distributor = HtmlDistributor(self.translator)
        list_sync_keys_keys = []
        list_specific_keys = []
        list_downtime_keys = []
        # OTHER
        html_distributor.add(HTML_TYPE.OTHER, u'help', HelpZone())
        html_distributor.add_button(u'delete_notify', u'delete_notify', LabelHtml(u'''<span class="shinicon-cross"></span>'''))
        html_distributor.add_button(u'close_reload_display', u'close_reload_display', LabelHtml(u'''<span class="shinken-icon shinkon-times-bold"></span>'''))
        html_distributor.add_button(u'test_connexion_ws-arbiter', u'test_connexion_ws-arbiter',
                                    LabelHtml(u'''<div class='btn-shinken btn-blue shinken-user-select-none'>%s</div>''' % self.translator.translate(u'tab_configuration.block_btn.test_connexion_ws-arbiter')))
        
        if source_info.state == SOURCE_STATE.NEED_CONFIRMATION:
            html_distributor.add_html(u'confirmation_configuration_html', ComponentManagerSyncui.get_internal_api_synchronizer_component().internal_call_by_instance_id(self.source_name, u'get_confirmation_configuration_html'))
            html_distributor.add_button(u'confirm_source_configuration', label=LabelHtml(u'''<span class="shinicon-check-circle"></span>%s''' % self.translator.translate(u'tab_configuration.confirm_changes.button_label')))
            _check_box = html_distributor.add_check_box_with_label(u'show_all_properties', name=u'show_all_properties',
                                                                   label=self.translator.translate(u'tab_configuration.confirm_changes.show_all_properties'))
            _check_box.add_url_param_key(u'show_all_properties')
        
        # PROPERTY
        use_sync_keys_prop_name = source_info.get_use_sync_keys_prop_name()
        for prop_name, prop_def in sorted(source_info.get_source_configuration_definition(), key=lambda (_key, value): value.order):
            if prop_def.html_category == HTML_CATEGORY.HIDDEN:
                continue
            if prop_def.html_category == HTML_CATEGORY.SYNC_KEYS:
                if prop_def.name in use_sync_keys_prop_name:
                    list_sync_keys_keys.append(prop_name)
                else:
                    continue
            if prop_def.html_category == HTML_CATEGORY.SPECIFIC:
                list_specific_keys.append(prop_name)
            elif prop_def.html_category == HTML_CATEGORY.DOWNTIME_ON_DELETE:
                list_downtime_keys.append(prop_name)
            
            _to_add = FrontEndPropertySource(self.translator, prop_name, source_info=source_info)
            _to_add.set_disabled(True)
            html_distributor.add(HTML_TYPE.PROPERTY, prop_name, _to_add.property)
        
        # CONTAINER COUNTER ERROR
        if source_info.validation_state.extra_counter_error_number:
            html_distributor.add(HTML_TYPE.OTHER, u'validation_state_extra_counter', Counter(u"shinken-error-counter shinken-hidden", number=source_info.validation_state.extra_counter_error_number, extra_attribute=u"data-is-discreet='1'"))
        
        return html_distributor, list_sync_keys_keys, list_specific_keys, list_downtime_keys


class RouteReloadSourceConfigurationProperties(AbstractRoute):
    
    def __init__(self, logger, source_name, tab_configuration, host_template_binding_rule_manager, mapper_origin_to_source):
        # type: (PartLogger, unicode, TabConfiguration, SourceRulesManager, MapperOriginToSource) -> None
        super(RouteReloadSourceConfigurationProperties, self).__init__(logger, source_name, u'source_configuration_properties/reload')
        self.tab_configuration = tab_configuration
        self.host_template_binding_rule_manager = host_template_binding_rule_manager
        self.mapper_origin_to_source = mapper_origin_to_source
    
    
    def controller(self):
        # type: () -> unicode
        _source_info = ComponentManagerSyncui.get_source_info_component().get_source_info(self.source_name)
        if _source_info.is_import_in_progress():
            return SOURCE_STATE.RUNNING
        
        instance_manager_distributor.reset(ORIGIN_DISTRIBUTOR_RESET.TAB_CONFIGURATION)
        try:
            ComponentManagerSyncui.get_source_info_component().reload(self.source_name)
        except BusinessException as e:
            raise HTTPSyncUIBusinessError(400, e.text)
        
        self.host_template_binding_rule_manager.set_translator(ComponentManagerSyncui.get_translate_component().source_translator(self.source_name))
        self.host_template_binding_rule_manager.reload_rule()
        self.mapper_origin_to_source.reload_mapper()
        controller = self.tab_configuration.controller()
        try:
            template_path = os.path.join(os.path.dirname(inspect.getfile(self.__class__)), u'http_template', u'tab_configuration.tpl')
            return template_format(template_path, HTML_TYPE=HTML_TYPE, **controller)
        except:
            self.logger.print_stack()
            raise


class RouteConfirmConfigurationProperties(AbstractRoute):
    
    def __init__(self, logger, source_name):
        # type: (PartLogger, unicode) -> None
        super(RouteConfirmConfigurationProperties, self).__init__(logger, source_name, u'source_configuration_properties/confirm')
    
    
    def controller(self):
        # type: () -> unicode
        _source_info = ComponentManagerSyncui.get_source_info_component().get_source_info(self.source_name)
        if _source_info.is_import_in_progress():
            return SOURCE_STATE.RUNNING
        try:
            ComponentManagerSyncui.get_source_info_component().confirm_configuration(self.source_name)
        except BusinessException as e:
            raise HTTPSyncUIBusinessError(400, e.text)
        return u''
