#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2017:
# This file is part of Shinken Enterprise, all rights reserved.

import json
import time

from shinken.log import logger, PartLogger
from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict


class Beacon:
    def __init__(self, name, from_info=None, enable=False, quiet=False):
        self.enable = enable
        self.name = name
        self.quiet = quiet
        self.raw_calls = []
        
        if not self.enable:
            return
        
        now = time.time()
        self.is_compute = True
        if from_info is None:
            self.is_compute = False
            from_info = {
                'last_call': now,
                'sons'     : [
                    {
                        'name': name,
                    },
                ]}
        self.times_infos = from_info
        if not self.quiet:
            logger.debug('[%s] Beacon start [%s]' % (self.name, now))
    
    
    def beacon(self, *argv):
        if not self.enable:
            return
        if self.is_compute:
            raise Exception('beacon is done')
        
        self.raw_calls.append((argv, time.time()))
    
    
    def _compute_call_time(self):
        if self.is_compute:
            return
        self.is_compute = True
        for names, now in self.raw_calls:
            names = list(names)
            names.insert(0, self.name)
            times_infos = self.times_infos
            beacon_names = ''
            for name in names:
                last_call = times_infos['last_call']
                sons_times_infos = times_infos['sons']
                current_times_info = next((t for t in sons_times_infos if t['name'] == name), None)
                if current_times_info is None:
                    current_times_info = {}
                    sons_times_infos.append(current_times_info)
                
                current_times_info['name'] = name
                current_times_info['duration'] = current_times_info.get('duration', 0) + now - last_call
                if 'last_call' not in current_times_info:
                    current_times_info['last_call'] = last_call
                
                current_times_info['sons'] = current_times_info.get('sons', [])
                beacon_names += '[%s] : [%0.5f]s ' % (name, current_times_info['duration'])
                times_infos = current_times_info
            
            times_infos = self.times_infos
            times_infos['last_call'] = now
            for name in names:
                sons_times_infos = times_infos['sons']
                current_times_info = next((t for t in sons_times_infos if t['name'] == name), None)
                current_times_info['last_call'] = now
                times_infos = current_times_info
    
    
    def dump(self):
        if not self.enable:
            return
        self._compute_call_time()
        logger.debug(json.dumps(self.times_infos))
    
    
    def print_time(self, only_level=None, _time_info=None, _name='', _level=0, csv_format=False, name_filter='', _logger=None):
        if not _logger:
            _logger = logger
        if not self.enable:
            return
        if _time_info is None:
            self._compute_call_time()
            _time_info = self.times_infos
        for son in _time_info.get('sons', []):
            if _name:
                new_name = _name + '.' + son['name']
            else:
                new_name = son['name']
            if (only_level is None or only_level == _level) and (not name_filter or name_filter == _name):
                if csv_format:
                    print ("%0.4f" % (son['duration'])).replace('.', ',') + ";%s;%s" % (_level, new_name)
                else:
                    _logger.debug("%5.2fms - %s" % (son['duration'] * 1000, new_name))
            self.print_time(only_level=only_level, _time_info=son, _name=new_name, _level=_level + 1, csv_format=csv_format, name_filter=name_filter)


class BeaconLogger(Beacon):
    def __init__(self, name, parent_logger, log_info=False, from_info=None, enable=False, quiet=False):
        # type: (unicode, PartLogger, bool, Dict, bool, bool) -> None
        Beacon.__init__(self, name, from_info, enable, quiet)
        self.last_time = 0
        self.last_step = None
        self.log_info = log_info
        self.parent_logger = parent_logger
    
    
    def _register_and_log(self, step_name, *argv):
        # type: (unicode, unicode) -> None
        if not self.enable:
            return
        if self.last_time and self.last_step:
            self.parent_logger.get_sub_part(self.last_step).get_sub_part(u'PERF').debug(u'took [ %.2f ] seconds' % (time.time() - self.last_time))
        self.beacon(step_name, *argv)
        self.last_time = time.time()
        self.last_step = step_name
    
    
    def info(self, to_log, step_name, *argv):
        # type: (unicode, unicode, unicode) -> None
        self._register_and_log(step_name, *argv)
        self.parent_logger.get_sub_part(step_name).info(to_log)
    
    
    def error(self, to_log, step_name, *argv):
        # type: (unicode, unicode, unicode) -> None
        self._register_and_log(step_name, *argv)
        self.parent_logger.get_sub_part(step_name).error(to_log)
    
    
    def warning(self, to_log, step_name, *argv):
        # type: (unicode, unicode, unicode) -> None
        self._register_and_log(step_name, *argv)
        self.parent_logger.get_sub_part(step_name).warning(to_log)
    
    
    def debug(self, to_log, step_name, *argv):
        # type: (unicode, unicode, unicode) -> None
        self._register_and_log(step_name, *argv)
        self.parent_logger.get_sub_part(step_name).debug(to_log)


if __name__ == '__main__':
    test_pre_optimisation = '''{"last_call": 1562157075.796761, "sons": [{"duration": 0.43555712699890137, "last_call": 1562157075.796761, "sons": [{"duration": 0.24370908737182617, "last_call": 1562157075.604913, "sons": [{"duration": 0.010084152221679688, "last_call": 1562157075.371288, "sons": [], "name": "arguments_parsing"}, {"duration": 0.05718588829040527, "last_call": 1562157075.428474, "sons": [], "name": "query_data"}, {"duration": 5.9604644775390625e-06, "last_call": 1562157075.42848, "sons": [], "name": "parse_filter"}, {"duration": 0.0677640438079834, "last_call": 1562157075.496244, "sons": [{"duration": 0.01815629005432129, "last_call": 1562157075.496235, "sons": [], "name": "pre_filter"}, {"duration": 0.045023441314697266, "last_call": 1562157075.496243, "sons": [], "name": "compute_acl"}, {"duration": 0.004584312438964844, "last_call": 1562157075.496244, "sons": [], "name": "apply_filter_on_one_element"}], "name": "filter_elements"}, {"duration": 0.061025142669677734, "last_call": 1562157075.557269, "sons": [], "name": "sorting_elements"}, {"duration": 0.00019693374633789062, "last_call": 1562157075.557466, "sons": [], "name": "pagination"}, {"duration": 0.026443958282470703, "last_call": 1562157075.58391, "sons": [{"duration": 0.026443958282470703, "last_call": 1562157075.58391, "sons": [{"duration": 0.004301548004150391, "last_call": 1562157075.583842, "sons": [], "name": "type"}, {"duration": 0.0029153823852539062, "last_call": 1562157075.583851, "sons": [], "name": "status"}, {"duration": 0.0041463375091552734, "last_call": 1562157075.583863, "sons": [], "name": "working_area_status"}, {"duration": 0.0007143020629882812, "last_call": 1562157075.583866, "sons": [], "name": "host_name"}, {"duration": 0.0008103847503662109, "last_call": 1562157075.583868, "sons": [], "name": "display_name"}, {"duration": 0.0006077289581298828, "last_call": 1562157075.58387, "sons": [], "name": "address"}, {"duration": 0.000705718994140625, "last_call": 1562157075.583872, "sons": [], "name": "realm"}, {"duration": 0.0006628036499023438, "last_call": 1562157075.583874, "sons": [], "name": "poller_tag"}, {"duration": 0.0031061172485351562, "last_call": 1562157075.583882, "sons": [], "name": "use"}, {"duration": 0.0006618499755859375, "last_call": 1562157075.583884, "sons": [], "name": "pack"}, {"duration": 0.0008478164672851562, "last_call": 1562157075.583888, "sons": [], "name": "comments"}, {"duration": 0.0006189346313476562, "last_call": 1562157075.583892, "sons": [], "name": "last_modification_who"}, {"duration": 0.0006368160247802734, "last_call": 1562157075.583893, "sons": [], "name": "last_modification_date"}, {"duration": 0.0007295608520507812, "last_call": 1562157075.583896, "sons": [], "name": "sources"}], "name": "get_extra_info_for_one_element"}], "name": "build_extra_info"}, {"duration": 0.021003007888793945, "last_call": 1562157075.604913, "sons": [], "name": "list_all_values_for_filter_dropdown"}], "name": "controller"}, {"duration": 0.19165492057800293, "last_call": 1562157075.796568, "sons": [{"duration": 0.04148411750793457, "last_call": 1562157075.646397, "sons": [], "name": "import"}, {"duration": 0.009676933288574219, "last_call": 1562157075.656074, "sons": [], "name": "top part"}, {"duration": 0.008014917373657227, "last_call": 1562157075.664089, "sons": [], "name": "header part"}, {"duration": 0.13247895240783691, "last_call": 1562157075.796568, "sons": [], "name": "content part"}], "name": "template"}, {"duration": 0.00019311904907226562, "last_call": 1562157075.796761, "sons": [], "name": "done"}], "name": "item_list"}]}'''
    from_info = json.loads(test_pre_optimisation)
    beacon = Beacon('loaded', from_info=from_info, enable=True)
    beacon.print_time()
    logger.debug('-----------------------------------------------------')
    # beacon.print_time(name_filter='item_list.controller')
    beacon.print_time(name_filter='item_list.controller.filter_elements')
    
    #
    # test_post_source_modif = '''{"last_call": 1509372495.4916849, "sons": [{"duration": 119.2716498374939, "last_call": 1509372495.4916849, "sons": [{"duration": 0.16718602180480957, "last_call": 1509372376.3872211, "sons": [], "name": "_load_source_items"}, {"duration": 21.230027198791504, "last_call": 1509372397.676477, "sons": [{"duration": 0.0018298625946044922, "last_call": 1509372376.389051, "sons": [{"duration": 0.0013818740844726562, "last_call": 1509372376.388603, "sons": [], "name": "find_linked_items"}, {"duration": 0.00044798851013183594, "last_call": 1509372376.389051, "sons": [], "name": "merge linked item"}], "name": "macromodulation"}, {"duration": 0.0013120174407958984, "last_call": 1509372376.390363, "sons": [{"duration": 0.00081992149353027344, "last_call": 1509372376.3898709, "sons": [], "name": "find_linked_items"}, {"duration": 0.000492095947265625, "last_call": 1509372376.390363, "sons": [], "name": "merge linked item"}], "name": "escalation"}, {"duration": 0.0018699169158935547, "last_call": 1509372376.3922329, "sons": [{"duration": 0.00074410438537597656, "last_call": 1509372376.3911071, "sons": [], "name": "find_linked_items"}, {"duration": 0.0011258125305175781, "last_call": 1509372376.3922329, "sons": [], "name": "merge linked item"}], "name": "notificationway"}, {"duration": 20.382250070571899, "last_call": 1509372396.774483, "sons": [{"duration": 2.6799411773681641, "last_call": 1509372379.0721741, "sons": [], "name": "find_linked_items"}, {"duration": 1.3013489246368408, "last_call": 1509372380.373523, "sons": [], "name": "merge linked item"}, {"duration": 16.363168954849243, "last_call": 1509372396.736692, "sons": [], "name": "ip-tags"}, {"duration": 0.037791013717651367, "last_call": 1509372396.774483, "sons": [], "name": "regexp-tags"}], "name": "host"}, {"duration": 0.024686098098754883, "last_call": 1509372396.7991691, "sons": [{"duration": 0.021485090255737305, "last_call": 1509372396.7959681, "sons": [], "name": "find_linked_items"}, {"duration": 0.0032010078430175781, "last_call": 1509372396.7991691, "sons": [], "name": "merge linked item"}], "name": "resultmodulation"}, {"duration": 0.0014698505401611328, "last_call": 1509372396.8006389, "sons": [{"duration": 0.00087785720825195312, "last_call": 1509372396.8000469, "sons": [], "name": "find_linked_items"}, {"duration": 0.00059199333190917969, "last_call": 1509372396.8006389, "sons": [], "name": "merge linked item"}], "name": "businessimpactmodulation"}, {"duration": 0.041846990585327148, "last_call": 1509372396.8424859, "sons": [{"duration": 0.0051441192626953125, "last_call": 1509372396.805783, "sons": [], "name": "find_linked_items"}, {"duration": 0.036702871322631836, "last_call": 1509372396.8424859, "sons": [], "name": "merge linked item"}], "name": "service"}, {"duration": 0.0047781467437744141, "last_call": 1509372396.8472641, "sons": [{"duration": 0.0010790824890136719, "last_call": 1509372396.843565, "sons": [], "name": "find_linked_items"}, {"duration": 0.0036990642547607422, "last_call": 1509372396.8472641, "sons": [], "name": "merge linked item"}], "name": "contactgroup"}, {"duration": 0.56259703636169434, "last_call": 1509372397.4098611, "sons": [{"duration": 0.013107061386108398, "last_call": 1509372396.8603711, "sons": [], "name": "find_linked_items"}, {"duration": 0.54948997497558594, "last_call": 1509372397.4098611, "sons": [], "name": "merge linked item"}], "name": "hostgroup"}, {"duration": 0.22457289695739746, "last_call": 1509372397.634434, "sons": [{"duration": 0.097469806671142578, "last_call": 1509372397.5073309, "sons": [], "name": "find_linked_items"}, {"duration": 0.12710309028625488, "last_call": 1509372397.634434, "sons": [], "name": "merge linked item"}], "name": "contact"}, {"duration": 0.040106058120727539, "last_call": 1509372397.67454, "sons": [{"duration": 0.0074069499969482422, "last_call": 1509372397.6418409, "sons": [], "name": "find_linked_items"}, {"duration": 0.032699108123779297, "last_call": 1509372397.67454, "sons": [], "name": "merge linked item"}], "name": "command"}, {"duration": 0.0019369125366210938, "last_call": 1509372397.676477, "sons": [{"duration": 0.0012249946594238281, "last_call": 1509372397.675765, "sons": [], "name": "find_linked_items"}, {"duration": 0.00071191787719726562, "last_call": 1509372397.676477, "sons": [], "name": "merge linked item"}], "name": "timeperiod"}], "name": "merge_items"}, {"duration": 0.018643856048583984, "last_call": 1509372397.6760881, "sons": [], "name": "_link_source_items"}, {"duration": 0.0023353099822998047, "last_call": 1509372397.676682, "sons": [], "name": "_merge_source_items"}, {"duration": 0.013409852981567383, "last_call": 1509372397.6769159, "sons": [], "name": "_apply_taggers_on_merge_items"}, {"duration": 0.025455713272094727, "last_call": 1509372397.677093, "sons": [], "name": "_reformat_merge_items"}, {"duration": 0.087790012359619141, "last_call": 1509372397.764883, "sons": [], "name": "_initialise_merge_item_datamanager"}, {"duration": 0.13270401954650879, "last_call": 1509372397.8975871, "sons": [], "name": "_valide_merge_items"}, {"duration": 24.899582862854004, "last_call": 1509372422.7971699, "sons": [], "name": "_build_double_link_for_merge_items"}, {"duration": 66.196922063827515, "last_call": 1509372488.994092, "sons": [], "name": "_compute_diff_with_existing_items"}, {"duration": 1.3444309234619141, "last_call": 1509372490.3385229, "sons": [], "name": "_saving_merge_items"}, {"duration": 5.1527061462402344, "last_call": 1509372495.4912291, "sons": [], "name": "_saving_retention_data"}, {"duration": 0.00028705596923828125, "last_call": 1509372495.4915161, "sons": [], "name": "_set_sources_as_merge_done"}, {"duration": 0.00016880035400390625, "last_call": 1509372495.4916849, "sons": [], "name": "end"}], "name": "beacon_merge_source"}]}'''
    # from_info = json.loads(test_post_source_modif)
    # beacon = Beacon('loaded', from_info)
    # beacon.print_time(1)
    #
    # test_post_source_modif = '''{"last_call": 1509439576.640569, "sons": [{"duration": 106.62402606010437, "last_call": 1509439576.640569, "sons": [{"duration": 0.15438508987426758, "last_call": 1509439470.170928, "sons": [], "name": "_load_source_items"}, {"duration": 7.1184203624725342, "last_call": 1509439477.3736529, "sons": [{"duration": 0.079040050506591797, "last_call": 1509439470.2499681, "sons": [{"duration": 0.078032016754150391, "last_call": 1509439470.24896, "sons": [], "name": "find_linked_items"}, {"duration": 0.0010080337524414062, "last_call": 1509439470.2499681, "sons": [], "name": "merge linked item"}], "name": "macromodulation"}, {"duration": 0.00091385841369628906, "last_call": 1509439470.2508819, "sons": [{"duration": 0.00061488151550292969, "last_call": 1509439470.2505829, "sons": [], "name": "find_linked_items"}, {"duration": 0.00029897689819335938, "last_call": 1509439470.2508819, "sons": [], "name": "merge linked item"}], "name": "escalation"}, {"duration": 0.0020291805267333984, "last_call": 1509439470.2529111, "sons": [{"duration": 0.00072312355041503906, "last_call": 1509439470.251605, "sons": [], "name": "find_linked_items"}, {"duration": 0.0013060569763183594, "last_call": 1509439470.2529111, "sons": [], "name": "merge linked item"}], "name": "notificationway"}, {"duration": 6.1561369895935059, "last_call": 1509439476.4090481, "sons": [{"duration": 2.6887810230255127, "last_call": 1509439472.9416921, "sons": [], "name": "find_linked_items"}, {"duration": 1.325657844543457, "last_call": 1509439474.26735, "sons": [], "name": "merge linked item"}, {"duration": 2.1400940418243408, "last_call": 1509439476.407444, "sons": [], "name": "ip-tags"}, {"duration": 0.0016040802001953125, "last_call": 1509439476.4090481, "sons": [], "name": "regexp-tags"}], "name": "host"}, {"duration": 0.045876979827880859, "last_call": 1509439476.4549251, "sons": [{"duration": 0.040053844451904297, "last_call": 1509439476.4491019, "sons": [], "name": "find_linked_items"}, {"duration": 0.0058231353759765625, "last_call": 1509439476.4549251, "sons": [], "name": "merge linked item"}], "name": "resultmodulation"}, {"duration": 0.0026838779449462891, "last_call": 1509439476.4576089, "sons": [{"duration": 0.0017600059509277344, "last_call": 1509439476.4566851, "sons": [], "name": "find_linked_items"}, {"duration": 0.00092387199401855469, "last_call": 1509439476.4576089, "sons": [], "name": "merge linked item"}], "name": "businessimpactmodulation"}, {"duration": 0.046489953994750977, "last_call": 1509439476.5040989, "sons": [{"duration": 0.0045521259307861328, "last_call": 1509439476.4621611, "sons": [], "name": "find_linked_items"}, {"duration": 0.041937828063964844, "last_call": 1509439476.5040989, "sons": [], "name": "merge linked item"}], "name": "service"}, {"duration": 0.004467010498046875, "last_call": 1509439476.5085659, "sons": [{"duration": 0.001354217529296875, "last_call": 1509439476.5054531, "sons": [], "name": "find_linked_items"}, {"duration": 0.00311279296875, "last_call": 1509439476.5085659, "sons": [], "name": "merge linked item"}], "name": "contactgroup"}, {"duration": 0.57904720306396484, "last_call": 1509439477.0876131, "sons": [{"duration": 0.013259172439575195, "last_call": 1509439476.5218251, "sons": [], "name": "find_linked_items"}, {"duration": 0.56578803062438965, "last_call": 1509439477.0876131, "sons": [], "name": "merge linked item"}], "name": "hostgroup"}, {"duration": 0.24057579040527344, "last_call": 1509439477.3281889, "sons": [{"duration": 0.091628789901733398, "last_call": 1509439477.1792419, "sons": [], "name": "find_linked_items"}, {"duration": 0.14894700050354004, "last_call": 1509439477.3281889, "sons": [], "name": "merge linked item"}], "name": "contact"}, {"duration": 0.044012069702148438, "last_call": 1509439477.372201, "sons": [{"duration": 0.010998010635375977, "last_call": 1509439477.3391869, "sons": [], "name": "find_linked_items"}, {"duration": 0.033014059066772461, "last_call": 1509439477.372201, "sons": [], "name": "merge linked item"}], "name": "command"}, {"duration": 0.0014519691467285156, "last_call": 1509439477.3736529, "sons": [{"duration": 0.000946044921875, "last_call": 1509439477.373147, "sons": [], "name": "find_linked_items"}, {"duration": 0.00050592422485351562, "last_call": 1509439477.3736529, "sons": [], "name": "merge linked item"}], "name": "timeperiod"}], "name": "merge_items"}, {"duration": 0.019610404968261719, "last_call": 1509439477.3733971, "sons": [], "name": "_link_source_items"}, {"duration": 0.0033605098724365234, "last_call": 1509439477.3743801, "sons": [], "name": "_merge_source_items"}, {"duration": 0.014766693115234375, "last_call": 1509439477.3747721, "sons": [], "name": "_apply_taggers_on_merge_items"}, {"duration": 0.048005104064941406, "last_call": 1509439477.3750911, "sons": [], "name": "_reformat_merge_items"}, {"duration": 0.075567960739135742, "last_call": 1509439477.450659, "sons": [], "name": "_initialise_merge_item_datamanager"}, {"duration": 0.12720298767089844, "last_call": 1509439477.577862, "sons": [], "name": "_valide_merge_items"}, {"duration": 24.308191061019897, "last_call": 1509439501.8860531, "sons": [], "name": "_build_double_link_for_merge_items"}, {"duration": 67.946712017059326, "last_call": 1509439569.8327651, "sons": [], "name": "_compute_diff_with_existing_items"}, {"duration": 1.8566949367523193, "last_call": 1509439571.68946, "sons": [], "name": "_saving_merge_items"}, {"duration": 4.9506649971008301, "last_call": 1509439576.640125, "sons": [], "name": "_saving_retention_data"}, {"duration": 0.00029802322387695312, "last_call": 1509439576.6404231, "sons": [], "name": "_set_sources_as_merge_done"}, {"duration": 0.00014591217041015625, "last_call": 1509439576.640569, "sons": [], "name": "end"}], "name": "beacon_merge_source"}]}'''
    # from_info = json.loads(test_post_source_modif)
    # beacon = Beacon('loaded', from_info)
    # beacon.print_time(1)
    #
    # test_post_source_modif = '''{"last_call": 1509440620.21488, "sons": [{"duration": 81.582485914230347, "last_call": 1509440620.21488, "sons": [{"duration": 0.17144083976745605, "last_call": 1509440538.8038349, "sons": [], "name": "_load_source_items"}, {"duration": 6.8842732906341553, "last_call": 1509440545.7318809, "sons": [{"duration": 0.079428195953369141, "last_call": 1509440538.8832631, "sons": [{"duration": 0.078700065612792969, "last_call": 1509440538.882535, "sons": [], "name": "find_linked_items"}, {"duration": 0.00072813034057617188, "last_call": 1509440538.8832631, "sons": [], "name": "merge linked item"}], "name": "macromodulation"}, {"duration": 0.00080895423889160156, "last_call": 1509440538.8840721, "sons": [{"duration": 0.0005397796630859375, "last_call": 1509440538.8838029, "sons": [], "name": "find_linked_items"}, {"duration": 0.00026917457580566406, "last_call": 1509440538.8840721, "sons": [], "name": "merge linked item"}], "name": "escalation"}, {"duration": 0.0019488334655761719, "last_call": 1509440538.8860209, "sons": [{"duration": 0.00074291229248046875, "last_call": 1509440538.884815, "sons": [], "name": "find_linked_items"}, {"duration": 0.0012059211730957031, "last_call": 1509440538.8860209, "sons": [], "name": "merge linked item"}], "name": "notificationway"}, {"duration": 5.9992871284484863, "last_call": 1509440544.885308, "sons": [{"duration": 2.6325922012329102, "last_call": 1509440541.5186131, "sons": [], "name": "find_linked_items"}, {"duration": 1.3276898860931396, "last_call": 1509440542.846303, "sons": [], "name": "merge linked item"}, {"duration": 2.0384480953216553, "last_call": 1509440544.8847511, "sons": [], "name": "ip-tags"}, {"duration": 0.00055694580078125, "last_call": 1509440544.885308, "sons": [], "name": "regexp-tags"}], "name": "host"}, {"duration": 0.020667076110839844, "last_call": 1509440544.9059751, "sons": [{"duration": 0.017545938491821289, "last_call": 1509440544.902854, "sons": [], "name": "find_linked_items"}, {"duration": 0.0031211376190185547, "last_call": 1509440544.9059751, "sons": [], "name": "merge linked item"}], "name": "resultmodulation"}, {"duration": 0.0012099742889404297, "last_call": 1509440544.9071851, "sons": [{"duration": 0.00071597099304199219, "last_call": 1509440544.9066911, "sons": [], "name": "find_linked_items"}, {"duration": 0.0004940032958984375, "last_call": 1509440544.9071851, "sons": [], "name": "merge linked item"}], "name": "businessimpactmodulation"}, {"duration": 0.045616865158081055, "last_call": 1509440544.9528019, "sons": [{"duration": 0.0033040046691894531, "last_call": 1509440544.9104891, "sons": [], "name": "find_linked_items"}, {"duration": 0.042312860488891602, "last_call": 1509440544.9528019, "sons": [], "name": "merge linked item"}], "name": "service"}, {"duration": 0.0047061443328857422, "last_call": 1509440544.9575081, "sons": [{"duration": 0.0012841224670410156, "last_call": 1509440544.9540861, "sons": [], "name": "find_linked_items"}, {"duration": 0.0034220218658447266, "last_call": 1509440544.9575081, "sons": [], "name": "merge linked item"}], "name": "contactgroup"}, {"duration": 0.51635479927062988, "last_call": 1509440545.4738629, "sons": [{"duration": 0.013588905334472656, "last_call": 1509440544.971097, "sons": [], "name": "find_linked_items"}, {"duration": 0.50276589393615723, "last_call": 1509440545.4738629, "sons": [], "name": "merge linked item"}], "name": "hostgroup"}, {"duration": 0.21509003639221191, "last_call": 1509440545.6889529, "sons": [{"duration": 0.078550100326538086, "last_call": 1509440545.552413, "sons": [], "name": "find_linked_items"}, {"duration": 0.13653993606567383, "last_call": 1509440545.6889529, "sons": [], "name": "merge linked item"}], "name": "contact"}, {"duration": 0.040112972259521484, "last_call": 1509440545.7290659, "sons": [{"duration": 0.0074021816253662109, "last_call": 1509440545.6963551, "sons": [], "name": "find_linked_items"}, {"duration": 0.032710790634155273, "last_call": 1509440545.7290659, "sons": [], "name": "merge linked item"}], "name": "command"}, {"duration": 0.0028150081634521484, "last_call": 1509440545.7318809, "sons": [{"duration": 0.0021550655364990234, "last_call": 1509440545.731221, "sons": [], "name": "find_linked_items"}, {"duration": 0.000659942626953125, "last_call": 1509440545.7318809, "sons": [], "name": "merge linked item"}], "name": "timeperiod"}], "name": "merge_items"}, {"duration": 0.017632722854614258, "last_call": 1509440545.7314479, "sons": [], "name": "_link_source_items"}, {"duration": 0.0023562908172607422, "last_call": 1509440545.732163, "sons": [], "name": "_merge_source_items"}, {"duration": 0.0029006004333496094, "last_call": 1509440545.7323029, "sons": [], "name": "_apply_taggers_on_merge_items"}, {"duration": 0.021444082260131836, "last_call": 1509440545.7324419, "sons": [], "name": "_reformat_merge_items"}, {"duration": 0.083088159561157227, "last_call": 1509440545.8155301, "sons": [], "name": "_initialise_merge_item_datamanager"}, {"duration": 0.15047597885131836, "last_call": 1509440545.966006, "sons": [], "name": "_valide_merge_items"}, {"duration": 0.019649028778076172, "last_call": 1509440545.9856551, "sons": [], "name": "_build_double_link_for_merge_items"}, {"duration": 67.473402976989746, "last_call": 1509440613.459058, "sons": [], "name": "_compute_diff_with_existing_items"}, {"duration": 1.9416308403015137, "last_call": 1509440615.4006889, "sons": [], "name": "_saving_merge_items"}, {"duration": 4.8137021064758301, "last_call": 1509440620.214391, "sons": [], "name": "_saving_retention_data"}, {"duration": 0.00027894973754882812, "last_call": 1509440620.2146699, "sons": [], "name": "_set_sources_as_merge_done"}, {"duration": 0.00021004676818847656, "last_call": 1509440620.21488, "sons": [], "name": "end"}], "name": "beacon_merge_source"}]}'''
    # from_info = json.loads(test_post_source_modif)
    # beacon = Beacon('loaded', from_info)
    # beacon.print_time(1)
    #
    # test_post_source_modif = '''{"last_call": 1509444618.3994811, "sons": [{"duration": 15.24434494972229, "last_call": 1509444618.3994811, "sons": [{"duration": 0.15823078155517578, "last_call": 1509444603.3133669, "sons": [], "name": "_load_source_items"}, {"duration": 7.2037239074707031, "last_call": 1509444610.591759, "sons": [{"duration": 0.072924137115478516, "last_call": 1509444603.386291, "sons": [{"duration": 0.072097063064575195, "last_call": 1509444603.385464, "sons": [], "name": "find_linked_items"}, {"duration": 0.00082707405090332031, "last_call": 1509444603.386291, "sons": [], "name": "merge linked item"}], "name": "macromodulation"}, {"duration": 0.00094199180603027344, "last_call": 1509444603.3873911, "sons": [{"duration": 0.00061297416687011719, "last_call": 1509444603.3870621, "sons": [], "name": "find_linked_items"}, {"duration": 0.00032901763916015625, "last_call": 1509444603.3873911, "sons": [], "name": "merge linked item"}], "name": "escalation"}, {"duration": 0.0018870830535888672, "last_call": 1509444603.3894291, "sons": [{"duration": 0.00067210197448730469, "last_call": 1509444603.3882141, "sons": [], "name": "find_linked_items"}, {"duration": 0.0012149810791015625, "last_call": 1509444603.3894291, "sons": [], "name": "merge linked item"}], "name": "notificationway"}, {"duration": 6.318000316619873, "last_call": 1509444609.7078791, "sons": [{"duration": 2.716123104095459, "last_call": 1509444606.105803, "sons": [], "name": "find_linked_items"}, {"duration": 1.2533071041107178, "last_call": 1509444607.3591101, "sons": [], "name": "merge linked item"}, {"duration": 2.346560001373291, "last_call": 1509444609.7056701, "sons": [], "name": "ip-tags"}, {"duration": 0.0022089481353759766, "last_call": 1509444609.7078791, "sons": [], "name": "regexp-tags"}], "name": "host"}, {"duration": 0.053472995758056641, "last_call": 1509444609.7613521, "sons": [{"duration": 0.049547910690307617, "last_call": 1509444609.757427, "sons": [], "name": "find_linked_items"}, {"duration": 0.0039250850677490234, "last_call": 1509444609.7613521, "sons": [], "name": "merge linked item"}], "name": "resultmodulation"}, {"duration": 0.00292205810546875, "last_call": 1509444609.7646739, "sons": [{"duration": 0.0018720626831054688, "last_call": 1509444609.763624, "sons": [], "name": "find_linked_items"}, {"duration": 0.0010499954223632812, "last_call": 1509444609.7646739, "sons": [], "name": "merge linked item"}], "name": "businessimpactmodulation"}, {"duration": 0.077511072158813477, "last_call": 1509444609.8424931, "sons": [{"duration": 0.0084130764007568359, "last_call": 1509444609.7733951, "sons": [], "name": "find_linked_items"}, {"duration": 0.069097995758056641, "last_call": 1509444609.8424931, "sons": [], "name": "merge linked item"}], "name": "service"}, {"duration": 0.0039370059967041016, "last_call": 1509444609.8467031, "sons": [{"duration": 0.00067400932312011719, "last_call": 1509444609.8434401, "sons": [], "name": "find_linked_items"}, {"duration": 0.0032629966735839844, "last_call": 1509444609.8467031, "sons": [], "name": "merge linked item"}], "name": "contactgroup"}, {"duration": 0.5055241584777832, "last_call": 1509444610.3523951, "sons": [{"duration": 0.012255191802978516, "last_call": 1509444609.8591261, "sons": [], "name": "find_linked_items"}, {"duration": 0.49326896667480469, "last_call": 1509444610.3523951, "sons": [], "name": "merge linked item"}], "name": "hostgroup"}, {"duration": 0.19678807258605957, "last_call": 1509444610.549325, "sons": [{"duration": 0.064295053482055664, "last_call": 1509444610.416832, "sons": [], "name": "find_linked_items"}, {"duration": 0.13249301910400391, "last_call": 1509444610.549325, "sons": [], "name": "merge linked item"}], "name": "contact"}, {"duration": 0.04027104377746582, "last_call": 1509444610.5897369, "sons": [{"duration": 0.0075421333312988281, "last_call": 1509444610.557008, "sons": [], "name": "find_linked_items"}, {"duration": 0.032728910446166992, "last_call": 1509444610.5897369, "sons": [], "name": "merge linked item"}], "name": "command"}, {"duration": 0.0016591548919677734, "last_call": 1509444610.5915461, "sons": [{"duration": 0.0010011196136474609, "last_call": 1509444610.590888, "sons": [], "name": "find_linked_items"}, {"duration": 0.0006580352783203125, "last_call": 1509444610.5915461, "sons": [], "name": "merge linked item"}], "name": "timeperiod"}], "name": "_merge_source_items"}, {"duration": 0.019427776336669922, "last_call": 1509444610.5911191, "sons": [], "name": "_link_source_items"}, {"duration": 0.0038893222808837891, "last_call": 1509444610.5919249, "sons": [], "name": "_apply_taggers_on_merge_items"}, {"duration": 0.05181121826171875, "last_call": 1509444610.5922191, "sons": [], "name": "_reformat_merge_items"}, {"duration": 0.077101945877075195, "last_call": 1509444610.6693211, "sons": [], "name": "_initialise_merge_item_datamanager"}, {"duration": 0.16834497451782227, "last_call": 1509444610.837666, "sons": [], "name": "_valide_merge_items"}, {"duration": 0.03189396858215332, "last_call": 1509444610.86956, "sons": [], "name": "_build_double_link_for_merge_items"}, {"duration": 0.04318690299987793, "last_call": 1509444610.9127469, "sons": [], "name": "_compute_diff_with_existing_items"}, {"duration": 2.5440680980682373, "last_call": 1509444613.456815, "sons": [], "name": "_saving_merge_items"}, {"duration": 4.92340087890625, "last_call": 1509444618.3802159, "sons": [], "name": "_saving_retention_data"}, {"duration": 0.00041103363037109375, "last_call": 1509444618.3806269, "sons": [], "name": "_set_sources_as_merge_done"}, {"duration": 0.018854141235351562, "last_call": 1509444618.3994811, "sons": [], "name": "end"}], "name": "beacon_merge_source"}]}'''
    # from_info = json.loads(test_post_source_modif)
    # beacon = Beacon('loaded', from_info)
    # beacon.print_time(1)
    #
    # test_post_source_modif = '''{"last_call": 1509446398.854248, "sons": [{"duration": 18.952978134155273, "last_call": 1509446398.854248, "sons": [{"duration": 0.16954302787780762, "last_call": 1509446380.0708129, "sons": [], "name": "_load_source_items"}, {"duration": 9.3876471519470215, "last_call": 1509446389.550384, "sons": [{"duration": 0.14450812339782715, "last_call": 1509446380.2153211, "sons": [{"duration": 0.12322998046875, "last_call": 1509446380.1940429, "sons": [], "name": "find_linked_items"}, {"duration": 0.021278142929077148, "last_call": 1509446380.2153211, "sons": [], "name": "merge linked item"}], "name": "macromodulation"}, {"duration": 0.0021300315856933594, "last_call": 1509446380.2178609, "sons": [{"duration": 0.0014040470123291016, "last_call": 1509446380.217135, "sons": [], "name": "find_linked_items"}, {"duration": 0.00072598457336425781, "last_call": 1509446380.2178609, "sons": [], "name": "merge linked item"}], "name": "escalation"}, {"duration": 0.0024888515472412109, "last_call": 1509446380.2206869, "sons": [{"duration": 0.00077199935913085938, "last_call": 1509446380.2189701, "sons": [], "name": "find_linked_items"}, {"duration": 0.0017168521881103516, "last_call": 1509446380.2206869, "sons": [], "name": "merge linked item"}], "name": "notificationway"}, {"duration": 8.4496719837188721, "last_call": 1509446388.6709881, "sons": [{"duration": 3.9598309993743896, "last_call": 1509446384.180985, "sons": [], "name": "find_linked_items"}, {"duration": 1.9956879615783691, "last_call": 1509446386.1766729, "sons": [], "name": "merge linked item"}, {"duration": 2.491016149520874, "last_call": 1509446388.6676891, "sons": [], "name": "ip-tags"}, {"duration": 0.0032989978790283203, "last_call": 1509446388.6709881, "sons": [], "name": "regexp-tags"}], "name": "host"}, {"duration": 0.045979022979736328, "last_call": 1509446388.7169671, "sons": [{"duration": 0.042714834213256836, "last_call": 1509446388.7137029, "sons": [], "name": "find_linked_items"}, {"duration": 0.0032641887664794922, "last_call": 1509446388.7169671, "sons": [], "name": "merge linked item"}], "name": "resultmodulation"}, {"duration": 0.002574920654296875, "last_call": 1509446388.719857, "sons": [{"duration": 0.0017900466918945312, "last_call": 1509446388.7190721, "sons": [], "name": "find_linked_items"}, {"duration": 0.00078487396240234375, "last_call": 1509446388.719857, "sons": [], "name": "merge linked item"}], "name": "businessimpactmodulation"}, {"duration": 0.046380996704101562, "last_call": 1509446388.7666609, "sons": [{"duration": 0.0068361759185791016, "last_call": 1509446388.7271161, "sons": [], "name": "find_linked_items"}, {"duration": 0.039544820785522461, "last_call": 1509446388.7666609, "sons": [], "name": "merge linked item"}], "name": "service"}, {"duration": 0.0039269924163818359, "last_call": 1509446388.7707789, "sons": [{"duration": 0.00092411041259765625, "last_call": 1509446388.767776, "sons": [], "name": "find_linked_items"}, {"duration": 0.0030028820037841797, "last_call": 1509446388.7707789, "sons": [], "name": "merge linked item"}], "name": "contactgroup"}, {"duration": 0.53203082084655762, "last_call": 1509446389.3030629, "sons": [{"duration": 0.014709949493408203, "last_call": 1509446388.785742, "sons": [], "name": "find_linked_items"}, {"duration": 0.51732087135314941, "last_call": 1509446389.3030629, "sons": [], "name": "merge linked item"}], "name": "hostgroup"}, {"duration": 0.19432687759399414, "last_call": 1509446389.497674, "sons": [{"duration": 0.070303916931152344, "last_call": 1509446389.373651, "sons": [], "name": "find_linked_items"}, {"duration": 0.1240229606628418, "last_call": 1509446389.497674, "sons": [], "name": "merge linked item"}], "name": "contact"}, {"duration": 0.049344062805175781, "last_call": 1509446389.5472081, "sons": [{"duration": 0.0082299709320068359, "last_call": 1509446389.506094, "sons": [], "name": "find_linked_items"}, {"duration": 0.041114091873168945, "last_call": 1509446389.5472081, "sons": [], "name": "merge linked item"}], "name": "command"}, {"duration": 0.0027210712432861328, "last_call": 1509446389.550123, "sons": [{"duration": 0.0020840167999267578, "last_call": 1509446389.5494859, "sons": [], "name": "find_linked_items"}, {"duration": 0.000637054443359375, "last_call": 1509446389.550123, "sons": [], "name": "merge linked item"}], "name": "timeperiod"}], "name": "_merge_source_items"}, {"duration": 0.039422988891601562, "last_call": 1509446389.549789, "sons": [], "name": "_link_source_items"}, {"duration": 0.0037767887115478516, "last_call": 1509446389.5505371, "sons": [], "name": "_apply_taggers_on_merge_items"}, {"duration": 0.049030065536499023, "last_call": 1509446389.5506899, "sons": [], "name": "_reformat_merge_items"}, {"duration": 0.08506011962890625, "last_call": 1509446389.6357501, "sons": [], "name": "_initialise_merge_item_datamanager"}, {"duration": 0.15010905265808105, "last_call": 1509446389.7858591, "sons": [], "name": "_valide_merge_items"}, {"duration": 0.02514195442199707, "last_call": 1509446389.8110011, "sons": [], "name": "_build_double_link_for_merge_items"}, {"duration": 0.037216901779174805, "last_call": 1509446389.848218, "sons": [], "name": "_compute_diff_with_existing_items"}, {"duration": 2.7891659736633301, "last_call": 1509446392.6373839, "sons": [], "name": "_saving_merge_items"}, {"duration": 6.1938130855560303, "last_call": 1509446398.831197, "sons": [], "name": "_saving_retention_data"}, {"duration": 0.0035929679870605469, "last_call": 1509446398.83479, "sons": [], "name": "_set_sources_as_merge_done"}, {"duration": 0.01945805549621582, "last_call": 1509446398.854248, "sons": [], "name": "end"}], "name": "beacon_merge_source"}]}'''
    # from_info = json.loads(test_post_source_modif)
    # beacon = Beacon('loaded', from_info)
    # beacon.print_time(1)
    #
    # test_post_source_modif = '''{"last_call": 1509985840.977422, "sons": [{"duration": 19.117752075195312, "last_call": 1509985840.977422, "sons": [{"duration": 0.63185310363769531, "last_call": 1509985822.491523, "sons": [], "name": "_load_source_items"}, {"duration": 0.026552915573120117, "last_call": 1509985822.5180759, "sons": [], "name": "_find_items_in_cache"}, {"duration": 0.41346383094787598, "last_call": 1509985822.9317009, "sons": [{"duration": 0.00087714195251464844, "last_call": 1509985822.532804, "sons": [{"duration": 0.00039911270141601562, "last_call": 1509985822.5184751, "sons": [], "name": "build items to merge"}, {"duration": 0.00017094612121582031, "last_call": 1509985822.518646, "sons": [], "name": "find_linked_items"}, {"duration": 0.014158010482788086, "last_call": 1509985822.532804, "sons": [], "name": "merge linked item"}], "name": "notificationway"}, {"duration": 0.0087389945983886719, "last_call": 1509985822.5329621, "sons": [{"duration": 0.0081090927124023438, "last_call": 1509985822.5267551, "sons": [], "name": "build items to merge"}, {"duration": 0.00047183036804199219, "last_call": 1509985822.5272269, "sons": [], "name": "find_linked_items"}, {"duration": 0.0057351589202880859, "last_call": 1509985822.5329621, "sons": [], "name": "merge linked item"}], "name": "host"}, {"duration": 0.0010440349578857422, "last_call": 1509985822.533288, "sons": [{"duration": 0.0005340576171875, "last_call": 1509985822.527761, "sons": [], "name": "build items to merge"}, {"duration": 0.00018405914306640625, "last_call": 1509985822.527945, "sons": [], "name": "find_linked_items"}, {"duration": 0.0053429603576660156, "last_call": 1509985822.533288, "sons": [], "name": "merge linked item"}], "name": "service"}, {"duration": 0.001171112060546875, "last_call": 1509985822.5337901, "sons": [{"duration": 0.00049304962158203125, "last_call": 1509985822.5284381, "sons": [], "name": "build items to merge"}, {"duration": 0.00017595291137695312, "last_call": 1509985822.528614, "sons": [], "name": "find_linked_items"}, {"duration": 0.0051760673522949219, "last_call": 1509985822.5337901, "sons": [], "name": "merge linked item"}], "name": "contactgroup"}, {"duration": 0.39818286895751953, "last_call": 1509985822.9311581, "sons": [{"duration": 0.00053691864013671875, "last_call": 1509985822.529151, "sons": [], "name": "build items to merge"}, {"duration": 0.00027799606323242188, "last_call": 1509985822.529429, "sons": [], "name": "find_linked_items"}, {"duration": 0.40172910690307617, "last_call": 1509985822.9311581, "sons": [], "name": "merge linked item"}], "name": "hostgroup"}, {"duration": 0.0020890235900878906, "last_call": 1509985822.931324, "sons": [{"duration": 0.0015580654144287109, "last_call": 1509985822.530987, "sons": [], "name": "build items to merge"}, {"duration": 0.00036501884460449219, "last_call": 1509985822.531352, "sons": [], "name": "find_linked_items"}, {"duration": 0.39997196197509766, "last_call": 1509985822.931324, "sons": [], "name": "merge linked item"}], "name": "contact"}, {"duration": 0.00075697898864746094, "last_call": 1509985822.9314411, "sons": [{"duration": 0.00047588348388671875, "last_call": 1509985822.5318279, "sons": [], "name": "build items to merge"}, {"duration": 0.000164031982421875, "last_call": 1509985822.531992, "sons": [], "name": "find_linked_items"}, {"duration": 0.39944911003112793, "last_call": 1509985822.9314411, "sons": [], "name": "merge linked item"}], "name": "command"}, {"duration": 0.00065684318542480469, "last_call": 1509985822.9315929, "sons": [{"duration": 0.00027894973754882812, "last_call": 1509985822.5322709, "sons": [], "name": "build items to merge"}, {"duration": 0.00022602081298828125, "last_call": 1509985822.5324969, "sons": [], "name": "find_linked_items"}, {"duration": 0.39909601211547852, "last_call": 1509985822.9315929, "sons": [], "name": "merge linked item"}], "name": "timeperiod"}], "name": "_merge_source_items"}, {"duration": 0.00016117095947265625, "last_call": 1509985822.5326581, "sons": [], "name": "_link_source_items"}, {"duration": 9.7990036010742188e-05, "last_call": 1509985822.9317989, "sons": [], "name": "_apply_taggers_on_merge_items"}, {"duration": 0.00026011466979980469, "last_call": 1509985822.932059, "sons": [], "name": "_reformat_merge_items"}, {"duration": 0.086343050003051758, "last_call": 1509985823.0184021, "sons": [], "name": "_initialise_merge_item_datamanager"}, {"duration": 0.0075159072875976562, "last_call": 1509985823.025918, "sons": [], "name": "_valide_merge_items"}, {"duration": 7.1071979999542236, "last_call": 1509985830.133116, "sons": [], "name": "_build_double_link_for_merge_items"}, {"duration": 7.226909875869751, "last_call": 1509985837.3600259, "sons": [], "name": "_compute_diff_with_existing_items"}, {"duration": 0.025546073913574219, "last_call": 1509985837.385572, "sons": [], "name": "_merge_merged_items"}, {"duration": 3.5907771587371826, "last_call": 1509985840.9763491, "sons": [], "name": "_saving_merge_items"}, {"duration": 0.00041294097900390625, "last_call": 1509985840.9767621, "sons": [], "name": "_saving_retention_data"}, {"duration": 0.00013804435729980469, "last_call": 1509985840.9769001, "sons": [], "name": "_set_sources_as_merge_done"}, {"duration": 0.0003337860107421875, "last_call": 1509985840.9772339, "sons": [], "name": "_populate_import_cache"}, {"duration": 0.00018811225891113281, "last_call": 1509985840.977422, "sons": [], "name": "end"}], "name": "beacon_merge_source"}]}'''
    # from_info = json.loads(test_post_source_modif)
    # beacon = Beacon('loaded', from_info)
    # beacon.print_time(1)
    #
    # test_post_source_modif = '''{"last_call": 1510734736.9647651, "sons": [{"duration": 8.2210350036621094, "last_call": 1510734736.9647651, "sons": [{"duration": 0.1775658130645752, "last_call": 1510734728.9212959, "sons": [], "name": "_load_source_items"}, {"duration": 0.031186103820800781, "last_call": 1510734728.952482, "sons": [], "name": "_find_items_in_cache"}, {"duration": 0.014868021011352539, "last_call": 1510734728.9674709, "sons": [{"duration": 0.00055909156799316406, "last_call": 1510734728.9665551, "sons": [{"duration": 0.00022411346435546875, "last_call": 1510734728.9527061, "sons": [], "name": "build items to merge"}, {"duration": 0.00011396408081054688, "last_call": 1510734728.9528201, "sons": [], "name": "find_linked_items"}, {"duration": 0.013735055923461914, "last_call": 1510734728.9665551, "sons": [], "name": "merge linked item"}], "name": "notificationway"}, {"duration": 0.0069208145141601562, "last_call": 1510734728.966644, "sons": [{"duration": 0.0065870285034179688, "last_call": 1510734728.9594071, "sons": [], "name": "build items to merge"}, {"duration": 0.00024485588073730469, "last_call": 1510734728.9596519, "sons": [], "name": "find_linked_items"}, {"duration": 0.0069921016693115234, "last_call": 1510734728.966644, "sons": [], "name": "merge linked item"}], "name": "host"}, {"duration": 0.0010249614715576172, "last_call": 1510734728.966732, "sons": [{"duration": 0.00080204010009765625, "last_call": 1510734728.960454, "sons": [], "name": "build items to merge"}, {"duration": 0.00013494491577148438, "last_call": 1510734728.9605889, "sons": [], "name": "find_linked_items"}, {"duration": 0.0061430931091308594, "last_call": 1510734728.966732, "sons": [], "name": "merge linked item"}], "name": "service"}, {"duration": 0.00030517578125, "last_call": 1510734728.9668291, "sons": [{"duration": 0.0001201629638671875, "last_call": 1510734728.9607091, "sons": [], "name": "build items to merge"}, {"duration": 8.7976455688476562e-05, "last_call": 1510734728.9607971, "sons": [], "name": "find_linked_items"}, {"duration": 0.0060319900512695312, "last_call": 1510734728.9668291, "sons": [], "name": "merge linked item"}], "name": "contactgroup"}, {"duration": 0.0011990070343017578, "last_call": 1510734728.966958, "sons": [{"duration": 0.00081801414489746094, "last_call": 1510734728.9616151, "sons": [], "name": "build items to merge"}, {"duration": 0.00025200843811035156, "last_call": 1510734728.9618671, "sons": [], "name": "find_linked_items"}, {"duration": 0.0050909519195556641, "last_call": 1510734728.966958, "sons": [], "name": "merge linked item"}], "name": "hostgroup"}, {"duration": 0.0026106834411621094, "last_call": 1510734728.9671819, "sons": [{"duration": 0.0019879341125488281, "last_call": 1510734728.963855, "sons": [], "name": "build items to merge"}, {"duration": 0.00039887428283691406, "last_call": 1510734728.9642539, "sons": [], "name": "find_linked_items"}, {"duration": 0.0029280185699462891, "last_call": 1510734728.9671819, "sons": [], "name": "merge linked item"}], "name": "contact"}, {"duration": 0.0015761852264404297, "last_call": 1510734728.9672821, "sons": [{"duration": 0.0011870861053466797, "last_call": 1510734728.965441, "sons": [], "name": "build items to merge"}, {"duration": 0.00028896331787109375, "last_call": 1510734728.96573, "sons": [], "name": "find_linked_items"}, {"duration": 0.0015521049499511719, "last_call": 1510734728.9672821, "sons": [], "name": "merge linked item"}], "name": "command"}, {"duration": 0.00069117546081542969, "last_call": 1510734728.9673691, "sons": [{"duration": 0.00046896934509277344, "last_call": 1510734728.9661989, "sons": [], "name": "build items to merge"}, {"duration": 0.00013518333435058594, "last_call": 1510734728.9663341, "sons": [], "name": "find_linked_items"}, {"duration": 0.0010349750518798828, "last_call": 1510734728.9673691, "sons": [], "name": "merge linked item"}], "name": "timeperiod"}], "name": "_merge_source_items"}, {"duration": 0.00012087821960449219, "last_call": 1510734728.966455, "sons": [], "name": "_link_source_items"}, {"duration": 7.915496826171875e-05, "last_call": 1510734728.96755, "sons": [], "name": "_apply_taggers_on_merge_items"}, {"duration": 0.02980804443359375, "last_call": 1510734728.9973581, "sons": [], "name": "_find_broken_precomputed_double_links"}, {"duration": 0.00028395652770996094, "last_call": 1510734728.997642, "sons": [], "name": "_reformat_merge_items"}, {"duration": 0.094753026962280273, "last_call": 1510734729.0923951, "sons": [], "name": "_initialise_merge_item_datamanager"}, {"duration": 0.0068359375, "last_call": 1510734729.099231, "sons": [], "name": "_valide_merge_items"}, {"duration": 0.012085914611816406, "last_call": 1510734729.1113169, "sons": [], "name": "_build_double_link_for_merge_items"}, {"duration": 0.0080251693725585938, "last_call": 1510734729.1193421, "sons": [], "name": "_compute_diff_with_existing_items"}, {"duration": 0.062807798385620117, "last_call": 1510734729.1821499, "sons": [], "name": "_merge_merged_items"}, {"duration": 7.7305350303649902, "last_call": 1510734736.9126849, "sons": [], "name": "_saving_merge_items"}, {"duration": 0.051105976104736328, "last_call": 1510734736.9637909, "sons": [], "name": "_saving_retention_data"}, {"duration": 0.00036406517028808594, "last_call": 1510734736.964155, "sons": [], "name": "_set_sources_as_merge_done"}, {"duration": 0.00052309036254882812, "last_call": 1510734736.964678, "sons": [], "name": "_populate_import_cache"}, {"duration": 8.7022781372070312e-05, "last_call": 1510734736.9647651, "sons": [], "name": "end"}], "name": "beacon_merge_source"}]}'''
    # from_info = json.loads(test_post_source_modif)
    # beacon = Beacon('loaded', from_info)
    # beacon.print_time(1)
