# !/usr/bin/python
# -*- coding: utf-8 -*-
# Copyright (C) 2013-2021:
# This file is part of Shinken Enterprise, all rights reserved.

import json
import time
import uuid

from shinken.misc.type_hint import TYPE_CHECKING
from shinken.thread_helper import Thread
from shinkensolutions.component.abstract_component import AbstractComponent
from shinkensolutions.data_hub.data_hub import DataHubConfig
from shinkensolutions.data_hub.data_hub_driver.abstract_data_hub_driver_file import DATA_HUB_FILE_DEFAULT_DIRECTORY
from shinkensolutions.data_hub.data_hub_driver.data_hub_driver_json import DataHubDriverConfigJson
from shinkensolutions.data_hub.data_hub_driver.data_hub_driver_mmap_file import DataHubDriverConfigMmapFile
from shinkensolutions.data_hub.data_hub_exception.data_hub_exception import DataHubItemNotFound, DataHubFatalException
from shinkensolutions.data_hub.data_hub_factory.data_hub_factory import DataHubFactory
from shinkensolutions.toolbox.box_tools_array import ToolsBoxArray

if TYPE_CHECKING:
    from shinken.log import PartLogger
    from shinken.misc.type_hint import List, Optional, Dict
    from shinkensolutions.data_hub.data_hub import DataHub
    from webui.components.configuration_component import ConfigurationComponent
    
    ItemForList = Dict
    Part = unicode


class ItemListSplitterDeletionReason(object):
    TIMEOUT = u'timeout'
    REQUEST_COMPLETED = u'request_completed'


class ItemListSplitterReturnStatus(object):
    NOT_ENOUGH_MEMORY = u'not_enough_memory'
    PART_DOES_NOT_EXIST = u'part_does_not_exist'
    ITEM_LIST_NOT_FOUND = u'item_list_not_found'


class ItemListSplitterKeys(object):
    END_VALIDITY_TIME = u'end_validity_time'
    NUMBER_OF_PART = u'number_of_parts'
    REMAINING_PARTS = u'remaining_parts'
    TIME_SPENT_TO_SPLIT = u'time_spent_to_split'
    CREATION_TIME = u'creation_time'
    TOTAL_SIZE = u'total_size'


class ItemListSplitterItemListNotFound(Exception):
    pass


class ItemListSplitterPartDoesNotExist(Exception):
    pass


class ItemListSplitterConf(object):
    def __init__(self, limit_time_to_keep_in_cache, limit_memory_to_use_in_cache, number_of_items_per_part):
        self.limit_time_to_keep_in_cache = limit_time_to_keep_in_cache
        self.limit_memory_to_use_in_cache = limit_memory_to_use_in_cache
        self.number_of_items_per_part = number_of_items_per_part


class ItemListSplitterIntoPartsComponent(AbstractComponent, Thread):
    
    def __init__(self, item_list_splitter_id, configuration_component, logger, data_hub_id, daemon_name, module_name, item_list_type=u'item list'):
        # type: (unicode, ConfigurationComponent, PartLogger, unicode, unicode, unicode, unicode) -> None
        self._id = item_list_splitter_id
        self.configuration_component = configuration_component
        self._limit_time_to_keep_in_cache = 0.0  # type: float
        self._limit_memory_to_use_in_cache = 0.0  # type: float
        self._real_limit_memory_to_use_in_cache_in_megabytes = 0.0  # type: float
        self._number_of_items_per_part = 0  # type: int
        self._item_lists_metadata_data_hub = None  # type: Optional[DataHub]
        self._parts_data_hub = None  # type: Optional[DataHub]
        self.item_list_type = item_list_type
        self._data_hub_id = data_hub_id
        self._daemon_name = daemon_name
        self._module_name = module_name
        Thread.__init__(self, logger=logger.get_sub_part(u'ITEM LIST SPLITTER'))
        self._logger_loop = self.logger.get_sub_part(u'LOOP')
    
    
    def init(self):
        # type: () -> None
        self._set_configuration()
        self._init_data_hub()
        self.start_thread()
    
    
    def _set_configuration(self):
        # type: () -> None
        configuration = self.configuration_component.get_item_list_splitter_configuration(self._id)
        self._limit_time_to_keep_in_cache = configuration.limit_time_to_keep_in_cache
        self._limit_memory_to_use_in_cache = configuration.limit_memory_to_use_in_cache * 90 / 100  # We remove ten percents to the limit, so we are safer regarding memory
        self._real_limit_memory_to_use_in_cache_in_megabytes = self._limit_memory_to_use_in_cache / 1000  # The limit displayed in log must still be the real limit
        self._number_of_items_per_part = configuration.number_of_items_per_part
    
    
    def get_thread_name(self):
        return u'broker-ils-cleaner'
    
    
    @property
    def _total_size(self):
        # type: () -> float
        if self._parts_data_hub:
            return float(self._parts_data_hub.get_total_size()) / 1000.0
        return 0
    
    
    def get_number_of_items_per_part(self):
        # type: () -> int
        return self._number_of_items_per_part
    
    
    def _init_data_hub(self):
        # type: () -> None
        self._init_meta_data_data_hub()
        self._init_export_parts_data_hub()
    
    
    def _init_meta_data_data_hub(self):
        # type: () -> None
        driver_config = DataHubDriverConfigJson(
            base_directory=DATA_HUB_FILE_DEFAULT_DIRECTORY.DEV_SHM,
            data_location_name=u'item_list_splitter__metadata',
            daemon_name=self._daemon_name,
            module_name=self._module_name
        )
        
        data_hub_config = DataHubConfig(
            data_hub_id=u'dev_shm__%s__metadata' % self._data_hub_id,
            data_hub_category=u'item_list_splitter__%s' % self._module_name,
            data_type=u'item_list_metadata',
            data_id_key_name=u'uuid',
            driver_config=driver_config,
            must_save_configuration=True
        )
        self._item_lists_metadata_data_hub = DataHubFactory.build_and_init_data_hub(self.logger, data_hub_config)
    
    
    def _init_export_parts_data_hub(self):
        # type: () -> None
        driver_config = DataHubDriverConfigMmapFile(
            base_directory=DATA_HUB_FILE_DEFAULT_DIRECTORY.DEV_SHM,
            data_location_name=u'item_list_splitter__parts',
            file_ext=u'part',
            daemon_name=self._daemon_name,
            module_name=self._module_name
        )
        
        data_hub_config = DataHubConfig(
            data_hub_id=u'dev_shm__%s__parts' % self._data_hub_id,
            data_hub_category=u'item_list_splitter__%s' % self._module_name,
            data_type=u'item_list_part',
            data_id_key_name=u'uuid',
            driver_config=driver_config,
            must_save_configuration=True
        )
        self._parts_data_hub = DataHubFactory.build_and_init_data_hub(self.logger, data_hub_config)
    
    
    @staticmethod
    def _generate_part_file_id(item_list_uuid, part_number):
        # type: (unicode, int) -> unicode
        return u'%s_part_%d' % (item_list_uuid, part_number)
    
    
    def _as_enough_free_memory(self, items_count):
        # type: (int) -> bool
        if self._total_size > self._limit_memory_to_use_in_cache:
            self.logger.warning(u'Can\'t add list of [ %d ] items. Item list splitter is already full [ %.2f/%dMB ]. Please try later' % (items_count, self._total_size / 1000.0, self._real_limit_memory_to_use_in_cache_in_megabytes))
            return False
        return True
    
    
    def _save_parts(self, item_list_uuid, items_count, parts, start_split_time):
        # type: (unicode, int, List[Part], float) -> None
        parts_count = len(parts)
        for i in range(parts_count):
            self._parts_data_hub.save_data(self._generate_part_file_id(item_list_uuid, i), parts[0])
            parts.pop(0)
        
        item_list_metadata = {
            ItemListSplitterKeys.NUMBER_OF_PART     : parts_count,
            ItemListSplitterKeys.END_VALIDITY_TIME  : int(time.time() + self._limit_time_to_keep_in_cache),
            ItemListSplitterKeys.REMAINING_PARTS    : parts_count - 1,
            ItemListSplitterKeys.TIME_SPENT_TO_SPLIT: time.time() - start_split_time,
            ItemListSplitterKeys.CREATION_TIME      : time.time(),
            ItemListSplitterKeys.TOTAL_SIZE         : self._get_total_size_of_item_list(item_list_uuid)
        }
        self._item_lists_metadata_data_hub.save_data(item_list_uuid, item_list_metadata)
        self.logger.debug(u'Adding %s with uuid [ %s ] and [ %d ] items for [ %d seconds ]. It is split in [ %d ] parts. [ %d ] %s are in memory' % (
            self.item_list_type, item_list_uuid, items_count, self._limit_time_to_keep_in_cache, parts_count, self.get_item_list_count(), self.item_list_type))
    
    
    def _delete_item_list(self, item_list_uuid, reason):
        # type: (unicode, unicode) -> None
        item_list_metadata = self._item_lists_metadata_data_hub.get_data(item_list_uuid)
        parts_id = self._parts_data_hub.get_all_data_id()
        for i in range(item_list_metadata.get(ItemListSplitterKeys.NUMBER_OF_PART, 0)):
            part_id = self._generate_part_file_id(item_list_uuid, i)
            if part_id in parts_id:
                self._parts_data_hub.remove_data_and_lock(part_id)
        self._item_lists_metadata_data_hub.remove_data_and_lock(item_list_uuid)
        self.logger.debug(u'Removing %s with uuid [ %s ] for reason [ %s ]. [ %d ] %s in memory' % (self.item_list_type, item_list_uuid, reason, self.get_item_list_count(), self.item_list_type))
    
    
    def _get_total_size_of_item_list(self, item_list_uuid):
        # type: (unicode) -> float
        _size = 0.0
        for part_uuid in self._get_all_parts_uuid(item_list_uuid):
            _size += self._parts_data_hub.get_size_of(part_uuid)
        return _size
    
    
    def _get_all_parts_uuid(self, item_list_uuid):
        # type: (unicode) -> List[unicode]
        return [part_uuid for part_uuid in self._parts_data_hub.get_all_data_id() if part_uuid.startswith(item_list_uuid)]
    
    
    def _get_part(self, item_list_uuid, part_index):
        # type: (unicode, int) -> Part
        # getting part infos
        try:
            item_list_metadata = self._item_lists_metadata_data_hub.get_data(item_list_uuid)
        except DataHubItemNotFound:
            raise ItemListSplitterItemListNotFound
        
        try:
            item_list_part = self._parts_data_hub.get_data(self._generate_part_file_id(item_list_uuid, part_index))
        except DataHubItemNotFound:
            raise ItemListSplitterPartDoesNotExist
        
        # remove part from dev shm
        self.logger.debug(u'Getting part [ %d ] from %s with uuid [ %s ]. Timeout delayed as we received a request.' % (part_index, self.item_list_type, item_list_uuid))
        if part_index > 0:
            try:
                item_list_metadata[ItemListSplitterKeys.REMAINING_PARTS] -= 1
                self._parts_data_hub.remove_data_and_lock(self._generate_part_file_id(item_list_uuid, part_index - 1))
            except DataHubFatalException:
                pass
            self.logger.debug(u'We remove part [ %d ] from the %s with uuid [ %s ] which will not be requested anymore.' % (part_index - 1, self.item_list_type, item_list_uuid))
        
        # update metadata
        item_list_metadata[ItemListSplitterKeys.END_VALIDITY_TIME] = int(time.time() + self._limit_time_to_keep_in_cache)
        self._item_lists_metadata_data_hub.save_data(item_list_uuid, item_list_metadata)
        return item_list_part
    
    
    def _get_parts_count_of_item_list(self, item_list_uuid):
        # type: (unicode) -> int
        try:
            item_list_metadata = self._item_lists_metadata_data_hub.get_data(item_list_uuid)
            return item_list_metadata.get(ItemListSplitterKeys.NUMBER_OF_PART, 0)
        except DataHubItemNotFound:
            raise ItemListSplitterItemListNotFound
    
    
    def get_part_size(self, item_list_uuid, part_index):
        # type: (unicode, int) -> int
        return self._parts_data_hub.get_size_of(self._generate_part_file_id(item_list_uuid, part_index))
    
    
    def get_remaining_parts_of_item_list(self, item_list_uuid):
        # type: (unicode) -> int
        try:
            item_list_metadata = self._item_lists_metadata_data_hub.get_data(item_list_uuid)
            return item_list_metadata.get(ItemListSplitterKeys.REMAINING_PARTS, 0)
        except DataHubItemNotFound:
            raise ItemListSplitterItemListNotFound
    
    
    def get_total_size_of_item_list(self, item_list_uuid):
        # type: (unicode) -> float
        try:
            item_list_metadata = self._item_lists_metadata_data_hub.get_data(item_list_uuid)
            return item_list_metadata.get(ItemListSplitterKeys.TOTAL_SIZE, 0.0)
        except DataHubItemNotFound:
            raise ItemListSplitterItemListNotFound
    
    
    def get_time_spent_to_split_of_item_list(self, item_list_uuid):
        # type: (unicode) -> float
        try:
            item_list_metadata = self._item_lists_metadata_data_hub.get_data(item_list_uuid)
            return item_list_metadata.get(ItemListSplitterKeys.TIME_SPENT_TO_SPLIT, 0.0)
        except DataHubItemNotFound:
            raise ItemListSplitterItemListNotFound
    
    
    def get_creation_time_of_item_list(self, item_list_uuid):
        # type: (unicode) -> float
        try:
            item_list_metadata = self._item_lists_metadata_data_hub.get_data(item_list_uuid)
            return item_list_metadata.get(ItemListSplitterKeys.CREATION_TIME, 0.0)
        except DataHubItemNotFound:
            raise ItemListSplitterItemListNotFound
    
    
    def get_item_list_count(self):
        # type: () -> int
        return self._item_lists_metadata_data_hub.get_number_of_stored_data()
    
    
    def split_item_list(self, item_list_uuid, item_list, item_list_key):
        # type: (unicode, List[ItemForList], unicode) -> None
        start_split = time.time()
        parts = [json.dumps({item_list_key: split_list}) for split_list in ToolsBoxArray.split_in_chunks(item_list, self._number_of_items_per_part)]
        self._save_parts(item_list_uuid, len(item_list), parts, start_split)
    
    
    def is_split_needed(self, item_list):
        # type: (List[ItemForList]) -> bool
        return 0 < self._number_of_items_per_part < len(item_list)
    
    
    def get_item_list_part(self, item_list_uuid, part_index):
        # type: (unicode, int) -> unicode
        try:
            return self._get_part(item_list_uuid, part_index)
        except ItemListSplitterPartDoesNotExist:
            self.logger.warning(u'Trying to get part [ %d ] of %s with uuid [ %s ] but this part does not exist' % (part_index, self.item_list_type, item_list_uuid))
            return json.dumps({
                u'export_error_status'         : ItemListSplitterReturnStatus.PART_DOES_NOT_EXIST,
                u'number_of_export_in_progress': self.get_item_list_count()
            })
        except ItemListSplitterItemListNotFound:
            self.logger.warning(u'Trying to get %s with uuid [ %s ] but this uuid does not exist' % (self.item_list_type, item_list_uuid))
            return json.dumps({
                u'export_error_status'         : ItemListSplitterReturnStatus.ITEM_LIST_NOT_FOUND,
                u'number_of_export_in_progress': self.get_item_list_count()
            })
    
    
    def delete_item_list(self, item_list_uuid):
        # type: (unicode) -> unicode
        try:
            self._delete_item_list(item_list_uuid, ItemListSplitterDeletionReason.REQUEST_COMPLETED)
            return u''
        except DataHubItemNotFound:
            self.logger.warning(u'Trying to get %s with uuid [ %s ] but this uuid does not exist' % (self.item_list_type, item_list_uuid))
            return json.dumps({
                u'export_error_status'         : ItemListSplitterReturnStatus.ITEM_LIST_NOT_FOUND,
                u'number_of_export_in_progress': self.get_item_list_count()
            })
    
    
    def item_list_can_be_split(self, item_list):
        # type: (List[ItemForList]) -> unicode
        items_count = len(item_list)
        if self.get_item_list_count() == 0:
            return u''
        
        if not self._as_enough_free_memory(items_count):
            return ItemListSplitterReturnStatus.NOT_ENOUGH_MEMORY
        
        return u''
    
    
    def split_item_list_if_needed(self, ret, item_list_key):
        # type: (Dict, unicode) -> Dict
        
        item_list = ret[item_list_key]
        if not self.is_split_needed(item_list):
            return ret
        item_list_uuid = uuid.uuid4().hex
        self.split_item_list(item_list_uuid, item_list, item_list_key)
        
        ret[item_list_key] = []
        ret[u'item_list_parts_count'] = self._get_parts_count_of_item_list(item_list_uuid)
        ret[u'item_list_uuid'] = item_list_uuid
        return ret
    
    
    def loop_turn(self):
        # type: () -> None
        if self.get_item_list_count() != 0:
            self._logger_loop.debug(u'There are actually [ %d ] %s in memory. [ %.2f/%dMB ] used' % (self.get_item_list_count(), self.item_list_type, self._total_size / 1000.0, self._real_limit_memory_to_use_in_cache_in_megabytes))
        actual_time = time.time()
        for item_list_uuid, item_list_metadata in self._item_lists_metadata_data_hub.get_all_data_with_id():
            if actual_time > item_list_metadata.get(ItemListSplitterKeys.END_VALIDITY_TIME):
                self._delete_item_list(item_list_uuid, ItemListSplitterDeletionReason.TIMEOUT)
