# !/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2021:
# This file is part of Shinken Enterprise, all rights reserved.
from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from shinken.misc.type_hint import List, Dict, Iterator, Sequence, Tuple, Any, Callable


class CacheStorageException(Exception):
    pass


class CacheItemNotFound(CacheStorageException):
    def __init__(self, object_id, message=u''):
        # type: (unicode, unicode) -> None
        if not message:
            message = u'Item with id [ %s ] not found in cache' % object_id
        self.object_id = object_id
        self.message = message


class CacheStorage(object):  # TODO (Python 3): typing.Generic
    def __init__(self):
        # type: () -> None
        self._cache_list = {}  # type: Dict[unicode, Any]
    
    
    def flush(self):
        self._cache_list.clear()
    
    
    def invalidate(self, predicate):
        # type: (Callable[[unicode, Any], bool]) -> List[unicode]
        to_delete = [object_id for object_id, object_in_cache in self._cache_list.iteritems() if predicate(object_id, object_in_cache)]
        for object_id in to_delete:
            self._cache_list.pop(object_id, None)
        return to_delete
    
    
    def invalidate_by_id(self, predicate):
        # type: (Callable[[unicode], bool]) -> List[unicode]
        to_delete = [object_id for object_id in self._cache_list.iterkeys() if predicate(object_id)]
        for object_id in to_delete:
            self._cache_list.pop(object_id, None)
        return to_delete
    
    
    def invalidate_by_object(self, predicate):
        # type: (Callable[[Any], bool]) -> List[unicode]
        to_delete = [object_id for object_id, object_in_cache in self._cache_list.iteritems() if predicate(object_in_cache)]
        for object_id in to_delete:
            self._cache_list.pop(object_id, None)
        return to_delete
    
    
    def get_all_ids(self):
        # type: () -> Sequence[unicode]
        return list(self._cache_list.iterkeys())
    
    
    def iter_objects_with_id(self, lazy):
        # type: (bool) -> Iterator[Tuple[unicode, Any]]
        return iter(self._cache_list.items() if not lazy else self._cache_list.iteritems())
    
    
    def iter_objects(self, lazy):
        # type: (bool) -> Iterator[Any]
        return iter(self._cache_list.values() if not lazy else self._cache_list.iteritems())
    
    
    def get_object(self, object_id, pop=False):
        # type: (unicode, bool) -> Any
        get_object = self._cache_list.__getitem__ if not pop else self._cache_list.pop
        try:
            object_cache = get_object(object_id)
        except (KeyError, TypeError):
            raise CacheItemNotFound(object_id)
        return object_cache
    
    
    def get_object_default(self, object_id, default, pop=False):
        # type: (unicode, Any, bool) -> Any
        try:
            return self.get_object(object_id, pop=pop)
        except CacheItemNotFound:
            return default
    
    
    def set_object_default(self, object_id, default):
        # type: (unicode, Any) -> Any
        return self._cache_list.setdefault(object_id, default)
    
    
    def store_object(self, object_id, object_to_cache):
        # type: (unicode, Any) -> None
        self._cache_list[object_id] = object_to_cache
    
    
    def delete_object(self, object_id, raising=False):
        try:
            del self._cache_list[object_id]
        except KeyError:
            if raising:
                raise CacheItemNotFound(object_id)
