#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

import inspect
import math
import os
import re
import shutil
import traceback
from os.path import abspath, dirname, join

from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict, List, Optional, Set

CFG_SEPARATOR = u';'
NEW_LINE_IN_TAG = u'(br)'
DEFAULT_FORMAT = u'───'

IGNORED_PROPERTIES = (u'module_type',)


class VERBOSE_LEVELS(object):
    NONE = 0
    ERROR = 1
    INFO = 2
    DEBUG = 3


VERBOSE_LEVEL = VERBOSE_LEVELS.NONE


def set_verbose_level(verbose_level):
    global VERBOSE_LEVEL
    VERBOSE_LEVEL = verbose_level


class ShinkenCfgFormatterException(Exception):
    pass


class FORMATTER_KEYS(object):
    SET_LINE_SIZE = u'set_line_size'
    DESCRIPTION_TITLE = u'description_title'
    DESCRIPTION_TEXT = u'description_text'
    DESCRIPTION_END_CFG_FORMAT_VERSION = u'description_end_cfg_format_version'
    DESCRIPTION_END = u'description_end'
    
    DEFINE_START = u'define_start'
    DEFINE_STOP = u'define_stop'
    
    PARAMETER_TITLE = u'parameter_title'
    PARAMETER_TITLE_MANDATORY = u'parameter_title_mandatory'
    PARAMETER_TEXT = u'parameter_text'
    PARAMETER_TEXT_WARNING = u'parameter_text_warning'
    PARAMETER_TEXT_INDENT = u'parameter_text_indent'
    PARAMETER_TEXT_DEFAULT = u'parameter_text_default'
    PARAMETER_TEXT_RECOMMENDED = u'parameter_text_recommended'
    
    PARAMETER_LIST_EXAMPLE = u'parameter_list_example'
    PARAMETER_LIST_EXAMPLE_2 = u'parameter_list_example_2'
    PARAMETER_LIST_EXAMPLE_3 = u'parameter_list_example_3'
    PARAMETER_LIST_DEFAULT = u'parameter_list_default'
    PARAMETER_LIST = u'parameter_list'
    PARAMETER_LIST_INDENT = u'parameter_list_indent'
    PARAMETER_LIST_INDENT_2 = u'parameter_list_indent_2'
    PARAMETER_LIST_INDENT_3 = u'parameter_list_indent_3'
    
    PARAMETER_VALUE = u'parameter_value'
    PARAMETER_COMMENTED = u'parameter_value_comment'
    
    OVERRIDE_FROM = u'override_from'
    OVERRIDE_TYPE = u'override_type'
    
    HEADER_1 = u'h1'
    HEADER_2 = u'h2'
    BREAK_LINE = u'br'
    SEPARATOR_PARAMETER = u'separator_parameter'
    
    SHOULD_NOT_BE_LAST_LINE = (PARAMETER_TEXT_DEFAULT, PARAMETER_TEXT_WARNING, PARAMETER_TEXT_RECOMMENDED, PARAMETER_LIST_EXAMPLE, PARAMETER_LIST, PARAMETER_LIST_INDENT)
    ALL_PARAMETER_KEYS = (PARAMETER_VALUE, PARAMETER_COMMENTED)


class FORMATTER_PATHS(object):
    SCRIPT_FOLDER = dirname(abspath(__file__))
    GENERATED_CFG_FOLDER = join(SCRIPT_FOLDER, u'cfg')
    CFG_DEF_FOLDER = join(SCRIPT_FOLDER, u'cfg_def')
    PREVIOUS_CFG_DEF_FOLDER = join(SCRIPT_FOLDER, u'previous_cfg_def')


class FORMATTER_FILES(object):
    ARCHITECTURE_EXPORT = u'architecture-export'
    BROKER_MODULE_LIVEDATA = u'broker-module-livedata'
    SOURCE_DISCOVERY = u'discovery'
    EVENT_MANAGER_READER = u'event_manager_reader'
    EVENT_MANAGER_WRITER = u'event_manager_writer'
    GRAPHITE = u'graphite'
    LIVEDATA_MODULE_SLA_PROVIDER = u'livedata-module-sla-provider'
    MODULE_MONGO = u'mongodb'
    MONGOS_SOCKET_KEEP_ALIVE = u'mongos_socket_keep_alive'
    MODULE_RETENTION_MONGO = u'retention_mongodb'
    SLA = u'sla'
    SYNCUI = u'syncui'
    WEBUI = u'webui'
    WEBUI_OVERRIDE = u'webui_cfg_override'
    WEBUI_MODULE_SERVICE_WEATHER = u'webui_module_service_weather'
    WEBUI_MODULE_SERVICE_WEATHER__DEFAULT_GRID_LAYOUT = u'webui_module_service_weather__default_grid_layout'
    WEBUI_MODULE_SERVICE_WEATHER__DEFAULT_INFO_BAR_LAYOUT = u'webui_module_service_weather__default_info_bar_layout'
    WEBUI_MODULE_SERVICE_WEATHER__DEFAULT_WIDGETS_LAYOUT = u'webui_module_service_weather__default_widgets_layout'
    WEBUI_MODULE_SERVICE_WEATHER__DEFAULT_NOTIFICATIONS_SETTINGS = u'webui_module_service_weather__default_notifications_settings'
    LISTENER_REST = u'listener-rest'
    
    INPUT_FILE_NAMES = (
        ARCHITECTURE_EXPORT,
        BROKER_MODULE_LIVEDATA,
        SOURCE_DISCOVERY,
        EVENT_MANAGER_READER,
        EVENT_MANAGER_WRITER,
        GRAPHITE,
        LIVEDATA_MODULE_SLA_PROVIDER,
        MODULE_MONGO,
        MONGOS_SOCKET_KEEP_ALIVE,
        MODULE_RETENTION_MONGO,
        SLA,
        SYNCUI,
        WEBUI,
        WEBUI_OVERRIDE,
        WEBUI_MODULE_SERVICE_WEATHER,
        WEBUI_MODULE_SERVICE_WEATHER__DEFAULT_GRID_LAYOUT,
        WEBUI_MODULE_SERVICE_WEATHER__DEFAULT_INFO_BAR_LAYOUT,
        WEBUI_MODULE_SERVICE_WEATHER__DEFAULT_WIDGETS_LAYOUT,
        WEBUI_MODULE_SERVICE_WEATHER__DEFAULT_NOTIFICATIONS_SETTINGS,
        LISTENER_REST,
    )
    
    CFG = u'cfg'
    CFG_DEF = u'cfg_def'


DEFAULT_SIZE_LINE = 107
ALIGN_VALUES = 50
DESCRIPTION_LENGTH = 80

need_separator = False
last_line_type = False

OVERRIDE_TYPE_KEY = u'__OVERRIDE_TYPE__'


def set_line_size(_buffer, data):
    # type: (List[unicode], unicode) -> None
    global SIZE_LINE
    SIZE_LINE = int(data)


def _append_to_buffer(_buffer, data, type_line):
    # type: (List[unicode], unicode, unicode) -> None
    global last_line_type
    _buffer.append(data)
    last_line_type = type_line


def _separator_description(_buffer):
    # type: (List[unicode]) -> None
    text = DESCRIPTION_LENGTH * u'='
    _append_to_buffer(_buffer, u'#%s' % text, inspect.stack()[0][3])


def _separator_default(_buffer, _=u''):
    # type: (List[unicode], unicode) -> None
    global last_line_type
    if last_line_type != inspect.stack()[0][3]:
        _append_to_buffer(_buffer, u'', inspect.stack()[0][3])


def separator_parameter(_buffer):
    # type: (List[unicode]) -> None
    global last_line_type
    _type_line = inspect.stack()[0][3]
    if last_line_type != _type_line and last_line_type not in FORMATTER_KEYS.SHOULD_NOT_BE_LAST_LINE:
        parameter_text(_buffer, u'', line_type=inspect.stack()[0][3])


def separator_parameter_value(_buffer):
    # type: (List[unicode]) -> None
    global last_line_type
    _type_line = inspect.stack()[0][3]
    if last_line_type in (
            _type_line, FORMATTER_KEYS.PARAMETER_TEXT_DEFAULT, FORMATTER_KEYS.PARAMETER_TEXT_WARNING, FORMATTER_KEYS.PARAMETER_TEXT_RECOMMENDED, FORMATTER_KEYS.PARAMETER_LIST_EXAMPLE, FORMATTER_KEYS.PARAMETER_LIST,
            FORMATTER_KEYS.PARAMETER_LIST_INDENT):
        _type_line = FORMATTER_KEYS.SEPARATOR_PARAMETER
    parameter_text(_buffer, u'', line_type=_type_line)


def description_title(_buffer, data):
    # type: (List[unicode], unicode) -> None
    _separator_description(_buffer)
    description_text(_buffer, data)
    _separator_description(_buffer)


def description_text(_buffer, data):
    # type: (List[unicode], unicode) -> None
    _append_to_buffer(_buffer, u'# %s' % data, inspect.stack()[0][3])


def description_end_cfg_format_version(_buffer, data):
    # type: (List[unicode], unicode) -> None
    _append_to_buffer(_buffer, u'# CFG_FORMAT_VERSION %s' % data, inspect.stack()[0][3])
    _separator_description(_buffer)


def description_end(_buffer, data):
    # type: (List[unicode], unicode) -> None
    description_text(_buffer, data)
    _separator_description(_buffer)


def define_start(_buffer, data):
    # type: (List[unicode], unicode) -> None
    _separator_default(_buffer)
    _append_to_buffer(_buffer, u'define %s {' % data, inspect.stack()[0][3])
    _separator_default(_buffer)


def define_stop(_buffer, _):
    # type: (List[unicode], unicode) -> None
    _separator_default(_buffer)
    _append_to_buffer(_buffer, u'}', inspect.stack()[0][3])
    _separator_default(_buffer)


def parameter_title(_buffer, data):
    # type: (List[unicode], unicode) -> None
    _separator_default(_buffer)
    parameter_text(_buffer, data, inspect.stack()[0][3])


def parameter_title_mandatory(_buffer, data):
    # type: (List[unicode], unicode) -> None
    _size = SIZE_LINE - len(u' [ MANDATORY ] ') - 2 * len(DEFAULT_FORMAT) - len(data) - 3
    _text_to_add = _size * u' '
    parameter_title(_buffer, u'%s%s [ MANDATORY ]' % (data, _text_to_add))


def parameter_text(_buffer, data, line_type=u''):
    # type: (List[unicode], unicode, unicode) -> None
    prefix_structure = u'' if line_type == FORMATTER_KEYS.SEPARATOR_PARAMETER else DEFAULT_FORMAT
    line_type = line_type or inspect.stack()[0][3]
    space_end = (SIZE_LINE - len(u'# %s  %s' % (prefix_structure, DEFAULT_FORMAT)) - len(data)) * u' '
    _append_to_buffer(_buffer, u'    # %s %s%s %s' % (prefix_structure, data, space_end, DEFAULT_FORMAT), line_type)


def override_from(_buffer, data):
    # type: (List[unicode], unicode) -> None
    line_type = inspect.stack()[0][3]
    _append_to_buffer(_buffer, u'    [OVERRIDE_FROM] %s' % data, line_type)


def override_type(_buffer, data):
    # type: (List[unicode], unicode) -> None
    line_type = inspect.stack()[0][3]
    size_line = SIZE_LINE - len(u'# ── #')
    _separator_default(_buffer)
    _append_to_buffer(_buffer, u'    # ─%s─ #' % (u'─' * size_line), inspect.stack()[0][3])
    _append_to_buffer(_buffer, u'    # This comment is used by Shinken to recognize this file, please do not edit or remove it.', inspect.stack()[0][3])
    _append_to_buffer(_buffer, u'    # If done so, several parts of Shinken, like sanatize, may not work properly.', inspect.stack()[0][3])
    _append_to_buffer(_buffer, u'    # %s %s' % (OVERRIDE_TYPE_KEY, data.strip()), line_type)
    _append_to_buffer(_buffer, u'    # ─%s─ #' % (u'─' * size_line), inspect.stack()[0][3])
    _separator_default(_buffer)


def parameter_text_warning(_buffer, data):
    # type: (List[unicode], unicode) -> None
    parameter_text_indent(_buffer, u'/!\\ %s' % data, line_type=FORMATTER_KEYS.PARAMETER_TEXT_WARNING)


def parameter_text_default(_buffer, data):
    # type: (List[unicode], unicode) -> None
    separator_parameter(_buffer)
    split = data.split(CFG_SEPARATOR)
    nb_split = len(split)
    prefix = formatter_prefix(data, is_default=True)
    tag = u''
    if nb_split == 3:
        value = u'%s' % split[1]
        tag = split[2]
    elif nb_split == 2:
        value = split[0]
        tag = split[1]
    else:
        value = split[0]
    
    formatter_values(_buffer, prefix=prefix, value=value, tag=tag, last_line_type=inspect.stack()[0][3])


def parameter_text_recommended(_buffer, data):
    # type: (List[unicode], unicode) -> None
    parameter_list_indent(_buffer, u'Recommended : %s' % data, 2)


def parameter_list_example(_buffer, data, nb_indent=1):
    # type: (List[unicode], unicode, int) -> None
    parameter_list_indent(_buffer, u'Example : %s' % data, nb_indent=nb_indent, line_type=FORMATTER_KEYS.PARAMETER_LIST_EXAMPLE)


def parameter_list_example_2(_buffer, data):
    # type: (List[unicode], unicode) -> None
    parameter_list_example(_buffer, data, 2)


def parameter_list_example_3(_buffer, data):
    # type: (List[unicode], unicode) -> None
    parameter_list_example(_buffer, data, 3)


def parameter_list(_buffer, data, suffix=u''):
    # type: (List[unicode], unicode, unicode) -> None
    separator_parameter(_buffer)
    split = data.split(CFG_SEPARATOR)
    nb_split = len(split)
    prefix = formatter_prefix(data)
    tag = u''
    if nb_split == 3:
        value = split[1]
        tag = split[2]
    elif nb_split == 2:
        value = split[0]
        tag = split[1]
    else:
        value = split[0]
    formatter_values(_buffer, prefix=prefix, value=value, tag=tag, suffix=suffix, last_line_type=inspect.stack()[0][3])


def parameter_list_indent(_buffer, data, nb_indent=1, line_type=FORMATTER_KEYS.PARAMETER_LIST_INDENT):
    # type: (List[unicode], unicode, int, unicode) -> None
    space_start = (int(nb_indent) * 5) * u' '
    prefix_indent = u' ->' if line_type == FORMATTER_KEYS.PARAMETER_LIST_INDENT else u''
    parameter_text(_buffer, u'%s%s %s' % (space_start, prefix_indent, data), line_type=line_type)


def parameter_text_indent(_buffer, data, nb_indent=1, line_type=FORMATTER_KEYS.PARAMETER_TEXT_INDENT):
    # type: (List[unicode], unicode, int, unicode) -> None
    space_start = (int(nb_indent) * 5) * u' '
    parameter_text(_buffer, u'%s %s' % (space_start, data), line_type=line_type)


def parameter_list_indent_2(_buffer, data):
    # type: (List[unicode], unicode) -> None
    parameter_list_indent(_buffer, data, 2)


def parameter_list_indent_3(_buffer, data):
    # type: (List[unicode], unicode) -> None
    parameter_list_indent(_buffer, data, 3)


def parameter_value(_buffer, data, is_commented=False):
    # type: (List[unicode], unicode, bool) -> None
    separator_parameter_value(_buffer)
    prop_name, value = data.split(CFG_SEPARATOR, 1)
    space = max((ALIGN_VALUES + 2 - len(prop_name)), 1) * u' '
    
    _append_to_buffer(_buffer, u'    %s%s%s%s' % (u'# ' if is_commented else u'', prop_name, space, value), inspect.stack()[0][3])
    _separator_default(_buffer)


def parameter_value_comment(_buffer, data):
    # type: (List[unicode], unicode) -> None
    parameter_value(_buffer, data, is_commented=True)


def formatter_h1(_buffer, data):
    # type: (List[unicode], unicode) -> None
    size_line = SIZE_LINE - len(u'# ┌┐ #')
    space_before = 1
    space_after = 4
    before = u'─' * int(math.floor((size_line - len(data) - (space_after + space_before) * 2) / 2.0))
    after = u'─' * int(math.ceil((size_line - len(data) - (space_after + space_before) * 2) / 2.0))
    _separator_default(_buffer)
    _append_to_buffer(_buffer, u'    # ┌%s┐ #' % (u'─' * size_line), inspect.stack()[0][3])
    _append_to_buffer(_buffer, u'    # │%s%s%s%s%s%s%s│ #' % (' ' * space_before, before, ' ' * space_after, data.upper(), ' ' * space_after, after, ' ' * space_before), inspect.stack()[0][3])
    _append_to_buffer(_buffer, u'    # └%s┘ #' % (u'─' * size_line), inspect.stack()[0][3])
    _separator_default(_buffer)


def formatter_h2(_buffer, data):
    # type: (List[unicode], unicode) -> None
    size_line = 58
    space_before = 1
    space_after = 2
    before = u'─' * int(math.floor((size_line - len(data) - (space_after + space_before) * 2) / 2.0))
    after = u'─' * (int(math.ceil((size_line - len(data) - (space_after + space_before) * 2) / 2.0)) + SIZE_LINE - size_line - 2)
    _separator_default(_buffer)
    _append_to_buffer(_buffer, u'    #%s%s%s%s%s%s%s#' % (' ' * space_before, before, ' ' * space_after, data, ' ' * space_after, after, ' ' * space_before), inspect.stack()[0][3])
    _separator_default(_buffer)


def formatter_values(_buffer, prefix=u'', value=u'', tag=u'', suffix=u'', last_line_type=u''):
    # type: (List[unicode], unicode, unicode, unicode, unicode, unicode) -> None
    last_line_type = last_line_type or inspect.stack()[0][3]
    
    new_lines = []
    next_line_align = 0
    next_line_align_tag = 0
    next_line_tag = u''
    if NEW_LINE_IN_TAG in tag:
        tag, next_line = tag.split(NEW_LINE_IN_TAG)
        tag = u' ( %s' % tag
        next_line_tag = u' %s )' % next_line
        next_line_align_tag = len(value) + len(prefix)
    elif tag:
        tag = u' ( %s )' % tag
    
    if NEW_LINE_IN_TAG in value:
        new_lines = value.split(NEW_LINE_IN_TAG)
        value = new_lines.pop(0)
        next_line_align = len(prefix)
    
    if new_lines:
        space_end = (SIZE_LINE - len(u'#  ───') - len(prefix) - len(value) - len(tag)) * u' '
        _append_to_buffer(_buffer, u'    # %s%s%s%s ───' % (prefix, value, tag, space_end), last_line_type)
        for _line in new_lines:
            space_end = (SIZE_LINE - len(u'#  ───') - next_line_align - len(_line) - len(suffix)) * u' '
            _append_to_buffer(_buffer, u'    # %s%s%s%s ───' % (next_line_align * u' ', _line, suffix, space_end), last_line_type)
    else:
        space_end = (SIZE_LINE - len(u'#  ───') - len(prefix) - len(value) - len(tag) - len(suffix)) * u' '
        _append_to_buffer(_buffer, u'    # %s%s%s%s%s ───' % (prefix, value, tag, suffix, space_end), last_line_type)
        if next_line_tag:
            space_end = (SIZE_LINE - len(u'#  ───') - next_line_align_tag - len(next_line_tag) - len(suffix) - 2) * u' '
            _append_to_buffer(_buffer, u'    #   %s%s%s%s ───' % (next_line_align_tag * u' ', next_line_tag, suffix, space_end), last_line_type)


def formatter_prefix(data, is_default=False, _=False):
    # type: (unicode, bool, bool) -> unicode
    split = data.split(CFG_SEPARATOR)
    nb_split = len(split)
    prefix = u''
    if nb_split == 3:
        prefix = split[0]
    
    if is_default and nb_split > 2:
        prefix = u'%s          Default : %s => ' % (u'', prefix)
    elif is_default:
        prefix = u'%s          Default : %s' % (u'', prefix)
    elif nb_split > 2:
        prefix = u'%s          ...     : %s => ' % (u'', prefix)
    else:
        prefix = u'%s          ...     : %s' % (u'', prefix)
    
    return prefix


FORMATTER_FUNCTIONS = {
    FORMATTER_KEYS.SET_LINE_SIZE                     : set_line_size,
    FORMATTER_KEYS.DESCRIPTION_TITLE                 : description_title,
    FORMATTER_KEYS.DESCRIPTION_TEXT                  : description_text,
    FORMATTER_KEYS.DESCRIPTION_END_CFG_FORMAT_VERSION: description_end_cfg_format_version,
    FORMATTER_KEYS.DESCRIPTION_END                   : description_end,
    
    FORMATTER_KEYS.DEFINE_START                      : define_start,
    FORMATTER_KEYS.DEFINE_STOP                       : define_stop,
    
    FORMATTER_KEYS.PARAMETER_TITLE                   : parameter_title,
    FORMATTER_KEYS.PARAMETER_TITLE_MANDATORY         : parameter_title_mandatory,
    FORMATTER_KEYS.PARAMETER_TEXT                    : parameter_text,
    FORMATTER_KEYS.PARAMETER_TEXT_WARNING            : parameter_text_warning,
    FORMATTER_KEYS.PARAMETER_TEXT_INDENT             : parameter_text_indent,
    FORMATTER_KEYS.PARAMETER_TEXT_DEFAULT            : parameter_text_default,
    FORMATTER_KEYS.PARAMETER_TEXT_RECOMMENDED        : parameter_text_recommended,
    
    FORMATTER_KEYS.PARAMETER_LIST_EXAMPLE            : parameter_list_example,
    FORMATTER_KEYS.PARAMETER_LIST_EXAMPLE_2          : parameter_list_example_2,
    FORMATTER_KEYS.PARAMETER_LIST_EXAMPLE_3          : parameter_list_example_3,
    FORMATTER_KEYS.PARAMETER_LIST_DEFAULT            : parameter_text_default,
    FORMATTER_KEYS.PARAMETER_LIST                    : parameter_list,
    FORMATTER_KEYS.PARAMETER_LIST_INDENT             : parameter_list_indent,
    FORMATTER_KEYS.PARAMETER_LIST_INDENT_2           : parameter_list_indent_2,
    FORMATTER_KEYS.PARAMETER_LIST_INDENT_3           : parameter_list_indent_3,
    
    FORMATTER_KEYS.PARAMETER_VALUE                   : parameter_value,
    FORMATTER_KEYS.PARAMETER_COMMENTED               : parameter_value_comment,
    
    FORMATTER_KEYS.OVERRIDE_FROM                     : override_from,
    FORMATTER_KEYS.OVERRIDE_TYPE                     : override_type,
    
    FORMATTER_KEYS.HEADER_1                          : formatter_h1,
    FORMATTER_KEYS.HEADER_2                          : formatter_h2,
    FORMATTER_KEYS.BREAK_LINE                        : _separator_default,
}


def _read_file_cfg_def(file_name, is_old_cfg=False):
    # type: (unicode, bool) -> unicode
    if is_old_cfg:
        folder_path = FORMATTER_PATHS.PREVIOUS_CFG_DEF_FOLDER
    else:
        folder_path = FORMATTER_PATHS.CFG_DEF_FOLDER
    file_path = join(folder_path, u'%s.%s' % (file_name, FORMATTER_FILES.CFG_DEF))
    if not os.path.isfile(file_path):
        raise ShinkenCfgFormatterException(u'This provided file name %s does not have a cfg_def file associated' % file_path)
    with open(file_path, u'r') as fd:
        ret = fd.read()
    ret.decode(u'utf-8')
    return ret


def _get_cfg_def_file_path_if_exists(cfg_def_without_ext, is_old_cfg=False):
    # type: (unicode, bool) -> Optional[unicode]
    folder_path = FORMATTER_PATHS.PREVIOUS_CFG_DEF_FOLDER if is_old_cfg else FORMATTER_PATHS.CFG_DEF_FOLDER
    file_path = join(folder_path, u'%s.%s' % (cfg_def_without_ext, FORMATTER_FILES.CFG_DEF))
    return file_path if os.path.isfile(file_path) else None


def does_cfg_def_file_exists(file_name_without_ext, is_old_cfg=False):
    # type: (unicode, bool) -> bool
    return bool(_get_cfg_def_file_path_if_exists(file_name_without_ext, is_old_cfg))


def get_cfg_def_name_from_override_file_name(file_path):
    # type: (unicode) -> Optional[unicode]
    
    if not os.path.isfile(file_path):
        return None
    
    with open(file_path, u'r') as override_file:
        # The override type should be at the top of the file, we do not want to parse the whole file at once by using "override_file.read()"
        for line in override_file.readlines():
            
            if OVERRIDE_TYPE_KEY.decode(u'utf-8', u'ignore') not in line.decode(u'utf-8', u'ignore'):
                continue
            
            # Our line contains the override type parameter key ! Getting it
            _, override_type_found = line.split(OVERRIDE_TYPE_KEY, 1)
            return override_type_found.strip()
    
    # The file has been fully parsed, but we did not find any "override_type_key"
    return None


def does_cfg_def_contains_key(cfg_def_name, key_to_search):
    # type: (unicode, unicode) -> bool
    cfg_def_path = _get_cfg_def_file_path_if_exists(cfg_def_name)
    
    if not cfg_def_path:
        return False
    
    # Matching all lines starting with our keyword and followed by spaces and the CFG delimiter and then anything
    # It will also work if the keyword itself has spaces / tabs before it
    # Example : KEYWORD    ;the_value_of_the_keyword
    key_regex = re.compile(r'^\s*%s\s*%s.*$' % (key_to_search, CFG_SEPARATOR), re.MULTILINE)
    with open(cfg_def_path, u'r') as cfg_def_file:
        cfg_def_content = cfg_def_file.read()
    
    return bool(key_regex.match(cfg_def_content))


def write_file(file_name, content, append_file=False):
    # type: (unicode, unicode, bool) -> None
    content = content.replace(u'\r\n', u'\n').encode(u'utf-8')
    if append_file:
        write_mode = u'a'
    else:
        write_mode = u'wb+'
    with open(file_name, write_mode) as fd:
        if append_file:
            fd.write('\n')
        fd.write(content)


def format_cfg_def(cfg_def):
    # type: (unicode) -> unicode
    
    global SIZE_LINE
    SIZE_LINE = DEFAULT_SIZE_LINE
    
    _buffer = []
    index = 0
    for raw_line in cfg_def.splitlines():
        index = index + 1
        if not raw_line:
            continue
        try:
            op, data = raw_line.split(CFG_SEPARATOR, 1)
        except ValueError:
            op = raw_line.split(CFG_SEPARATOR, 1)[0]
            data = u''
        
        op = op.strip()
        try:
            FORMATTER_FUNCTIONS.get(op)(_buffer, data)
        except TypeError:
            print u'Formatter key [ %s ] does not exist :' % op
            print raw_line
            traceback.print_exc()
        except Exception:
            print u'error in line : '
            print raw_line
            traceback.print_exc()
    
    return u'\n'.join(_buffer)


def _replace_def_with_conf_values(cfg_def, conf, cfg_params_new_name_mapping=None):
    # type: (unicode, Dict, Optional[Dict[unicode, unicode]]) -> unicode
    file_lines = cfg_def.splitlines()
    computed_conf = []
    # This flag here is set to True when encountering a "define stop" in cfg_def file, in this case we need to append this key to the result
    # In overrides types of cfg_def, we do not have this key
    need_to_stop_definition = False
    for line in file_lines:
        line_data = line.split(CFG_SEPARATOR)
        
        # The line does not contain any "data" or is an empty line
        if len(line_data) == 1:
            # If we find a "define_stop" here, we want to remember that we saw one to append it at the end of the file because we sometime want to add some treatment just before (like overrides)
            if line.strip() == FORMATTER_KEYS.DEFINE_STOP:
                need_to_stop_definition = True
            else:
                computed_conf.append(line)
            
            continue
        
        formatter_file_key = line_data[0].strip()
        # The line parsed is a blank line
        if not formatter_file_key:
            computed_conf.append(u'')
            continue
        
        parameter_name = line_data[1].strip()
        # The current line is a parameter or a commented parameter
        if formatter_file_key in FORMATTER_KEYS.ALL_PARAMETER_KEYS:
            
            # This key should not be replaced
            if formatter_file_key in IGNORED_PROPERTIES:
                computed_conf.append(line)
            
            # The name of the parameter should be renamed
            if cfg_params_new_name_mapping:
                parameter_name = cfg_params_new_name_mapping.get(parameter_name, parameter_name)
                if isinstance(parameter_name, list):
                    parameter_name = next((param_name for param_name in conf.keys() if param_name in parameter_name), None)
            
            conf_value = conf.get(parameter_name, None)
            # The loaded configuration does not override the parameter, adding it to result without treatment
            if conf_value is None:
                computed_conf.append(line)
            # The parameter is found in the loaded conf, we need to replace the default value
            else:
                # Sometimes the parameter is a list of only one element because of the way we are parsing the configuration
                if isinstance(conf_value, list):
                    conf_value = conf_value[0]
                line_data[2] = conf_value
                # The parameter has been changed by the user, we want to uncomment this in the new configuration file
                if formatter_file_key == FORMATTER_KEYS.PARAMETER_COMMENTED:
                    line_data[0] = FORMATTER_KEYS.PARAMETER_VALUE
                computed_conf.append(CFG_SEPARATOR.join(line_data))
        
        elif formatter_file_key == FORMATTER_KEYS.OVERRIDE_FROM:
            # We want to ignore the override_from keys because we will treat them separately at the end
            continue
        
        # The line is not a parameter, adding it to result
        else:
            computed_conf.append(line)
    
    # Parsing the configuration overrides to update with the new override paths
    overrides = conf.get(u'overrides', [])
    if overrides:
        for override_conf in overrides:
            imported_from = override_conf.get(u'imported_from', u'')
            computed_conf.append(u'%s%s%s' % (FORMATTER_KEYS.OVERRIDE_FROM, CFG_SEPARATOR, imported_from))
    
    # We encountered a "define_stop" key, that is probably because we are not in an override file, so we need to close the definition.
    if need_to_stop_definition:
        computed_conf.append(FORMATTER_KEYS.DEFINE_STOP)
    
    return u'\n'.join(computed_conf)


def get_parameters_from_formatter_key(cfg_def, formatter_key):
    # type: (unicode, unicode) -> List[unicode]
    file_lines = cfg_def.splitlines()
    parameters = []
    
    for line in file_lines:
        line_data = line.split(CFG_SEPARATOR)
        if line_data[0].strip() == formatter_key:
            parameters.append(line_data[1])
    
    return parameters


def format_file_from_conf(file_name, conf, cfg_params_new_name_mapping=None, is_old_cfg=False):
    # type: (unicode, Dict, Optional[Dict[unicode, unicode]], bool) -> unicode
    cfg_def = _read_file_cfg_def(file_name, is_old_cfg)
    computed_cfg_def = _replace_def_with_conf_values(cfg_def, conf, cfg_params_new_name_mapping)
    return computed_cfg_def


def format_config_file(config, cfg_params_new_name_mapping, already_written_paths, cfg_def_to_use, is_old_cfg=False):
    # type: (Dict, Optional[Dict[unicode, unicode]], Set[unicode],unicode, bool) -> None
    imported_from = config.get(u'imported_from', u'')
    if isinstance(imported_from, list):
        imported_from = imported_from[0]
    
    if u':' not in imported_from:
        file_path = imported_from
    else:
        file_path, _ = imported_from.strip().rsplit(u':', 1)
    
    cfg_def_file = format_file_from_conf(cfg_def_to_use, config, cfg_params_new_name_mapping, is_old_cfg)
    cfg_file = format_cfg_def(cfg_def_file)
    
    # If we detect that the file we currently want to use has already been written, we want to append the content and not override it.
    # This is because the file contains multiple module definition.
    if file_path in already_written_paths:
        write_file(file_path, cfg_file, append_file=True)
    else:
        # Will override the existing file
        write_file(file_path, cfg_file)
    
    already_written_paths.add(file_path)


def run(file_name):
    # type: (unicode) -> None
    if VERBOSE_LEVEL == VERBOSE_LEVELS.INFO:
        print u'formatting file %s' % file_name
    cfg_def = _read_file_cfg_def(file_name)
    _buffer = format_cfg_def(cfg_def)
    file_path = join(FORMATTER_PATHS.GENERATED_CFG_FOLDER, u'%s.%s' % (file_name, FORMATTER_FILES.CFG))
    write_file(file_path, _buffer)


def format_all_files():
    # type: () -> None
    files = os.listdir(FORMATTER_PATHS.CFG_DEF_FOLDER)
    
    try:
        shutil.rmtree(FORMATTER_PATHS.GENERATED_CFG_FOLDER)
    except Exception:
        pass
    os.makedirs(FORMATTER_PATHS.GENERATED_CFG_FOLDER)
    
    for _file in files:
        _file_name, _ = os.path.splitext(_file)
        if _file_name in FORMATTER_FILES.INPUT_FILE_NAMES:
            run(_file_name)
