#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019:
# This file is part of Shinken Enterprise, all rights reserved.
from shinken.misc.type_hint import TYPE_CHECKING
from ..constant import FRONT_HTML_FORMAT
from ..object.counter import Counter
from ..object.empty_object import EmptyObject
from ..object.shinken_object import ShinkenObject
from ..component.button.generic_button import GenericButton
from ..object.html.tag import TagHtml

if TYPE_CHECKING:
    from shinkensolutions.api.synchronizer import TranslatePart
    from shinken.misc.type_hint import NoReturn


class InputStringComp(ShinkenObject):
    def __init__(self, translate=None, name=u''):
        # type: (TranslatePart, unicode)-> None
        super(InputStringComp, self).__init__(translate)
        self.name = name
        self.type = 'text'
        self.value = ''
        self.place_holder = ''
        self.is_protected = False
        self.data_type = False
        self.is_lock = False
        self.extra_class = ''
        self.extra_class_table = ''
        self.extra_attribute = ''
        self.extra_attribute_input = ''
        # self.warning_counter = Counter('shinken-error-counter shinicon-cross-circle')
        # self.error_counter = Counter('shinken-warning-counter shinicon-cross-circle')
        self.user_counter = EmptyObject()
        self.total_counter = EmptyObject()
        self.clean_button = None
    
    
    def set_value(self, to_set):
        # type: (unicode)-> NoReturn
        self.value = to_set
    
    
    def set_type(self, to_set):
        # type: (str)-> NoReturn
        self.type = to_set
    
    
    def set_place_holder(self, to_set):
        # type: (unicode)-> NoReturn
        self.place_holder = to_set
    
    
    def add_clean_button(self):
        # type: ()-> NoReturn
        self.clean_button = GenericButton(name='input-clean-button', label='<span class="shinkon-broom"></span>')
        self.clean_button.set_tag(TagHtml('div', internal_class='shinken-button-container'))
        self.clean_button.add_tooltip()
        self.clean_button.tooltip.set_html(self._('main.action.input_clean_button_tooltip'))
    
    
    def add_user_counter(self):
        # type: ()-> NoReturn
        self.user_counter = Counter('shinken-user-set-counter')
        self.total_counter = Counter('shinken-window-total-counter')

    # *****************************   HTML  ************************************************************#
    def get_data_name_html(self):
        # type: ()-> NoReturn
        if self.name:
            return 'data-name="%s"' % self.name
        return ''
    
        
    def get_html(self, extra_content='', html_format=FRONT_HTML_FORMAT.NORMAL, is_disabled=False, is_lock=False):
        # type: (str, str, bool, bool)-> str
        focus_html = ''
        if self.is_protected:
            focus_html = ' onfocus="MANAGER.EventManager.onFocus(this)" onfocusout="MANAGER.EventManager.onFocusOut(event)"'
        if type(self.value) == str or isinstance(self.value, unicode):
            value = self.value.replace("'", "&apos;")
        else:
            value = str(self.value)
        _to_return = []
        
        if value == '':
            self.extra_class_table += ' shinken-empty-input'
        
        _to_return.append('''<table class="shinken-input-table shinken-width-100 %s" %s %s><tr>''' % (self.extra_class_table, self.tooltip.get_html(), '''data-url-param-key="%s" test''' % self.url_param_key if self.url_param_key else ''))
        if is_lock:
            _to_return.append(self.get_html_lock())
        _to_return.append('''   <td class="shinken-input-container-cell" %s>''' % self.extra_attribute)
        if html_format == FRONT_HTML_FORMAT.NOT_EDITABLE:
            _to_return.append('''<div class="shinken-label shinken-input"''')
            _to_return.append(''' %s %s ''' % (self.extra_attribute_input, focus_html))
            _to_return.append('''> %s </div>''' % self.value)
        else:
            _read_only_attribute = 'readonly' if is_disabled else ''
            _to_return.append('''       <div class="shinken-input-container">''')
            if self.type == 'text' or self.type == 'password':
                _value = self._('element.password_protected') if self.type == 'password' else self.value
                _to_return.append('''<input class="shinken-input %s" type="%s" value="%s" %s''' % (self.extra_class, self.type, _value, self.get_data_name_html()))
                _to_return.append('''placeholder="%s" oncut="MANAGER.EventManager.onChangeOnInput(event)" onpaste="MANAGER.EventManager.onChangeOnInput(event)" %s onkeyup="MANAGER.EventManager.onChangeOnInput(event)" %s ''' % (
                    self.place_holder, self.extra_attribute_input, focus_html))
                _to_return.append('''onclick="MANAGER.EventManager.clickOnInput(event)" %s> ''' % _read_only_attribute)
            elif self.type == 'textarea':
                _to_return.append('''<textarea rows="5" cols="80" class="shinken-input %s" %s data-type="string" ''' % (self.extra_class, self.get_data_name_html()))
                _to_return.append(
                    '''type="text" value="%s" oncut="MANAGER.EventManager.onChangeOnInput(event)" onpaste="MANAGER.EventManager.onChangeOnInput(event)" onkeyup="MANAGER.EventManager.onChangeOnInput(event)" shi-tip-placement="float-top-left"''' % self.value)
                _to_return.append('''placeholder="%s" %s %s>%s</textarea >''' % (self.place_holder, focus_html, _read_only_attribute, self.value))
                _to_return.append('''%s''' % extra_content)
            _to_return.append('''</div>''')
        # _to_return.append('''%s''' % self.error_counter.get_html())
        # _to_return.append('''%s''' % self.warning_counter.get_html())
        _to_return.append('''%s''' % self.user_counter.get_html())
        _to_return.append('''%s''' % self.total_counter.get_html())
        _to_return.append('''</td>''')
        if self.clean_button:
            _to_return.append('''<td>''')
            _to_return.append(self.clean_button.get_html())
            _to_return.append('''</td>''')
        _to_return.append('''</tr></table>''')
        return ' '.join(_to_return)
