#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2019:
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.misc.type_hint import TYPE_CHECKING
from shinken.toolbox.pickledb import ShinkenPickleableMeta
from .empty_object import EmptyObject
from .html.tooltip import Tooltip
from ..constant import FRONT_HTML_FORMAT
from ..toolbox.string import escape_XSS

if TYPE_CHECKING:
    from shinken.misc.type_hint import Union, Dict, Optional, Callable, Tuple
    from shinkensolutions.component.abstract_translate_component import TranslateKey


class ShinkenObject(object):
    __metaclass__ = ShinkenPickleableMeta
    
    def __init__(self, translate=None):
        # type: (Optional[Callable[[TranslateKey, Optional[Tuple]], unicode]]) -> None
        self._translate = translate
        self.tooltip = EmptyObject()  # type: Union[EmptyObject,Tooltip]
        self.url_param_key = u''  # type: unicode
        self._data = {}  # type: Dict
        self._attribute = {}  # type: Dict
        self._class = u''  # type: unicode
        self._class_container = u''  # type: unicode
        self._class_content = u''  # type: unicode
        self.html_format = FRONT_HTML_FORMAT.NORMAL  # type: unicode
    
    
    @staticmethod
    def wrap_in_middle_vertical_align_container(content):
        # type: (unicode) -> unicode
        return u'''<div class="shinken-display-table"><div class="shinken-display-table-cell">%s</div></div>''' % content
    
    
    # ****************************************************** GETTER SETTER  ***************************************
    def add_class_container(self, to_set):
        # type: (unicode) -> None
        if not to_set:
            return
        if self._class_container:
            self._class_container = u'%s %s' % (self._class_container, to_set)
        else:
            self._class_container = to_set
    
    
    def add_class_content(self, to_set):
        # type: (unicode)-> None
        if not to_set:
            return
        if self._class_content:
            self._class_content = u'%s %s' % (self._class_content, to_set)
        else:
            self._class_content = to_set
    
    
    def add_class(self, to_set):
        # type: (unicode) -> None
        if not to_set:
            return
        if self._class:
            self._class = u'%s %s' % (self._class, to_set)
        else:
            self._class = to_set
    
    
    def add_data(self, key, value):
        # type: (Union[str, unicode], Union[str, int, unicode])-> None
        self._data[key] = value
    
    
    def add_attribute(self, key, value):
        # type: (unicode, unicode)-> None
        self._attribute[key] = value
    
    
    # ****************************************************** OTHER  ***************************************
    def add_tooltip(self):
        # type: () -> Tooltip
        self.tooltip = Tooltip()
        return self.tooltip
    
    
    def add_url_param_key(self, key_name):
        # type: (unicode) -> None
        self.url_param_key = key_name
    
    
    # ****************************************************** MISC  ***************************************
    def translate(self, key, *args):
        # type: (TranslateKey, Optional[Tuple]) -> unicode
        if not callable(self._translate):
            raise TypeError(u'There is no translate function')
        return self._translate(key, *args)

    def _(self, key, *args):
        # type: (unicode, Optional[Tuple]) -> unicode
        return self.translate(key, *args)
    
    
    @staticmethod
    def escape_xss(value):
        return escape_XSS(value)
    
    
    # ****************************************************** HTML  ***************************************
    def get_object_tag_html(self, class_to_add=u'', extra_attribute=None, skip_tooltip=False):
        # type: ( Optional[unicode], Optional[Dict], Optional[bool])-> unicode
        tooltip_html = u''
        if not skip_tooltip:
            tooltip_html = self.tooltip.get_html()
        if self._class or class_to_add:
            class_to_add = u"%s %s" % (self._class, class_to_add)
            class_to_add = class_to_add.strip()
        if class_to_add:
            return u'''class="%s" %s %s %s''' % (class_to_add, self.get_data_html(), self.get_attribute_html(extra_attribute), tooltip_html)
        return u'''%s %s %s''' % (self.get_data_html(), self.get_attribute_html(), tooltip_html)
    
    
    def get_data_html(self):
        # type: ()-> unicode
        return u' '.join([u'data-%s="%s"' % (k, v) for k, v in self._data.iteritems()])
    
    
    def get_attribute_html(self, extra_attribute=None):
        # type: (Optional[Dict])-> unicode
        _to_return = [u'%s="%s"' % (k, v) for k, v in self._attribute.iteritems()]
        if extra_attribute:
            _to_return.extend([u'%s="%s"' % (k, v) for k, v in extra_attribute.iteritems()])
        return u' '.join(_to_return)
    
    
    @staticmethod
    def get_html_tooltip(text=u'', message_type=u'', placement=u''):
        # type: (Optional[unicode], Optional[unicode], Optional[unicode]) -> unicode
        _attribute_type = u'''shi-tip-type="%s"''' % message_type if message_type else u''
        _attribute_placement = u'''shi-tip-placement="%s"''' % placement if placement else u''
        return u'''onmouseenter="SHINKEN_TOOLTIP.showTooltip(this)" onmouseleave="SHINKEN_TOOLTIP.hideTooltip()" %s shi-tip-html="%s" %s''' % (_attribute_type, text.replace(u'&', u'&amp;'), _attribute_placement)
    
    
    @staticmethod
    def get_html_lock():
        # type: () -> unicode
        return u'''<td class="shinken-lock-cell"><span class="shinkon-lock"></span></td>'''
    
    
    def get_html(self):
        # type: () -> unicode
        return u''
    
    
    def get_html_with_format(self, html_format):
        # type: (unicode) -> unicode
        self.html_format = html_format
        return self.get_html()


class ShinkenObjectContainer(ShinkenObject):
    
    def __init__(self, html):
        # type: (unicode) -> None
        super(ShinkenObjectContainer, self).__init__()
        self.html_to_return = html
    
    
    def get_html(self):
        # type: () -> unicode
        return self.html_to_return
