#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.common_ui.object.shinken_ui_composite.abstract_shinken_ui_composite import AbstractShinkenUIComposite
from shinkensolutions.common_ui.object.shinken_ui_composite.constants import SHINKEN_UI_VALUE
from shinkensolutions.common_ui.object.shinken_ui_composite.shinken_ui_property import AbstractShinkenUIProperty, ShinkenUIAnyProperty
from shinkensolutions.common_ui.object.shinken_ui_composite.shinken_ui_property_bad_value import ShinkenUIPropertyBadValue

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict, Type, List, Any
    from shinkensolutions.common_ui.object.messages import ValidatorMessages


class AbstractShinkenUIContainer(AbstractShinkenUIComposite):
    
    __slots__ = (u'_missing',)
    
    
    def __init__(self, associated_key, validator_message_type, mandatory):
        super(AbstractShinkenUIContainer, self).__init__(associated_key, validator_message_type, mandatory)
        self._missing = False
    
    
    def is_source_missing(self):
        # type: () -> bool
        return self._missing
    
    
    def _set_source_value_from_source(self, source):
        raise NotImplementedError()
    
    
    def to_flattened_format(self, required_value):
        # type: (unicode) -> Any
        raise NotImplementedError()
    
    
    def _get_stored_values_to_dict(self, computed_value, source_value, config_value, include_validation_messages):
        # type: (bool, bool, bool, bool) -> Dict[unicode, Any]
        raise NotImplementedError()
    
    
    def get_unknown_keys(self):
        # type: () -> List
        raise NotImplementedError()
    
    
    def apply_config_value_to_computed_if_needed(self):
        raise NotImplementedError()
    
    
    def apply_source_value_to_computed_if_needed(self):
        raise NotImplementedError()
    
    
    def apply_config_or_source_value_to_computed_if_needed(self):
        raise NotImplementedError()
    
    
    def replace_properties_default_values_from(self, other):
        raise NotImplementedError()


def setup_shinken_ui_objects_tuple_if_needed(func):
    
    def wrapper(self, *args, **kwargs):
        if self.__shinken_ui_composite_tuple__ is None:
            self.__shinken_ui_composite_tuple__ = [(_name, _attr) for _name, _attr in vars(self).iteritems() if isinstance(_attr, AbstractShinkenUIComposite)]
        return func(self, *args, **kwargs)
    
    
    return wrapper


class ShinkenUIObject(AbstractShinkenUIContainer):
    __slots__ = (u'_unknown_keys', u'__shinken_ui_composite_tuple__')
    
    KEY_VALUE = SHINKEN_UI_VALUE.KEY_VALUE
    
    # We do not want to skip 0 and empty string/list
    EMPTY_VALUES = (None, {})
    
    
    def __init__(self, associated_key, validator_message_type, mandatory=False):
        # type: (unicode, Type[ValidatorMessages], bool) -> None
        super(ShinkenUIObject, self).__init__(associated_key, validator_message_type, mandatory=mandatory)
        self._unknown_keys = {}
        self.__shinken_ui_composite_tuple__ = None
    
    
    def _clear_empty_key(self, _dict):
        # type: (Dict) -> None
        keys_to_delete = []
        for remaining_keys, remaining_value in _dict.iteritems():
            if isinstance(remaining_value, dict):
                self._clear_empty_key(remaining_value)
            if remaining_value in (None, {}, []):
                keys_to_delete.append(remaining_keys)
        
        for _empty_key in keys_to_delete:
            _dict.pop(_empty_key)
    
    
    @setup_shinken_ui_objects_tuple_if_needed
    def _set_source_value_from_source(self, source):
        # type: (Dict) -> None
        _my_dictionary = self._get_associated_value(source, None)
        
        if _my_dictionary is None:
            self._missing = True
            return
        
        if not isinstance(_my_dictionary, dict):
            self._source_value = ShinkenUIPropertyBadValue(_my_dictionary)
            return
        
        for _, _attr in self.__shinken_ui_composite_tuple__:
            _attr.set_source_value_from_source(_my_dictionary)
        
        self._clear_empty_key(_my_dictionary)
        
        for _key, _value in _my_dictionary.iteritems():
            _unknown_property = ShinkenUIAnyProperty(_key, validator_message_type=self.validator_message_type)
            _unknown_property.set_source_value(_value)
            self._unknown_keys[_key] = _unknown_property
    
    
    def get_unknown_keys(self):
        # type: () -> Dict
        return self._unknown_keys
    
    
    @classmethod
    def _recursively_add_associated_value_to_dict(cls, _dict, associated_key, value, wrap_in_dict):
        # type: (Dict, unicode, Any, bool) -> None
        if value in cls.EMPTY_VALUES:
            return
        _associated_key_steps, _, _associated_key_last_step = associated_key.rpartition(u'.')
        if _associated_key_steps:
            _associated_key_steps = _associated_key_steps.split(u'.')
        else:
            _associated_key_steps = []
        _step = _dict
        for _associated_key_step in _associated_key_steps:
            _step = _step.setdefault(_associated_key_step, {})
            if wrap_in_dict:
                _step = _step.setdefault(cls.KEY_VALUE, {})
        _step[_associated_key_last_step] = value
    
    
    @setup_shinken_ui_objects_tuple_if_needed
    def to_flattened_format(self, required_value):
        # type: (unicode) -> Any
        if not self.is_source_valid():
            return self._source_value.to_json()
        _dict_result = dict(((_key, _prop.to_flattened_format(required_value)) for _key, _prop in self._unknown_keys.iteritems())) if required_value == SHINKEN_UI_VALUE.KEY_SOURCE_VALUE else {}
        for _, _attr in self.__shinken_ui_composite_tuple__:
            self._recursively_add_associated_value_to_dict(_dict_result, _attr.associated_key, _attr.to_flattened_format(required_value), wrap_in_dict=False)
        return _dict_result
    
    
    @setup_shinken_ui_objects_tuple_if_needed
    def _get_stored_values_to_dict(self, computed_value, source_value, config_value, include_validation_messages):
        # type: (bool, bool, bool, bool) -> Dict[unicode, Any]
        _dict_result = {}
        if self.is_source_valid():
            _value_dict = dict(((_key, _prop._get_stored_values_to_dict(computed_value=False, source_value=source_value, config_value=False, include_validation_messages=True)) for _key, _prop in self._unknown_keys.iteritems()))
            for _, _attr in self.__shinken_ui_composite_tuple__:
                _value = _attr.to_dict(computed_value=computed_value, source_value=source_value, config_value=config_value, include_validation_messages=include_validation_messages)
                self._recursively_add_associated_value_to_dict(_value_dict, _attr.associated_key, _value, wrap_in_dict=True)
            if _value_dict:
                _dict_result[self.KEY_VALUE] = _value_dict
        else:
            _dict_result[self.KEY_VALUE] = self._source_value.to_json()
        if include_validation_messages:
            _validation_messages = self.validator_messages.as_dict()
            if _validation_messages:
                _dict_result[self.validator_messages.KEY_ROOT] = _validation_messages
        return _dict_result
    
    
    @setup_shinken_ui_objects_tuple_if_needed
    def has_critical_errors(self):
        # type: () -> bool
        for _, _attr in self.__shinken_ui_composite_tuple__:
            if _attr.has_critical_errors():
                return True
        return self.validator_messages.has_critical()
    
    
    @setup_shinken_ui_objects_tuple_if_needed
    def has_errors(self):
        # type: () -> bool
        for _, _attr in self.__shinken_ui_composite_tuple__:
            if _attr.has_errors():
                return True
        return self.validator_messages.has_error()
    
    
    @setup_shinken_ui_objects_tuple_if_needed
    def has_warnings(self):
        # type: () -> bool
        for _, _attr in self.__shinken_ui_composite_tuple__:
            if _attr.has_warnings():
                return True
        return self.validator_messages.has_warning()
    
    
    @setup_shinken_ui_objects_tuple_if_needed
    def get_critical_errors_nb(self):
        # type: () -> int
        nb_of_critical = self.validator_messages.get_nb_critical()
        for _, _attr in self.__shinken_ui_composite_tuple__:
            nb_of_critical += _attr.get_critical_errors_nb()
        return nb_of_critical
    
    
    @setup_shinken_ui_objects_tuple_if_needed
    def get_errors_nb(self):
        # type: () -> int
        nb_of_errors = self.validator_messages.get_nb_error()
        for _, _attr in self.__shinken_ui_composite_tuple__:
            nb_of_errors += _attr.get_errors_nb()
        return nb_of_errors
    
    
    @setup_shinken_ui_objects_tuple_if_needed
    def get_warnings_nb(self):
        # type: () -> int
        nb_of_warning = self.validator_messages.get_nb_warning()
        for _, _attr in self.__shinken_ui_composite_tuple__:
            nb_of_warning += _attr.get_warnings_nb()
        return nb_of_warning
    
    
    @setup_shinken_ui_objects_tuple_if_needed
    def get_critical_copy(self):
        # type: () -> List
        critical = self.validator_messages.get_critical_copy()
        for _, _attr in self.__shinken_ui_composite_tuple__:
            critical.extend(_attr.get_critical_copy())
        return critical
    
    
    @setup_shinken_ui_objects_tuple_if_needed
    def get_errors_copy(self):
        # type: () -> List
        errors = self.validator_messages.get_errors_copy()
        for _, _attr in self.__shinken_ui_composite_tuple__:
            errors.extend(_attr.get_errors_copy())
        return errors
    
    
    @setup_shinken_ui_objects_tuple_if_needed
    def get_warnings_copy(self):
        # type: () -> List
        warnings = self.validator_messages.get_warnings_copy()
        for _, _attr in self.__shinken_ui_composite_tuple__:
            warnings.extend(_attr.get_warnings_copy())
        return warnings
    
    
    @setup_shinken_ui_objects_tuple_if_needed
    def __eq__(self, other):
        # type: (ShinkenUIObject) -> bool
        if type(self) is not type(other):
            return NotImplemented
        
        return self.__shinken_ui_composite_tuple__ == other.__shinken_ui_composite_tuple__
    
    
    def __ne__(self, other):
        return not self == other
    
    
    @setup_shinken_ui_objects_tuple_if_needed
    def apply_config_value_to_computed_if_needed(self):
        for _, _attr in self.__shinken_ui_composite_tuple__:
            _attr.apply_config_value_to_computed_if_needed()
    
    
    @setup_shinken_ui_objects_tuple_if_needed
    def apply_source_value_to_computed_if_needed(self):
        for _, _attr in self.__shinken_ui_composite_tuple__:
            _attr.apply_source_value_to_computed_if_needed()
    
    
    @setup_shinken_ui_objects_tuple_if_needed
    def apply_config_or_source_value_to_computed_if_needed(self):
        for _, _attr in self.__shinken_ui_composite_tuple__:
            _attr.apply_config_or_source_value_to_computed_if_needed()
    
    
    @setup_shinken_ui_objects_tuple_if_needed
    def replace_properties_default_values_from(self, other):
        if not isinstance(self, type(other)):
            return
        
        for _attr_name, _attr in self.__shinken_ui_composite_tuple__:
            if isinstance(_attr, (ShinkenUIObject, AbstractShinkenUIProperty)):  # List cannot be merged that way.
                _attr.replace_properties_default_values_from(getattr(other, _attr_name, None))


class ShinkenUIList(AbstractShinkenUIContainer):
    KEY_LIST_CONTENT = u'list_content'
    
    # We do not want to skip 0 and empty string
    EMPTY_VALUES = (None, [], {})
    
    
    def __init__(self, associated_key, validator_message_type, mandatory=False, list_elements_type=None):
        # type: (unicode, Type[ValidatorMessages], bool, Type[AbstractShinkenUIComposite]) -> None
        super(ShinkenUIList, self).__init__(associated_key, validator_message_type, mandatory=mandatory)
        self._list = []  # type: List[AbstractShinkenUIComposite]
        self.list_elements_type = list_elements_type
    
    
    def _create_object(self, _list, idx):
        # type: (List, int) -> AbstractShinkenUIComposite
        if self.list_elements_type:
            return self.list_elements_type(associated_key=u'%d' % idx, validator_message_type=self.validator_message_type, mandatory=False)
        else:
            raise NotImplementedError(u'No list_element and _create_object is not override')
    
    
    def _set_source_value_from_source(self, source):
        # type: (Dict) -> None
        _my_list = self._get_associated_value(source)
        
        if _my_list is None:
            self._missing = True
            return
        
        if not isinstance(_my_list, list):
            self._source_value = ShinkenUIPropertyBadValue(_my_list)
            self.valid = False
            return
        
        for idx in xrange(len(_my_list)):
            _shinken_ui_object = self._create_object(_my_list, idx)
            self._list.append(_shinken_ui_object)
        
        for _item in self._list:
            _item.set_source_value_from_source(_my_list)
    
    
    def apply_config_value_to_computed_if_needed(self):
        for _attr in self._list:
            _attr.apply_config_value_to_computed_if_needed()
    
    
    def apply_source_value_to_computed_if_needed(self):
        for _attr in self._list:
            _attr.apply_source_value_to_computed_if_needed()
    
    
    def apply_config_or_source_value_to_computed_if_needed(self):
        for _attr in self._list:
            _attr.apply_config_or_source_value_to_computed_if_needed()
    
    
    def replace_properties_default_values_from(self, *others):
        # We merge list this way :
        # We take an object, and we apply it on every object that has the same type in our list
        if not others:
            return
        for _attr in self._list:
            for _other in others:
                if isinstance(_attr, type(_other)):
                    _attr.replace_properties_default_values_from(_other)
                    break
    
    
    def get_unknown_keys(self):
        # type: () -> List
        return []
    
    
    def __iter__(self):
        return self._list.__iter__()
    
    
    def to_flattened_format(self, required_value):
        # type: (unicode) -> Any
        if not self.is_source_valid():
            return self._source_value.to_json()
        
        _result_list = []
        _EMPTY_VALUES = self.EMPTY_VALUES  # retrieve list once
        for _item in self._list:
            _value = _item.to_flattened_format(required_value)
            if _value not in _EMPTY_VALUES:
                _result_list.append(_value)
        return _result_list
    
    
    def _get_stored_values_to_dict(self, computed_value, source_value, config_value, include_validation_messages):
        # type: (bool, bool, bool, bool) -> Dict[unicode, Any]
        _validation_messages = self.validator_messages.as_dict()
        _dict_result = {}
        _EMPTY_VALUES = self.EMPTY_VALUES  # retrieve list once
        
        if self.is_source_valid():
            _result_list = []
            for _item in self._list:
                _value = _item.to_dict(computed_value=computed_value, source_value=source_value, config_value=config_value, include_validation_messages=include_validation_messages)
                if _value not in _EMPTY_VALUES:
                    _result_list.append(_value)
            
            _dict_result[self.KEY_LIST_CONTENT] = _result_list
        else:
            _dict_result[self.KEY_LIST_CONTENT] = self._source_value.to_json()
        if include_validation_messages:
            _validation_messages = self.validator_messages.as_dict()
            if _validation_messages:
                _dict_result[self.validator_messages.KEY_ROOT] = _validation_messages
        return _dict_result
    
    
    def has_critical_errors(self):
        # type: () -> bool
        for _attr in self._list:
            if _attr.has_critical_errors():
                return True
        return self.validator_messages.has_critical()
    
    
    def has_errors(self):
        # type: () -> bool
        for _attr in self._list:
            if _attr.has_errors():
                return True
        return self.validator_messages.has_error()
    
    
    def has_warnings(self):
        # type: () -> bool
        for _attr in self._list:
            if _attr.has_warnings():
                return True
        return self.validator_messages.has_warning()
    
    
    def get_critical_errors_nb(self):
        # type: () -> int
        nb_of_critical = self.validator_messages.get_nb_critical()
        for _attr in self._list:
            nb_of_critical += _attr.get_critical_errors_nb()
        return nb_of_critical
    
    
    def get_errors_nb(self):
        # type: () -> int
        nb_of_errors = self.validator_messages.get_nb_error()
        for _attr in self._list:
            nb_of_errors += _attr.get_errors_nb()
        return nb_of_errors
    
    
    def get_warnings_nb(self):
        # type: () -> int
        nb_of_warning = self.validator_messages.get_nb_warning()
        for _attr in self._list:
            nb_of_warning += _attr.get_warnings_nb()
        return nb_of_warning
    
    
    def get_critical_copy(self):
        # type: () -> List
        critical = self.validator_messages.get_critical_copy()
        for _attr in self._list:
            critical.extend(_attr.get_critical_copy())
        return critical
    
    
    def get_errors_copy(self):
        # type: () -> List
        errors = self.validator_messages.get_errors_copy()
        for _attr in self._list:
            errors.extend(_attr.get_errors_copy())
        return errors
    
    
    def get_warnings_copy(self):
        # type: () -> List
        warnings = self.validator_messages.get_warnings_copy()
        for _attr in self._list:
            warnings.extend(_attr.get_warnings_copy())
        return warnings
    
    
    def __eq__(self, other):
        # type: (ShinkenUIList) -> bool
        if type(self) is not type(other):
            return NotImplemented
        
        return self._list == other._list
    
    
    def __ne__(self, other):
        return not self == other
    
    
    def __len__(self):
        return len(self._list)
    
    
    def __getitem__(self, item):
        if not isinstance(item, int):
            raise TypeError(u'Expected integer index')
        
        return self._list[item]
    
    
    def __bool__(self):
        return bool(self._list)
    
    
    __nonzero__ = __bool__
