#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.misc.type_hint import TYPE_CHECKING
from shinken.util import to_bool
from shinkensolutions.common_ui.object.shinken_ui_composite.abstract_shinken_ui_composite import AbstractShinkenUIComposite
from shinkensolutions.common_ui.object.shinken_ui_composite.constants import SHINKEN_UI_VALUE
from shinkensolutions.common_ui.object.shinken_ui_composite.shinken_ui_property_bad_value import ShinkenUIPropertyBadValue, ShinkenUIPropertyBadInheritedValue, ShinkenUIPropertyBadInheritedValueFromConfig
from shinkensolutions.toolbox.tool_box_integer import ToolBoxInteger

if TYPE_CHECKING:
    from shinkensolutions.common_ui.object.messages import ValidatorMessages
    from shinken.misc.type_hint import Any, Union, List, Tuple, Optional, Type, Sequence, Dict


class ShinkenUIPropertyBadKey(ValueError):
    pass


class AbstractShinkenUIProperty(AbstractShinkenUIComposite):
    default_values = (u'default', u'', None)  # type: Sequence[Any]
    __slots__ = (u'_computed_value', u'_config_value', u'allow_default')
    
    KEY_VALUE = SHINKEN_UI_VALUE.KEY_VALUE
    KEY_SOURCE_VALUE = SHINKEN_UI_VALUE.KEY_SOURCE_VALUE
    KEY_CFG_VALUE = SHINKEN_UI_VALUE.KEY_CFG_VALUE
    
    
    def __init__(self, associated_key, validator_message_type, mandatory, allow_default):
        # type: (unicode, Type[ValidatorMessages], bool, bool) -> None
        super(AbstractShinkenUIProperty, self).__init__(associated_key, validator_message_type=validator_message_type, mandatory=mandatory)
        self._computed_value = None  # type: Any
        self._config_value = None  # type: Any
        self.allow_default = allow_default
    
    
    def _serialize_value(self, value):
        # type: (unicode) -> Any
        raise NotImplementedError()
    
    
    def _set_source_value_from_source(self, source):
        _value = self._get_associated_value(source)
        self.set_source_value(_value)
    
    
    def override_source_value_with_computed(self):
        if isinstance(self._computed_value, ShinkenUIPropertyBadValue):
            self._source_value = self._computed_value.__copy__()
        else:
            self._source_value = self._computed_value
    
    
    def apply_config_value_to_computed_if_needed(self):
        if self._computed_value is None and self._config_value is not None:
            if self.allow_default and self._source_value is None:
                if isinstance(self._config_value, ShinkenUIPropertyBadValue):
                    self._computed_value = ShinkenUIPropertyBadInheritedValueFromConfig.from_bad_value(self._config_value)
                else:
                    self._computed_value = self._config_value
    
    
    def apply_source_value_to_computed_if_needed(self):
        if self._computed_value is None and self._source_value is not None:
            if isinstance(self._source_value, ShinkenUIPropertyBadValue):
                self._computed_value = self._source_value.__copy__()
            else:
                self._computed_value = self._source_value
    
    
    def apply_config_or_source_value_to_computed_if_needed(self):
        self.apply_config_value_to_computed_if_needed()
        self.apply_source_value_to_computed_if_needed()
    
    
    def replace_properties_default_values_from(self, other):
        # type: (AbstractShinkenUIProperty) -> None
        if type(self) != type(other):
            return
        
        if self._computed_value is None:
            if self.allow_default and self._source_value is None:
                if isinstance(other.get_computed_value(), ShinkenUIPropertyBadValue):
                    self._computed_value = ShinkenUIPropertyBadInheritedValue.from_bad_value(other.get_computed_value())
                else:
                    self._computed_value = other.get_computed_value()
    
    
    def __eq__(self, other):
        # type: (Any) -> bool
        if isinstance(other, AbstractShinkenUIProperty):
            return self._computed_value == other._computed_value and self._source_value == other._source_value and self._config_value == other._config_value
        else:
            return self._computed_value == other
    
    
    def __repr__(self):
        # type: () -> unicode
        return u'<%s value=%s>' % (type(self).__name__, self._computed_value)
    
    
    def __unicode__(self):
        if self._computed_value:
            return unicode(self._computed_value)
        return u''
    
    
    def __str__(self):
        return unicode(self).encode('utf-8')
    
    
    def __ne__(self, other):
        # type: (AbstractShinkenUIProperty) -> bool
        return not (self == other)
    
    
    def set_source_and_computed_value(self, value):
        self.set_source_value(value)
        self.set_computed_value(value)
    
    
    def set_source_value(self, _source_value):
        if self.allow_default and _source_value in self.default_values:
            self._source_value = None
        elif _source_value is not None:
            self._source_value = self._serialize_value(_source_value)
    
    
    def set_computed_value(self, _computed_value):
        if self.allow_default and _computed_value in self.default_values:
            self._computed_value = None
        elif _computed_value is not None:
            self._computed_value = self._serialize_value(_computed_value)
    
    
    def set_config_value(self, _config_value):
        if not isinstance(_config_value, ShinkenUIPropertyBadValue):
            self._config_value = self._serialize_value(_config_value)
        else:
            self._config_value = _config_value
    
    
    def get_source_value(self):
        return self._source_value
    
    
    @staticmethod
    def _get_real_value(value):
        if isinstance(value, ShinkenUIPropertyBadValue):
            return value.value
        return value
    
    
    def get_computed_value(self):
        return self._computed_value
    
    
    def get_config_value(self):
        return self._config_value
    
    
    def to_flattened_format(self, required_value):
        # type: (unicode) -> Any
        if required_value == self.KEY_VALUE:
            return self._get_real_value(self.get_computed_value())
        if required_value == self.KEY_SOURCE_VALUE:
            return self._get_real_value(self.get_source_value())
        if required_value == self.KEY_CFG_VALUE:
            return self._get_real_value(self.get_config_value())
        raise ShinkenUIPropertyBadKey(u'Unknown value type [ %s ]' % required_value)
    
    
    def _get_stored_values_to_dict(self, computed_value, source_value, config_value, include_validation_messages):
        # type: (bool, bool, bool, bool) -> Dict[unicode, Any]
        _dict_result = {}
        if source_value:
            _real_source_value = self._get_real_value(self.get_source_value())
            if _real_source_value is not None:
                _dict_result[self.KEY_SOURCE_VALUE] = _real_source_value
        if computed_value:
            _real_computed_value = self._get_real_value(self.get_computed_value())
            if _real_computed_value is not None:
                _dict_result[self.KEY_VALUE] = _real_computed_value
        if config_value:
            _real_config_value = self._get_real_value(self.get_config_value())
            if _real_config_value is not None:
                _dict_result[self.KEY_CFG_VALUE] = _real_config_value
        if include_validation_messages:
            _validation_messages = self.validator_messages.as_dict()
            if _validation_messages:
                _dict_result[self.validator_messages.KEY_ROOT] = _validation_messages
        return _dict_result
    
    
    def _handle_bad_value(self, validation_status, trad_key, trad_args):
        # type: (unicode, unicode, Union[Tuple, List]) -> None
        if not isinstance(self._computed_value, ShinkenUIPropertyBadValue):
            new_value = ShinkenUIPropertyBadValue(self._computed_value)
            new_value.set_translate_variables(validation_status, trad_key, trad_args)
            self._computed_value = new_value
    
    
    def is_source_missing(self):
        # type: () -> bool
        return self._source_value is None
    
    
    def is_valid(self):
        # type: () -> bool
        return not (self.mandatory and self._computed_value is None) and not isinstance(self._computed_value, ShinkenUIPropertyBadValue)
    
    
    def is_incorrect(self):
        # type: () -> bool
        return isinstance(self._computed_value, ShinkenUIPropertyBadValue)
    
    
    def is_missing(self):
        # type: () -> bool
        return self._computed_value is None
    
    
    def is_incorrect_or_missing(self):
        # type: () -> bool
        return self.is_incorrect() or self.is_missing()


class ShinkenUIAnyProperty(AbstractShinkenUIProperty):
    def __init__(self, associated_key, validator_message_type=None, mandatory=False, allow_default=False):
        # type: (unicode, Type[ValidatorMessages], bool, bool) -> None
        super(ShinkenUIAnyProperty, self).__init__(associated_key, validator_message_type, mandatory=mandatory, allow_default=allow_default)
    
    
    def _serialize_value(self, value):
        # type: (unicode) -> Union[ShinkenUIPropertyBadValue, unicode]
        return value


class ShinkenUIStringProperty(AbstractShinkenUIProperty):
    __slots__ = (u'must_be_filled',)
    
    
    def __init__(self, associated_key, validator_message_type=None, mandatory=False, allow_default=False, must_be_filled=False):
        # type: (unicode, Type[ValidatorMessages], bool, bool, bool) -> None
        super(ShinkenUIStringProperty, self).__init__(associated_key, validator_message_type, mandatory=mandatory, allow_default=allow_default)
        self.must_be_filled = must_be_filled
    
    
    def _serialize_value(self, value):
        # type: (unicode) -> Union[ShinkenUIPropertyBadValue, unicode]
        try:
            if not isinstance(value, basestring):
                return ShinkenUIPropertyBadValue(value)
            value = unicode(value).strip(u' ')
            if self.must_be_filled and not value:
                return ShinkenUIPropertyBadValue(value)
            return value
        except ValueError:
            return ShinkenUIPropertyBadValue(value)


class ShinkenUIEnumProperty(AbstractShinkenUIProperty):
    __slots__ = (u'enum',)
    
    
    def __init__(self, associated_key, enum, validator_message_type=None, mandatory=False, allow_default=False):
        # type: (unicode, Sequence, Type[ValidatorMessages], bool, bool) -> None
        super(ShinkenUIEnumProperty, self).__init__(associated_key, validator_message_type=validator_message_type, mandatory=mandatory, allow_default=allow_default)
        self.enum = enum
    
    
    def _serialize_value(self, value):
        # type: (unicode) -> Union[ShinkenUIPropertyBadValue, unicode]
        if value in self.enum:
            if isinstance(value, str):
                try:
                    return value.encode(u'utf-8')
                except ValueError:
                    return ShinkenUIPropertyBadValue(value)
            return value
        return ShinkenUIPropertyBadValue(value)


class AbstractShinkenUINumberProperty(AbstractShinkenUIProperty):
    __slots__ = (u'must_be_strict_negative', u'must_be_strict_positive', u'must_be_positive', u'must_be_negative', u'range_allowed')
    
    
    def __init__(self, associated_key, validator_message_type=None, mandatory=False, must_be_strict_positive=False, must_be_strict_negative=False, must_be_positive=False, must_be_negative=False, range_allowed=None, allow_default=False):
        # type: (unicode, Type[ValidatorMessages], bool, bool, bool, bool, bool, Optional[List], bool) -> None
        super(AbstractShinkenUINumberProperty, self).__init__(associated_key, validator_message_type, mandatory=mandatory, allow_default=allow_default)
        _number_of_bool = 0
        if must_be_strict_positive:
            _number_of_bool += 1
        if must_be_strict_negative:
            _number_of_bool += 1
        if must_be_positive:
            _number_of_bool += 1
        if must_be_negative:
            _number_of_bool += 1
        
        if _number_of_bool > 1 or (_number_of_bool == 1 and range_allowed):
            raise TypeError(u'Only 1 boolean can be true and range_allowed can\'t be used with other options')
        
        self.must_be_strict_positive = must_be_strict_positive
        self.must_be_strict_negative = must_be_strict_negative
        self.must_be_positive = must_be_positive
        self.must_be_negative = must_be_negative
        self.range_allowed = range_allowed
    
    
    def _check_value(self, value):
        if self.must_be_strict_positive:
            if value <= 0:
                raise ValueError(u'Negative or null value')
        elif self.must_be_strict_negative:
            if value >= 0:
                raise ValueError(u'Positive or null value')
        elif self.must_be_positive:
            if value < 0:
                raise ValueError(u'Negative value')
        elif self.must_be_negative:
            if value > 0:
                raise ValueError(u'Positive value')
        if self.range_allowed:
            if not (self.range_allowed[0] <= value <= self.range_allowed[1]):
                raise ValueError(u'Value not in range [%s:%s]' % (self.range_allowed[0], self.range_allowed[1]))
        return value
    
    
    def _serialize_value(self, value):
        raise NotImplementedError


class ShinkenUIIntegerProperty(AbstractShinkenUINumberProperty):
    __slots__ = ()
    
    
    def __init__(self, associated_key, validator_message_type=None, mandatory=False, must_be_strict_positive=False, must_be_strict_negative=False, must_be_positive=False, must_be_negative=False, range_allowed=None, allow_default=False):
        # type: (unicode, Type[ValidatorMessages], bool, bool, bool, bool, bool, Optional[List], bool) -> None
        super(ShinkenUIIntegerProperty, self).__init__(associated_key, validator_message_type, mandatory, must_be_strict_positive, must_be_strict_negative, must_be_positive, must_be_negative, range_allowed, allow_default)
    
    
    def _serialize_value(self, value):
        # type: (unicode) -> Union[ShinkenUIPropertyBadValue, int]
        try:
            if not ToolBoxInteger.is_integer_or_string_integer(value):
                return ShinkenUIPropertyBadValue(value)
            return self._check_value(int(value))
        except (ValueError, TypeError):
            return ShinkenUIPropertyBadValue(value)


class ShinkenUIFloatProperty(AbstractShinkenUINumberProperty):
    __slots__ = ()
    
    
    def __init__(self, associated_key, validator_message_type=None, mandatory=False, must_be_strict_positive=False, must_be_strict_negative=False, must_be_positive=False, must_be_negative=False, range_allowed=None, allow_default=False):
        # type: (unicode, Type[ValidatorMessages], bool, bool, bool, bool, bool, Optional[List], bool) -> None
        super(ShinkenUIFloatProperty, self).__init__(associated_key, validator_message_type, mandatory, must_be_strict_positive, must_be_strict_negative, must_be_positive, must_be_negative, range_allowed, allow_default)
    
    
    def _serialize_value(self, value):
        # type: (unicode) -> Union[ShinkenUIPropertyBadValue, float]
        try:
            return self._check_value(float(value))
        except (ValueError, TypeError):
            return ShinkenUIPropertyBadValue(value)


class ShinkenUIIntegerOrEnumProperty(ShinkenUIIntegerProperty):
    __slots__ = (u'enum',)
    
    
    def __init__(self, associated_key, enum=None, validator_message_type=None, mandatory=False, must_be_strict_positive=False, must_be_strict_negative=False, must_be_positive=False, must_be_negative=False, range_allowed=None, allow_default=False):
        # type: (unicode, Sequence, Type[ValidatorMessages], bool, bool, bool, bool, bool, Optional[List], bool) -> None
        super(ShinkenUIIntegerOrEnumProperty, self).__init__(associated_key, validator_message_type, mandatory, must_be_strict_positive, must_be_strict_negative, must_be_positive, must_be_negative, range_allowed, allow_default)
        self.enum = () if enum is None else enum
    
    
    def _serialize_value(self, value):
        # type: (unicode) -> Union[ShinkenUIPropertyBadValue, int, unicode]
        if value in self.enum:
            if isinstance(value, str):
                try:
                    return value.encode(u'utf-8')
                except (ValueError, TypeError):
                    return ShinkenUIPropertyBadValue(value)
            return value
        return super(ShinkenUIIntegerOrEnumProperty, self)._serialize_value(value)


class ShinkenUIBoolProperty(AbstractShinkenUIProperty):
    __slots__ = ()
    
    
    def __init__(self, associated_key, validator_message_type=None, mandatory=False, allow_default=False):
        # type: (unicode, Type[ValidatorMessages], bool, bool) -> None
        super(ShinkenUIBoolProperty, self).__init__(associated_key, validator_message_type, mandatory=mandatory, allow_default=allow_default)
    
    
    def _serialize_value(self, value):
        # type: (unicode) -> Union[ShinkenUIPropertyBadValue, bool]
        try:
            return to_bool(value, with_strict_check=True)
        except (ValueError, TypeError):
            return ShinkenUIPropertyBadValue(value)
