# !/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2021:
# This file is part of Shinken Enterprise, all rights reserved.
import time
from collections import namedtuple

from shinken.misc.type_hint import TYPE_CHECKING, cast
from shinkensolutions.component.abstract_component import AbstractComponent

if TYPE_CHECKING:
    from shinken.log import PartLogger
    from shinken.misc.type_hint import Dict, List, Optional, Sequence
    from webui.module import WebuiBroker
    from shinken.webui.bottlewebui import MultiDict

FilterQuery = namedtuple('FilterQuery', ['valid_keys'])


class FilterParserError(Exception):
    def __init__(self, error_list, message=u'Error occurred when parsing HTTP query params for filters'):
        # type: (List[unicode], unicode) -> None
        self.message = message
        self.error_list = error_list


class FilterParserComponent(AbstractComponent):
    def __init__(self, logger, webui):
        # type: (PartLogger, WebuiBroker) -> None
        self.logger = logger.get_sub_part(u'FILTER_PARSER')
        self.webui = webui
        self._filter_query = {}  # type: Dict[unicode, FilterQuery]
    
    
    def init(self):
        # type: () -> None
        self._init_filter_query()
    
    
    def _init_filter_query(self):
        # type: () -> None
        pass
    
    
    def set_default_values(self, filter_dict):
        # type: (Dict[unicode, Dict[unicode, unicode]]) -> None
        pass
    
    
    def make_filter_dict_from_request(self, default_values=None):
        # type: (Optional[Dict[unicode, Dict[unicode, unicode]]]) -> Dict[unicode, Dict[unicode, unicode]]
        
        start_time = time.time()
        query_params = cast('MultiDict', self.webui.request.GET)  # type: MultiDict
        
        filter_dict = {}  # type: Dict[unicode, Dict[unicode, unicode]]
        self.set_default_values(filter_dict)
        if default_values:
            for filter_name, filter_item_dict in default_values.iteritems():
                filter_dict.setdefault(filter_name, {}).update(filter_item_dict)
        error_list = []  # type: List[unicode]
        for filter_name, filter_query in self._filter_query.iteritems():
            filter_string = query_params.get(filter_name, u'')
            if filter_string:
                try:
                    filter_dict[filter_name] = self.parse_filter_param(filter_string, filter_query.valid_keys, filter_dict=filter_dict.get(filter_name, None), logger=self.logger)
                except FilterParserError as exc:
                    error_list.extend(u'Error when parsing [ %s ]: %s' % (filter_name, error_msg) for error_msg in exc.error_list)
        self.logger.log_perf(start_time, u'make_filter_manager_from_request', u'filter_dict = [ %s ]' % filter_dict)
        if error_list:
            raise FilterParserError(error_list)
        return filter_dict
    
    
    @staticmethod
    def parse_filter_param(filter_string_to_parse, valid_keys, filter_dict=None, logger=None):
        # type: (unicode, Sequence[unicode], Optional[Dict[unicode, unicode]], Optional[PartLogger]) -> Dict[unicode, unicode]
        if filter_dict is None:
            filter_dict = {}
        error_list = []  # type: List[unicode]
        for key_value_pair_string in filter_string_to_parse.strip().split(u'~'):
            key_value_pair_string = key_value_pair_string.strip()
            key_value_pair = [element.strip() for element in key_value_pair_string.split(u':')]  # type: List[unicode]
            if len(key_value_pair) != 2:
                msg = u'Invalid format, expected:[ name:string ], got:[ %s ]' % key_value_pair_string
                if logger:
                    logger.error(msg)
                error_list.append(msg)
                continue
            key, value = key_value_pair
            if not key or not value:
                msg = u'Invalid format, expected:[ name:string ], got:[ %s ]' % key_value_pair_string
                if logger:
                    logger.error(msg)
                error_list.append(msg)
                continue
            if not valid_keys or key not in valid_keys:
                msg = u'Unknown key:[ %s ]' % key
                if logger:
                    logger.error(msg)
                error_list.append(msg)
                continue
            filter_dict[key] = value
        if error_list:
            raise FilterParserError(error_list)
        return filter_dict
