#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

import collections
import datetime
import os
import re

from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from shinken.misc.type_hint import List, Literal, Final

DEFAULT_INSTALL_ARTIFACT_DIRECTORY = u'~/shinken/versions_and_patch_installations'

INSTALL_TIME_LOG_FILENAME = u'install_time.log'

INSTALL_DIRECTORY_REGEX = re.compile(r'(\d+-\d+-\d+-\d+h\d+m\d+)-(install|update)-(.+)')
INSTALL_DATETIME_FORMAT = u'%Y-%m-%d-%Hh%Mm%S'


class InstallationArtifact(collections.namedtuple('Installation', ['directory', 'datetime', 'type', 'version'])):
    if TYPE_CHECKING:
        directory = None  # type: Final[unicode]
        datetime = None  # type: Final[datetime.datetime]
        type = None  # type: Final[Literal[u'install', u'update']]
        version = None  # type: Final[unicode]


def get_installation_logs_directory(install_dir):
    # type: (unicode) -> unicode
    logs_dir = os.path.join(install_dir, u'update_logs')
    if not os.path.isdir(logs_dir):
        logs_dir = os.path.join(install_dir, u'installation_logs')
    return logs_dir


def get_installation_artifacts(base_install_dir=DEFAULT_INSTALL_ARTIFACT_DIRECTORY):
    # type: (unicode) -> List[InstallationArtifact]
    
    artifacts = []  # type: List[InstallationArtifact]
    
    for install_dirname in sorted(os.listdir(base_install_dir)):
        install_dirname_match = INSTALL_DIRECTORY_REGEX.match(install_dirname)
        if install_dirname_match is None:
            continue
        install_dirname = os.path.join(base_install_dir, install_dirname)
        artifacts.append(InstallationArtifact(
            datetime=datetime.datetime.strptime(install_dirname_match.group(1), INSTALL_DATETIME_FORMAT),
            type=install_dirname_match.group(2),
            version=install_dirname_match.group(3),
            directory=os.path.join(base_install_dir, install_dirname),
        ))
    
    return artifacts
