#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

from abc import ABCMeta, abstractmethod
from collections import OrderedDict

from shinken.misc.fast_copy import fast_deepcopy
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.mass_actions.exception import ActionAlreadyRegistered
from shinkensolutions.validators.validation import ValidationMessages

if TYPE_CHECKING:
    from shinken.log import PartLogger
    from shinken.misc.type_hint import Any, Dict, List, Union, Optional
    from .validator import AbstractActionValidator
    
    ErrorReport = List[unicode]
    CallbackErrorReport = Union[ErrorReport, None]


class ActionResult(object):
    def __init__(self, action_done, messages=None, extra_information=None):
        # type: (bool, Optional[ValidationMessages], Optional[Dict]) -> None
        self.messages = messages or ValidationMessages(warnings=[], errors=[], critical=[])
        self.action_done = action_done
        self.extra_information = extra_information
    
    
    def as_dict(self):
        # type: () -> Dict[unicode, Any]
        to_return = dict(action_done=self.action_done, **self.messages.as_dict())
        if self.extra_information:
            to_return.update(self.extra_information)
        return to_return


class MassActionsResult(object):
    def __init__(self):
        self._result_dict = OrderedDict()  # type: OrderedDict[unicode, ActionResult]
    
    
    def as_dict(self):
        # type: () -> Dict[unicode, Dict[unicode, Any]]
        return dict((action, validation.as_dict()) for action, validation in self._result_dict.iteritems())
    
    
    def add_result(self, action, result):
        # type: (unicode, ActionResult) -> None
        self._result_dict[action] = result
    
    
    def get_nb_critical(self):
        # type: () -> int
        return sum(v.messages.get_nb_critical() for v in self._result_dict.itervalues())
    
    
    def get_nb_error(self, include_critical=False):
        # type: (bool) -> int
        return sum(v.messages.get_nb_error(include_critical=include_critical) for v in self._result_dict.itervalues())
    
    
    def get_nb_warning(self):
        # type: () -> int
        return sum(v.messages.get_nb_warning() for v in self._result_dict.itervalues())
    
    
    def has_critical(self):
        # type: () -> bool
        return bool(self.get_nb_critical())
    
    
    def has_error(self, include_critical=False):
        # type: (bool) -> bool
        return bool(self.get_nb_error(include_critical=include_critical))
    
    
    def has_warning(self):
        # type: () -> bool
        return bool(self.get_nb_warning())


class AbstractMassActionsHandler(object):
    __metaclass__ = ABCMeta
    
    
    def __init__(self):
        self.__actions_register = OrderedDict()  # type: OrderedDict[unicode, AbstractActionValidator]
    
    
    def register(self, action, validator):
        # type: (unicode, AbstractActionValidator) -> None
        if action in self.__actions_register:
            raise ActionAlreadyRegistered(action)
        self.__actions_register[action] = validator
    
    
    def _setup(self):
        # type: () -> None
        pass
    
    
    @abstractmethod
    def _execute(self, all_actions, logger):
        # type: (Dict[unicode, Any], Optional[PartLogger]) -> Dict[unicode, ActionResult]
        """
        Execute all registered actions
        The implementation assumed the action object is validated and parsed
        """
        raise NotImplementedError()
    
    
    def _teardown(self):
        # type: () -> None
        pass
    
    
    def apply(self, action_dict, logger=None):
        # type: (Dict[unicode, Dict[unicode, Any]], Optional[PartLogger]) -> MassActionsResult
        actions_register = self.__actions_register
        unknown_actions = [action for action in action_dict.iterkeys() if action not in actions_register]
        valid_actions = [action for action in action_dict.iterkeys() if action in actions_register]
        result_register = MassActionsResult()
        
        if unknown_actions and logger:
            logger.error(u'Unknown actions caught: %s' % unknown_actions)
        for action in unknown_actions:
            result_register.add_result(action, ActionResult(action_done=False, messages=ValidationMessages(errors=[u'Unknown action'])))
        
        self._setup()
        try:
            all_actions = {}  # type: Dict[unicode, Any]
            validation_messages_dict = {}  # type: Dict[unicode, ValidationMessages]
            for action, action_validator in ((action, actions_register[action]) for action in valid_actions):
                action_obj = fast_deepcopy(action_dict[action])
                validation_result = action_validator.validate(action_obj, logger=logger)
                if validation_result.should_be_done:
                    all_actions[action] = action_validator.pythonize(action_obj, logger=logger)
                validation_messages_dict[action] = validation_result.messages
            execution_validation_dict = self._execute(all_actions, logger)
            for action, validation_messages in validation_messages_dict.iteritems():
                if action in execution_validation_dict:
                    execution_validation_dict[action].messages += validation_messages
                else:
                    if logger:
                        logger.warning(u'Action [ %s ] do not have result from execution' % action)
                    execution_validation_dict[action] = ActionResult(action_done=False, messages=validation_messages)
            for action, action_result in execution_validation_dict.iteritems():
                result_register.add_result(action, action_result)
        finally:
            self._teardown()
        return result_register
