# !/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2021:
# This file is part of Shinken Enterprise, all rights reserved.

import os

import errno
import sys

# DO NOT IMPORT SHINKEN IN THIS FILE
TYPE_HINT = False
if TYPE_HINT:
    from shinken.misc.type_hint import List, Tuple
    from shinken.log import PartLogger

if sys.platform.startswith(u'win'):
    pwd = None
    grp = None
    
    import tempfile
    
    SHARE_ITEM_SYNC_FILE_PATH = os.path.join(tempfile.gettempdir(), u'share_item')
    
    
    def rename_file(source, target):
        # type: (unicode, unicode) -> None
        if os.path.exists(target):
            os.remove(target)
        os.rename(source, target)
    
    
    def get_cur_user_id():
        # type: () -> int
        return 0
    
    
    def get_cur_group_id():
        # type: () -> int
        return 0
    
    
    def get_cur_user_name():
        # type: () -> unicode
        return u'shinken'
    
    
    def get_cur_group_name():
        # type: () -> unicode
        return u'shinken'
    
    
    # noinspection PyUnusedLocal
    def get_user_name_from_id(user_id):
        # type: (int) -> unicode
        return u'shinken'
    
    
    # noinspection PyUnusedLocal
    def get_group_name_from_id(group_id):
        # type: (int) -> unicode
        return u'shinken'
    
    
    # noinspection PyUnusedLocal
    def get_user_id_from_name(user_name):
        # type: (unicode) -> int
        return 0
    
    
    # noinspection PyUnusedLocal
    def get_group_id_from_name(group_name):
        # type: (unicode) -> int
        return 0
    
    
    # noinspection PyUnusedLocal
    def get_file_owner_and_group_name(path):
        # type: (unicode) -> Tuple[unicode,unicode]
        return u'shinken', u'shinken'
    
    
    # noinspection PyUnusedLocal
    def set_ownership(path, user=u'shinken', group=u'shinken', is_link=False):
        # type: (unicode, unicode, unicode, bool) -> None
        pass
    
    
    def get_all_group():
        # type: () -> List
        return []
    
    
    # noinspection PyUnusedLocal
    def get_file_info(path):
        return 777, u'shinken', u'shinken'

else:
    import pwd
    import grp
    
    SHARE_ITEM_SYNC_FILE_PATH = u'/dev/shm/share_item'
    
    
    def rename_file(source, target):
        # type: (unicode, unicode) -> None
        os.rename(source, target)
    
    
    def get_cur_user_id():
        # type: () -> int
        return os.getuid()
    
    
    def get_cur_group_id():
        # type: () -> int
        return os.getgid()
    
    
    def get_cur_user_name():
        # type: () -> unicode
        return get_user_name_from_id(get_cur_user_id())
    
    
    def get_cur_group_name():
        # type: () -> unicode
        return get_group_name_from_id(get_cur_group_id())
    
    
    def get_user_name_from_id(user_id):
        # type: (int) -> unicode
        return pwd.getpwuid(user_id).pw_name
    
    
    def get_group_name_from_id(group_id):
        # type: (int) -> unicode
        return grp.getgrgid(group_id).gr_name
    
    
    def get_user_id_from_name(user_name):
        # type: (unicode) -> int
        try:
            return pwd.getpwnam(user_name.encode(u'utf-8')).pw_uid
        except KeyError:
            raise OSError(errno.EINVAL, u'This user does not exists', user_name)
    
    
    def get_group_id_from_name(group_name):
        # type: (unicode) -> int
        try:
            return grp.getgrnam(group_name.encode(u'utf-8')).gr_gid
        except KeyError:
            raise OSError(errno.EINVAL, u'This group does not exists', group_name)
    
    
    def get_file_owner_and_group_name(path):
        # type: (unicode) -> Tuple[unicode,unicode]
        stat = os.stat(path)
        user_id = stat.st_uid
        group_id = stat.st_gid
        
        return get_user_name_from_id(user_id), get_group_name_from_id(group_id)
    
    
    def set_ownership(path, user=u'shinken', group=u'shinken', is_link=False):
        # type: (unicode, unicode, unicode, bool) -> None
        
        if get_cur_user_id() != 0 and not (get_cur_user_name() == u'shinken' and user == u'shinken' and group in (u'shinken', u'apache')):
            return
        uid = get_user_id_from_name(user)
        gid = get_group_id_from_name(group)
        os.chown(path, uid, gid)
        if is_link:
            os.lchown(path, uid, gid)
    
    
    def get_all_group():
        # type: () -> List
        return grp.getgrall()
    
    
    def get_file_info(path):
        file_stat = os.stat(path)
        # thanks to https://github.com/naparuba/opsbro/blob/master/opsbro/compliancemgr.py#L69
        file_permissions = int(oct(file_stat.st_mode & 0o777)[1:])  # => to have something like 644
        user_id = file_stat.st_uid
        group_id = file_stat.st_gid
        return file_permissions, get_user_name_from_id(user_id), get_group_name_from_id(group_id)


def make_dirs_and_chown_shinken(directory, mode=0o777):
    # type: (unicode, int) -> None
    head, tail = os.path.split(directory)
    if not tail:
        head, tail = os.path.split(head)
    if head and tail and not os.path.exists(head):
        make_dirs_and_chown_shinken(head, mode)
        if tail == os.curdir:
            return
    
    try:
        os.mkdir(directory, mode)
    except OSError as e:
        # be happy if someone already created the path
        if e.errno != errno.EEXIST:
            raise
    
    if get_cur_user_id() == 0:
        set_ownership(directory, u'shinken', u'shinken')


def make_file_and_chown_shinken(file_path):
    # type: (unicode) -> None
    with open(file_path, u'w') as _file:
        _file.write('')
    set_ownership(file_path, u'shinken', u'shinken')


def set_shinken_owner_on_file_or_directory(path, logger=None):
    # type: (unicode, PartLogger) -> None
    try:
        set_ownership(path, u'shinken', u'shinken')
    except Exception as e:
        if logger:
            logger.error(u'Can\'t give ownership to shinken:shinken on [ %s ]' % path)
        raise e


def verify_if_shinken_own_file_or_directory(path, logger=None):
    # type: (unicode, PartLogger) -> bool
    if get_cur_user_id() != 0:
        # Non-root user cannot create file with different owner, no need to crash if write operation is possible
        return True
    
    user_owner, group_owner = get_file_owner_and_group_name(path)
    
    result = True
    if user_owner != u'shinken':
        if logger:
            logger.error(u'[ %s ] user owner is not [ shinken ] but [ %s ]' % (path, user_owner))
        result = False
    
    if group_owner != u'shinken':
        if logger:
            logger.error(u'[ %s ] group owner is not [ shinken ] but [ %s ]' % (path, group_owner))
        result = False
    
    return result


def verify_if_dir_or_file_exist(path, logger=None):
    if not os.path.exists(path):
        if logger:
            logger.error(u'[ %s ] file does not exist' % path)
        return False
    return True


def verify_file_exist_and_can_be_read_by_shinken(path, logger=None):
    # type: (unicode, PartLogger) -> bool
    if not verify_if_dir_or_file_exist(path, logger=logger):
        return False
    return verify_if_shinken_own_file_or_directory(path, logger=logger)
