#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019:
# This file is part of Shinken Enterprise, all rights reserved.
import random
import re

from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.toolbox.box_tools_array import ToolsBoxArray

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict, Optional, Any, Union
    from sre_parse import Pattern


class NEWLINE(object):
    UNIX = u'\n'
    HTML = u'<BR>'


XSS_SAFE = {
    u'<': '&lt;',
    u'>': '&gt;',
    u'"': '&quot;',
    u'/': '&#x2F;',
    u"'": '&#39;',
    
    # u"'": '&apos;', not working in outlook 2016 ...
    # u'″': '&#8243;',
    # u'′': '&#8242;'
}

XSS_SAFE_SPECIAL_QUOTE = {
    u'<': '&lt;',
    u'>': '&gt;',
    u'"': '&quot;',
    u"'": '&#39;',
    u'/': '&#x2F;',
    # it's for outlook 2016
}


class ToolsBoxString(object):
    @staticmethod
    def format_for_message(string, separator=u', ', last_separator=u'or'):
        # type: (unicode, unicode, unicode) -> unicode
        _to_return = u'%s %s %s' % (separator.join(string[:-1]), last_separator, string[len(string) - 1],)
        return _to_return
    
    
    @staticmethod
    def escape_XSS(value, to_test=None):
        # type: (Any, Dict) -> unicode
        if not isinstance(value, basestring):
            return value
        if not to_test:
            to_test = XSS_SAFE
        # Deactivated because of SEF-3389 and SEF-3394
        # value = value.replace('&', '&amp;')
        for k, v in to_test.iteritems():
            value = value.replace(k, v)
        return value
    
    
    @staticmethod
    def escape_XSS_special_quote(value):
        # type: (unicode) -> unicode
        return ToolsBoxString.escape_XSS(value, XSS_SAFE_SPECIAL_QUOTE)
    
    
    @staticmethod
    def unescape_XSS(value):
        # type: (unicode) -> unicode
        if not isinstance(value, basestring):
            return value
        for k, v in XSS_SAFE.iteritems():
            value = value.replace(v, k)
        return value
    
    
    @staticmethod
    def escape_XSS_not_rentrant(value):
        # type: (unicode) -> unicode
        # Same of escape XSS
        if not isinstance(value, basestring):
            return value
        return value.replace('&', '&amp;').replace('<', '&lt;').replace('>', '&gt;').replace('"', '&quot;').replace("'", '&#039;')
    
    
    @staticmethod
    def convert_newlines_to_html_line_break(value):
        # type: (unicode) -> unicode
        return value.replace(NEWLINE.UNIX, NEWLINE.HTML)
    
    
    @staticmethod
    def atoi(text):
        return int(text) if text.isdigit() else text.lower()
    
    
    @staticmethod
    def natural_keys(text):
        # sorts in human order
        # http://nedbatchelder.com/blog/200712/human_sorting.html
        return [ToolsBoxString.atoi(c) for c in re.split(r'(\d+)', text)]
    
    
    @staticmethod
    def parse_string_integer_to_integer_if_possible(value):
        # type: (unicode) -> Optional[int, unicode]
        if value is not None:
            try:
                return int(value)
            except ValueError:
                pass
        return value
    
    
    @staticmethod
    def parse_string_float_to_float_if_possible(value):
        # type: (unicode) -> Optional[float, unicode]
        if value is not None:
            try:
                return float(value)
            except ValueError:
                pass
        return value
    
    
    @staticmethod
    def random_string_4():
        return format(int((1 + random.random()) * 0x10000), u'x')[1:]
    
    
    @staticmethod
    def generate_ui_visu_uuid():
        return u'%s%s-%s-%s-%s-%s%s%s' % (
            ToolsBoxString.random_string_4(),
            ToolsBoxString.random_string_4(),
            ToolsBoxString.random_string_4(),
            ToolsBoxString.random_string_4(),
            ToolsBoxString.random_string_4(),
            ToolsBoxString.random_string_4(),
            ToolsBoxString.random_string_4(),
            ToolsBoxString.random_string_4()
        )


class IllegalString(object):
    
    def __init__(self, list_illegal_str, regex=u'', accent_authorized=True):
        # type: (unicode, Union[unicode, Pattern], bool) -> None
        self.value = list_illegal_str
        self.regex = regex or re.compile(r'''[%s]''' % u''.join(list_illegal_str))
        self.value_to_message = ToolsBoxArray.replace(self.value, u' ', u'SPACE')
        self.accent_authorized = accent_authorized
    
    
    def contains_illegal(self, string_to_test):
        # type: (unicode) -> bool
        if bool(string_to_test and self.regex.search(string_to_test)):
            return True
        if not self.accent_authorized and bool(string_to_test and re.findall(ur'[À-ÿ]', string_to_test)):
            return True
        return False
    
    
    def get_message_first_part(self, separator=u'''</span>, <span class='shinken-data-user'>'''):
        # type: (unicode) -> unicode
        return ToolsBoxString.format_for_message(self.value_to_message[:-1], separator=separator, last_separator=separator)
    
    
    def get_message_last_part(self):
        # type: () -> unicode
        return self.value_to_message[len(self.value_to_message) - 1]
    
    
    def get_message(self, separator=u'''</span>, <span class='shinken-data-user'>'''):
        # type: (unicode) -> unicode
        return ToolsBoxString.format_for_message(self.value_to_message, separator=separator, last_separator=separator)
