#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2023:
# This file is part of Shinken Enterprise, all rights reserved.

import json
import logging
import os

from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.hash_helper import hash_file
from shinkensolutions.os_helper import verify_file_exist_and_can_be_read_by_shinken

if TYPE_CHECKING:
    from shinken.misc.type_hint import List, Set, Optional, Dict
    from shinken.log import PartLogger
    from .resource_mapping_helper import LoadResourceInformation

RESOURCES_INFO_PATH = u'/etc/shinken-user/configuration/daemons/brokers/modules/webui/resources_info/notifications/sounds/'
USER_RESOURCE_DIRECTORY = u'/etc/shinken-user/resources/daemons/brokers/modules/webui/notifications/sounds/'


class ResourceInformation(object):
    fields = ('original_name', 'display_name', 'id', 'path', 'file_info_path', 'resource_hash')
    __slots__ = ('original_name', 'display_name', 'id', 'path', 'file_info_path', 'resource_hash')
    
    
    def __init__(self, **kwargs):
        for field in ResourceInformation.fields:
            setattr(self, field, kwargs.get(field, None))


class ResourcesID(object):
    GLOBAL_SOUND_NOTIFICATION = u'global_sound_notification'


KNOWN_RESOURCES_ID = (ResourcesID.GLOBAL_SOUND_NOTIFICATION,)


def load_all_resource_information(files_information_folder_path, logger):
    # type: (unicode, PartLogger) -> List[ResourceInformation]
    if not verify_file_exist_and_can_be_read_by_shinken(files_information_folder_path, logger=logger):
        return []
    
    already_load_resource_id = set()  # type: Set[unicode]
    all_resource_information = []  # type: List[ResourceInformation]
    for file_information_name in os.listdir(files_information_folder_path):
        file_information_path = os.path.join(files_information_folder_path, file_information_name)
        if not os.path.isfile(file_information_path):
            continue
        if not verify_file_exist_and_can_be_read_by_shinken(file_information_path, logger=logger):
            continue
        
        logger.debug(u'Found file information at : 〖%s〗' % file_information_path)
        try:
            resource_information_as_json = json.load(open(file_information_path, 'r'))
        except Exception as e:
            logger.warning(u'Fail to read file 〖%s〗 because : 〖%s〗' % (file_information_path, e))
            logger.print_stack(level=logging.WARNING)
            continue
        
        resource_information_as_json[u'file_info_path'] = file_information_path
        resource_path = resource_information_as_json.get(u'path', u'')
        resource_hash = None
        if resource_path and verify_file_exist_and_can_be_read_by_shinken(resource_path):
            try:
                resource_hash = hash_file(resource_path)
            except Exception as e:
                logger.warning(u'Fail to compute hash of file 〖%s〗 because 〖%s〗.' % (resource_path, e))
        resource_information_as_json[u'resource_hash'] = resource_hash
        for field in ResourceInformation.fields:
            if field not in resource_information_as_json:
                logger.warning(u'Missing field 〖%s〗 in file 〖%s〗' % (field, file_information_path))
                continue
        
        resource_information = ResourceInformation(**resource_information_as_json)
        if resource_information.id not in KNOWN_RESOURCES_ID:
            logger.warning(u'The id 〖%s〗 in file 〖%s〗 is not a known resource id. Known id are : 〖%s〗' % (resource_information.id, file_information_path, u','.join(KNOWN_RESOURCES_ID)))
            continue
        
        if resource_information.id in already_load_resource_id:
            logger.warning(u'There are 2 file with the same id 〖%s〗 in folder 〖%s〗.' % (resource_information.id, files_information_folder_path))
            for ri in all_resource_information[:]:
                if ri.id == resource_information.id:
                    all_resource_information.remove(ri)
            continue
        
        already_load_resource_id.add(resource_information.id)
        all_resource_information.append(resource_information)
    
    return all_resource_information


def compute_resource_id_to_reload(resource_mapping, all_resource_information):
    # type: (Dict[unicode, LoadResourceInformation], List[ResourceInformation]) -> Set[unicode]
    
    resource_id_to_reload = set()
    for resource_id in KNOWN_RESOURCES_ID:
        load_resource_information = resource_mapping.get(resource_id, None)  # type:Optional[LoadResourceInformation]
        
        # No load resources information for this resource_id so we must make one -> Must reload resource.
        if not load_resource_information:
            resource_id_to_reload.add(resource_id)
            continue
        
        # No resource information and the one loaded is not the delivery one -> Must reload resource.
        resource_information = next((ri for ri in all_resource_information if ri.id == resource_id), None)
        if not resource_information and load_resource_information.use_file_provided_by_shinken is False:
            resource_id_to_reload.add(resource_id)
            continue
        
        # New resource information and the one loaded is the delivery one -> Must reload resource.
        if resource_information and load_resource_information.use_file_provided_by_shinken is True:
            resource_id_to_reload.add(resource_id)
            continue
        
        # No resource information and the one loaded is the delivery one -> LOAD RESOURCE IS OK.
        if not resource_information and load_resource_information.use_file_provided_by_shinken is True:
            continue
        
        # original_name have change -> Must reload resource.
        if resource_information.original_name != load_resource_information.original_name:
            resource_id_to_reload.add(resource_id)
            continue
        
        # display_name have change -> Must reload resource.
        if resource_information.display_name != load_resource_information.display_name:
            resource_id_to_reload.add(resource_id)
            continue
        
        # path have change -> Must reload resource.
        if resource_information.path != load_resource_information.resource_path:
            resource_id_to_reload.add(resource_id)
            continue
        
        # resource hash have change -> Must reload resource.
        if resource_information.resource_hash != load_resource_information.resource_hash:
            resource_id_to_reload.add(resource_id)
            continue
    
    return resource_id_to_reload
