#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2023:
# This file is part of Shinken Enterprise, all rights reserved.

import json
import os
import time
from collections import namedtuple

import shinkensolutions.common_ui as common_ui_python_module
from shinken.exceptions.system import FailedToReadFile
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.os_helper import verify_file_exist_and_can_be_read_by_shinken

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict
    from shinken.log import PartLogger

SOUND_MAPPING_FILE_PATH = u'/var/lib/shinken/persistent_data/modules/webui/mappings/sounds/webui_sound_mapping.json'
LoadResourceInformation = namedtuple('LoadResourceInformation', ['use_file_provided_by_shinken', 'original_name', 'display_name', 'file_info_path', 'resource_hash', 'loading_date_time', 'resource_path'])

GLOBAL_SOUND_NOTIFICATION_DELIVERED_BY_SHINKEN_INFORMATION = LoadResourceInformation(
    use_file_provided_by_shinken=True,
    original_name=u'shinken_basic_notification.mp3',
    display_name=u'shinken basic notification',
    file_info_path=None,
    resource_hash=u'shinken_basic_notification',
    loading_date_time=int(time.time()),
    resource_path=os.path.join(common_ui_python_module.COMMON_UI_HTDOCS_DIRECTORY, u'sounds', u'shinken_basic_notification.mp3')
)


def load_resource_mapping_file(mapping_file_path, logger=None):
    # type: (unicode, PartLogger) -> Dict[unicode, LoadResourceInformation]
    if not verify_file_exist_and_can_be_read_by_shinken(mapping_file_path, logger):
        return {}
    
    all_mapping = {}
    try:
        all_mapping_as_json = json.load(open(mapping_file_path, 'r'))
    except Exception as e:
        raise FailedToReadFile(u'Fail to read file 〖%s〗 because : 〖%s〗' % (mapping_file_path, e))
    
    for resource_id, load_resource_information_as_json in all_mapping_as_json.iteritems():
        for field in LoadResourceInformation._fields:
            if field not in load_resource_information_as_json:
                raise FailedToReadFile(u'File 〖%s〗 is malformed. Missing field : 〖%s〗 in load resource information.' % (mapping_file_path, field))
        
        all_mapping[resource_id] = LoadResourceInformation(**load_resource_information_as_json)
    return all_mapping
