#!/usr/bin/python-shinken
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

import os
import re
import traceback

from shinken.log import LoggerFactory
from shinken.misc.type_hint import TYPE_CHECKING
from .versions.patch_decorator import is_weather_format_patch

if TYPE_CHECKING:
    from shinken.log import PartLogger
    from shinken.misc.type_hint import Any, List, Callable, Dict, Optional
    
    WeatherVersionType = int
    WeatherDictType = Dict[unicode, Any]
    MigrationHookFunctionType = Callable[[WeatherDictType], None]

default_logger = LoggerFactory.get_logger(u'WEATHER JSON MIGRATOR')

VERSION_DIRECTORY_NAME_PATTERN = re.compile(r'^(version_([0-9]{4,}))$')
VERSIONS_PKG_NAME = u'versions'
PATCHES_MODULE_NAME = u'patches'


class WeatherJSONMigrator(object):
    """
    The weather JSON migrator will load all the migration hooks stored in the subpackage 'versions'

    Each migration versions must have the following tree:
    versions/
    ├─ version_XXXX/    # Where 'XXXX' is the target version (ex: 0001)
    │  ├─ __init__.py
    │  ├─ patches.py    # All functions decorated with @weather_format_patches must be stored here
    ├─ __init__.py
    ├─ patch_decorator.py
    """
    
    __migration_dict = {}  # type: Dict[WeatherVersionType, List[MigrationHookFunctionType]]
    
    
    def __new__(cls):
        # Lazy singleton instance
        try:
            instance = getattr(cls, u'__instance__')
        except AttributeError:
            instance = object.__new__(cls)
            instance.init_migration_dict()
            setattr(cls, u'__instance__', instance)
        return instance
    
    
    def apply_migrations(self, weather_dict, version, logger=None):
        # type: (WeatherDictType, WeatherVersionType, Optional[PartLogger]) -> None
        """
        Apply migrations loaded from plugins with the version :version: to :weather_dict: WITHOUT verifying the actual version
        The function will not do anything if the version does not exist
        """
        if not logger:
            logger = default_logger
        else:
            logger = logger.get_sub_part(u'WEATHER JSON MIGRATOR')
        logger.info(u'Applying migration patches for version [ %d ]' % version)
        for patch in self.__migration_dict.get(version, []):
            patch_name = getattr(patch, u'__name__', repr(patch))
            try:
                patch(weather_dict)
            except:
                logger.error(u'--> Error when applying patch [ %s ]' % patch_name)
                logger.error(u'--> Backtrace: %s' % traceback.format_exc())
                raise
            logger.info(u'--> Successfully applied patch [ %s ]' % patch_name)
        
        weather_identity = weather_dict.setdefault(u'identification', {})
        if not isinstance(weather_identity, dict):
            weather_dict[u'identification'] = weather_identity = dict()
        weather_identity[u'weather_format_version'] = version
    
    
    @classmethod
    def init_migration_dict(cls):
        # type: () -> None
        
        cls.__migration_dict.clear()
        
        # Represents folders in ./versions/*
        versions_directory = os.path.join(os.path.dirname(__file__), VERSIONS_PKG_NAME)
        
        logger = default_logger
        logger.debug(u'Version directory: [ %s ]' % versions_directory)
        
        for filename in os.listdir(versions_directory):
            match = VERSION_DIRECTORY_NAME_PATTERN.match(filename)
            if not match:
                continue
            module_name = match.group(1)
            if not os.path.join(versions_directory, filename, u'__init__.py'):  # Not a regular Python package
                continue
            patches_filepath = os.path.join(versions_directory, filename, u'%s.py' % PATCHES_MODULE_NAME)
            if not os.path.isfile(patches_filepath):  # Not a regular migration package
                continue
            patch_version = int(match.group(2))
            module_name = u'.'.join([__package__, VERSIONS_PKG_NAME, module_name, PATCHES_MODULE_NAME])  # Full module name from top-level
            logger.debug(u'Loading patches for weather format version [ %d ]' % patch_version)
            logger.debug(u'--> Loading patches from module [ %s ]' % module_name)
            patches_module = cls._import_module(module_name)
            for func_ptr in filter(is_weather_format_patch, vars(patches_module).itervalues()):
                migration_list = cls.__migration_dict.setdefault(patch_version, [])
                migration_list.append(func_ptr)
                logger.debug(u'----> Patch [ %s ] imported' % getattr(func_ptr, u'__name__', repr(func_ptr)))
    
    
    @staticmethod
    def _import_module(name):
        # type: (unicode) -> Any
        """Helper to import a module programmatically"""
        # noinspection PyTypeChecker
        # Give [None] to 'from_list' parameter is required to get the imported module and not the top-level package
        return __import__(name, globals(), None, [None], -1)
