#!/usr/bin/python-shinken
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.misc.type_hint import TYPE_CHECKING
from ....utils.weather_tools import travel_dictionaries, change_existing_value_if_equals_to

if TYPE_CHECKING:
    from shinken.misc.type_hint import Any, Dict, List, Optional


# WEATHER CONSTANTS ARE FORBIDDEN HERE

def _parse_existing(grid, path, default_value=u'default'):
    # type: (Dict, unicode, Optional[Any]) -> Optional[Any]
    return travel_dictionaries(grid, path, default_value=default_value)


def _safe_integer_conversion(value):
    # type: (unicode) -> unicode
    if value in [u'default', u'all']:  # Special case -> Specific values that are not integers
        return value
    try:
        integer_value = int(value)
    except (TypeError, ValueError):  # Conversion could not be done, keep the actual value
        return value
    return unicode(integer_value)


def generate_new_grids(weather_grids):
    # type: (List[Dict]) -> List[Dict]
    
    to_return = []
    for grid in weather_grids:
        to_return.append({
            u'grid_elements'     : _parse_grid_elements(grid) or _parse_existing(grid, u'grid_elements', []),
            u'nb_tiles_in_height': _parse_nb_tiles_in_height(grid) or _parse_existing(grid, u'nb_tiles_in_height'),
            u'nb_tiles_in_width' : _parse_nb_tiles_in_width(grid) or _parse_existing(grid, u'nb_tiles_in_width'),
            u'layouts'           : _parse_grid_layouts(grid) or _parse_existing(grid, u'layouts', {}),
            u'type'              : _parse_grid_type(grid) or _parse_existing(grid, u'type', u'grid')
        })
    return to_return


def _parse_grid_elements(grid):
    # type: (Dict) -> List
    to_return = []
    for grid_element in _parse_existing(grid, u'grid_elements', []):
        widget_type = _parse_existing(grid_element, u'content.type')
        to_return.append({
            u'position_x': grid_element[u'position_x'],
            u'position_y': grid_element[u'position_y'],
            u'width'     : _safe_integer_conversion(_parse_existing(grid_element, u'width')),
            u'height'    : _safe_integer_conversion(_parse_existing(grid_element, u'height')),
            u'content'   : _parse_widget_content(widget_type, grid_element)
        })
    
    return to_return


def _parse_ordering(grid):
    # type: (Dict) -> unicode
    to_return = u''
    # if weather_version == 1:
    #     to_return = grid[WEATHER_CONSTS.WEATHER__GRIDS__ORDERING]
    
    return to_return


def _parse_nb_tiles_in_height(grid):
    # type: (Dict) -> unicode
    to_return = u''
    # if weather_version == 1:
    #     to_return = grid[WEATHER_CONSTS.WEATHER__GRIDS__NB_TILES_IN_HEIGHT]
    
    return to_return


def _parse_nb_tiles_in_width(grid):
    # type: (Dict) -> unicode
    to_return = u''
    # if weather_version == 1:
    #     to_return = grid[WEATHER_CONSTS.WEATHER__GRIDS__NB_TILES_IN_WIDTH]
    
    return to_return


def _parse_grid_layouts(grid):
    # type: (Dict) -> Dict
    return {
        u'weather_widget_layout'  : _parse_weather_widget_layouts(grid),
        u'title_widget_layout'    : _parse_title_widget_layouts(grid),
        u'separator_widget_layout': _parse_separator_widget_layouts(grid)
    }


def _parse_grid_type(grid):
    # type: (Dict) -> unicode
    to_return = u''
    # if weather_version == 1:
    #     to_return = grid[WEATHER_CONSTS.WEATHER__GRIDS__TYPE]
    
    return to_return


_BACKGROUND_VALUES_RENAMES = {
    u'from_webui': u'status_context_from_webui',
    u'none'      : u'no_background'
}

_STATE_ICON_VALUES_RENAMES = {
    u'from_webui': u'status_context'
}


def _replace_value_according_to_dict(value, reference_dict):
    # type: (unicode, Dict[unicode, unicode]) -> unicode
    
    for former_value, new_value in reference_dict.iteritems():
        value = change_existing_value_if_equals_to(value, former_value, new_value)
    
    return value


def _parse_existing_layout(grid, widget_type, dotted_keys_from_layout, default_value=u'default'):
    # type: (Dict, unicode, unicode, Any) -> Any
    """
    Due to some magical happening thing, from Weather format 1 to 2,
    some weathers already have the 'layout' field whereas others have the former field 'default_values_for_grid_elements' :)
    """
    dotted_keys_format_1 = u'default_values_for_grid_elements.%s' % widget_type
    dotted_keys_format_2 = u'layouts.%s_layout' % widget_type
    
    return _parse_existing(
        grid,
        '.'.join([dotted_keys_format_2, dotted_keys_from_layout]),
        _parse_existing(grid, '.'.join([dotted_keys_format_1, dotted_keys_from_layout]), default_value)
    )


def _parse_weather_widget_layouts(grid):
    # type: (Dict) -> Dict
    widget_type = u'weather_widget'
    return {
        u'width'  : _safe_integer_conversion(_parse_existing_layout(grid, widget_type, u'width')),
        u'height' : _safe_integer_conversion(_parse_existing_layout(grid, widget_type, u'height')),
        u'content': {
            u'layout': {
                u'background'            : _replace_value_according_to_dict(_parse_existing_layout(grid, widget_type, u'content.background'), _BACKGROUND_VALUES_RENAMES),
                u'information_to_display': {
                    u'left_area'     : {
                        u'icon_type': _replace_value_according_to_dict(_parse_existing_layout(grid, widget_type, u'content.state_icon'), _STATE_ICON_VALUES_RENAMES)
                    },
                    u'top_right_area': {
                        u'displayed': u'default',
                        u'icon_type': u'default'
                    },
                    u'sla_area'      : {
                        u'displayed': _parse_existing_layout(grid, widget_type, u'content.display_sla'),
                        u'show_icon': u'default'
                    },
                    u'name_area'     : {
                        u'property_used_as_name': change_existing_value_if_equals_to(_parse_existing_layout(grid, widget_type, u'content.property_used_as_name'), u'definition_name', u'item_name'),
                        u'nb_lines_max'         : u'default',
                        u'vertical_align'       : u'default',
                        u'font_zoom_percent'    : u'default'
                    }
                }
            }
        }
    }


def _parse_title_widget_layouts(grid):
    # type: (Dict) -> Dict
    widget_type = u'title_widget'
    return {
        u'width'  : _safe_integer_conversion(_parse_existing_layout(grid, widget_type, u'width')),
        u'height' : _safe_integer_conversion(_parse_existing_layout(grid, widget_type, u'height')),
        u'content': {
            u'layout': {
                u'information_to_display': {
                    u'title_area': {
                        u'font_zoom_percent': u'default',
                        u'text_color'       : _parse_existing_layout(grid, widget_type, u'content.text_color')
                    }
                }
            }
        }
    }


def _parse_separator_widget_layouts(grid):
    # type: (Dict) -> Dict
    widget_type = u'separator_widget'
    return {
        u'width'  : _safe_integer_conversion(_parse_existing_layout(grid, widget_type, u'width')),
        u'height' : _safe_integer_conversion(_parse_existing_layout(grid, widget_type, u'height')),
        u'content': {
            u'layout': {
                u'information_to_display': {
                    u'left_area' : {
                        u'displayed'    : u'default',
                        u'width_percent': _parse_existing_layout(grid, widget_type, u'content.start_zone_text_percent')
                    },
                    u'title_area': {
                        u'text_color'       : _parse_existing_layout(grid, widget_type, u'content.text_color'),
                        u'text_align'       : _parse_existing_layout(grid, widget_type, u'content.text_align'),
                        u'font_zoom_percent': u'default'
                    },
                    u'right_area': {
                        u'displayed'    : u'default',
                        u'width_percent': _parse_existing_layout(grid, widget_type, u'content.end_zone_text_percent')
                    }
                }
            }
        }
    }


def _parse_widget_content(widget_type, grid_element):
    # type: (unicode, Dict) -> Dict
    
    to_return = {
        u'type'  : widget_type,
        u'layout': _parse_widget_content_layout(widget_type, grid_element) or _parse_existing(grid_element, 'content', {})
    }
    if widget_type == u'weather_widget':
        to_return[u'item'] = _parse_existing(grid_element, u'content.item', {})
    
    return to_return


def _parse_widget_content_layout(widget_type, grid_element):
    # type: (unicode, Dict) -> Dict
    to_return = {}
    if widget_type == u'title_widget':
        to_return = _parse_title_widget_content_layout(grid_element)
    elif widget_type == u'separator_widget':
        to_return = _parse_separator_widget_content_layout(grid_element)
    elif widget_type == u'weather_widget':
        to_return = _parse_weather_widget_content_layout(grid_element)
    
    return to_return


def _parse_title_widget_content_layout(grid_element):
    # type: (Dict) -> Dict
    return {
        u'information_to_display': {
            u'title_area': {
                u'label'            : _parse_existing(grid_element, u'content.title'),
                u'text_color'       : _parse_existing(grid_element, u'content.text_color'),
                u'font_zoom_percent': u'default'
            }
        }
    }


def _parse_separator_widget_content_layout(grid_element):
    # type: (Dict) -> Dict
    return {
        u'information_to_display': {
            u'left_area' : {
                u'width_percent': _parse_existing(grid_element, u'content.start_zone_text_percent')
            },
            u'title_area': {
                u'text_color': _parse_existing(grid_element, u'content.text_color'),
                u'text_align': _parse_existing(grid_element, u'content.text_align'),
                u'label'     : _parse_existing(grid_element, u'content.title')
            },
            u'right_area': {
                u'width_percent': _parse_existing(grid_element, u'content.end_zone_text_percent')
            }
        }
    }


def _parse_weather_widget_content_layout(grid_element):
    # type: (Dict) -> Dict
    return {
        u'background'            : _replace_value_according_to_dict(_parse_existing(grid_element, u'content.background'), _BACKGROUND_VALUES_RENAMES),
        u'information_to_display': {
            u'left_area': {
                u'icon_type': _replace_value_according_to_dict(_parse_existing(grid_element, u'content.state_icon'), _STATE_ICON_VALUES_RENAMES)
            },
            u'sla_area' : {
                u'displayed': _parse_existing(grid_element, u'content.display_sla'),
            },
            u'name_area': {
                u'property_used_as_name': change_existing_value_if_equals_to(_parse_existing(grid_element, u'content.property_used_as_name'), u'definition_name', u'item_name'),
            }
        }
    }
