#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.
import uuid
from copy import deepcopy

from shinken.misc.type_hint import TYPE_CHECKING, Union
from webui_module_service_weather.objects.weather_objects.weather_constants import WEATHER_CONSTS
from .version import WEATHER_VERSION

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict, Literal, Optional
    
    DefaultLiteral = Literal[u'default']
    
    IntDefault = Union[int, DefaultLiteral]
    BoolDefault = Union[bool, DefaultLiteral]
    WidthType = Union[IntDefault, Literal[u'all']]


class WeatherGenerator(object):
    @classmethod
    def new_weather_skeleton(cls, weather_uuid=None, weather_shared_group=None):
        # type: (Optional[unicode], Optional[unicode]) -> Dict
        if not weather_uuid:
            weather_uuid = uuid.uuid4().hex
        return {
            WEATHER_CONSTS.WEATHER__IDENTITY: {
                WEATHER_CONSTS.WEATHER__IDENTITY__TYPE          : u'service_weather',
                WEATHER_CONSTS.WEATHER__IDENTITY__FORMAT_VERSION: WEATHER_VERSION.WEATHER_FORMAT_VERSION,
                WEATHER_CONSTS.WEATHER__IDENTITY__VERSION       : WEATHER_VERSION.DEFAULT_WEATHER_VERSION,
                WEATHER_CONSTS.WEATHER__IDENTITY__UUID          : weather_uuid,
                WEATHER_CONSTS.WEATHER__IDENTITY__SHARED_GROUP  : weather_shared_group or u'service_weather'
            }
        }
    
    
    @classmethod
    def create_widget(cls, widget_type, position_x, position_y, width=u'default', height=u'default'):
        # type: (unicode, int, int, WidthType, IntDefault) -> Dict
        return {
            WEATHER_CONSTS.WEATHER__WIDGETS__POSITION_X: position_x,
            WEATHER_CONSTS.WEATHER__WIDGETS__POSITION_Y: position_y,
            WEATHER_CONSTS.WEATHER__WIDGETS__WIDTH     : width,
            WEATHER_CONSTS.WEATHER__WIDGETS__HEIGHT    : height,
            WEATHER_CONSTS.WEATHER__WIDGETS__CONTENT   : {
                WEATHER_CONSTS.WEATHER__WIDGETS__TYPE  : widget_type,
                WEATHER_CONSTS.WEATHER__WIDGETS__LAYOUT: {}
            }
        }
    
    
    @classmethod
    def add_title_widget_to_grid(cls, grid_dict, position_x, position_y, label, width=u'default', height=u'default', text_color=u'default', text_align=u'default', font_zoom_percent=u'default'):
        # type: (Dict, int, int, unicode, WidthType, IntDefault, unicode, unicode, IntDefault) -> Dict
        grid_elements = grid_dict.setdefault(WEATHER_CONSTS.WEATHER__GRID_ELEMENTS, [])
        _title_widget = cls.create_widget(WEATHER_CONSTS.WEATHER__WIDGETS__TITLE_WIDGET, position_x, position_y, width, height)
        _title_widget[WEATHER_CONSTS.WEATHER__WIDGETS__CONTENT][WEATHER_CONSTS.WEATHER__WIDGETS__LAYOUT][WEATHER_CONSTS.WEATHER__WIDGETS__INFORMATION_TO_DISPLAY] = {
            WEATHER_CONSTS.WEATHER__WIDGETS__TITLE_WIDGET__TITLE_AREA: {
                WEATHER_CONSTS.WEATHER__WIDGETS__TITLE_WIDGET__LABEL             : label,
                WEATHER_CONSTS.WEATHER__WIDGETS__COMMON_WIDGET__TEXT_COLOR       : text_color,
                WEATHER_CONSTS.WEATHER__WIDGETS__COMMON_WIDGET__TEXT_ALIGN       : text_align,
                WEATHER_CONSTS.WEATHER__WIDGETS__COMMON_WIDGET__FONT_ZOOM_PERCENT: font_zoom_percent
            }
        }
        grid_elements.append(_title_widget)
        return _title_widget
    
    
    @classmethod
    def add_separator_widget_to_grid(cls, grid_dict, position_x, position_y, label, width=u'default', height=u'default', left_area_displayed=u'default', left_area_width_percent=u'default', text_color=u'default', text_align=u'default',
                                     font_zoom_percent=u'default', right_area_displayed=u'default', right_area_width_percent=u'default'):
        # type: (Dict, int, int, unicode, WidthType, IntDefault, BoolDefault, IntDefault, unicode, unicode, IntDefault, BoolDefault, IntDefault) -> Dict
        grid_elements = grid_dict.setdefault(WEATHER_CONSTS.WEATHER__GRID_ELEMENTS, [])
        _separator_widget = cls.create_widget(WEATHER_CONSTS.WEATHER__WIDGETS__SEPARATOR_WIDGET, position_x, position_y, width, height)
        _separator_widget[WEATHER_CONSTS.WEATHER__WIDGETS__CONTENT][WEATHER_CONSTS.WEATHER__WIDGETS__LAYOUT][WEATHER_CONSTS.WEATHER__WIDGETS__INFORMATION_TO_DISPLAY] = {
            WEATHER_CONSTS.WEATHER__WIDGETS__SEPARATOR_WIDGET__LEFT_AREA : {
                WEATHER_CONSTS.WEATHER__WIDGETS__COMMON_WIDGET__DISPLAY         : left_area_displayed,
                WEATHER_CONSTS.WEATHER__WIDGETS__SEPARATOR_WIDGET__WIDTH_PERCENT: left_area_width_percent
            },
            WEATHER_CONSTS.WEATHER__WIDGETS__SEPARATOR_WIDGET__TITLE_AREA: {
                WEATHER_CONSTS.WEATHER__WIDGETS__SEPARATOR_WIDGET__LABEL         : label,
                WEATHER_CONSTS.WEATHER__WIDGETS__COMMON_WIDGET__TEXT_COLOR       : text_color,
                WEATHER_CONSTS.WEATHER__WIDGETS__COMMON_WIDGET__TEXT_ALIGN       : text_align,
                WEATHER_CONSTS.WEATHER__WIDGETS__COMMON_WIDGET__FONT_ZOOM_PERCENT: font_zoom_percent
            },
            WEATHER_CONSTS.WEATHER__WIDGETS__SEPARATOR_WIDGET__RIGHT_AREA: {
                WEATHER_CONSTS.WEATHER__WIDGETS__COMMON_WIDGET__DISPLAY         : right_area_displayed,
                WEATHER_CONSTS.WEATHER__WIDGETS__SEPARATOR_WIDGET__WIDTH_PERCENT: right_area_width_percent
            }
        }
        grid_elements.append(_separator_widget)
        return _separator_widget
    
    
    @classmethod
    def add_weather_widget_to_grid(cls, grid_dict, position_x, position_y, item_type, width=u'default', height=u'default', item_uuid=u'', item_name=u'', background=u'default', left_area_icon_type=u'default', top_right_area_displayed=u'default',
                                   top_right_area_icon_type=u'default',
                                   sla_area_displayed=u'default',
                                   sla_area_show_icon=u'default',
                                   name_area_property_used_as_name=u'default',
                                   name_area_nb_lines_max=u'default',
                                   name_area_vertical_align=u'default',
                                   name_area_font_zoom_percent=u'default'):
        # type: (Dict, int, int, unicode, WidthType, IntDefault, unicode, unicode, unicode, unicode, BoolDefault, unicode, BoolDefault, BoolDefault, unicode, IntDefault, unicode, IntDefault) -> Dict
        grid_elements = grid_dict.setdefault(WEATHER_CONSTS.WEATHER__GRID_ELEMENTS, [])
        _weather_widget = cls.create_widget(WEATHER_CONSTS.WEATHER__WIDGETS__WEATHER_WIDGET, position_x, position_y, width, height)
        _weather_widget[WEATHER_CONSTS.WEATHER__WIDGETS__CONTENT][WEATHER_CONSTS.WEATHER__WIDGETS__WEATHER_WIDGET__ITEM] = {
            WEATHER_CONSTS.WEATHER__WIDGETS__WEATHER_WIDGET__ITEM__ITEM_NAME: item_name,
            WEATHER_CONSTS.WEATHER__WIDGETS__WEATHER_WIDGET__ITEM__ITEM_UUID: item_uuid,
            WEATHER_CONSTS.WEATHER__WIDGETS__WEATHER_WIDGET__ITEM__ITEM_TYPE: item_type
        }
        _weather_widget[WEATHER_CONSTS.WEATHER__WIDGETS__CONTENT][WEATHER_CONSTS.WEATHER__WIDGETS__LAYOUT] = {
            WEATHER_CONSTS.WEATHER__WIDGETS__WEATHER_WIDGET__BACKGROUND: background,
            WEATHER_CONSTS.WEATHER__WIDGETS__INFORMATION_TO_DISPLAY    : {
                WEATHER_CONSTS.WEATHER__WIDGETS__WEATHER_WIDGET__LEFT_AREA     : {
                    WEATHER_CONSTS.WEATHER__WIDGETS__WEATHER_WIDGET__ICON_TYPE: left_area_icon_type
                },
                WEATHER_CONSTS.WEATHER__WIDGETS__WEATHER_WIDGET__TOP_RIGHT_AREA: {
                    WEATHER_CONSTS.WEATHER__WIDGETS__COMMON_WIDGET__DISPLAY   : top_right_area_displayed,
                    WEATHER_CONSTS.WEATHER__WIDGETS__WEATHER_WIDGET__ICON_TYPE: top_right_area_icon_type
                },
                WEATHER_CONSTS.WEATHER__WIDGETS__WEATHER_WIDGET__SLA_AREA      : {
                    WEATHER_CONSTS.WEATHER__WIDGETS__COMMON_WIDGET__DISPLAY   : sla_area_displayed,
                    WEATHER_CONSTS.WEATHER__WIDGETS__WEATHER_WIDGET__SHOW_ICON: sla_area_show_icon
                },
                WEATHER_CONSTS.WEATHER__WIDGETS__WEATHER_WIDGET__NAME_AREA     : {
                    WEATHER_CONSTS.WEATHER__WIDGETS__WEATHER_WIDGET__PROPERTY_USED_AS_NAME: name_area_property_used_as_name,
                    WEATHER_CONSTS.WEATHER__WIDGETS__COMMON_WIDGET__NB_LINES_MAX          : name_area_nb_lines_max,
                    WEATHER_CONSTS.WEATHER__WIDGETS__COMMON_WIDGET__VERTICAL_ALIGN        : name_area_vertical_align,
                    WEATHER_CONSTS.WEATHER__WIDGETS__COMMON_WIDGET__FONT_ZOOM_PERCENT     : name_area_font_zoom_percent
                }
            }
        }
        grid_elements.append(_weather_widget)
        return _weather_widget
    
    
    @classmethod
    def create_widget_layout(cls, width=u'default', height=u'default'):
        # type: (WidthType, IntDefault) -> Dict
        return {
            WEATHER_CONSTS.WEATHER__WIDGETS__WIDTH  : width,
            WEATHER_CONSTS.WEATHER__WIDGETS__HEIGHT : height,
            WEATHER_CONSTS.WEATHER__WIDGETS__CONTENT: {
                WEATHER_CONSTS.WEATHER__WIDGETS__LAYOUT: {}
            }
        }
    
    
    @classmethod
    def add_title_widget_layout_to_grid(cls, grid_dict, width=u'default', height=u'default', text_color=u'default', text_align=u'default', font_zoom_percent=u'default'):
        # type: (Dict, WidthType, IntDefault, unicode, unicode, IntDefault) -> Dict
        grid_default_layouts = grid_dict.setdefault(WEATHER_CONSTS.WEATHER__GRIDS__LAYOUTS, {})
        _title_widget_layout = cls.create_widget_layout(width, height)
        _title_widget_layout[WEATHER_CONSTS.WEATHER__WIDGETS__CONTENT][WEATHER_CONSTS.WEATHER__WIDGETS__LAYOUT][WEATHER_CONSTS.WEATHER__WIDGETS__INFORMATION_TO_DISPLAY] = {
            WEATHER_CONSTS.WEATHER__WIDGETS__TITLE_WIDGET__TITLE_AREA: {
                WEATHER_CONSTS.WEATHER__WIDGETS__COMMON_WIDGET__TEXT_COLOR       : text_color,
                WEATHER_CONSTS.WEATHER__WIDGETS__COMMON_WIDGET__TEXT_ALIGN       : text_align,
                WEATHER_CONSTS.WEATHER__WIDGETS__COMMON_WIDGET__FONT_ZOOM_PERCENT: font_zoom_percent
            }
        }
        grid_default_layouts[WEATHER_CONSTS.WEATHER__WIDGETS__TITLE_WIDGET_LAYOUT] = _title_widget_layout
        return _title_widget_layout
    
    
    @classmethod
    def add_separator_widget_layout_to_grid(cls, grid_dict, width=u'default', height=u'default', left_area_displayed=u'default', left_area_width_percent=u'default', text_color=u'default', text_align=u'default',
                                            font_zoom_percent=u'default', right_area_displayed=u'default', right_area_width_percent=u'default'):
        # type: (Dict, WidthType, IntDefault, BoolDefault, IntDefault, unicode, unicode, IntDefault, BoolDefault, IntDefault) -> Dict
        grid_default_layouts = grid_dict.setdefault(WEATHER_CONSTS.WEATHER__GRIDS__LAYOUTS, {})
        _separator_widget_layout = cls.create_widget_layout(width, height)
        _separator_widget_layout[WEATHER_CONSTS.WEATHER__WIDGETS__CONTENT][WEATHER_CONSTS.WEATHER__WIDGETS__LAYOUT][WEATHER_CONSTS.WEATHER__WIDGETS__INFORMATION_TO_DISPLAY] = {
            WEATHER_CONSTS.WEATHER__WIDGETS__SEPARATOR_WIDGET__LEFT_AREA : {
                WEATHER_CONSTS.WEATHER__WIDGETS__COMMON_WIDGET__DISPLAY         : left_area_displayed,
                WEATHER_CONSTS.WEATHER__WIDGETS__SEPARATOR_WIDGET__WIDTH_PERCENT: left_area_width_percent
            },
            WEATHER_CONSTS.WEATHER__WIDGETS__SEPARATOR_WIDGET__TITLE_AREA: {
                WEATHER_CONSTS.WEATHER__WIDGETS__COMMON_WIDGET__TEXT_COLOR       : text_color,
                WEATHER_CONSTS.WEATHER__WIDGETS__COMMON_WIDGET__TEXT_ALIGN       : text_align,
                WEATHER_CONSTS.WEATHER__WIDGETS__COMMON_WIDGET__FONT_ZOOM_PERCENT: font_zoom_percent
            },
            WEATHER_CONSTS.WEATHER__WIDGETS__SEPARATOR_WIDGET__RIGHT_AREA: {
                WEATHER_CONSTS.WEATHER__WIDGETS__COMMON_WIDGET__DISPLAY         : right_area_displayed,
                WEATHER_CONSTS.WEATHER__WIDGETS__SEPARATOR_WIDGET__WIDTH_PERCENT: right_area_width_percent
            }
        }
        grid_default_layouts[WEATHER_CONSTS.WEATHER__WIDGETS__SEPARATOR_WIDGET_LAYOUT] = _separator_widget_layout
        
        return _separator_widget_layout
    
    
    @classmethod
    def add_weather_widget_layout_to_grid(cls, grid_dict, width=u'default', height=u'default', background=u'default', left_area_icon_type=u'default', top_right_area_displayed=u'default',
                                          top_right_area_icon_type=u'default',
                                          sla_area_displayed=u'default',
                                          sla_area_show_icon=u'default',
                                          name_area_property_used_as_name=u'default',
                                          name_area_nb_lines_max=u'default',
                                          name_area_vertical_align=u'default',
                                          name_area_font_zoom_percent=u'default'):
        # type: (Dict, WidthType, IntDefault, unicode, unicode, BoolDefault, unicode, BoolDefault, BoolDefault, unicode, IntDefault, unicode, IntDefault) -> Dict
        grid_default_layouts = grid_dict.setdefault(WEATHER_CONSTS.WEATHER__GRIDS__LAYOUTS, {})
        _weather_widget_layout = cls.create_widget_layout(width, height)
        _weather_widget_layout[WEATHER_CONSTS.WEATHER__WIDGETS__CONTENT][WEATHER_CONSTS.WEATHER__WIDGETS__LAYOUT] = {
            WEATHER_CONSTS.WEATHER__WIDGETS__WEATHER_WIDGET__BACKGROUND: background,
            WEATHER_CONSTS.WEATHER__WIDGETS__INFORMATION_TO_DISPLAY    : {
                WEATHER_CONSTS.WEATHER__WIDGETS__WEATHER_WIDGET__LEFT_AREA     : {
                    WEATHER_CONSTS.WEATHER__WIDGETS__WEATHER_WIDGET__ICON_TYPE: left_area_icon_type
                },
                WEATHER_CONSTS.WEATHER__WIDGETS__WEATHER_WIDGET__TOP_RIGHT_AREA: {
                    WEATHER_CONSTS.WEATHER__WIDGETS__COMMON_WIDGET__DISPLAY   : top_right_area_displayed,
                    WEATHER_CONSTS.WEATHER__WIDGETS__WEATHER_WIDGET__ICON_TYPE: top_right_area_icon_type
                },
                WEATHER_CONSTS.WEATHER__WIDGETS__WEATHER_WIDGET__SLA_AREA      : {
                    WEATHER_CONSTS.WEATHER__WIDGETS__COMMON_WIDGET__DISPLAY   : sla_area_displayed,
                    WEATHER_CONSTS.WEATHER__WIDGETS__WEATHER_WIDGET__SHOW_ICON: sla_area_show_icon
                },
                WEATHER_CONSTS.WEATHER__WIDGETS__WEATHER_WIDGET__NAME_AREA     : {
                    WEATHER_CONSTS.WEATHER__WIDGETS__WEATHER_WIDGET__PROPERTY_USED_AS_NAME: name_area_property_used_as_name,
                    WEATHER_CONSTS.WEATHER__WIDGETS__COMMON_WIDGET__NB_LINES_MAX          : name_area_nb_lines_max,
                    WEATHER_CONSTS.WEATHER__WIDGETS__COMMON_WIDGET__VERTICAL_ALIGN        : name_area_vertical_align,
                    WEATHER_CONSTS.WEATHER__WIDGETS__COMMON_WIDGET__FONT_ZOOM_PERCENT     : name_area_font_zoom_percent
                }
            }
        }
        grid_default_layouts[WEATHER_CONSTS.WEATHER__WIDGETS__WEATHER_WIDGET_LAYOUT] = _weather_widget_layout
        return _weather_widget_layout
    
    
    @classmethod
    def add_grid_to_weather(cls, weather_dict, nb_tiles_in_width=u'default', nb_tiles_in_height=u'default', separation_grid_element_max=u'default', separation_grid_element_min=u'default', separation_grid_element_percent=u'default'):
        # type: (Dict, IntDefault, IntDefault, IntDefault, IntDefault, IntDefault) -> Dict
        weather_grids = weather_dict.setdefault(WEATHER_CONSTS.WEATHER__GRIDS, [])
        _grid = {
            WEATHER_CONSTS.WEATHER__WIDGETS__TYPE                         : WEATHER_CONSTS.WEATHER__GRIDS__GRID,
            WEATHER_CONSTS.WEATHER__GRIDS__TILES_IN_WIDTH                 : nb_tiles_in_width,
            WEATHER_CONSTS.WEATHER__GRIDS__TILES_IN_HEIGHT                : nb_tiles_in_height,
            WEATHER_CONSTS.WEATHER__GRIDS__SEPARATION_GRID_ELEMENT_MAX    : separation_grid_element_max,
            WEATHER_CONSTS.WEATHER__GRIDS__SEPARATION_GRID_ELEMENT_MIN    : separation_grid_element_min,
            WEATHER_CONSTS.WEATHER__GRIDS__SEPARATION_GRID_ELEMENT_PERCENT: separation_grid_element_percent,
            WEATHER_CONSTS.WEATHER__GRID_ELEMENTS                         : [],
            WEATHER_CONSTS.WEATHER__GRIDS__LAYOUTS                        : {}
        }
        weather_grids.append(_grid)
        return _grid
    
    
    @classmethod
    def create_info_bar(cls, background_color=u'default',
                        logo_displayed=u'default',
                        position=u'default',
                        refresh_chrono_displayed=u'default',
                        refresh_generation_time_displayed=u'default'):
        # type: (unicode, BoolDefault, unicode, BoolDefault, BoolDefault) -> Dict
        return {
            WEATHER_CONSTS.WEATHER__EXTERNAL_LINKS__INFO_BAR__BACKGROUND_COLOR: background_color,
            WEATHER_CONSTS.WEATHER__EXTERNAL_LINKS__INFO_BAR__LOGO_DISPLAYED  : logo_displayed,
            WEATHER_CONSTS.WEATHER__EXTERNAL_LINKS__INFO_BAR__POSITION        : position,
            WEATHER_CONSTS.WEATHER__EXTERNAL_LINKS__INFO_BAR__REFRESH         : {
                WEATHER_CONSTS.WEATHER__EXTERNAL_LINKS__INFO_BAR__REFRESH_CHRONO_DISPLAYED         : refresh_chrono_displayed,
                WEATHER_CONSTS.WEATHER__EXTERNAL_LINKS__INFO_BAR__REFRESH_GENERATION_TIME_DISPLAYED: refresh_generation_time_displayed
            }
        }
    
    
    @classmethod
    def add_external_links_to_weather(cls, weather_dict, default_link_protocol=u'default', default_base_url=u'default', default_external_part_url=u'default',
                                      default_authentication_needed=u'default',
                                      default_info_bar_background_color=u'default',
                                      default_info_bar_logo_displayed=u'default',
                                      default_info_bar_position=u'default',
                                      default_info_bar_refresh_chrono_displayed=u'default',
                                      default_info_bar_refresh_generation_time_displayed=u'default'):
        # type: (Dict, unicode, unicode, unicode, BoolDefault, unicode, BoolDefault, unicode, BoolDefault, BoolDefault) -> Dict
        _external_links = {
            WEATHER_CONSTS.WEATHER__EXTERNAL_LINKS__LINKS_LIST  : [],
            WEATHER_CONSTS.WEATHER__EXTERNAL_LINKS__DEFAULT_LINK: {
                WEATHER_CONSTS.WEATHER__EXTERNAL_LINKS__EXTERNAL_LINK__PROTOCOL             : default_link_protocol,
                WEATHER_CONSTS.WEATHER__EXTERNAL_LINKS__EXTERNAL_LINK__BASE_URL             : default_base_url,
                WEATHER_CONSTS.WEATHER__EXTERNAL_LINKS__EXTERNAL_LINK__EXTERNAL_PART_URL    : default_external_part_url,
                WEATHER_CONSTS.WEATHER__EXTERNAL_LINKS__EXTERNAL_LINK__AUTHENTICATION_NEEDED: default_authentication_needed,
                WEATHER_CONSTS.WEATHER__EXTERNAL_LINKS__INFO_BAR                            : cls.create_info_bar(
                    background_color=default_info_bar_background_color,
                    logo_displayed=default_info_bar_logo_displayed,
                    position=default_info_bar_position,
                    refresh_chrono_displayed=default_info_bar_refresh_chrono_displayed,
                    refresh_generation_time_displayed=default_info_bar_refresh_generation_time_displayed,
                )
            }
        }
        weather_dict[WEATHER_CONSTS.WEATHER__EXTERNAL_LINKS] = _external_links
        return _external_links
    
    
    @classmethod
    def add_link_to_external_links(cls, external_links_dict,
                                   link_name,
                                   link_uuid=None,
                                   link_protocol=u'default',
                                   link_base_url=u'default',
                                   link_external_part_url=u'default',
                                   authentication_needed=u'default',
                                   info_bar_background_color=u'default',
                                   info_bar_logo_displayed=u'default',
                                   info_bar_position=u'default',
                                   info_bar_refresh_chrono_displayed=u'default',
                                   info_bar_refresh_generation_time_displayed=u'default'):
        # type: (Dict, unicode, unicode, unicode, unicode, unicode, BoolDefault, unicode, BoolDefault, unicode, BoolDefault, BoolDefault) -> Dict
        external_links_list = external_links_dict.setdefault(WEATHER_CONSTS.WEATHER__EXTERNAL_LINKS__LINKS_LIST, [])
        _external_link = {
            WEATHER_CONSTS.WEATHER__EXTERNAL_LINKS__EXTERNAL_LINK__NAME                 : link_name,
            WEATHER_CONSTS.WEATHER__EXTERNAL_LINKS__EXTERNAL_LINK__PROTOCOL             : link_protocol,
            WEATHER_CONSTS.WEATHER__EXTERNAL_LINKS__EXTERNAL_LINK__BASE_URL             : link_base_url,
            WEATHER_CONSTS.WEATHER__EXTERNAL_LINKS__EXTERNAL_LINK__EXTERNAL_PART_URL    : link_external_part_url,
            WEATHER_CONSTS.WEATHER__EXTERNAL_LINKS__EXTERNAL_LINK__AUTHENTICATION_NEEDED: authentication_needed,
            WEATHER_CONSTS.WEATHER__EXTERNAL_LINKS__INFO_BAR                            : cls.create_info_bar(
                background_color=info_bar_background_color,
                logo_displayed=info_bar_logo_displayed,
                position=info_bar_position,
                refresh_chrono_displayed=info_bar_refresh_chrono_displayed,
                refresh_generation_time_displayed=info_bar_refresh_generation_time_displayed,
            )
        }
        if link_uuid:
            _external_link[WEATHER_CONSTS.WEATHER__EXTERNAL_LINKS__EXTERNAL_LINK__UUID] = link_uuid
        
        external_links_list.append(_external_link)
        return _external_link
    
    
    @classmethod
    def set_current_info_bar_to_weather(cls, weather_dict,
                                        background_color=u'default',
                                        logo_displayed=u'default',
                                        position=u'default',
                                        refresh_chrono_displayed=u'default',
                                        refresh_generation_time_displayed=u'default'):
        # type: (Dict, unicode, BoolDefault, unicode, BoolDefault, BoolDefault) -> Dict
        _info_bar = cls.create_info_bar(
            background_color=background_color,
            logo_displayed=logo_displayed,
            position=position,
            refresh_chrono_displayed=refresh_chrono_displayed,
            refresh_generation_time_displayed=refresh_generation_time_displayed,
        )
        weather_dict[WEATHER_CONSTS.WEATHER__CURRENT_INFO_BAR] = _info_bar
        return _info_bar
    
    
    @classmethod
    def set_current_info_bar_to_weather_from_link(cls, weather_dict, link_dict):
        # type: (Dict, Dict) -> Dict
        _info_bar = deepcopy(link_dict[WEATHER_CONSTS.WEATHER__EXTERNAL_LINKS__INFO_BAR])
        weather_dict[WEATHER_CONSTS.WEATHER__CURRENT_INFO_BAR] = _info_bar
        return _info_bar
    
    
    @classmethod
    def set_users_to_weather(cls, weather_dict, owner_user_name=u'', owner_user_uuid=u'', previous_owner_user_name=u'', previous_owner_user_uuid=u'', creator_user_name=u'', creator_user_uuid=u''):
        # type: (Dict, unicode, unicode, unicode, unicode, unicode, unicode) -> Dict
        _users = {
            WEATHER_CONSTS.WEATHER__RIGHTS__OWNER_USER         : {
                WEATHER_CONSTS.WEATHER__RIGHTS__USER_NAME: owner_user_name,
                WEATHER_CONSTS.WEATHER__RIGHTS__USER_UUID: owner_user_uuid
            },
            WEATHER_CONSTS.WEATHER__RIGHTS__PREVIOUS_OWNER_USER: {
                WEATHER_CONSTS.WEATHER__RIGHTS__USER_NAME: previous_owner_user_name,
                WEATHER_CONSTS.WEATHER__RIGHTS__USER_UUID: previous_owner_user_uuid
            },
            WEATHER_CONSTS.WEATHER__RIGHTS__CREATOR_USER       : {
                WEATHER_CONSTS.WEATHER__RIGHTS__USER_NAME: creator_user_name,
                WEATHER_CONSTS.WEATHER__RIGHTS__USER_UUID: creator_user_uuid
            }
        }
        weather_dict[WEATHER_CONSTS.WEATHER__RIGHTS] = _users
        return _users
    
    
    @classmethod
    def add_notification_to_weather(cls, weather_dict, sound_enabled=False, blink_enabled=True):
        # type: (Dict, BoolDefault, BoolDefault) -> Dict
        weather_notification = weather_dict.setdefault(WEATHER_CONSTS.WEATHER__NOTIFICATIONS, {
            WEATHER_CONSTS.WEATHER__NOTIFICATIONS__SOUND : {
                WEATHER_CONSTS.WEATHER__NOTIFICATIONS__COMMON__ENABLED: sound_enabled
            },
            WEATHER_CONSTS.WEATHER__NOTIFICATIONS__VISUAL: {
                WEATHER_CONSTS.WEATHER__NOTIFICATIONS__VISUAL__BLINK: {
                    WEATHER_CONSTS.WEATHER__NOTIFICATIONS__COMMON__ENABLED: blink_enabled
                }
            }
        })
        return weather_notification
