#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

from texttable import Texttable

from .log import logger, get_section_string
from .property import IntegerProp, StringProp, BoolProp
from .satellitelink import SatelliteLinks
from .util import get_obj_name_possibly_none
from .withinventorysatellitelink import WithInventorySatelliteLink

try:
    from collections import Counter
except ImportError:  # python 2.6
    from .misc.counter import Counter

# Typing
from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    pass

DEFAULT_BROKS_PACKET_SIZE = 200 * 1024  # 200 MB


# IMPORTANT: When adding a new property, think about the retention in the arbiter SPARE!
class BrokerLink(WithInventorySatelliteLink):
    """TODO: Add some comment about this class for the doc"""
    id = 0
    my_type = 'broker'
    
    # spare_daemon = None  # type: Optional[BrokerLink] => breaks default value!
    
    properties = WithInventorySatelliteLink.properties.copy()
    properties.update({
        'broker_name'                                                              : StringProp(fill_brok=['full_status'], to_send=True),
        'port'                                                                     : IntegerProp(default='7772', fill_brok=['full_status']),
        'broks_packet_size'                                                        : IntegerProp(default='%d' % DEFAULT_BROKS_PACKET_SIZE, fill_brok=['full_status'], to_send=True),  # Currrently do not take risk and disable it by setting to 200Mb
        'broker__manage_brok__enable_sub_processes_memory_usage_protection'        : BoolProp(default='1', fill_brok=['full_status'], to_send=True),
        'broker__manage_brok__sub_process_memory_usage_system_reserved_memory'     : IntegerProp(default='0', fill_brok=['full_status'], to_send=True),
        'broker__manage_brok__sub_processes_memory_usage_protection_max_retry_time': IntegerProp(default='5', fill_brok=['full_status'], to_send=True),
        # For broks pusher process
        'broker__manage_brok__sub_process_broks_pusher_max_execution_timeout'      : IntegerProp(default='240', fill_brok=['full_status'], to_send=True),
        'broker__manage_brok__sub_process_broks_pusher_security_ratio'             : IntegerProp(default='5', fill_brok=['full_status'], to_send=True),
        'broker__manage_brok__sub_process_broks_pusher_min_execution_timeout'      : IntegerProp(default='5', fill_brok=['full_status'], to_send=True),
        'broker__manage_brok__sub_process_broks_pusher_max_retry'                  : IntegerProp(default='3', fill_brok=['full_status'], to_send=True),
        'broker__manage_brok__sub_process_broks_pusher_queue_batch_size'           : IntegerProp(default='100000', fill_brok=['full_status'], to_send=True),  # How much to send MAX for each batch
        
        'broker__manage_spare__spare_must_have_the_same_list_of_module_type'       : BoolProp(default='1', to_send=True),
        'spare_daemon'                                                             : StringProp(default='', to_send=True, conf_send_preparation=get_obj_name_possibly_none),
        
    })
    
    running_properties = WithInventorySatelliteLink.running_properties.copy()
    running_properties.update({
        'master_daemon': StringProp(default=None, to_send=True, conf_send_preparation=get_obj_name_possibly_none),
    })
    
    
    def get_name(self):
        return getattr(self, 'broker_name', 'unknown')
    
    
    def register_to_my_realm(self):
        self.realm.brokers.append(self)
    
    
    def get_short_spare_display(self):
        if not self.spare:
            have_spare = '' if self.spare_daemon is None else 'spare=>%s' % self.spare_daemon.get_name()
            return have_spare
        else:
            spare_for = 'spare for=>%s' % self.master_daemon.get_name() if self.master_daemon else 'UNUSED SPARE'
            return spare_for
    
    
    def linkify_with_brokers(self, all_brokers):
        # type: (BrokerLinks) -> None
        if self.spare_daemon == '':
            self.spare_daemon = None
            return
        
        # There is a name, must exists or I will be not so happy
        spare_daemon = all_brokers.find_by_name(self.spare_daemon)
        if spare_daemon is None:
            self.configuration_errors.append('Cannot find a broker with the name "%s" for the property "spare_daemon"' % self.spare_daemon)
            self.spare_daemon = None
            return
        logger.debug('[SPARING] Success fully link brokers %s => %s' % (self.get_name(), spare_daemon.get_name()))
        self.set_spare_daemon(spare_daemon)
        spare_daemon.set_master_daemon(self)
    
    
    def set_spare_daemon(self, spare_daemon):
        self.spare_daemon = spare_daemon
    
    
    def set_master_daemon(self, master_daemon):
        self.master_daemon = master_daemon
    
    
    # For broker, there are other protection: a spare cannot have a spare daemon
    def is_correct(self):
        if self.spare and self.spare_daemon is not None:
            self.configuration_errors.append('The broker spare "%s" is not authorized to have a daemon_spare ("%s")' % (self.get_name(), self.spare_daemon.get_name()))
        
        r = super(BrokerLink, self).is_correct()  # will raise the error in it
        
        return r
    
    
    # Override set state from pickle to catch case where the broks_packet_size was not save in retention
    # from #SEF-6352
    def __setstate__(self, state):
        super(BrokerLink, self).__setstate__(state)
        if 'broks_packet_size' not in state:
            self.broks_packet_size = DEFAULT_BROKS_PACKET_SIZE


class BrokerLinks(SatelliteLinks):
    """TODO: Add some comment about this class for the doc"""
    name_property = "broker_name"
    inner_class = BrokerLink
    
    
    # Brokers must link with:
    # * realms
    # * modules
    # AND spare_daemon => other brokers
    def linkify(self, realms, modules):
        super(BrokerLinks, self).linkify(realms, modules)
        self._linkify_brokers_with_brokers()
    
    
    def _linkify_brokers_with_brokers(self):
        for broker in self:
            broker.linkify_with_brokers(self)
    
    
    # We will look if a spare was assigned twice as a spare, if so, it's
    # and error
    def _check_spare_assigned_only_once(self):
        to_spare_mapping = {}
        for broker in self:
            if broker.spare_daemon is not None:
                spare_name = broker.spare_daemon.get_name()
                if spare_name not in to_spare_mapping:
                    to_spare_mapping[spare_name] = []
                to_spare_mapping[spare_name].append(broker.get_name())
        r = True
        for spare_name, master_daemons in to_spare_mapping.items():
            if len(master_daemons) >= 2:  # only one master can link to a spare
                err = '%s %s daemons (%s) have "%s" as spare_daemon. A daemon can be the spare of only one other.' % (get_section_string('TWO MASTERS FOR ONE SLAVE'), len(master_daemons), ', '.join(master_daemons), spare_name)
                self.configuration_errors.append(err)
                r = False
        return r
    
    
    # We will look if a spare have the same module types than the master
    # note: we may have not the exact same module, because the spare network access is different
    #       so maybe the module configuration is different
    def _check_spare_have_valid_modules(self):
        r = True
        for broker in self:
            if broker.spare and broker.master_daemon is not None:
                spare_daemon = broker
                master_daemon = broker.master_daemon
                
                # Get all counters
                spare_modules_types = Counter([module.get_type() for module in spare_daemon.modules])
                master_modules_types = Counter([module.get_type() for module in master_daemon.modules])
                
                # Maybe the master server is allowing a degraded spare (not the same module types)
                # NOTE: this is not the case by default
                if not master_daemon.broker__manage_spare__spare_must_have_the_same_list_of_module_type:
                    continue
                
                # Ok, it's the same
                if spare_modules_types == master_modules_types:
                    continue
                
                r = False  # bad configuration
                self.configuration_errors.append(
                    '%s The Broker master "%s" and its spare "%s" do not have the same types of modules (cf "module_type" property in the .cfg):' % (get_section_string('SPARE MODULE NOT MATCHING'), master_daemon.get_name(), spare_daemon.get_name()))
                
                # We will use a texttable to do the table print for us
                table = Texttable()
                table.set_cols_align(["l", "c", "c"])  # align left, center, center
                table.add_row(['Module type', 'Master [ %s ]' % master_daemon.get_name(), 'Spare [ %s ]' % spare_daemon.get_name()])
                # Oups, wrong configuration, show differences
                all_modules = spare_modules_types | master_modules_types
                
                for module_type in all_modules:
                    master_count = master_modules_types.get(module_type, 0)
                    spare_count = spare_modules_types.get(module_type, 0)
                    
                    if master_count == spare_count:
                        continue
                    
                    table.add_row([module_type, '%d' % master_count, '%d' % spare_count])
                lines = table.draw()
                for line in lines.splitlines():
                    self.configuration_errors.append('%s  %s' % (get_section_string('SPARE MODULE NOT MATCHING'), line))
        
        return r
    
    
    def is_correct(self):
        r = True
        r &= self._check_spare_assigned_only_once()
        
        r &= self._check_spare_have_valid_modules()
        
        r &= super(BrokerLinks, self).is_correct()
        
        return r
