#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#     Gabes Jean, naparuba@gmail.com
#     Gerhard Lausser, Gerhard.Lausser@consol.de
#     Gregory Starck, g.starck@gmail.com
#     Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import json
import socket
import threading
import urllib.request, urllib.parse, urllib.error

# Pycurl part
import pycurl
import io
from shinken.log import logger
from shinkensolutions.localinstall import VERSION
from .compat import cPickle
from .compresser import compresser

pycurl.global_init(pycurl.GLOBAL_ALL)
PYCURL_VERSION = pycurl.version_info()[1]


class HTTPException(Exception):
    SSL_FAILED = 35


HTTPExceptions = (HTTPException,)


class HTTPClient(object):
    def __init__(self, address='', port=0, use_ssl=False, timeout=3, data_timeout=120, uri='', strong_ssl=False):
        self.address = address
        self.port = port
        self.timeout = timeout
        self.data_timeout = data_timeout
        
        if not uri:
            if use_ssl:
                self.uri = "https://%s:%s/" % (self.address, self.port)
            else:
                self.uri = "http://%s:%s/" % (self.address, self.port)
        else:
            self.uri = uri
        
        self.con = pycurl.Curl()
        self.lock = threading.RLock()
        
        self._set_con_opt(strong_ssl)
    
    
    def _set_con_opt(self, strong_ssl):
        with self.lock:
            
            # Remove the Expect: 100-Continue default behavior of pycurl, because swsgiref do not manage it
            headers = ['Expect:', 'Keep-Alive: 300', 'Connection: Keep-Alive']
            try:
                host_name = socket.getfqdn() or socket.gethostname()
                headers.append('X-Forwarded-For: %s' % host_name)
            except:
                pass
            # Set the daemon name into the header, so the http daemon can display it in debug mode if asked
            headers.append('X-Caller-Name: %s' % str(logger.name))
            self.con.setopt(pycurl.HTTPHEADER, headers)
            self.con.setopt(pycurl.USERAGENT, 'shinken:%s pycurl:%s' % (VERSION, PYCURL_VERSION))
            self.con.setopt(pycurl.FOLLOWLOCATION, 1)
            self.con.setopt(pycurl.FAILONERROR, True)
            self.con.setopt(pycurl.CONNECTTIMEOUT, self.timeout)
            self.con.setopt(pycurl.HTTP_VERSION, pycurl.CURL_HTTP_VERSION_1_1)
            
            # Also set the SSL options to do not look at the certificates too much
            # unless the admin asked for it
            if strong_ssl:
                self.con.setopt(pycurl.SSL_VERIFYPEER, 1)
                self.con.setopt(pycurl.SSL_VERIFYHOST, 2)
            else:
                self.con.setopt(pycurl.SSL_VERIFYPEER, 0)
                self.con.setopt(pycurl.SSL_VERIFYHOST, 0)
    
    
    # Try to get an URI path
    def get(self, path, args={}, wait='short'):
        with self.lock:
            c = self.con
            c.setopt(c.POST, 0)
            c.setopt(pycurl.HTTPGET, 1)
            
            # For the TIMEOUT, it will depends if we are waiting for a long query or not
            # long:data_timeout, like for huge broks receptions
            # short:timeout, like for just "ok" connection
            if wait == 'short':
                c.setopt(c.TIMEOUT, self.timeout)
            else:
                c.setopt(c.TIMEOUT, self.data_timeout)
            
            # if proxy:
            #    c.setopt(c.PROXY, proxy)
            # print "GO TO", self.uri+path+'?'+urllib.urlencode(args)
            c.setopt(c.URL, str(self.uri + path + '?' + urllib.parse.urlencode(args)))
            # Ok now manage the response
            response = io.BytesIO()
            c.setopt(pycurl.WRITEFUNCTION, response.write)
            
            # c.setopt(c.VERBOSE, 1)
            try:
                c.perform()
            except pycurl.error as error:
                errno, errstr = error.args
                exp = HTTPException('Connection error to %s : %s' % (self.uri, errstr))
                exp.errno = errno
                exp.errstr = errstr
                raise exp
            r = c.getinfo(pycurl.HTTP_CODE)
            # Do NOT close the connection, we want a keep alive
            
            if r != 200:
                err = response.getvalue()
                logger.error("There was a critical error : %s" % err)
                exception = Exception('Connection error to %s : %s' % (self.uri, r))
                exception.errno = r
                raise exception
            else:
                # Manage special return of pycurl
                ret = json.loads(response.getvalue())
                # print "GOT RAW RESULT", ret, type(ret)
                return ret
    
    
    # Try to get an URI path
    def post(self, path, args, wait='short'):
        with self.lock:
            # Take args, pickle them and then compress the result
            for (k, v) in args.items():
                _to_compress = cPickle.dumps(v, 2)
                args[k] = compresser.compress(_to_compress)
            # Ok go for it!
            
            c = self.con
            c.setopt(pycurl.HTTPGET, 0)
            c.setopt(c.POST, 1)
            
            # For the TIMEOUT, it will depends if we are waiting for a long query or not
            # long:data_timeout, like for huge broks receptions
            # short:timeout, like for just "ok" connection
            if wait == 'short':
                c.setopt(c.TIMEOUT, self.timeout)
            else:
                c.setopt(c.TIMEOUT, self.data_timeout)
            # if proxy:
            #    c.setopt(c.PROXY, proxy)
            # Pycurl want a list of tuple as args
            postargs = [(k, v) for (k, v) in args.items()]
            c.setopt(c.HTTPPOST, postargs)
            c.setopt(c.URL, str(self.uri + path))
            # Ok now manage the response
            response = io.BytesIO()
            c.setopt(pycurl.WRITEFUNCTION, response.write)
            # c.setopt(c.VERBOSE, 1)
            try:
                c.perform()
            except pycurl.error as error:
                errno, errstr = error.args
                exp = HTTPException('Connection error to %s : %s' % (self.uri, errstr))
                exp.errno = c.getinfo(pycurl.HTTP_CODE)
                exp.errstr = errstr
                raise exp
            
            r = c.getinfo(pycurl.HTTP_CODE)
            # Do NOT close the connection
            # c.close()
            if r != 200:
                err = response.getvalue()
                logger.error("There was a critical error : %s" % err)
                exception = Exception('Connection error to %s : %s' % (self.uri, r))
                exception.errno = r
                raise exception
            else:
                # Manage special return of pycurl
                # ret  = json.loads(response.getvalue().replace('\\/', '/'))
                ret = response.getvalue()
                return ret
            
            # Should return us pong string
            return ret
