#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import os
import threading
import time
import traceback

from shinken.compat import Empty
from shinken.misc.type_hint import List, Union
from .basesubprocess import BaseSubProcess, CommandQueueHandler
from .log import LoggerFactory
from .message import Message
from .misc.type_hint import TYPE_CHECKING
from .modules.base_module.module_info_reader import ModuleInfoReaderMixin
from .objects.inventory import ElementsInventoryInBrokerModule
from .objects.itemsummary import HostSummary, CheckSummary
from .withworkersandinventorymodule import FROM_DAEMON_TO_MODULE_MESSAGE_TYPES

if TYPE_CHECKING:
    from .ipc.shinken_queue.shinken_queue import ShinkenQueue
    from .misc.type_hint import Callable, Optional
    from .objects.module import Module as ShinkenModuleDefinition
    from .withworkersandinventorymodule import FromModuleToWorkerContainer

WORKER_INFO_LOGGER_NAME = 'WORKER-INFO'


class ModuleWorker(BaseSubProcess, ModuleInfoReaderMixin):
    def __init__(self, worker_id, mod_conf, name, from_module_to_main_daemon_queue, queue_factory, daemon_display_name):
        # type: (int, ShinkenModuleDefinition, str, ShinkenQueue, Callable[[Optional[str],Optional[str],Optional[str]],ShinkenQueue], str) -> None
        BaseSubProcess.__init__(self, name, daemon_display_name)
        ModuleInfoReaderMixin.__init__(self, mod_conf)
        
        self._worker_id = worker_id
        self._queue_factory = queue_factory
        self._init_inventories()
        self._main_process_to_worker_inventory_queue = self._queue_factory('INV', name, 'W[%s]' % worker_id)
        self._from_module_to_main_daemon_queue = from_module_to_main_daemon_queue
        self.myconf = mod_conf
        self.logger = LoggerFactory.get_logger(mod_conf.get_name())
        self.daemon_display_name = daemon_display_name
        self.command_from_module_to_worker = self._queue_factory('CMD', name, 'W[%s]' % worker_id)
        self.command_from_worker_to_module = self.command_from_module_to_worker
        self.command_queue_handler = CommandQueueHandler('CQH-%s' % self.get_name(), LoggerFactory.get_logger(self.get_name()), self.command_from_module_to_worker, self.command_from_worker_to_module, self)
        self._worker_is_init = False  # some threads will wait until the worker is fully init (like manage broks)
    
    
    def get_name(self):
        return self.name
    
    
    def in_main_process_tick(self):
        pass
    
    
    def _start_sub_process_threads(self):
        super(ModuleWorker, self)._start_sub_process_threads()
        
        self.command_queue_handler.start_thread()
    
    
    def get_raw_stats(self):
        return {}
    
    
    def _init_inventories(self):
        self._global_inventory = ElementsInventoryInBrokerModule()
        self._realms_inventory = {}
        self._inventory_lock = threading.RLock()
    
    
    # Set by worker based modules to display their worker in the process list
    def _get_worker_display_name(self):
        return '[ Worker: %s ]' % self._worker_id
    
    
    def get_process_name(self):
        return "%s [ - Module: %s ]%s" % (self.father_name, self.name, self._get_worker_display_name())
    
    
    def get_logger_name(self):
        # type: () -> List[str]
        return [self.father_name.replace('shinken-', '').strip(), self.name, 'WORKER: %s (pid:%s)' % (self._worker_id, os.getpid())]
    
    
    # By default workers start no more threads than the inventory one
    def start_worker_specific_treads(self):
        pass
    
    
    # Can be implemented by end-user
    def do_stop(self):
        pass
    
    
    def send_object_to_main_daemon(self, obj):
        self._from_module_to_main_daemon_queue.put(obj)
    
    
    # Important: others threads cannot work until the worker is fully init
    def _wait_for_worker_init_to_be_done(self):
        while not self._worker_is_init:
            time.sleep(0.1)
    
    
    def _worker_main(self):
        self.logger.debug('In the worker with pid: %s' % os.getpid())
        try:
            
            # set up name, priority, father lookup and such things
            # TODO: get back this
            self._sub_process_common_warmup()
            
            # Let the user code prepare itself
            self.init_worker(self.myconf)
            
            # Also start a thread that will manage all inventory updates
            # Start a thread that is need to load/compute inventory
            thr = threading.Thread(target=self._do_in_worker_inventory_thread, name='worker-inventory-thread')
            thr.daemon = True
            thr.start()
            
            # If the class want, it can ask to start specific threads (like broks for broker worker)
            self.start_worker_specific_treads()
            
            # Let the user code prepare itself, but after we did prepare the threads and worker structure
            self.init_worker_before_main(self.myconf)
            self._worker_is_init = True
            
            self.logger.info('Now running..')
            
            # Will block here!
            self.logger.debug('go main')
            self.worker_main()
            self.do_stop()
            self.logger.info('exiting now..')
            # noinspection PyUnresolvedReferences, PyProtectedMember
            os._exit(0)
        except Exception as exp:
            self.logger.error('[ %s ] The worker %s did crash. Exiting. Back trace of it:' % (self.name, self._worker_id))
            self.logger.print_stack()
            msg = Message(id=0, type='ICrash', data={'name': self.get_name(), 'exception': str(exp), 'trace': traceback.format_exc()})
            self._from_module_to_main_daemon_queue.put(msg)
            # wait a bit so we know that the receiver got our message, and die
            time.sleep(5)
            # And force the full module process to go down
            # noinspection PyUnresolvedReferences, PyProtectedMember
            os._exit(2)
    
    
    def _send_exception_error_to_main_process_and_exit(self, trace, exception, thread_name):
        self.logger.error('The module %s does have a fail %s thread (%s). Cannot continue.' % (self.get_name(), thread_name, trace))
        try:
            msg = Message(id=0, type='ICrash', data={'name': self.get_name(), 'exception': str(exception), 'trace': trace})
            self._from_module_to_main_daemon_queue.put(msg)
            # wait a bit so we know that the receiver got our message, and die
            time.sleep(5)
        except Exception as exp:  # maybe the queue was close, but in all ca
            self.logger.error('Cannot send back thread error to the main process because of an error: %s' % exp)
        # And force the full module process to go down
        # noinspection PyUnresolvedReferences, PyProtectedMember
        os._exit(2)
    
    
    def _do_in_worker_inventory_thread(self):
        # Important: we cannot manage broks until the worker is fully init
        self._wait_for_worker_init_to_be_done()
        
        while True:
            try:
                self._do_in_worker_inventory_thread_loop()
            except Exception as exp:
                self._send_exception_error_to_main_process_and_exit(traceback.format_exc(), exp, 'inventory update')
    
    
    def _send_inventory_message_to_worker(self, message):
        queue = self._main_process_to_worker_inventory_queue
        try:
            queue.put(message)
        except Exception as exp:
            raise Exception('[%s] [INVENTORY] Cannot send the inventory message to the worker %s: %s' % (self.get_name(), self._worker_id, exp))
    
    
    def _do_in_worker_inventory_thread_loop(self):
        # self.logger.debug('_do_in_worker_inventory_thread_loop:: %s' % self._worker_id)
        # A bit of debug but not too much
        if int(time.time() % 3600 == 0):
            for (realm_name, inventory) in list(self._realms_inventory.items()):
                self.logger.debug('Dumping realm %s about inventory: %s elements' % (realm_name, inventory.get_len()))
        
        # We get from the Queue update from the receiver.
        # NOTE: as we are blocking, we are not using CPU and so no need for sleep
        try:
            inventory_update_message = self._main_process_to_worker_inventory_queue.get(block=True, timeout=1)
        except Empty:
            return
        
        # Load all form it and update our inventories
        self._in_worker_load_inventory_update_message(inventory_update_message)
    
    
    # We are in the worker and the master process did send us a inventory update message
    # with new/deleted and updated elements
    def _in_worker_load_inventory_update_message(self, inventory_update_message):
        message_type = inventory_update_message.get_type()
        payload = inventory_update_message.get_payload()
        realm_name = payload['realm_name']
        logger_inventory = self.logger.get_sub_part('INVENTORY').get_sub_part('WORKER:%s' % self._worker_id)
        logger_inventory.debug('We did receive an inventory update (%s) from the daemon about realm %s' % (message_type, realm_name))
        
        # New realm => add
        if message_type == FROM_DAEMON_TO_MODULE_MESSAGE_TYPES.ADD_REALM:
            realm_inventory = payload['inventory']
            logger_inventory.info('Adding the new realm %s in inventory' % realm_name)
            logger_inventory.info('Elements handle by worker : %s' % len(realm_inventory.get_all()))
            
            # self.logger_inventory_worker.info('Adding the new realm %s in inventory' % realm_name)
            with self._inventory_lock:
                self._realms_inventory[realm_name] = realm_inventory
                self._global_inventory.add_elements_from_realm_inventory(realm_name, realm_inventory)
            self._warn_user_code_about_new_inventory(realm_name, realm_inventory)
        
        # Delete ream => del
        elif message_type == FROM_DAEMON_TO_MODULE_MESSAGE_TYPES.DELETE_REALM:
            logger_inventory.info('removing the realm %s from inventory' % realm_name)
            old_realm_inventory = self._realms_inventory.get(realm_name, None)
            if old_realm_inventory is None:
                raise Exception('Asking for to delete a not exiting realm entry? %s %s' % (payload, self._realms_inventory))
            with self._inventory_lock:
                del self._realms_inventory[realm_name]
                self._global_inventory.delete_elements_from_realm_inventory(realm_name, old_realm_inventory)
            self._warn_user_code_about_deleted_inventory(realm_name, old_realm_inventory)
        
        # Update => reset
        elif message_type == FROM_DAEMON_TO_MODULE_MESSAGE_TYPES.UPDATE_REALM:
            logger_inventory.info('Updating the realm %s in inventory' % realm_name)
            current_realm_inventory = self._realms_inventory.get(realm_name, None)
            if current_realm_inventory is None:
                raise Exception('Asking for a update on a not exiting realm entry? %s %s' % (payload, self._realms_inventory))
            inventory_differences = payload['inventory_differences']
            self.logger.debug('Applying realm difference: %s' % inventory_differences)
            
            current_realm_inventory.update_from_differences(inventory_differences)
            with self._inventory_lock:
                self._global_inventory.update_from_differences(inventory_differences)
            
            # Differences have new/changed/deleted elements
            self._warn_user_code_about_changed_inventory(realm_name, inventory_differences)
        
        else:
            raise Exception('Error: the message type %s is not managed' % message_type)
    
    
    # The module did get a new inventory, let the user code know about it
    def _warn_user_code_about_new_inventory(self, realm_name, realm_inventory):
        host_uuids = realm_inventory.get_hosts_uuids()
        for host_uuid in host_uuids:
            self.callback__a_new_host_added(host_uuid)
        self.callback__a_new_realm_added(realm_name)
    
    
    # The module did get a realm/inventory deletion, let the user code know about it
    def _warn_user_code_about_deleted_inventory(self, realm_name, old_realm_inventory):
        host_uuids = old_realm_inventory.get_hosts_uuids()
        for host_uuid in host_uuids:
            self.callback__a_host_deleted(host_uuid)
        self.callback__a_realm_deleted(realm_name)
    
    
    # The module did get an updated inventory, so need to let the user code know about new/deleted/changed elements
    def _warn_user_code_about_changed_inventory(self, realm_name, inventory_differences):
        for host_uuid in list(inventory_differences['new'].keys()):
            self.callback__a_new_host_added(host_uuid)
        
        for host_uuid in list(inventory_differences['changed'].keys()):
            self.callback__a_host_updated(host_uuid)
        
        for host_uuid in inventory_differences['deleted']:
            self.callback__a_host_deleted(host_uuid)
        
        self.callback__a_realm_updated(realm_name)
    
    
    # ######## call backs that can be implemented by the final module
    def callback__a_new_host_added(self, host_uuid):
        pass
    
    
    def callback__a_host_updated(self, host_uuid):
        pass
    
    
    def callback__a_host_deleted(self, host_uuid):
        pass
    
    
    def callback__a_new_realm_added(self, realm_name):
        pass
    
    
    def callback__a_realm_deleted(self, realm_name):
        pass
    
    
    def callback__a_realm_updated(self, realm_name):
        pass
    
    
    def get_inventory_lock(self):
        return self._inventory_lock
    
    
    def get_host_from_uuid(self, host_uuid):
        # type: (str) -> HostSummary
        with self._inventory_lock:
            return self._global_inventory.get_host(host_uuid)
    
    
    def get_all_inventory(self):
        # type: () -> List[Union[HostSummary, CheckSummary]]
        with self._inventory_lock:
            return self._global_inventory.get_all()
    
    
    def get_inventory_len(self):
        # type: () -> int
        with self._inventory_lock:
            return self._global_inventory.get_len()
    
    
    def known_item_uuid(self, item_uuid):
        # type: (str) -> bool
        with self._inventory_lock:
            return self._global_inventory.have_item_uuid(item_uuid)
    
    
    def get_hosts_uuids(self):
        with self._inventory_lock:
            return self._global_inventory.get_hosts_uuids()
    
    
    def worker_main(self):
        raise NotImplementedError()
    
    
    def init_worker(self, module_configuration):
        pass
    
    
    def init_worker_before_main(self, module_configuration):
        pass
    
    
    def do_loop_turn(self):
        pass
    
    
    def loop_turn(self):
        pass
    
    
    def set_from_module_to_worker_container(self, from_module_to_worker_container):
        # type: (FromModuleToWorkerContainer) -> None
        pass
