#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2022:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

from shinken.misc.type_hint import TYPE_CHECKING
from .inventoryhasher import get_hasher

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict, Tuple


class ITEM_TYPE(object):
    CLUSTER = 'cluster'
    HOST = 'host'
    CHECK = 'check'


class CheckSummary(object):
    TO_REMOVE_DATA_KEYS = ['_ID', '_SE_UUID', '_SE_UUID_HASH']  # these keys are purely internal
    _host = None  # type: HostSummary
    
    
    def __init__(self):
        self._name = None
        self._uuid = None
        self._hash = None
        self._data = {}
        self.monitoring_start_time = None
        self.sla_warning_threshold = None
        self.sla_critical_threshold = None
    
    
    def create_from_service(self, host_summary, service):
        self._name = service.get_name()
        self._uuid = service.get_instance_uuid()
        self.monitoring_start_time = service.monitoring_start_time
        self.sla_warning_threshold = service.sla_warning_threshold
        self.sla_critical_threshold = service.sla_critical_threshold
        self._data = service.customs.copy()
        self._host = host_summary
        for key_to_remove in self.TO_REMOVE_DATA_KEYS:
            if key_to_remove in self._data:
                del self._data[key_to_remove]
        self._compute_hash()
    
    
    def _compute_hash(self):
        hasher = get_hasher()
        hasher.update(self._name)
        hasher.update(self._uuid)
        hasher.update(str(self.monitoring_start_time))
        hasher.update(str(self.sla_warning_threshold))
        hasher.update(str(self.sla_critical_threshold))
        
        # Same for data: always sorted
        data_keys = sorted(list(self._data.keys()))
        for data_key in data_keys:
            data_value = self._data[data_key]
            hasher.update(data_key)
            hasher.update(data_value)
        self._hash = hasher.hexdigest()
    
    
    def get_uuid(self):
        return self._uuid
    
    
    def get_hash(self):
        if self._hash is None:
            raise Exception('Asking for an hash on a CheckInReceiver object that did not compute it.')
        return self._hash
    
    
    def get_data(self):
        return self._data
    
    
    def get_name(self):
        return self._name
    
    
    def get_full_name(self):
        return '%s/%s' % (self.get_host_name(), self._name)
    
    
    def get_host_name(self):
        return self._host.get_name()
    
    
    def get_host(self):
        return self._host
    
    
    def get_realm(self):
        return self._host.get_realm()
    
    
    @staticmethod
    def get_type():
        return ITEM_TYPE.CHECK
    
    
    def get_instance_name(self):
        return '%s-%s' % (self._host.get_name(), self.get_name())
    
    
    def get_instance_uuid(self):
        # type: () -> str
        return self._uuid
    
    
    def get_names(self):
        # type: () -> Tuple[str, str]
        return self._host.get_name(), self.get_name()


class HostSummary(object):
    _checks = {}  # type: Dict[str,CheckSummary]
    
    
    def __init__(self):
        self._name = None
        self._templates = []
        self._uuid = None
        self._realm = None
        self.address = None
        self._data = {}
        self._checks = {}
        self._is_cluster = False
        # IMPORTANT: if you are adding a new property, update the compute_hash!
        self._hash = None  # used to know if we already know about it or not
        self.monitoring_start_time = None
        self.sla_warning_threshold = None
        self.sla_critical_threshold = None
    
    
    # Get a hash from our object to know if another node know about us or not
    # IMPORTANT: if you are adding a new element or change an order, you will force a SEND
    #            all for the daemons
    def _compute_hash(self):
        hasher = get_hasher()
        hasher.update(self._name)
        hasher.update(self.address)
        hasher.update(self._realm)
        hasher.update(str(self.monitoring_start_time))
        hasher.update(str(self.sla_warning_threshold))
        hasher.update(str(self.sla_critical_threshold))
        for template in self._templates:
            hasher.update(template)
        hasher.update(self._uuid)
        
        # NOTE: be sure to sort to al<ays have the same hash
        check_uuids = sorted(list(self._checks.keys()))
        for check_uuid in check_uuids:
            check_entry = self._checks[check_uuid]
            hasher.update(check_entry.get_hash())
        
        # Same for data: always sorted
        data_keys = list(self._data.keys())
        data_keys.sort()
        for data_key in data_keys:
            data_value = self._data[data_key]
            hasher.update(data_key)
            hasher.update(data_value)
        
        hasher.update(str(self._is_cluster))
        
        # We have all, we can compute final hash
        self._hash = hasher.hexdigest()
    
    
    def get_hash(self):
        if self._hash is None:
            raise Exception('Asking for an hash on a HostInReceiver object that did not compute it.')
        return self._hash
    
    
    def get_uuid(self):
        return self._uuid
    
    
    def get_name(self):
        return self._name
    
    
    def get_full_name(self):
        return self._name
    
    
    def get_host_name(self):
        return self._name
    
    
    def get_realm(self):
        return self._realm
    
    
    def get_data(self):
        return self._data
    
    
    def get_checks(self):
        return self._checks
    
    
    def get_templates(self):
        return self._templates
    
    
    def is_cluster(self):
        return self._is_cluster
    
    
    def get_type(self):
        return ITEM_TYPE.CLUSTER if self._is_cluster else ITEM_TYPE.HOST
    
    
    def get_instance_name(self):
        return self.get_name()
    
    
    def get_instance_uuid(self):
        # type: () -> str
        return self._uuid
    
    
    def get_names(self):
        # type: () -> Tuple[str, str]
        return self.get_name(), ''
    
    
    def _import_data_from_host(self, host, inventory_filters):
        # Get DATA/MACROS from the elements_sharding_add_data_of_templates
        data_keys_to_load = set()
        
        # Get ALL templates from the tree, not just the higher level
        for tpl in host.get_templates_as_flatten_list():
            template_name = tpl.get_name()
            
            do_match = False
            for inventory_filter in inventory_filters:
                if inventory_filter.template_match_data_filter(template_name):
                    do_match = True
                    break
            if do_match:
                for data_key in list(tpl.customs.keys()):
                    data_keys_to_load.add(data_key)
        host_macros = host.customs
        for data_key in data_keys_to_load:
            if data_key in host_macros:
                self._data[data_key] = host_macros[data_key]
    
    
    def create_from_host(self, host, inventory_filters):
        # Get basic data
        self._name = host.get_name()
        # NOTE: we do not want shinken-host template to be shown to the external part (modules, etc)
        self._templates = [tpl.get_name() for tpl in host.templates if tpl.get_name() != 'shinken-host']
        self._uuid = host.get_instance_uuid()
        self._realm = host.realm
        self.address = host.address
        self.monitoring_start_time = host.monitoring_start_time
        self.sla_warning_threshold = host.sla_warning_threshold
        self.sla_critical_threshold = host.sla_critical_threshold
        
        # Create checks from the host services
        for service in host.services:
            check_entry = CheckSummary()
            check_entry.create_from_service(self, service)
            self._checks[check_entry.get_uuid()] = check_entry
        
        self._is_cluster = host.is_cluster
        
        # Now look at host data and take only the one receiver wants
        # but in the broke case, skip it because broker don't need data
        if inventory_filters is not None:
            self._import_data_from_host(host, inventory_filters)
        
        # Our hash should be re-computed now
        self._compute_hash()
