import copy
import os
import time
import pickle

from shinken.util import set_process_name, to_mb_size
from shinken.property import UnusedProp
from shinken.log import logger
from .config import Config
from .hostgroup import Hostgroups
from .host import Hosts
from .service import Services
from .servicegroup import Servicegroups
from .proxyitem import proxyitemsgraph
from .inventory import ElementsInventory


class ShardedConfiguration(Config):
    
    def load_from_global_configuration(self, part_id, global_configuration):
        # Now we copy all properties of conf into the new ones
        for prop, entry in Config.properties.items():
            if entry.managed and not isinstance(entry, UnusedProp):
                val = getattr(global_configuration, prop)
                setattr(self, prop, val)
        
        # also copy static_macros from source conf
        self.static_macros = copy.copy(global_configuration.static_macros)
        
        # Also copy default properties set by the user
        self.default_properties_values = copy.copy(global_configuration.default_properties_values)
        
        # we need a deepcopy because each conf
        # will have new hostgroups
        self.id = part_id
        self.commands = global_configuration.commands
        self.timeperiods = global_configuration.timeperiods
        # Create hostgroups with just the name and same id, but no members
        new_hostgroups = []
        for hg in global_configuration.hostgroups:
            new_hostgroups.append(hg.copy_shell())
        self.hostgroups = Hostgroups(new_hostgroups)
        self.notificationways = global_configuration.notificationways
        self.checkmodulations = global_configuration.checkmodulations
        self.macromodulations = global_configuration.macromodulations
        self.contactgroups = global_configuration.contactgroups
        self.contacts = global_configuration.contacts
        # Create hostgroups with just the name and same id, but no members
        new_servicegroups = []
        for sg in global_configuration.servicegroups:
            new_servicegroups.append(sg.copy_shell())
        self.servicegroups = Servicegroups(new_servicegroups)
        self.hosts = []  # will be filled after
        self.services = []  # will be filled after
        # if a scheduler have accepted the conf
        self.is_assigned = False
    
    
    def load_hosts_and_checks_from_pack(self, pack_id, pack):
        for h in pack:
            h.pack_id = pack_id
            self.hosts.append(h)
            for s in h.services:
                self.services.append(s)
    
    
    def finish_load_and_relink(self, global_configuration):
        # Link our proxies items
        self.proxy_items_graph = proxyitemsgraph
        
        # Create ours classes
        self.hosts = Hosts(self.hosts)
        self.hosts.create_reversed_list()
        self.services = Services(self.services)
        self.services.create_reversed_list()
        # Fill host groups
        for ori_hg in global_configuration.hostgroups:
            hg = self.hostgroups.find_by_name(ori_hg.get_name())
            mbrs = ori_hg.members
            mbrs_id = []
            for host in mbrs:
                if host is not None:
                    mbrs_id.append(host.id)
            for host in self.hosts:
                if host.id in mbrs_id:
                    hg.members.append(host)
        
        # And also relink the hosts with the valid hostgroups
        for host in self.hosts:
            orig_hgs = host.hostgroups
            nhgs = []
            for ohg in orig_hgs:
                nhg = self.hostgroups.find_by_name(ohg.get_name())
                nhgs.append(nhg)
            host.hostgroups = nhgs
        
        # Fill servicegroup
        for ori_sg in global_configuration.servicegroups:
            sg = self.servicegroups.find_by_name(ori_sg.get_name())
            mbrs = ori_sg.members
            mbrs_id = []
            for s in mbrs:
                if s is not None:
                    mbrs_id.append(s.id)
            for s in self.services:
                if s.id in mbrs_id:
                    sg.members.append(s)
        
        # And also relink the services with the valid servicegroups
        for service in self.services:
            orig_hgs = service.servicegroups
            nhgs = []
            for ohg in orig_hgs:
                nhg = self.servicegroups.find_by_name(ohg.get_name())
                nhgs.append(nhg)
            service.servicegroups = nhgs
    
    
    def _get_host_names_as_list(self):
        return [host.get_name() for host in self.hosts]
    
    
    def parallel_sharded_configuration_serialization(self, to_master_process_return, rname, shard_id, receivers_inventory_filters):
        # Let this process ultra low level
        os.nice(20)
        _daemon_display_name = 'shinken-arbiter'
        full_name = '%s - worker - serializing shard %s [%d]' % (_daemon_display_name, rname, shard_id)
        set_process_name(full_name)
        
        logger.info('SHARDING OF %s' % type(self))
        
        # Remember to protect the local conf hostgroups too!
        self.hostgroups.prepare_for_sending()
        logger.info('[%s] Serializing the shard %d' % (rname, shard_id))
        t0 = time.time()
        serialized_shard = pickle.dumps(self, pickle.HIGHEST_PROTOCOL)
        
        # The arbiter also need the host names to redirect external commands to the valid shard
        host_names = self._get_host_names_as_list()
        
        # We need to have some data for the receivers & brokers so they can:
        # * receiver: take some of the nodes, and manage them
        # * broker: at boot create objects of exiting objects without waiting for schedulers and always
        #           have a one full packet of all realm elements, so not need to manage moving elements
        sharded_hosts_and_checks_receiver_inventory, sharded_hosts_and_checks_broker_inventory = self.get_sharded_hosts_and_checks_inventories(receivers_inventory_filters)
        
        # Put into to_master_process_return
        logger.info("[config] Shard %s (in realm %s) serialization took %.2fs and is %.3fMB" % (shard_id, rname, time.time() - t0, to_mb_size(len(serialized_shard))))
        try:
            to_master_process_return.append((rname, shard_id, serialized_shard, host_names, sharded_hosts_and_checks_receiver_inventory, sharded_hosts_and_checks_broker_inventory))
        except Exception as exp:  # noqa
            logger.error('[%s] [Worker=%s] The serialization of the shard %s did fail with the error: %s. Exiting.' % (rname, full_name, shard_id, exp))
            os._exit(2)  # noqa: foce kill all threads
    
    
    # We are asking this sharded configuration 2 inventories:
    # * one with all the hosts/checks for the brokers
    # * one with only the elements for the receivers (can be void if no receiver is interested)
    def get_sharded_hosts_and_checks_inventories(self, receiver_inventory_filters):
        receiver_sharded_inventory = ElementsInventory()
        # We only want elements if at least one receiver did ask for it currently
        # NOTE: when we will use inventory for broker, we will need to change this condition
        if len(receiver_inventory_filters) != 0:
            receiver_sharded_inventory.create_from_sharded_configuration(self, receiver_inventory_filters)
        
        # TODO: create only one inventory and use the broker to filter and get the receiver one
        broker_sharded_inventory = ElementsInventory()
        broker_sharded_inventory.create_from_sharded_configuration(self, inventory_filters=None)
        logger.info('DID CREATE %s / %s' % (receiver_sharded_inventory.get_len(), broker_sharded_inventory.get_len()))
        return receiver_sharded_inventory, broker_sharded_inventory
